/*
 * LXCF - LXC Facility
 * Copyright (C) 2013-2014 FUJITSU LIMITED
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */

/*
 * lxcf batch scheduler
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <sys/file.h>
#include <sys/wait.h>

// Control of multiplicity of job
#define MAXJOB 10

int MaxMultiJobs = 1;
pid_t JobPid[MAXJOB];

// Setting of queue
#define HQUEUE "/var/lib/lxcf/hqueue"
#define QQUEUE "/var/lib/lxcf/qqueue"
#define LQUEUE "/var/lib/lxcf/lqueue"
#define EXJOB "/var/lib/lxcf/exjob"

#define LXCF_LOG "/var/log/lxcf/lxcf-messages"
#define MAXBUF 4096

#define MULTIQUEUENUM "/etc/lxcf/multiqueue"

char *buf;
char jobbuf[MAXBUF];
char cmdbuf[MAXBUF + 16];

void usage(void);
void init(void);
void launch_job(void);
void read_multiplicity(void);
void write_multiplicity(void);
int job_cleanup(void);
void clear_exec_job(char *filename, pid_t pid);
char *get_job(void);
char *get_job_queue(char *filequeue);
char *date_time(void);
void fork_multi_job(char *cmd);
void cmd_exec(char *cmd);
void lxcf_ex(char *arg);
void set_exec_job(char *filequeue, pid_t pid, char *s);

// The multiplicity is preserved in the file.
void write_multiplicity(void)
{
	FILE *fd;
	int fh;

	fd = fopen(MULTIQUEUENUM, "w");
	if (fd == NULL) {
		fprintf(stderr, "error: lxcf-sched: can't open %s\n",
				MULTIQUEUENUM);
		exit(-1);
	}

	// lock QUEUE
	fh = fileno(fd);
	flock(fh, LOCK_EX);

	if (ftruncate(fh, 0) == -1) {
		fprintf(stderr, "error: lxcf-sched: ftrancate\n");
	}

	fprintf(fd, "%d", MaxMultiJobs);

	// unlock QUEUE
	flock(fh, LOCK_UN);
	fclose(fd);

	return;
}

// The multiplicity is read from the file.
// MaxMultiJobs is written in the file if there is no file.
void read_multiplicity(void)
{
	FILE *fd;
	int fh, multi;
	char buf[1024];

	fd = fopen(MULTIQUEUENUM, "r");
	if (fd == NULL) {
		write_multiplicity();
		fd = fopen(MULTIQUEUENUM, "r");
		if (fd == NULL) {
			fprintf(stderr, "error: lxcf-sched: can't open %s\n",
					MULTIQUEUENUM);
			exit(-1);
		}
	}

	// lock QUEUE
	fh = fileno(fd);
	flock(fh, LOCK_EX);

	buf[0] = 0;
	multi = MaxMultiJobs;
	if (fgets(buf, 1024 - 1, fd) == NULL) {
		write_multiplicity();
	} else {
		buf[1024 - 1] = 0;
		multi = (int) strtol(buf, NULL, 10);
		if ((multi < 0) || (multi > MAXJOB)) {
			multi = MaxMultiJobs;
		}
	}
	if (multi <= 0) {
		multi = 1;
	}
	MaxMultiJobs = multi;

	// unlock QUEUE
	flock(fh, LOCK_UN);
	fclose(fd);

	return;
}

// Preservation to file of job when being executing it
void set_exec_job(char *filequeue, pid_t pid, char *s)
{
	FILE *fdh;
	int fh;

	// get job from QUEUE
	fdh = fopen(filequeue, "a");
	if (fdh == NULL) {
		fprintf(stderr, "error: lxcf-sched: can't open %s\n",
				filequeue);
		exit(-1);
	}

	// lock QUEUE
	fh = fileno(fdh);
	flock(fh, LOCK_EX);

	fprintf(fdh, "%010d %s\n", pid, s);

	// unlock QUEUE
	flock(fh, LOCK_UN);
	fclose(fdh);

	return;
}

// Deletion from file of completion job
void clear_exec_job(char *filename, pid_t pid)
{
	char buf[4096];

	snprintf(buf, 4096 - 1,
			"/usr/bin/flock %s /bin/sed -i \"/^%010d/d\" %s",
			filename, pid, filename);
	buf[4096 - 1] = 0;
	if (system(buf) == -1) {
		printf("error: system function\n");
	}
}

// exec of a job
void lxcf_ex(char *arg)
{
	execlp("/bin/bash", "bash", "-c", arg, NULL);
	fprintf(stderr, "error: exec %s\n", arg);
	exit(-1);
}

void cmd_exec(char *cmd)
{
	lxcf_ex(cmd);
}

char *date_time(void)
{
	time_t timer;
	struct tm *timep;
	char *s;

	time(&timer);
	timep = localtime(&timer);
	s = asctime(timep);
	if (s[strlen(s) - 1] == '\n') {
		s[strlen(s) - 1] = 0;
	}
	return s;
}

// salvage completion jobs
int job_cleanup(void)
{
	pid_t p;
	int status, i, jobs;

	p = waitpid(0, &status, WNOHANG);
	for (i = jobs = 0; i < MAXJOB; i++) {
		if (JobPid[i] == 0) {
			continue;
		}
		jobs++;
		if (p != JobPid[i]) {
			continue;
		}
		if (WIFEXITED(status) || WIFSIGNALED(status)) {
			clear_exec_job(EXJOB, p);
			JobPid[i] = 0;
			jobs--;
		}
	}

	return jobs;
}

void fork_multi_job(char *cmd)
{
	int pid, i, jobs;

	// spawn job
	pid = fork();
	if (pid < 0) {
		printf("### fork ERROR ###\n");
	} else if (pid == 0) { // child proccess
		cmd_exec(cmdbuf);
		exit(0);
	} else { // parent process
		set_exec_job(EXJOB, pid, cmd);
		for (i = 0; i < MAXJOB; i++) {
			if (JobPid[i] == 0) {
				JobPid[i] = pid;
				break;
			}
		}
	}

	while (1) {
		// It loops until becoming empty JobPid.
		read_multiplicity();

		jobs = job_cleanup();

		// The multiplicity has been eased.
		if (jobs < MaxMultiJobs) {
			break;
		}
		sleep(1);
	}
}

char *get_job_queue(char *filequeue)
{
	FILE *fdh;
	int fh, maxqueue, c, i, j;

	// get job from QUEUE
	fdh = fopen(filequeue, "r");
	if (fdh == NULL) {
		fprintf(stderr, "error: lxcf-sched: can't open %s\n",
				filequeue);
		return NULL;
	}

	// lock QUEUE
	fh = fileno(fdh);
	flock(fh, LOCK_EX);

	// count jobfile lines
	maxqueue = 0;
	while (1) {
		c = fgetc(fdh);
		if (c == '\n') {
			maxqueue++;
		} else if (c == EOF) {
			break;
		}
	}
	rewind(fdh);

	// The buffer where the job is put is allocated.
	buf = (char *) malloc(maxqueue * MAXBUF);

	// get all QUEUE
	for (i = 0; i < maxqueue; i++) {
		if (fgets(&buf[i * MAXBUF], MAXBUF - 1, fdh) == NULL) {
			buf[maxqueue * MAXBUF - 1] = 0;
			break;
		}
		buf[(i + 1) * MAXBUF - 1] = 0;
	}

	// find job in QUEUE
	if (i != 0) {
		fdh = freopen(filequeue, "w", fdh);
		if (fdh == NULL) {
			fprintf(stderr, "error: lxcf-sched: "
					"can't re-open %s\n", filequeue);
			return NULL;
		}

		for (j = 1; j < i; j++) {
			fprintf(fdh, "%s", &buf[j * MAXBUF]);
		}

		// unlock QUEUE
		flock(fh, LOCK_UN);
		fclose(fdh);

		// The first job is put in jobbuf.
		strncpy(jobbuf, buf, MAXBUF - 1);
		jobbuf[strlen(jobbuf) - 1] = 0;

		// free buf
		free(buf);
		buf = NULL;

		return jobbuf;
	}

	// unlock QUEUE
	flock(fh, LOCK_UN);
	fclose(fdh);

	// free buf
	free(buf);

	// Because the job is not found, NULL is returned.
	return NULL;
}

char *get_job(void)
{
	char *s;

	s = get_job_queue(HQUEUE);
	if (s != NULL) {
		printf("### %s H-QUEUE: %s ###\n", date_time(), s);
		fflush(stdout), fflush(stderr);
		return s;
	}

	s = get_job_queue(QQUEUE);
	if (s != NULL) {
		printf("### %s Q-QUEUE: %s ###\n", date_time(), s);
		fflush(stdout), fflush(stderr);
		return s;
	}

	s = get_job_queue(LQUEUE);
	if (s != NULL) {
		printf("### %s L-QUEUE: %s ###\n", date_time(), s);
		fflush(stdout), fflush(stderr);
		return s;
	}

	return NULL;
}

void launch_job(void)
{
	char *s;
	int i;

	while (1) {
		// A broken container is found, and it erases it.
//		int r;
//		r = system("/usr/lib64/lxcf/lxcf-collection-broken-container");
//		if (r == -1) {
//			printf("error: system function\n");
//		}

		sleep(1);

		read_multiplicity();

		job_cleanup();

		// get a job from all queue
		s = get_job();
		if (s == NULL) {
			continue;
		}

		for (i = 0; i < strlen(s); i++) {
			if (s[i] == ' ') {
				snprintf(cmdbuf, MAXBUF + 16 - 1,
						"/usr/sbin/lxcf %s", &s[i]);
				cmdbuf[MAXBUF + 16 - 1] = 0;

				fork_multi_job(s);

				break;
			}
		}

		printf("### %s JOB END ###\n", date_time());
		fflush(stdout);
		fflush(stderr);
	}
}

void usage(void)
{
	fprintf(stderr, "usage: lxcf-sched\n");
	exit(-1);
}

void init(void)
{
	FILE *fd;
	pid_t i;

	// clear exjob
	fd = fopen(EXJOB, "w");
	if (fd != NULL) {
		fclose(fd);
	}

	// clear jobstatus
	for (i = 0; i < MAXJOB; i++) {
		JobPid[i] = 0;
	}
}

int main(int argc, char **argv)
{
	// euid check
	if (geteuid() != 0) {
		fprintf(stderr, "error: Because you are not root, "
				"you cannot execute this command.\n");
		exit(-1);
	}

	// args check
	if (argc != 1) {
		usage();
	}

	// reopen stdout
	if (freopen(LXCF_LOG, "a", stdout) == NULL) {
		fprintf(stderr, "error: can't open %s\n", LXCF_LOG);
		exit(-1);
	}

	// reopen stderr
	if (freopen(LXCF_LOG, "a", stderr) == NULL) {
		fprintf(stderr, "error: can't open %s\n", LXCF_LOG);
		exit(-1);
	}

	init();

	launch_job();

	// close stdout & stderr
	fclose(stdout);
	fclose(stderr);

	exit(0);
}

