/*
 *  m2m_listener.c:
 *     pthread(POSIX thread) routines.
 *
 *  Copyright (C) Taichi Nakamura <pdf30044@biglobe.ne.jp> - Feb 2000
 *
 *
 *  This file is part of m2m, a free MPEG2-Program-Stream player.
 *  It's a frontend of mpeg2dec.
 *    
 *  m2m is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  m2m is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 
 *
 */
#ifndef _LARGEFILE64_SOURCE
#define _LARGEFILE64_SOURCE
#endif
#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 199506L
#endif
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE  500
#endif

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <stdint.h>
#include <fcntl.h>
#include <errno.h>
#include "m2m.h"

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <pthread.h>
#include <mpeg2dec/mm_accel.h>
#include "video_out/video_out.h"
#include <mpeg2dec/mpeg2.h>


#define BUFFER_SIZE 2048

static int start_title=0;
static char *dvd_device=NULL;
static int is_stdin=0;
static int n_arg_files=0;
static FILE **arg_files=NULL;
static int pipe_in=-1;
static int pipe_out=-1;
static struct ip_mreq stMreq;
mpeg2dec_t mpeg2dec;
vo_instance_t * vo_instance=NULL;
char *vo_driver=NULL;
char *ao_driver=NULL;
int m2m_brightness =135;
int m2m_contrast =127;
int m2m_ad_sync =100;
int m2m_ad_late =500;
int m2m_ad_skip =500000;
int audio_no=1;
int audio_rate=0;
uint32_t commandline_params=0;
#ifdef HAVE_ESOUND
char *esd_host=NULL;
#endif

static inline void m2m_leave_caster(){
  setsockopt(pipe_in, IPPROTO_IP, IP_DROP_MEMBERSHIP, (char*) &stMreq, sizeof(stMreq));
  close(pipe_in);
  if ( pipe_in != pipe_out ) close(pipe_out);
  pipe_in=pipe_out=-1;
}

static void m2m_sighanhler( int signum )
{
fprintf( stderr,"catched signal %d\n",signum );
  mpeg2_close (&mpeg2dec);
  vo_close (vo_instance);
  m2m_leave_caster();
  kill( getpid(),SIGQUIT );
}

static inline int m2m_join2caster( char ip_addr[] , unsigned short port ) {
  struct sockaddr_in addr;
  int status;
  int len;
  char hello_word[2048];

  if ( start_title > 1 ) sprintf(hello_word,"HELLO title=%d",start_title );
  else strcpy(hello_word,"HELLO" );
  if ( dvd_device ){
    strcat(hello_word," dvd=");
    strcat(hello_word,dvd_device);
  }
  strcat(hello_word,"\r\n");

  if ( pipe_in != -1 ) m2m_leave_caster();

  pipe_in = socket(PF_INET, SOCK_STREAM , 0);
  if ( pipe_in == -1 ){ perror("socket"); return -1; }
  pipe_out=pipe_in;
  memset(&addr, 0, sizeof(addr));
  addr.sin_port = htons(port);
  addr.sin_family = AF_INET;
  status = inet_pton( AF_INET,ip_addr,&addr.sin_addr );
  if ( status == 0 ){
    struct hostent * entry=gethostbyname(ip_addr);
    if ( entry ){
      addr.sin_addr.s_addr =*((in_addr_t *)entry->h_addr_list[0]);
      //free(entry);
    }
  }
  status = connect( pipe_in, (struct sockaddr *)&addr, sizeof(addr));
  if ( status == -1 ) {
    int sockdesc[4];
    if ( pipe(sockdesc) ){ perror("pipe:1"); return -2; }
    if ( pipe(sockdesc+2) ){
      perror("pipe:2");
      close(sockdesc[0]);
      close(sockdesc[1]);
      return -3;
    }
    pipe_in=sockdesc[0];
    pipe_out=sockdesc[3];
    fcntl(sockdesc[1],F_SETFD,0);
    fcntl(sockdesc[2],F_SETFD,0);
    if ( (status=fork()) == 0 ){ // child
      static char arg1[32];
      static char arg2[32];
      static char *args[]={"m2caster","-",arg1,arg2,NULL};
      snprintf( arg1,sizeof(arg1),"%d",sockdesc[2] );
      snprintf( arg2,sizeof(arg2),"%d",sockdesc[1] );
      execvp("m2caster",args);
      perror("execvp");
      exit(-1);
    }else if ( status == -1 ){
      perror("fork:");
      close(sockdesc[0]);
      close(sockdesc[1]);
      close(sockdesc[3]);
      close(sockdesc[4]);
      return -4;
    }
  }
  if ( pipe_in == pipe_out ){
    status = send(pipe_out,hello_word,strlen(hello_word),0);
  }else{
    status = write(pipe_out,hello_word,strlen(hello_word));
  }
  if ( status < 1 ){ perror("send" ); return -5; }
  fcntl( pipe_in,F_SETFL,O_NONBLOCK );
  return 0;
}

int read_sector( uint8_t buf[] ){
    int ret=0;
    int status;
    FILE *fp=NULL;
    if ( n_arg_files ) {
        int file_idx=n_arg_files?0:-1;
        while( ret < 2048 ){
            status =  fread( buf+ret,1,2048-ret,arg_files[file_idx] );
            if ( status == 0 ) {
                if ( feof(arg_files[file_idx]) == 0 ){
                    perror("read" );
                    exit(1);
                }
                clearerr( arg_files[file_idx] );
                if ( ++file_idx >= n_arg_files ) exit(0);
                memset( buf,0xff,sizeof(buf) ); // END OF TITLE
                buf[0]=buf[1]=0x0; buf[2]=0x01; buf[3]=0x0f2;
                buf[4]=0x08; buf[5]=0x0;
            }
/*
            status = read( arg_files[file_idx],buf+ret, 2048-ret);
            if ( status == -1 ) {
                if ( errno == EAGAIN ){
                  sched_yield();
                  continue;
                }
                perror("RECV:" );
                exit(1);
            }
*/
            ret += status;
        }
    }else
    if( pipe_in == pipe_out ){
        while( ret < 2048 ){
            status = recv( pipe_in,buf+ret, 2048-ret,MSG_WAITALL );
            if ( status == -1 ) {
                if ( errno == EAGAIN ){
                  sched_yield();
                  continue;
                }
                perror("RECV:" );
                exit(1);
            }
            ret += status;
        }
    }else{
        while( ret < 2048 ){
            status = read( pipe_in,buf+ret, 2048-ret);
            if ( status == -1 ) {
                if ( errno == EAGAIN ){
                  sched_yield();
                  continue;
                }
                perror("RECV:" );
                exit(1);
            }
            ret += status;
        }
    }
    return ret;
}

static inline int m2m_init( int argc,char *argv[] )
{
    int i;
    uint32_t accel = mm_accel () | MM_ACCEL_MLIB;
    vo_driver_t * drivers = vo_drivers();

    printf(" mpeg2dec: (C) 1999 Aaron Holtzman <aholtzma@ess.engr.uvic.ca>\n");
    printf(" m2m : (C) 2000 Taichi Nakamura <pdf30044@biglobe.ne.jp>\n");

    vo_accel (accel);
    vo_instance = NULL;
    for ( i=0 ; vo_driver && drivers[i].name ; i++ )
    {
        if ( strcmp(vo_driver,drivers[i].name)==0 )
        {
            vo_instance = vo_open (drivers[i].open);
if ( vo_instance ) fprintf( stderr,"select video driver=%s\n",drivers[i].name);
            break;
        }
    }
    if ( vo_instance == NULL ) vo_instance = vo_open (drivers[0].open);
    if ( vo_instance == NULL ) return -1;
    mpeg2_init( &mpeg2dec , accel, vo_instance );
    for( i=0 ; i<100 ; i++ ) m2m_fill_buffer(0);
}

static inline int handle_args( int argc,char *argv[] )
{
  int i;
  int show_prop=0;
  int help=0;
  int is_stdin=0;
  uint16_t port=0;
  char * host=NULL;
  char * prop=NULL;

  arg_files=(FILE **)malloc( argc*sizeof(FILE *) );
  n_arg_files=0;

  for ( i=1 ; i< argc ; i++ )
  {
      if( argv[i][0] != '-' )
      {
          if( (arg_files[n_arg_files]=fopen(argv[i],"rb")) != NULL )
              n_arg_files++;
          continue;
      }
      if( argv[i][1] == 0x0 )
      {
          is_stdin=1;
          //fcntl( 0,F_SETFL,O_NONBLOCK );
      }else
      if( strncmp(argv[i],"--title=",8) == 0 && argv[i][8] != 0x0 )
      {
          start_title=atol(argv[i]+8);
      }else
      if( ( strcmp(argv[i],"-t") == 0 || strcmp(argv[i],"--title") == 0 )
        && i+1 < argc && argv[i+1][0] != '-' )
      {
          i++;
          start_title=atol(argv[i]);
      }else
      if( strncmp(argv[i],"--device=",9) == 0 && argv[i][9] != 0x0 )
      {
          dvd_device = argv[i]+9;
      }else
      if ( strcmp(argv[i],"--device") == 0 || strcmp(argv[i],"-d") == 0 )
      {
          dvd_device="/dev/dvd";
          if ( i+1 < argc && argv[i+1][0] != '-' )
          {
              i++;
              dvd_device=argv[i];
          }
      }else
      if( strncmp(argv[i],"--host=",9) == 0 && argv[i][9] != 0x0 )
      {
          host = argv[i]+9;
      }else
      if ( strcmp(argv[i],"--host") == 0 || strcmp(argv[i],"-h") == 0 
       &&  i+1 < argc && argv[i+1][0] != '-' )
      {
          i++;
          host=argv[i];
      }else
      if( strncmp(argv[i],"--port=",9) == 0 && argv[i][9] != 0x0 )
      {
          port = atol(argv[i]+9);
      }else
      if ( strcmp(argv[i],"--port") == 0 || strcmp(argv[i],"-p") == 0 
       &&  i+1 < argc && argv[i+1][0] != '-' )
      {
          i++;
          port=atol(argv[i]);
      }else
      if ( strcmp(argv[i],"--conf") == 0 && i+1 < argc && argv[i+1][0] != '-' )
      {
          i++;
          prop=argv[i];
      }else
      if ( strcmp(argv[i],"--showconf") == 0 )
      {
          show_prop=1;
      }else
      if ( strcmp(argv[i],"--help") == 0 || strcmp(argv[i],"-h") == 0 )
      {
          help=1;
      }
  }
  if (is_stdin )
  {
      arg_files[0]=stdin;
      n_arg_files=1;
  }
  m2m_load_properties( prop );
  if( host ) m2m_set_property( "CAST_IP",host );
  if( port ){
    char wk[8];
    snprintf(wk,8,"%d",port);
    m2m_set_property( "CAST_PORT",wk );
  }
  if ( show_prop ) m2m_write_properties( 1 );
  if ( help ) {
    printf( "Usage:%s - \n",argv[0] );
    printf( "Usage:%s files... \n",argv[0] );
    printf( "Usage:%s options... \n",argv[0] );
    printf( "options:\n" );
    printf( "  -d device                 : dvd device.\n" );
    printf( "  --device=devicename       : dvd device.\n" );
    printf( "  -t titleno                : start title no.\n" );
    printf( "  -title=titleno            : start title no.\n" );
    printf( "  -h caster_host            : connect caster.\n" );
    printf( "  --host=caster_host        : connect caster.\n" );
    printf( "  -p caster_port            : connect caster.\n" );
    printf( "  --port=caster_port        : connect caster.\n" );
    printf( "  --help                    : this message.\n" );
    exit(0);
  }
  return n_arg_files?1:0;
}

int main( int argc,char *argv[] ){
    if ( handle_args( argc,argv ) == 0
      && m2m_join2caster( m2m_get_property("CAST_IP") ,
                          atoi(m2m_get_property("CAST_PORT"))) )
    {
        exit(1);
    }
    signal( SIGINT,SIG_IGN);
    signal( SIGHUP,m2m_sighanhler);
    signal( SIGABRT,m2m_sighanhler);
    signal( SIGPIPE,m2m_sighanhler);
    signal( SIGTSTP,m2m_sighanhler);
    m2m_init(argc,argv);
    m2m_thread_control( M2M_AUDIO_THREAD_ID,1 );
    video_thread(NULL);
    m2m_leave_caster();
    mpeg2_close (&mpeg2dec);
    vo_close (vo_instance);
}




#define ARCH_X86
#ifdef ARCH_X86
static uint32_t x86_accel (void)
{
    uint32_t eax, ebx, ecx, edx;
    int AMD;
    uint32_t caps;

#define cpuid(op,eax,ebx,ecx,edx)	\
    asm ("cpuid"			\
	 : "=a" (eax),			\
	   "=b" (ebx),			\
	   "=c" (ecx),			\
	   "=d" (edx)			\
	 : "a" (op)			\
	 : "cc")

    asm ("pushfl\n\t"
	 "popl %0\n\t"
	 "movl %0,%1\n\t"
	 "xorl $0x200000,%0\n\t"
	 "pushl %0\n\t"
	 "popfl\n\t"
	 "pushfl\n\t"
	 "popl %0"
         : "=a" (eax),
	   "=b" (ebx)
	 :
	 : "cc");

    if (eax == ebx)		/* no cpuid */
	return 0;

    cpuid (0x00000000, eax, ebx, ecx, edx);
    if (!eax)			/* vendor string only */
	return 0;

    AMD = (ebx == 0x68747541) && (ecx == 0x444d4163) && (edx == 0x69746e65);

    cpuid (0x00000001, eax, ebx, ecx, edx);
    if (! (edx & 0x00800000))	/* no MMX */
	return 0;

    caps = MM_ACCEL_X86_MMX;
    if (edx & 0x02000000)	/* SSE - identical to AMD MMX extensions */
	caps = MM_ACCEL_X86_MMX | MM_ACCEL_X86_MMXEXT;

    cpuid (0x80000000, eax, ebx, ecx, edx);
    if (eax < 0x80000001)	/* no extended capabilities */
	return caps;

    cpuid (0x80000001, eax, ebx, ecx, edx);

    if (edx & 0x80000000)
	caps |= MM_ACCEL_X86_3DNOW;

    if (AMD && (edx & 0x00400000))	/* AMD MMX extensions */
	caps |= MM_ACCEL_X86_MMXEXT;

    return caps;
}
#endif

uint32_t mm_accel (void)
{
#ifdef ARCH_X86
    static int got_accel = 0;
    static uint32_t accel;

    if (!got_accel) {
	got_accel = 1;
	accel = x86_accel ();
    }

    return accel;
#else
    return 0;
#endif
}

