/*
 *  m2m_caster.c:
 *     pthread(POSIX thread) routines.
 *
 *  Copyright (C) Taichi Nakamura <pdf30044@biglobe.ne.jp> - Feb 2000
 *
 *
 *  This file is part of m2m, a free MPEG2-Program-Stream player.
 *  It's a frontend of mpeg2dec.
 *    
 *  m2m is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  m2m is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 
 *
 */
#ifndef _LARGEFILE64_SOURCE
#define _LARGEFILE64_SOURCE
#endif
#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 199506L
#endif
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE  500
#endif

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <stdint.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netinet/in.h>
#include "m2m.h"
#include "m2m_descripter.h"
#include "m2m_playlist.h"

static int m2m_read_file( struct m2m_desctipter_s *_desc,uint8_t buf[],int * size )
{
  m2m_filedesc_t *desc = (m2m_filedesc_t *)_desc ;
  int bytes_read ;
  if ( desc->pos < _desc->st || desc->pos > _desc->ed ) return 0;
  if ( (bytes_read = read( desc->fd , buf , *size )) > 0 ){
    desc->pos += bytes_read;
  }
  *size = bytes_read;
  return bytes_read;
}

static int m2m_seek_file( struct m2m_desctipter_s *_desc,off64_t pos )
{
  m2m_filedesc_t *desc = (m2m_filedesc_t *)_desc ;
  if ( pos < _desc->st || pos > _desc->ed ) return -1;
  lseek64( desc->fd , pos , SEEK_SET );
  desc->pos=pos;
  return 0;
}

static void m2m_close_file( struct m2m_desctipter_s *_desc )
{
    m2m_filedesc_t *desc = (m2m_filedesc_t *)_desc ;
    close ( desc->fd );
    free ( desc );
}

m2m_playlist_entry_t * m2m_get_playlist_entry_by_fn( const char *filename )
{
    struct stat statbuf;
    m2m_filedesc_t *desc;
    m2m_playlist_entry_t *ret =(m2m_playlist_entry_t*)
                                   malloc(sizeof(m2m_playlist_entry_t));
    int i,j;

    if ( ret == NULL ) return NULL;
    for( i=j=0; filename[i] != 0x0 ; i++ ){
      if ( filename[i] == '/' ) j=i+1;
    }
    ret->path=(char *)malloc( (strlen(filename)+1)*sizeof(char) );
    ret->name =(char *)malloc( (strlen(filename)+1-j)*sizeof(char) );
    desc=(m2m_filedesc_t *)malloc(sizeof(m2m_filedesc_t));
    if ( ret->name == NULL || ret->path == NULL || desc == NULL ){
      if ( ret->path ) free(ret->path);
      if ( ret->name ) free(ret->name);
      if ( desc ) free(desc);
      free(ret);
      return NULL;
    }
    desc->fd=open(filename,O_RDONLY);
    if ( desc->fd < 0 ){
       perror(" - file open:");
       printf( "filename[%s]\n",filename);
       free(ret->path);
       free(ret->name);
       free(desc);
       free(ret);
       return NULL;
    }
    strcpy(ret->path,filename);
    strcpy(ret->name,filename+j);

    fstat( desc->fd , &statbuf );
    desc->pos=0;
    desc->desc.read=m2m_read_file;
    desc->desc.seek=m2m_seek_file;
    desc->desc.close=m2m_close_file;
    desc->desc.eject=m2m_close_file;
    desc->desc.type=0;
    desc->desc.st=0;
    desc->desc.ed=statbuf.st_size;
    ret->descripter=(m2m_descripter_t *)desc;
    ret->next=NULL;
    ret->isES=1;
    ret->st=0;
    ret->ed=0;
    return ret;
}

int m2m_add_list( m2m_listener_handle_t *listener, char *in_next_name , char *in_path ){
  struct m2m_playlist_entry_s *add=NULL;
  struct m2m_playlist_entry_s *cur;
  char *wk;
  int i,j;

  for( i=0; in_path[i] != 0x0 && in_path[i] != '#' ; i++ );
  if ( in_path[i] == '#' ){
    char wkpath[1024];
    strncpy(wkpath,in_path,1024);
    wkpath[1023]=wkpath[i]=0x0;
    add=m2m_get_playlist_entry_by_chapter(wkpath,wkpath+(i+1) );
  }
  if ( add == NULL ){
    add= m2m_get_playlist_entry_by_fn( in_path );
  }

  if ( listener->playlist == NULL ) listener->playlist=add;
  else
  for( cur=listener->playlist ; cur ; cur=cur->next){
    if ( cur->next == NULL ){
      cur->next=add;
      break;
    }else
    if ( cur->next && in_next_name && strcmp(in_next_name,cur->next->name) == 0 ){
      add->next=cur->next;
      cur->next=add;
      break;
    }
  }

}

int m2m_remove_list( m2m_listener_handle_t *listener,char *in_name ){
  struct m2m_playlist_entry_s *cur=listener->playlist;
  struct m2m_playlist_entry_s *pre=NULL;
  for( ; cur != NULL && strcmp(in_name,cur->name) != 0 ; pre=cur,cur=cur->next);
  if( cur == NULL ) return -1;

  if ( pre ) pre->next=cur->next;
  else  listener->playlist=cur->next;

  free(cur->name);
  free(cur->path);
  free(cur);

  return 0;
}

m2m_playlist_entry_t * m2m_add_playlist_from_properties(m2m_listener_handle_t *listener )
{
  char name[256];
  int i;
  for ( i=0 ; ; i++ ){
    char *value;
    sprintf( name,"PLAYLIST.%03d",i+1 );
    value=m2m_get_property(name);
    if ( value == NULL ) break;
    m2m_add_list( listener, NULL , value );
  }
  return listener->playlist;
}

m2m_playlist_entry_t * m2m_add_playlist_from_ifo( m2m_listener_handle_t *listener, char *devicename )
{
  extern m2m_playlist_entry_t * m2m_get_playlist_by_ifo( char *);
  m2m_playlist_entry_t *cur;
  m2m_playlist_entry_t *add= m2m_get_playlist_by_ifo( devicename );
  if ( listener->playlist == NULL ) listener->playlist=add;
  else{
    for( cur=listener->playlist ; cur->next ; cur=cur->next) ;
    cur->next=add;
  }
  return listener->playlist;
}

off64_t m2m_get_playlist_length( m2m_listener_handle_t *listener ){
  off64_t ret=0;
  struct m2m_playlist_entry_s *cur;

  for( cur=listener->playlist ; cur ; cur=cur->next ){
    cur->st = ret;
    cur->ed = ret = cur->st + (cur->descripter->ed - cur->descripter->st);
  }
  return ret;
}

