import java.awt.BorderLayout;
import java.awt.Polygon;
import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Properties;

import javax.swing.JFrame;
import javax.swing.UIManager;

import map.IsjFactory;
import map.KsjFactory;
import map.data.CityMap;
import map.map25000.Map25000Factory;
import map.map25000.Map25000Storage;
import map.map25000.cell.CellSearch;
import map.store.Store;
import util.Log;
import util.PluginLoader;
import view.MapPanel;
import view.StatusBar;
import controller.Controller;

/**
 * 地図描画アプリケーション起動のためのクラス
 * 
 * @author Masayasu Fujiwara
 */
public class Main {

	/**
	 * フレーム高さ
	 */
	private static final int F_WIDTH = 800;

	/**
	 * フレーム幅
	 */
	private static final int F_HEIGHT = 600;

	/**
	 * args[0]には地図データを格納するディレクトリのルートディレクトリの位置を指定できます。
	 * 引数によって指定されない場合は、"./"（同一ディレクトリ）をルートディレクトリとしてします。
	 * 
	 * @param args 引数
	 */
	public static void main(String[] args) {
		if (args.length == 0) {
			args = new String[] { ".digital_map" + File.separatorChar, "plugin" + File.separatorChar};
		} else if (args.length == 1){
			args = new String[] { args[0], "plugin" + File.separatorChar};			
		}

		// 都道府県データの相対位置リスト
		final String list = "/data/prefecture.csv";

		// 都道府県界のデータ
		final String prefecture = "/data/prefecture.dat";

		// 都道府県のバウンディングボックスのデータリスト
		final String cell = "/data/prefecture.cell";
		
		try {
			Properties properties = System.getProperties();
			if (((String) properties.get("java.version")).toLowerCase().contains("1.6")) {
				if (((String) properties.get("sun.desktop")).toLowerCase().contains("windows")){
					properties.setProperty("sun.java2d.d3d", "true");
					Log.out(Main.class, "setProperty sun.java2d.d3d=true");
				}
			}
//			properties.list(System.out);
		} catch (Exception e) {
			Log.err(Main.class, e);
		}

		new Main(args[0], list, prefecture, cell, args[1]);
	}

	/**
	 * アプリケーション起動のための初期設定
	 * 
	 * @param dir 地図データディレクトリのルート
	 * @param list
	 * @param prefData
	 * @param cellData 
	 * @param pluginDir プラグインディレクトリ
	 */
	public Main(String dir, String list, String prefData, String cellData, String pluginDir) {

		final JFrame frame = new JFrame("Digital Map");
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setLayout(new BorderLayout());
		
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception e) {
			Log.err(this, e);
		}

		final MapPanel panel = new MapPanel();
		Log.out(this, "created panel.");

		final StatusBar statusbar = new StatusBar(" ");
		Log.out(this, "created statusbar");

		
		// Listener
		final Controller listener = new Controller(panel);
		Log.out(this, "created listener.");

		frame.add(statusbar, BorderLayout.SOUTH);
		frame.add(panel, BorderLayout.CENTER);
		frame.add(listener.getMenuBar(), BorderLayout.NORTH);

		frame.setSize(Main.F_WIDTH, Main.F_HEIGHT);
		
		frame.setVisible(true);
		Log.out(this, "frame set visible.");

		
		try {
			statusbar.setMessage("初期設定");

			final Collection<Store> shops = PluginLoader.loadStorePlugin(pluginDir, dir + "store" + File.separatorChar);
			Log.out(this, "loaded plugins.");

			
			final CityMap maps = new CityMap(panel, new CellSearch(cellData),
					new Map25000Factory(new Map25000Storage(dir + "data25k" + File.separatorChar, list, statusbar)),
					new IsjFactory(dir + "isj" + File.separatorChar), shops, statusbar);
			statusbar.clearReading();
			Log.out(this, "created maps.");

			statusbar.setMessage("国土数値情報 都道府県");
			final Polygon[] prefecture = KsjFactory.readPolygon(prefData);
			// int[] coast = new int[]{82, 83, 84, 143, 288, 457, 505, 582, 725, 878, 941, 967, 1007, 1008, 1032, 1592, 1593, 1595, 1694, 1828, 1909, 1910, 1911, 1912, 1913, 1914, 1915, 2073, 2253, 2345, 2641, 3335, 3466, 3503, 3595, 3705, 4191, 4418, 4531, 4600, 5193};
			Log.out(this, "created prefecture.");

			// 湖などを繰りぬく
			final int[] coast = new int[] { 2345 };
			final Polygon[] prefectureCoast = new Polygon[coast.length];
			int count = 0;
			for (final int i : coast) {
				prefectureCoast[count++] = prefecture[i];
			}
			statusbar.clearReading();

			panel.init(maps, prefecture, prefectureCoast);
			maps.start();
			Log.out(this, "started maps.");

		} catch (IOException e) {
			statusbar.setMessage(e.getMessage());
			Log.err(this, e);
		}
		statusbar.clearReading();

		frame.addKeyListener(listener);
		panel.addKeyListener(listener);
		panel.addMouseListener(listener);
		panel.addMouseMotionListener(listener);
		panel.addMouseWheelListener(listener);

		statusbar.setThreadPriority(Thread.MIN_PRIORITY);
		
		Log.out(this, "finished main().");
	}
}
