package controller;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;

import javax.print.PrintException;
import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.KeyStroke;

import output.Output;


import util.Log;
import view.MapPanel;
import map.data.Road;

/**
 * マウス、キーボードによる操作クラス
 * 
 * @author ma38su
 */
public class Controller implements MouseListener, MouseMotionListener,
		MouseWheelListener, KeyListener, ActionListener {

	/**
	 * キーボードによる平行移動感度
	 */
	private static final int MOVE_SENSE = 8;

	/**
	 * 地図表示パネル
	 */
	private final MapPanel panel;

	/**
	 * メニューバー
	 */
	private final JMenuBar menu;

	private int x;

	private int y;

	private int dx;

	private int dy;

	/**
	 * マウスがWindow内に入っているかどうか
	 */
	private boolean isMouseEntered;

	/**
	 * 
	 * @param panel
	 *            地図表示パネル
	 */
	public Controller(final MapPanel panel) {
		this.panel = panel;
		this.x = panel.getWidth() / 2;
		this.y = panel.getHeight() / 2;

		this.menu = new JMenuBar();

		final JMenu menu0 = new JMenu("ファイル(F)");
		menu0.setMnemonic(KeyEvent.VK_F);

		final JMenu menu1 = new JMenu("探索設定(S)");
		menu1.setMnemonic(KeyEvent.VK_S);

		final JMenu menu2 = new JMenu("表示設定(V)");
		menu2.setMnemonic(KeyEvent.VK_V);

		final JMenu helpMenu = new JMenu("ヘルプ(H)");
		helpMenu.setMnemonic(KeyEvent.VK_H);

		this.menu.add(menu0);
		this.menu.add(menu1);
		this.menu.add(menu2);
		this.menu.add(helpMenu);

		final JMenuItem item0_1 = new JMenuItem("印刷");
		item0_1.setActionCommand("print");
		item0_1.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P,
				InputEvent.CTRL_DOWN_MASK));

		final JMenuItem item0_2 = new JMenuItem("エクスポート");
		item0_2.setActionCommand("export");
		item0_2.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S,
				InputEvent.CTRL_DOWN_MASK));

		final JMenuItem item0_3 = new JMenuItem("終了");
		item0_3.setActionCommand("exit");
		item0_3.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X,
				InputEvent.CTRL_DOWN_MASK));

		menu0.add(item0_1);
		menu0.add(item0_2);
		menu0.addSeparator();
		menu0.add(item0_3);

		final JCheckBoxMenuItem item1_1 = new JCheckBoxMenuItem("Dijkstra",
				false);
		item1_1.setActionCommand("path_dijkstra");
		final JCheckBoxMenuItem item1_2 = new JCheckBoxMenuItem("A*", true);
		item1_2.setActionCommand("path_a*");

		final JCheckBoxMenuItem item1_3 = new JCheckBoxMenuItem("高速道路", true);
		item1_3.setActionCommand("highway");

		ButtonGroup group = new ButtonGroup();
		group.add(item1_1);
		group.add(item1_2);

		menu1.add(item1_1);
		menu1.add(item1_2);
		menu1.addSeparator();
		menu1.add(item1_3);

		final JMenu item2_1 = new JMenu("ラベル表示");

		final JCheckBoxMenuItem item2_1_1 = new JCheckBoxMenuItem("駅", true);
		item2_1_1.setActionCommand("labelStation");
		final JCheckBoxMenuItem item2_1_2 = new JCheckBoxMenuItem("地名", true);
		item2_1_2.setActionCommand("labelCity");
		final JCheckBoxMenuItem item2_1_3 = new JCheckBoxMenuItem("施設", true);
		item2_1_3.setActionCommand("labelFacility");

		item2_1.add(item2_1_1);
		item2_1.add(item2_1_2);
		item2_1.add(item2_1_3);

		final JCheckBoxMenuItem item2_2 = new JCheckBoxMenuItem("頂点表示", false);
		item2_2.setActionCommand("node");
		final JCheckBoxMenuItem item2_3 = new JCheckBoxMenuItem("アンチエイリアス",
				true);
		item2_3.setActionCommand("alias");
		final JCheckBoxMenuItem item2_4 = new JCheckBoxMenuItem("水域区間の塗りつぶしβ",
				false);
		item2_4.setActionCommand("fill");

		menu2.add(item2_1);
		menu2.add(item2_2);
		menu2.add(item2_3);
		menu2.add(item2_4);

		item0_1.addActionListener(this);
		item0_2.addActionListener(this);
		item0_3.addActionListener(this);

		item1_1.addActionListener(this);
		item1_2.addActionListener(this);
		item1_3.addActionListener(this);

		item2_1_1.addActionListener(this);
		item2_1_2.addActionListener(this);
		item2_1_3.addActionListener(this);

		item2_2.addActionListener(this);
		item2_3.addActionListener(this);
		item2_4.addActionListener(this);

		JCheckBoxMenuItem help01 = new JCheckBoxMenuItem("操作マニュアル", true);
		help01.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F1,
				InputEvent.CTRL_DOWN_MASK));
		help01.setActionCommand("help");

		helpMenu.add(help01);

		help01.addActionListener(this);
	}

	public JMenuBar getMenuBar() {
		return this.menu;
	}

	public void mouseClicked(MouseEvent e) {
		if (e.getButton() == MouseEvent.BUTTON3) {
			this.panel.searchBoundary(e.getX(), e.getY(),
					(e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) != 0);
		}
	}

	public void mouseEntered(MouseEvent e) {
		this.isMouseEntered = true;
	}

	public void mouseExited(MouseEvent e) {
		this.isMouseEntered = false;
		if (!this.panel.isOperation()) {
			this.x = this.panel.getWidth() / 2;
			this.y = this.panel.getHeight() / 2;
		}
	}

	public void mousePressed(MouseEvent e) {
		this.panel.setOperation(true);
	}

	public void mouseReleased(MouseEvent e) {
		if (!this.isMouseEntered
				&& (e.getModifiersEx() & (InputEvent.BUTTON1_DOWN_MASK
						| InputEvent.BUTTON2_DOWN_MASK | InputEvent.BUTTON3_DOWN_MASK)) == 0) {
			this.x = this.panel.getWidth() / 2;
			this.y = this.panel.getHeight() / 2;
		}
		this.panel.setOperation(false);
	}

	public void mouseDragged(MouseEvent e) {
		// ポインター座標変化量を計算
		this.dx = e.getX() - this.x;
		this.dy = e.getY() - this.y;
		// 変化時のポインター座標
		this.x = e.getX();
		this.y = e.getY();
		// 平行移動
		this.panel.moveLocation(this.dx, this.dy);
		this.panel.setOperation(true);
	}

	public void mouseMoved(MouseEvent e) {
		if (this.isMouseEntered) {
			this.x = e.getX();
			this.y = e.getY();
		}
	}

	public void mouseWheelMoved(MouseWheelEvent e) {
		int d = e.getWheelRotation();
		this.panel.zoom(e.getX(), e.getY(), d);
	}

	public void keyTyped(KeyEvent e) {
	}

	public void keyPressed(KeyEvent e) {
		this.panel.setOperation(true);
		switch (e.getKeyCode()) {
		case KeyEvent.VK_RIGHT:
		case KeyEvent.VK_KP_RIGHT:
			this.panel.moveLocation(-Controller.MOVE_SENSE, 0);
			break;
		case KeyEvent.VK_LEFT:
		case KeyEvent.VK_KP_LEFT:
			this.panel.moveLocation(Controller.MOVE_SENSE, 0);
			break;
		case KeyEvent.VK_UP:
		case KeyEvent.VK_KP_UP:
			this.panel.moveLocation(0, Controller.MOVE_SENSE);
			break;
		case KeyEvent.VK_DOWN:
		case KeyEvent.VK_KP_DOWN:
			this.panel.moveLocation(0, -Controller.MOVE_SENSE);
			break;
		case KeyEvent.VK_PLUS:
		case KeyEvent.VK_SEMICOLON:
		case KeyEvent.VK_PAGE_UP:
			this.panel.zoom(this.x, this.y, 1);
			break;
		case KeyEvent.VK_MINUS:
		case KeyEvent.VK_PAGE_DOWN:
		case KeyEvent.VK_EQUALS:
			this.panel.zoom(this.x, this.y, -1);
			break;
		case KeyEvent.VK_ASTERISK:
		case KeyEvent.VK_COLON:
			this.panel.moveToNishiwaki();
			break;
		}
	}

	public void keyReleased(KeyEvent e) {
		this.panel.setOperation(false);
	}

	public void actionPerformed(ActionEvent e) {
		String command = e.getActionCommand();
		if (command.equals("exit")) {
			// 即終了しても大丈夫だろうか・・・
			System.exit(0);
		} else if (command.equals("export")) {
			try {
				while (!this.panel.isLoaded()) {
						Thread.sleep(2L);
				}
				Output.export(this.panel);
			} catch (InterruptedException ex) {
				Log.err(this, ex);
			}
		} else if (command.equals("print")) {
			try {
				while (!this.panel.isLoaded()) {
						Thread.sleep(1L);
				}
				Output.print(this.panel);
			} catch (InterruptedException ex) {
				Log.err(this, ex);
			} catch (PrintException ex) {
				Log.err(this, ex);
			}
		} else if (command.startsWith("path_")) {
			this.panel.switchShortestPathAlgorithm(command);
		} else if (command.startsWith("label")) {
			if (command.endsWith("Station")) {
				this.panel.switchLabel(1);
			} else if (command.endsWith("City")) {
				this.panel.switchLabel(2);
			} else if (command.endsWith("Facility")) {
				this.panel.switchLabel(4);
			}
		} else if (command.equals("node")) {
			this.panel.switchNodeView();
		} else if (command.equals("alias")) {
			this.panel.switchRendering();
		} else if (command.equals("fill")) {
			this.panel.switchFill();
		} else if (command.equals("highway")) {
			Road.switchHighway();
			this.panel.reroute();
		} else if (command.equals("help")) {
			this.panel.switchHelp();
		}
	}
}
