package map.map25000;

import java.awt.Rectangle;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.StringTokenizer;
import java.util.regex.Pattern;

import util.Log;
import view.StatusBar;

import map.data.City;

import jp.gr.java_conf.dangan.util.lha.LhaHeader;
import jp.gr.java_conf.dangan.util.lha.LhaInputStream;

/**
 * 数値地図25000のデータ管理
 * @author Masayasu Fujiwara
 *
 */
public class Map25000Storage {

	/**
	 * 文字列の分割
	 */
	private static final Pattern CSV_SPLITER = Pattern.compile(",");

	public static void main(String[] args) throws IOException {
		Map25000Storage storage = new Map25000Storage("./aaa/", "./data/prefecture.csv", null);
		storage.getStreaming(28102);
	}

	/**
	 * 保存フォルダ
	 */
	private final String CACHE_DIR;

	/**
	 * 国土数値情報のベースとなるURL
	 */
	private final String MAP25K_URL = "http://sdf.gsi.go.jp/data25k/";

	/**
	 * 都道府県のURL補助
	 */
	private final String[] prefecture;
	
	/**
	 * ステータスバー
	 */
	private final StatusBar status;

	/**
	 * 直列化したファイルの保存先ディレクトリ
	 */
	private String SERIALIZE_DIR;

	/**
	 * コンストラクタ
	 * @param cacheDir データ格納ディレクトリ
	 * @param list リスト
	 * @param status ステータスバー
	 * @throws IOException 入出力エラー
	 */
	public Map25000Storage(String cacheDir, String list, StatusBar status) throws IOException {
		this.CACHE_DIR = cacheDir;
		this.SERIALIZE_DIR = cacheDir + "serialize" + File.separatorChar;

		File dir = new File(cacheDir);
		File serializeDir = new File(this.SERIALIZE_DIR);
		
		Log.out(this, "init Cache Directory "+ dir.getCanonicalPath());
		Log.out(this, "init Cache Serialize Directory "+ serializeDir.getCanonicalPath());
		if(!dir.isDirectory()) {
			dir.mkdirs();
		}

		// 都道府県番号（都道府県数47）
		this.prefecture = new String[48];
		this.status = status;

		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(Map25000Storage.class.getResourceAsStream(list), "SJIS"));
			int i = 1;
			while(bi.ready()) {
				String line = bi.readLine();
				this.prefecture[i++] = Map25000Storage.CSV_SPLITER.split(line)[1];
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
	}
	
	/**
	 * ファイルのコピーを行います。
	 * 入出力のストリームは閉じないので注意が必要です。
	 * 
	 * @param in 入力ストリーム
	 * @param out 出力ストリーム
	 * @throws IOException 入出力エラー
	 */
	private void copy(InputStream in, OutputStream out) throws IOException {
		final byte buf[] = new byte[1024];
		int size;
		while ((size = in.read(buf)) != -1) {
			out.write(buf, 0, size);
			out.flush();
		}
	}

	/**
	 * ファイルをダウンロードします。
	 * @param code 市区町村番号
	 * @param file ダウンロード先のファイル
	 * @throws IOException 入出力エラー
	 */
	private void download(int code, File file) throws IOException {
		URL url = new URL(this.MAP25K_URL + this.prefecture[code / 1000] + City.cityFormat(code) + ".lzh");
		InputStream in = null;
		OutputStream out = null;
		try {
			URLConnection connect = url.openConnection();
			// ファイルのチェック（ファイルサイズの確認）
			int contentLength = connect.getContentLength();
			if (contentLength != file.length()) {
				if(this.status != null) {
					this.status.setDLFile(code, contentLength, file);
				}
				final long start = System.currentTimeMillis();
				// ダウンロード
				in = connect.getInputStream();
				out = new FileOutputStream(file);
				this.copy(in, out);
				Log.out(this, "download "+ url + " / " + (System.currentTimeMillis() - start) + "ms");
			}
		} finally {
			if(in != null) {
				in.close();
			}
			if(out != null) {
				out.close();
			}
		}
	}

	/**
	 * 圧縮ファイルを展開します。
	 * @param file 展開するファイル
	 * @return 展開したファイル配列
	 * @throws IOException 入出力エラー
	 */
	private File[] extract(File file) throws IOException {
		final long start = System.currentTimeMillis();

		boolean isExtracted = false;

		LhaInputStream in = null;
		FileOutputStream out = null;
		Collection<File> extracted = new ArrayList<File>();
		try {
			in = new LhaInputStream(new FileInputStream(file));
			LhaHeader entry;
			while ((entry = in.getNextEntry()) != null) {
				// 出力先ファイル
				File outFile;
				// 13421のときは例外
				if(entry.getPath().startsWith("1342")) {
					outFile = new File(this.CACHE_DIR + "13421/" + entry.getPath());
					final File dirParent = new File(this.CACHE_DIR + "13421/" + entry.getPath()).getParentFile();
					if(!dirParent.isDirectory()) {
						dirParent.mkdir();
						Log.out(this, "mkdirs " + dirParent);
					}
				} else {
					outFile = new File(this.CACHE_DIR + entry.getPath());
				}
				// 出力先ファイル
				if (!outFile.exists() || entry.getOriginalSize() != outFile.length()) {
					out = new FileOutputStream(outFile);
					this.copy(in, out);
					isExtracted = true;
				}
				extracted.add(outFile);
			}
		} finally {
			if (in != null) {
				in.close();
			}
			if (out != null) {
				out.close();
			}
		}
		final long end = System.currentTimeMillis();
		if (isExtracted) {
			Log.out(this, "extract " + file + " / " + (end - start) + "ms");
		}
		return extracted.toArray(new File[]{});
	}
	
	/**
	 * 市区町村番号に対応するファイル配列を取得します。
	 * @param code 市区町村番号
	 * @return 市区町村番号に対応したファイル配列
	 * @throws IOException 入出力エラー
	 */
	public synchronized File[] get(int code) throws IOException {
		String stringCode = City.cityFormat(code);
		
		File[] file = null;
		// 圧縮ファイル
		final File lzh = new File(this.CACHE_DIR + stringCode + ".lzh");
		// 保存先ディレクトリ
		final File dir = new File(this.CACHE_DIR + stringCode + File.separatorChar);

		if (lzh.exists() || !dir.isDirectory() || dir.list().length == 0) {
			dir.mkdir();
			this.download(code, lzh);
		}
		if(lzh.exists()) {
			// ファイルの展開
			file = this.extract(lzh);
			if(lzh.delete()){
				Log.out(this, "delete "+ lzh.getCanonicalPath());
			}
		} else if(dir.isDirectory()) {
			if(code == 13421) {
				Collection<File> subFileList = new ArrayList<File>();
				for(File subdir : dir.listFiles()) {
					for(File subfile : subdir.listFiles()) {
						subFileList.add(subfile);
					}
				}
				file = subFileList.toArray(new File[]{});
			}else{
				file = dir.listFiles();
			}
		}
		
		if(file == null || file.length == 0) {
			throw new IOException();
		}

		return file;
	}
	
	/**
	 * 頂点の外部への接続情報の取得
	 * @param code 市町村番号
	 * @return 市町村番号に対応する頂点の外部への接続情報
	 */
	public InputStream getBoundaryNode(int code) {
		String codeStr = City.cityFormat(code);
		return this.getClass().getResourceAsStream("/data/" + codeStr.substring(0, 2) + "/" + codeStr +".nod");
	}

	/**
	 * 市区町村番号に対応するファイル配列をストリーミングで取得します。
	 * @param code 市区町村番号
	 * @return 市区町村番号に対応したファイル配列
	 * @throws IOException 入出力エラー
	 */
	public synchronized File[] getStreaming(int code) throws IOException {

		// 保存先ディレクトリ
		final File dir = new File(this.CACHE_DIR + City.cityFormat(code));
		
		List<File> list = null;
		if (!dir.isDirectory() || dir.list().length == 0) {
			dir.mkdirs();
			list = new ArrayList<File>();
			
			if (this.prefecture[code / 1000] == null) {
				throw new IOException(City.cityFormat(code) + " is not found.");
			}

			URL url = new URL(this.MAP25K_URL + this.prefecture[code / 1000] + City.cityFormat(code) + ".lzh");
			if (this.status != null) {
				this.status.setDLFile(code);
			}
			LhaInputStream in = null;
			try {
				in = new LhaInputStream(url.openStream());
				LhaHeader header;
				while ((header = in.getNextEntry()) != null) {
					File file = new File(this.CACHE_DIR + header.getPath());
					if (!file.getParentFile().isDirectory()) {
						file.getParentFile().mkdirs();
					}
					list.add(file);
					Log.out(this, "get " + file.getCanonicalPath());
					OutputStream out = null;
					try {
						out = new FileOutputStream(file);
						this.copy(in, out);
					} finally {
						if (out != null) {
							out.close();
						}
					}
				}
			} catch (IOException e) {
				Log.err(this, e);
				if (list != null) {
					for (File file : dir.listFiles()) {
						if (file.exists()) {
							file.delete();
						}
					}
					dir.delete();
				}
			} finally {
				if (in != null) {
					in.close();
				}
				if (this.status != null) {
					this.status.clearDLFile();
				}
			}
			return list.toArray(new File[]{});
		} else {
			return dir.listFiles();
		}
	}

	/**
	 * 直列化されたメッシュ標高が存在するか確認します。
	 * @param code 市区町村番号
	 * @param label ラベル
	 * @return 直列化されたメッシュ標高が存在すればtrue
	 */
	public boolean hasSerializable(int code, String label) {
		File file = new File(this.SERIALIZE_DIR + City.cityFormat(code) + File.separatorChar + label +".dat");
		return file.exists();
	}
	
	/**
	 * 市区町村番号に対応するファイルが取得済みであるかどうかを確認します。
	 * @param code 市区町村番号
	 * @return 取得済みであればtrue
	 */
	public boolean isAvailable(String code) {
		final File lzh = new File(this.CACHE_DIR + code + ".lzh");
		final File dir = new File(this.CACHE_DIR + code);
		return (!lzh.exists() && dir.isDirectory() && dir.list().length != 0);
	}

	/**
	 * 直列化されたメッシュ標高を読み込みます。
	 * @param code 市区町村番号
	 * @param label ラベル
	 * @return メッシュ標高
	 */
	public Object readSerializable(int code, String label) {
		synchronized (label) {
			Log.out(this, "read Serialize "+ label);
			Object obj = null;
			File file = new File(this.SERIALIZE_DIR + City.cityFormat(code) + File.separatorChar + label +".dat");
			try {
				ObjectInputStream in = null;
				try {
					in = new ObjectInputStream(new FileInputStream(file));
					obj = in.readObject();
				} finally {
					if (in != null) {
						in.close();
					}
				}
			} catch (Exception e) {
				obj = null;
				Log.err(this, e);
				if (file.exists()) {
					file.delete();
				}
			}
			return obj;
		}
	}

	/**
	 * 市区町村の範囲をLZHファイルからストリーミングで取得します。
	 * @param code 市区町村番号
	 * @return 市区町村の範囲
	 * @throws IOException 入出力エラー
	 */
	public Rectangle readSLMStreaming(int code) throws IOException {
		Rectangle rect = null;
		URL url = new URL(this.MAP25K_URL + this.prefecture[code / 1000] + City.cityFormat(code) + ".lzh");
		LhaInputStream in = null;
		BufferedReader bi = null;
		try {
			in = new LhaInputStream(url.openStream());
			bi = new BufferedReader(new InputStreamReader(in));
			LhaHeader header;
			while ((header = in.getNextEntry()) != null) {
				File entry = new File(header.getPath());
				if (!entry.isDirectory() && entry.getName().endsWith(".slm")) {
					StringTokenizer st = new StringTokenizer(bi.readLine(), ",");
					int x = (int) (Long.parseLong(st.nextToken()) / 10L);
					int y = (int) (Long.parseLong(st.nextToken()) / 10L);

					// 領域読み込み
					st = new StringTokenizer(bi.readLine(), ",");
					int width = Integer.parseInt(st.nextToken()) / 10;
					int height = Integer.parseInt(st.nextToken()) / 10;
					rect = new Rectangle(x, y, width, height);
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			} else if (in != null) {
				in.close();
			}
		}
		return rect;
	}

	/**
	 * オブジェクトを直列化して保存します。
	 * @param code 市区町村番号
	 * @param label ラベル
	 * @param obj 直列化可能なオブジェクト
	 */
	public void writeSerializable(int code, String label, Object obj) {
		synchronized (label) {
			Log.out(this, "save "+ code +" Serialize "+ label);
			File file = new File(this.SERIALIZE_DIR + City.cityFormat(code) + File.separatorChar + label + ".dat");
			if (!file.getParentFile().isDirectory()) {
				file.getParentFile().mkdirs();
			}
			try {
				ObjectOutputStream out = null;
				try {
					out = new ObjectOutputStream(new FileOutputStream(file));
					out.writeObject(obj);
					out.flush();
				} finally {
					if (out != null) {
						out.close();
					}
				}
			} catch (Exception e) {
				Log.err(this, e);
				if (file.exists()) {
					file.delete();
				}
			}
		}
	}
}
