package map.map25000.cell;

import java.awt.Rectangle;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * セルによる領域検索
 * @author Masayasu Fujiwara
 */
public class CellSearch {

	/**
	 * 都道府県セル
	 */
	private final Prefecture[] prefecture;
	
	/**
	 * コンストラクタ
	 * @param file
	 * @throws IOException
	 */
	public CellSearch(String file) throws IOException {
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(CellSearch.class.getResourceAsStream(file), "SJIS"));
			this.prefecture = new Prefecture[47];
			for (int i = 0; i < 47; i++) {
				String line = bi.readLine();
				this.prefecture[i] = new Prefecture(i + 1, line.split(","));
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
	}

	/**
	 * 都道府県番号を取得する
	 * @param rectangle 検索範囲
	 * @return 検索範囲と重なる都道府県の番号
	 */
	public Set<Integer> searchPrefectureCode(Rectangle rectangle) {
		Set<Integer> codes = new HashSet<Integer>();
		for(Prefecture cell : this.prefecture) {
			if(rectangle.intersects(cell)) {
				codes.add(cell.getCode());
			}
		}
		return codes;
	}

	/**
	 * 都道府県番号と市区町村番号を取得する
	 * @param rectangle 検索範囲
	 * @return 検索範囲内の都道府県番号と市区町村番号
	 * @throws IOException 入出力エラー
	 */
	public Map<Integer, Set<Integer>> search(Rectangle rectangle) throws IOException {
		Map<Integer, Set<Integer>> code = new HashMap<Integer, Set<Integer>>();
		for(Prefecture prefecture : this.prefecture) {
			if(rectangle.intersects(prefecture)) {
				prefecture.ready();
				code.put(prefecture.getCode(), prefecture.search(rectangle));
			}else{
				prefecture.dump();
			}
		}
		return code;
	}

	/**
	 * 市区町村番号を取得する
	 * @param rectangle 検索範囲
	 * @return 検索範囲と重なる市区町村の番号
	 * @throws IOException 
	 */
	public Set<Integer> searchCityCode(Rectangle rectangle) throws IOException {
		Set<Integer> codes = new HashSet<Integer>();
		for(Prefecture cell : this.prefecture) {
			if(rectangle.intersects(cell)) {
				cell.ready();
				cell.search(codes, rectangle);
			}else{
				cell.dump();
			}
		}
		return codes;
	}
}