package view;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Stroke;
import java.awt.geom.GeneralPath;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.swing.JPanel;

import util.Log;

import map.data.Curve;
import map.data.City;
import map.data.CityMap;
import map.data.Label;
import map.data.Labels;
import map.data.Mesh;
import map.data.Node;
import map.data.Road;
import map.labeling.Labeling;
import map.labeling.SimpleLabeling;
import map.route.SearchThread;
import map.route.ShortestPathAlgorithm;
import map.route.DirectDistance;
import map.route.UnidirectShortestPathSearch;

/**
 * 地図データを表示するためのパネル
 * 
 * 2005/09/23
 * 
 * @author ma38su
 */
public class MapPanel extends JPanel implements Printable {

	private static final int MAX_SCREEN_X = 154 * 3600000;
	private static final int MIN_SCREEN_X = 122 * 3600000;
	private static final int MAX_SCREEN_Y = 46  * 3600000;
	private static final int MIN_SCREEN_Y = 20  * 3600000;

	/**
	 * 国土数値情報による塗りつぶし色
	 */
	private static final Color FILLCOLOR = Color.GREEN.brighter();
	
	/**
	 * 地図の背景色（海の色）
	 */
	private static final Color MAP_BGCOLOR = Color.CYAN;
	
	/**
	 * フォントの種類
	 */
	private static final String FONT_FAMILY = "GOTHIC";
	
	/**
	 * 最大フォントサイズ
	 */
	private static final int MAX_FONT_SIZE = 100;

	/**
	 * 最短経路探索アルゴリズム
	 */
	private ShortestPathAlgorithm algorithm;

	/**
	 * 市町村まで表示する倍率
	 */
	private final float CITY_SCALE = 0.0002f;

	/**
	 * 国土数値情報と数値地図25000を切り替える倍率
	 */
	private final float DATA25K_SCALE = 0.0020f;

	/**
	 * 描画する道路のレベル
	 */
	private int drawlevel;

	/**
	 * アンチエイリアスのフラグ
	 */
	private boolean isAntialias;

	/**
	 * 水域区間の塗りつぶしフラグ
	 */
	private boolean isFill;

	/**
	 * 操作マニュアルの表示フラグ
	 */
	private boolean isHelp;

	/**
	 * ラベル表示のフラグ
	 */
	private int isLabel;

	/**
	 * 頂点表示のフラグ
	 */
	private boolean isNodeView;

	/**
	 * マウス操作のフラグ
	 */
	private boolean isOperation;
	
	/**
	 * 地図情報管理マップ
	 */
	private CityMap maps;

	/**
	 * 頂点のサイズ
	 */
	private int nodeSize = 3;

	/**
	 * オフスクリーンイメージ
	 */
	private Image offs;

	/**
	 * 都道府県ポリゴン
	 */
	private Polygon[] prefecture;

	/**
	 * 一般道のルート色
	 */
	private final Color ROUTE_COLOR_GENERAL = Color.YELLOW;

	/**
	 * 高速道路のルート色
	 */
	private final Color ROUTE_COLOR_HIGHWAY = Color.RED;

	/**
	 * 地図の表示倍率
	 */
	private float scale = 0.005f;

	/**
	 * 表示倍率の上限
	 */
	private float SCALE_MAX = 0.1f;
	
	/**
	 * 表示倍率の下限
	 */
	private float SCALE_MIN = 0.000005f;

	/**
	 * 表示倍率の変更精度
	 */
	private float SCALE_SENSE = 0.08f;

	/**
	 * スクリーンサイズ
	 */
	private Rectangle screen;
	
	/**
	 * 探索の始点
	 */
	private Node start;

	/**
	 * 道路区間，鉄道区間の描画の基本の幅
	 */
	private static final int STROKE_WIDTH = 130;
	
	/**
	 * 探索の終点
	 */
	private Node terminal;
	
	/**
	 * 探索を行うスレッド
	 */
	private SearchThread thread;

	/**
	 * 再描画フラグ
	 */
	private boolean isRepaint;

	/**
	 * 水域
	 */
	private Polygon[] prefectureCoast;

	/**
	 * 市区町村の行政界を描画します。
	 * @param g 描画するGraphics2D
	 * @param city 描画する行政界
	 * @param color 
	 */
	private void drawBorder(Graphics2D g, City city, Color color) {
		Polygon[] polygons = city.getPolygon();
		if (polygons != null) {
			g.setColor(color);
			int x = 0;
			int y = 0;
			int maxN = 0;
			for (int i = 0; i < polygons.length; i++) {
				if(polygons[i].intersects(this.screen)) {
					int[] aryX = polygons[i].xpoints;
					int[] aryY = polygons[i].ypoints;
	
					int[] polygonX = new int[polygons[i].npoints];
					int[] polygonY = new int[polygons[i].npoints];
	
					for (int j = 0; j < polygons[i].npoints; j++) {
						polygonX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
						polygonY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
						x += polygonX[j];
						y += polygonY[j];
					}

					g.drawPolygon(polygonX, polygonY, polygons[i].npoints);
					if (maxN < polygons[i].npoints) {
						maxN = polygons[i].npoints;
						Rectangle rect = polygons[i].getBounds();
						x = (int)((rect.x + rect.width / 2 - this.screen.x) * this.scale);
						y = this.getHeight() - (int)((rect.y + rect.height / 2 - this.screen.y) * this.scale);
					}
				}
			}
			if (maxN > 0 && city.getName() != null) {
				Font defaultFont = g.getFont();
				int fontSize = 12 + (int)(this.scale * 10000 + 0.5f);
				if (fontSize > MapPanel.MAX_FONT_SIZE) {
					fontSize = MapPanel.MAX_FONT_SIZE;
				}
				g.setFont(new Font(MapPanel.FONT_FAMILY, Font.PLAIN, fontSize));
				FontMetrics metrics = g.getFontMetrics();
				y += metrics.getHeight() / 2;
				x -= metrics.stringWidth(city.getName()) / 2;
				g.drawString(city.getName(), x, y);
				g.setFont(defaultFont);
			}
		}
	}

	/**
	 * 行政界の描画
	 * 
	 * @param g
	 *            描画するGraphics
	 * @param curves
	 *            描画する行政界
	 * @param color
	 *            描画する色
	 */
	private void drawBorderCurve(final Graphics2D g, final Curve[] curves, final Color color) {
		if (curves != null) {
			g.setColor(color);
			for (final Curve curve : curves) {
				final int[] aryX = curve.getArrayX();
				final int[] aryY = curve.getArrayY();
				int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
				int y0 = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);
				for (int k = 1; k < aryX.length; k++) {
					int x = (int) ((aryX[k] - this.screen.x) * this.scale);
					int y = this.getHeight() - (int) ((aryY[k] - this.screen.y) * this.scale);
					// 表示領域内
					if (this.screen.intersectsLine(aryX[k - 1], aryY[k - 1], aryX[k], aryY[k])) {
						g.drawLine(x0, y0, x, y);
					}
					x0 = x;
					y0 = y;
				}
			}
		}
	}

	/**
	 * 水域界の描画
	 * @param g 描画するGraphics2D
	 * @param coast 描画する水域界
	 * @param color
	 */
	private void drawCoast(Graphics2D g, Map<Curve, Boolean> coast, Color color) {
		if (coast != null) {
			g.setColor(color);
			for (Map.Entry<Curve, Boolean> entry : coast.entrySet()) {
				Curve curve = entry.getKey();
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();
				if (this.isFill && entry.getValue() && curve.getType() == 0) {
					Polygon polygon = new Polygon(aryX, aryY, aryX.length);
					if (polygon.intersects(this.screen)) {
						int[] x = new int[aryX.length];
						int[] y = new int[aryY.length];
						for (int i = 0; i < aryX.length; i++) {	
							x[i] = (int) ((aryX[i] - this.screen.x) * this.scale);
							y[i] = this.getHeight() - (int) ((aryY[i] - this.screen.y) * this.scale);
						}
						g.fillPolygon(x, y, aryX.length);
					}
				} else {
					int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
					int y0 = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);
					for (int i = 1; i < aryX.length; i++) {
						int x = (int) ((aryX[i] - this.screen.x) * this.scale);
						int y = this.getHeight() - (int) ((aryY[i] - this.screen.y) * this.scale);

						// 表示領域内
						if (this.screen.intersectsLine(aryX[i - 1], aryY[i - 1], aryX[i], aryY[i])) {
							g.drawLine(x0, y0, x, y);
						}
						x0 = x;
						y0 = y;
					}
				}
			}
		}
	}
	
	/**
	 * 折れ線の描画をおこないます。
	 * @param g
	 * @param path
	 * @param color
	 * @param stroke
	 */
	private void drawGeneralPath(Graphics2D g, GeneralPath path, Color color, Stroke stroke) {
		g.setColor(color);
		g.setStroke(stroke);
		g.draw(path);
	}

	/**
	 * 操作説明を描画します。
	 * @param g
	 * @param x
	 * @param y
	 */
	public void drawHelp(Graphics2D g, int x, int y) {

		int width = 305;
		int height = 220;

		g.setColor(new Color(255, 255, 255, 200));

		g.fillRect(x, y, width, height);

		g.setColor(Color.BLACK);
		g.setFont(new Font(MapPanel.FONT_FAMILY, Font.BOLD, 14));
		FontMetrics metrics = g.getFontMetrics();
		String msg = "操作マニュアル";
		g.drawString(msg,  x + (width - metrics.stringWidth(msg)) / 2, y + 25);
		g.setFont(new Font(MapPanel.FONT_FAMILY, Font.BOLD, 12));
		
		g.drawString("↑ / ↓ / ← / →", x + 180, y + 50);
		g.drawString("＋ / ； / PgUp", x + 180, y + 95);
		g.drawString("－ / ＝ / PgDn", x + 180, y + 140);
		g.drawString("Ctrl + F1", x + 180, y + 185);

		g.drawString("マウス ドラッグ", x + 20, y + 50);
		g.drawString("マウス ホイール", x + 20, y + 95);
		g.drawString("右クリック", x + 20, y + 140);
		g.drawString("Shift + 右クリック", x + 20, y + 185);


		g.setFont(new Font(MapPanel.FONT_FAMILY, Font.PLAIN, 12));
		g.drawString("地図の平行移動",  x + 180, y + 70);
		g.drawString("地図の拡大",     x + 180, y + 115);
 		g.drawString("地図の縮小",     x + 180, y + 160);
		g.drawString("マニュアル表示切替", x + 180, y + 205);

 		g.drawString("地図の平行移動", x + 20, y + 70);
 		g.drawString("地図の拡大縮小", x + 20, y + 115);
 		g.drawString("探索始点の選択 / 探索開始", x + 20, y + 160);
 		g.drawString("探索終点の選択 / 探索中止", x + 20, y + 205);
	}

	/**
	 * 数値地図25000を描画する
	 * 
	 * @param g
	 */
	public void drawMap(Graphics2D g) {

		int meshSize = (int) (Mesh.SIZE * this.scale) + 1;

		// メッシュの描画
		synchronized (this.maps) {
			for (final City map : this.maps.values()) {
				if(!map.hasData()) {
					this.fillPolygon(g, map.getPolygon(), MapPanel.FILLCOLOR);
					this.drawBorder(g, map, Color.BLACK);
				} else {
					int halfMesh = Mesh.SIZE / 2;
					for (Mesh mesh : map.getMesh()) {
						if (!this.screen.intersects(mesh.getX() - halfMesh, mesh.getY() - halfMesh, Mesh.SIZE, Mesh.SIZE)) {
							continue;
						}
						int x = (int) ((mesh.getX() - halfMesh - this.screen.x) * this.scale);
						int y = this.getHeight() - (int) ((mesh.getY() + halfMesh - this.screen.y) * this.scale);
		
						g.setColor(mesh.getColor());
						g.fillRect(x, y, meshSize, meshSize);
					}
				}
			}
			for (final City map : this.maps.values()) {
	
				if (!map.hasData()) {
					continue;
				}
				// 行政界を描画
				this.drawBorderCurve(g, map.getBorder(), Color.GREEN);
	
				// 河川を描画
				this.drawBorderCurve(g, map.getRiver(), Color.BLUE);
	
				// 沿岸を描画
				this.drawCoast(g, map.getCoast(), Color.BLUE);
			}
		}
		{
			// 道路を描画する
			int[] roadWidth = new int[6];
			
			GeneralPath[] road = new GeneralPath[12];
			for (int i = 5; i >= 0; i--) {
				roadWidth[i] = (int)((i + 1) * MapPanel.STROKE_WIDTH * this.scale);
				if(roadWidth[i] > 0) {
					road[i] = new GeneralPath();
					this.drawlevel = i;
				}
				road[i + 6] = new GeneralPath();
			}
			
//			List<int[]> jrX = new ArrayList<int[]>();
//			List<int[]> jrY = new ArrayList<int[]>();
			GeneralPath jr = new GeneralPath();
			List<int[]> otherX = new ArrayList<int[]>();
			List<int[]> otherY = new ArrayList<int[]>();
			List<int[]> stationX = new ArrayList<int[]>();
			List<int[]> stationY = new ArrayList<int[]>();

			synchronized (this.maps) {
				for (final City map : this.maps.values()) {
					if (!map.hasData()) {
						continue;
					}
					for (int i = 5; i >= 0; i--) {
						this.extractRoadway(road[i], map.getRoad()[i]);
						this.extractRoadway(road[i + 6], map.getRoad()[i + 6]);
					}
					this.extractGeneralPath(map.getJRRailway(), jr);
//					this.extractPolyLine(map.getJRRailway(), jrX, jrY);
					this.extractPolyLine(map.getOtherRailway(), otherX, otherY);
					this.extractPolyLine(map.getStation(), stationX, stationY);
				}
			}
			this.drawRoadway(g, road, roadWidth);
			int w = (int)(MapPanel.STROKE_WIDTH * this.scale * 4f + 0.5f);
			if (w == 0) {
				w = 1;
			}

			Stroke stroke = new BasicStroke(w + 2, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_ROUND);
			this.drawPolyLine(g, otherX, otherY, Color.DARK_GRAY, stroke);
			this.drawGeneralPath(g, jr, Color.BLACK, stroke);

			this.drawPolyLine(g, otherX, otherY, Color.BLACK, new BasicStroke(w, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			this.drawGeneralPath(g, jr, Color.WHITE, new BasicStroke(w, BasicStroke.CAP_BUTT, BasicStroke.JOIN_ROUND, 10f, new float[]{w * 4, w * 4}, 0.25f));

			w *= 3;
			this.drawPolyLine(g, stationX, stationY, Color.BLACK, new BasicStroke(w + 2, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_ROUND));
			this.drawPolyLine(g, stationX, stationY, Color.WHITE, new BasicStroke(w, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_ROUND));
		}
		synchronized (this.maps) {
			for (final City map : this.maps.values()) {
				if (!map.hasData()) {
					continue;
				}	
				if (this.isNodeView) {
					int nodeSize = (int) (this.nodeSize / this.scale);
	
					g.setColor(Color.GRAY);
	
					for (Node node : map.getNode()) {
						// 表示領域外であれば次へ
						if (!this.screen.intersects(node.getX() - nodeSize, node.getY() - nodeSize, nodeSize * 2, nodeSize * 2)) {
							continue;
						}
	
						int x = (int) ((node.getX() - this.screen.x) * this.scale);
						int y = this.getHeight() - (int) ((node.getY() - this.screen.y) * this.scale);
	
						// draw node
						g.fillRect(x - this.nodeSize, y - this.nodeSize, this.nodeSize * 2, this.nodeSize * 2);
					}
				}
			}
		}
		this.drawRoute(g);
		g.setStroke(new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
		if (this.isLabel > 0) {
			Labeling labeling = new SimpleLabeling(g, this.screen, this.getWidth(), this.getHeight(), this.scale);
			synchronized (this.maps) {
				for (City map : this.maps.values()) {
					if (map.hasData()) {	
						if ((this.isLabel & 1) != 0) {
							labeling.add(map.getStation());
						}
						if ((this.isLabel & 2) != 0) {
							labeling.add(map.getLabel(Label.KEY_CM));
						}
						if ((this.isLabel & 4) != 0) {
							labeling.add(map.getLabel(Label.KEY_KO));
						}
						for (Labels labels : map.getLabels()) {
							labeling.add(labels);
						}
					}
				}
			}
			labeling.draw();
		}
	}

	/**
	 * ポリゴンを描画します。
	 * @param g 描画するGraphics2D
	 * @param polygons 描画するポリゴン
	 * @param color 
	 */
	private void drawPolygon(Graphics2D g, Polygon[] polygons, Color color) {
		if (polygons == null) {
			return;
		}
		g.setColor(color);
		for (int i = 0; i < polygons.length; i++) {
			if(polygons[i].intersects(this.screen)) {
				int[] aryX = polygons[i].xpoints;
				int[] aryY = polygons[i].ypoints;

				int[] polygonX = new int[polygons[i].npoints];
				int[] polygonY = new int[polygons[i].npoints];

				for (int j = 0; j < polygons[i].npoints; j++) {
					polygonX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
					polygonY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
				}
				g.drawPolygon(polygonX, polygonY, polygons[i].npoints);
			}
		}
	}

	/**
	 * 折れ線の描画をおこないます。
	 * @param g
	 * @param listX
	 * @param listY
	 * @param color
	 * @param stroke
	 */
	private void drawPolyLine(Graphics2D g, List<int[]> listX, List<int[]> listY, Color color, Stroke stroke) {
		g.setColor(color);
		g.setStroke(stroke);
		for (int i = 0; i < listX.size(); i++) {
			int[] aryX = listX.get(i);
			g.drawPolyline(aryX, listY.get(i), aryX.length);
		}
	}
	
	/**
	 * 道路の描画
	 * 
	 * @param g Graphics2D
	 * @param path 
	 * @param width 
	 */
	private void drawRoadway(Graphics2D g, GeneralPath[] path, int[] width) {
		g.setColor(Color.LIGHT_GRAY);
		for (int i = 0; i < 6; i++) {
			if(path[i] != null) {
				g.setStroke(new BasicStroke(width[i] + 2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				g.draw(path[i]);
			}
		}
		g.setColor(Color.WHITE);
		for (int i = 0; i < 6; i++) {
			if(path[i] != null) {
				g.setStroke(new BasicStroke(width[i], BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				g.draw(path[i]);
			}
		}
		g.setColor(Color.DARK_GRAY);
		for (int i = 0; i < 6; i++) {
			g.setStroke(new BasicStroke(width[i] + 3, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(path[i + 6]);
		}
		g.setColor(Color.GRAY);
		for (int i = 0; i < 6; i++) {
			g.setStroke(new BasicStroke(width[i] + 1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(path[i + 6]);
		}
	}

	/**
	 * 探索済み頂点を描画する
	 * 
	 * @param g
	 */
	private void drawRoute(Graphics2D g) {
		if (this.thread != null) {
			Collection<Road> route = this.thread.getRoute();
			if (route != null) {
				int[] width = new int[12];
				for (int i = 0; i < 6; i++) {
					width[i] = (int)((i + 1) * MapPanel.STROKE_WIDTH * this.scale) + 1;
				}
				for (Road road : route) {
					int[] aryX = road.getArrayX();
					int[] aryY = road.getArrayY();
		
					int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
					int y0 = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);

					if (road.getType() == 3) {
						g.setColor(this.ROUTE_COLOR_HIGHWAY);
					} else {
						g.setColor(this.ROUTE_COLOR_GENERAL);
					}
					g.setStroke(new BasicStroke(width[road.getWidth()], BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
		
					for (int k = 1; k < aryX.length; k++) {
		
						int x = (int) ((aryX[k] - this.screen.x) * this.scale);
						int y = this.getHeight() - (int) ((aryY[k] - this.screen.y) * this.scale);

						// 表示領域内
						if (this.screen.intersectsLine(aryX[k - 1], aryY[k - 1], aryX[k], aryY[k])) {
							g.drawLine(x0, y0, x, y);
						}
						x0 = x;
						y0 = y;
					}
				}
			}
		}
		int r = (int)(this.scale * 500) + 2;
		g.setStroke(new BasicStroke(r / 2f));
		if (this.start != null) {
			int x = (int) ((this.start.getX() - this.screen.x) * this.scale) - r;
			int y = this.getHeight() - (int) ((this.start.getY() - this.screen.y) * this.scale) - r;
			g.setColor(Color.YELLOW);
			g.fillOval(x, y, r * 2, r * 2);
			g.setColor(Color.BLACK);
			g.drawOval(x, y, r * 2, r * 2);
		}
		if (this.terminal != null) {
			int x = (int) ((this.terminal.getX() - this.screen.x) * this.scale) - r;
			int y = this.getHeight() - (int) ((this.terminal.getY() - this.screen.y) * this.scale) - r;
			g.setColor(Color.RED);
			g.fillOval(x, y, r * 2, r * 2);
			g.setColor(Color.BLACK);
			g.drawOval(x, y, r * 2, r * 2);
		}
	}

	/**
	 * GeneralPathに展開
	 * 
	 * @param curves 展開する曲線の配列
	 * @param path 展開されたGeneralPath
	 */
	private void extractGeneralPath(final Curve[] curves, final GeneralPath path) {
		if (curves != null) {
			for (final Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();

				int x = (int) ((aryX[0] - this.screen.x) * this.scale);
				int y = this.getHeight() - (int) ((aryY[0] - this.screen.y) * this.scale);
				path.moveTo(x, y);

				for (int i = 1; i < aryX.length; i++) {
					x = (int) ((aryX[i] - this.screen.x) * this.scale);
					y = this.getHeight() - (int) ((aryY[i] - this.screen.y) * this.scale);
					path.lineTo(x, y);
				}
			}
		}
	}

	/**
	 * 折れ線に展開
	 * 
	 * @param curves
	 * @param polyX 
	 * @param polyY 
	 */
	private void extractPolyLine(final Curve[] curves, final List<int[]> polyX, final List<int[]> polyY) {
		if (curves != null) {
			for (final Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();
				int[] lineX = new int[aryX.length];
				int[] lineY = new int[aryY.length];
				for (int i = 0; i < aryX.length; i++) {
					lineX[i] = (int) ((aryX[i] - this.screen.x) * this.scale);
					lineY[i] = this.getHeight() - (int) ((aryY[i] - this.screen.y) * this.scale);
				}
				polyX.add(lineX);
				polyY.add(lineY);
			}
		}
	}

	/**
	 * 道路データをGeneralPathに展開する
	 * @param path 展開先のGeneralPath
	 * @param curves 道路データ
	 */
	private void extractRoadway(GeneralPath path, Curve[] curves) {
		if (path != null && curves != null) {
			for (Curve curve : curves) {
				int[] aryX = curve.getArrayX();
				int[] aryY = curve.getArrayY();
	
				int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
				int y0 = this.getHeight()
						- (int) ((aryY[0] - this.screen.y) * this.scale);
	
				path.moveTo(x0, y0);
				
				for (int k = 1; k < aryX.length; k++) {
	
					int x = (int) ((aryX[k] - this.screen.x) * this.scale);
					int y = this.getHeight() - (int) ((aryY[k] - this.screen.y) * this.scale);

					// 表示領域外であれば次へ
					if (this.screen.intersectsLine(aryX[k - 1], aryY[k - 1], aryX[k], aryY[k])) {
						path.lineTo(x, y);
					} else {
						path.moveTo(x, y);
					}
				}
			}
		}
	}

	/**
	 * ポリゴンを描画します。
	 * @param g 描画するGraphics2D
	 * @param polygons 描画するポリゴン
	 * @param color 
	 */
	private void fillPolygon(Graphics2D g, Polygon[] polygons, Color color) {
		if (polygons == null) {
			return;
		}
		g.setColor(color);
		for (int i = 0; i < polygons.length; i++) {
			if(polygons[i].intersects(this.screen)) {
				int[] aryX = polygons[i].xpoints;
				int[] aryY = polygons[i].ypoints;

				int[] polygonX = new int[polygons[i].npoints];
				int[] polygonY = new int[polygons[i].npoints];

				for (int j = 0; j < polygons[i].npoints; j++) {
					polygonX[j] = (int)((aryX[j] - this.screen.x) * this.scale);
					polygonY[j] = this.getHeight() - (int)((aryY[j] - this.screen.y) * this.scale);
				}
				g.fillPolygon(polygonX, polygonY, polygons[i].npoints);
			}
		}
	}

	
	/**
	 * スクリーン情報の取得
	 * 
	 * @return スクリーンのRectangle
	 */
	public Rectangle getScreen() {
		return this.screen;
	}

	/**
	 * 初期設定
	 * @param map 
	 * @param prefecture 
	 * @param prefectureCoast 
	 */
	public void init(CityMap map, Polygon[] prefecture, Polygon[] prefectureCoast) {
		Log.out(this, "init called.");
		this.algorithm = new UnidirectShortestPathSearch(new DirectDistance(), map);

		this.maps = map;
		
		this.prefecture = prefecture;
		this.prefectureCoast = prefectureCoast;
		
		this.screen = new Rectangle();
		this.screen.x = MapPanel.MIN_SCREEN_X;
		this.screen.y = MapPanel.MIN_SCREEN_Y;
		
		this.isLabel = 7;
		this.isNodeView = false;
		this.isAntialias = true;

		this.isFill = false;
		
		this.isHelp = true;

		this.isOperation = false;
		
		super.setBackground(MapPanel.MAP_BGCOLOR);

		double widthScale = (double) this.getWidth() / (MapPanel.MAX_SCREEN_X - MapPanel.MIN_SCREEN_X);
		double heightScale = (double) this.getHeight() / (MapPanel.MAX_SCREEN_Y - MapPanel.MIN_SCREEN_Y);
		this.scale = (float) ((widthScale < heightScale) ? widthScale : heightScale);
		this.repaint();
	}
	
	public boolean isLoaded() {
		return this.maps != null;
	}

	/**
	 * 操作しているかどうか
	 * @return 操作していればtrue
	 */
	public boolean isOperation() {
		return this.isOperation;
	}

	/**
	 * 地図の表示状態
	 * 数値地図25000 : 2
	 * 国土数値情報   : 1
	 * 都道府県      : 0
	 * @return 地図の表示状態
	 */
	public int mode() {
		if(this.scale > this.DATA25K_SCALE) {
			return 2;
		} else if(this.scale > this.CITY_SCALE) {
			return 1;
		} else {
			return 0;
		}
	}
	
	/**
	 * 平行移動を行う
	 * 
	 * @param dx
	 * @param dy
	 */
	public void moveLocation(int dx, int dy) {
		this.screen.x -= dx / this.scale;
		this.screen.y += dy / this.scale;
		this.repaint();
	}

	/**
	 * 地図の中心経緯度を指定して移動します。
	 * @param x 東経
	 * @param y 北緯
	 */
	public void moveToLocation(double x, double y) {
		this.screen.x = (int)(x * 3600000) - (int)(this.getWidth() / 2 / this.scale + 0.5f);
		this.screen.y = (int)(y * 3600000) - (int)(this.getHeight() / 2 / this.scale + 0.5f);
		this.repaint();
	}
	
	/**
	 * 西脇市へ移動します（デバッグ用）
	 *
	 */
	public void moveToNishiwaki() {
		this.scale = this.DATA25K_SCALE * 1.2f;
		this.moveToLocation(135, 35);
		this.repaint();
	}

	@Override
	public void paintComponent(Graphics g) {
		if(this.maps == null) {
			this.isRepaint = false;
			this.offs = this.createImage(this.getWidth(), this.getHeight());
			Graphics2D offg = (Graphics2D)this.offs.getGraphics();
			super.paintComponent(offg);
			offg.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
			// オフスクリーンバッファ
			Font defaultFont = g.getFont();
			offg.setFont(new Font(MapPanel.FONT_FAMILY, Font.PLAIN, 20));
			String msg = "Now Loading...";
			FontMetrics metrics = offg.getFontMetrics();
			offg.drawString(msg, (this.getWidth() - metrics.stringWidth(msg)) / 2, (this.getHeight() - metrics.getHeight()) / 2);
			offg.setFont(defaultFont);
			offg.dispose();
		} else if (this.isRepaint) {
			this.isRepaint = false;
			if (this.offs == null) {
				this.offs = this.createImage(this.getWidth(), this.getHeight());
			}
			if (this.getWidth() != this.offs.getWidth(null) || this.getHeight() != this.offs.getHeight(null)) {
				this.offs.flush();
				// オフスクリーンバッファ
				this.offs = super.createImage(this.getWidth(), this.getHeight());
			}
			Graphics2D offg = (Graphics2D)this.offs.getGraphics();
			super.paintComponent(offg);
			// なぜか海が描画されないので、とりあえずむりやり描画する。
			offg.setColor(MapPanel.MAP_BGCOLOR);
			offg.fillRect(0, 0, this.getWidth(), this.getHeight());
			this.paintMap(offg);

			offg.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);			
			offg.setColor(Color.BLACK);
			offg.drawString("SCALE : " + Float.toString(this.scale * 1000) + "m", 5, 15);
			// アルゴリズム名の表示
			int vp = this.thread != null ? this.thread.getVP() : 0;
			if (vp == 0) {
				offg.drawString(this.algorithm.toString(), 5, 35);
			} else {
				offg.drawString(this.algorithm.toString() + " : " + vp, 5, 35);
			}
			if (this.isHelp) {
				this.drawHelp(offg, this.getWidth() - 325, this.getHeight() - 240);
			}
			offg.dispose();
		}
		g.drawImage(this.offs, 0, 0, null);
	}

	/**
	 * 地図の描画
	 * @param g
	 */
	public void paintMap(Graphics2D g) {
		// スクリーン情報を指定する
		this.screen.width = (int) (this.getWidth() / this.scale);
		this.screen.height = (int) (this.getHeight() / this.scale);

		if(this.scale > this.CITY_SCALE) {
			if (!this.isOperation && this.isAntialias) {
				g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			}
			if (this.scale >= this.DATA25K_SCALE) {
				this.drawMap(g);
			} else {
				g.setStroke(new BasicStroke(2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				this.fillPolygon(g, this.prefecture, MapPanel.FILLCOLOR);
				this.drawPolygon(g, this.prefecture, Color.BLACK);

				this.fillPolygon(g, this.prefectureCoast, Color.CYAN);
				this.drawPolygon(g, this.prefectureCoast, Color.BLACK);

				g.setStroke(new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				synchronized (this.maps) {
					for (City city : this.maps.values()) {
//						this.drawPolygon(g, city.getPolygon(), Color.BLACK);
						this.drawBorder(g, city, Color.BLACK);
					}
				}
				this.drawRoute(g);
			}
		} else {
			this.fillPolygon(g, this.prefecture, MapPanel.FILLCOLOR);
			this.drawPolygon(g, this.prefecture, Color.BLACK);

			this.fillPolygon(g, this.prefectureCoast, Color.CYAN);
			this.drawPolygon(g, this.prefectureCoast, Color.BLACK);
		}
	}

	/**
	 * 変換して描画します。
	 * @param g 
	 * @param width 
	 * @param height 
	 */
	public void paintTranslate(Graphics2D g, double width, double height) {
		final double newScale = Math.min(width / this.getWidth(), height / this.getHeight());
		g.scale(newScale, newScale);
		g.setClip(0, 0, this.getWidth(), this.getHeight());
		g.setColor(MapPanel.MAP_BGCOLOR);
		g.fillRect(0, 0, this.getWidth(), this.getHeight());
		Log.out(this, "painted sea.");
		this.paintMap(g);
	}

	/**
	 * Printable インターフェースの実装
	 * by Kumano
	 * 
	 * @param graphics
	 * @param pageFormat
	 * @param pageIndex
	 * @return 状態
	 */
	public int print(Graphics graphics, PageFormat pageFormat, int pageIndex) {
		if (pageIndex == 0 && this.maps != null) {
			final Graphics2D g = (Graphics2D) graphics;
			g.translate(pageFormat.getImageableX(), pageFormat.getImageableY());
			this.paintTranslate(g, pageFormat.getImageableWidth(), pageFormat.getImageableHeight());
			return Printable.PAGE_EXISTS;
		} else {
			return Printable.NO_SUCH_PAGE;
		}
	}

	@Override
	public void repaint() {
		if (this.maps == null || !this.isRepaint) {
			this.isRepaint = true;
			super.repaint();
		}
	}

	/**
	 * ルートの再計算を行います。
	 *
	 */
	public void reroute() {
		if(this.start != null && this.terminal != null && this.start != this.terminal) {
			if (this.thread != null) {
				this.thread.kill();
			}
			this.thread = this.algorithm.search(this.start, this.terminal);
		}
		this.repaint();
	}

	/**
	 * 探索端点の変更を行います。
	 * @param ex マウスのX座標
	 * @param ey マウスのY座標
	 * @param flag trueなら終点の変更、falseなら始点の変更
	 */
	public void searchBoundary(final int ex, final int ey, final boolean flag) {
		int x = (int)(ex / this.scale + this.screen.x);
		int y = (int)((this.getHeight() - ey) / this.scale + this.screen.y);
		
		Node point = null;
		double nodeDist = Double.POSITIVE_INFINITY;
		synchronized (this.maps) {
			for (final City map : this.maps.values()) {
				if (!map.hasData() || !map.getArea().contains(x, y)) {
					continue;
				}
				for (final Node node : map.getNode()) {
					final double d = node.distance(x, y, this.drawlevel);
					if (nodeDist > d) {
						point = node;
						nodeDist = d;
					}
				}
			}
		}
		if (point != null) {
			if (flag) {
				this.terminal = point;
				this.start = null;
				if (this.thread != null) {
					this.thread.kill();
				}
			} else {
				this.start = point;
				if (this.terminal != null) {
					if (this.thread != null) {
						this.thread.kill();
					}
					if (this.start != this.terminal) {
						this.thread = this.algorithm.search(this.start, this.terminal);
					}
				}
			}
		}
	}

	/**
	 * マウス操作の状態を設定する
	 * @param flag マウス操作していればtrue
	 */
	public void setOperation(boolean flag) {
		this.isOperation = flag;
		if(!flag) {
			this.repaint();
		}
	}

	/**
	 * 水域界の塗りつぶしを切り替える
	 */
	public void switchFill() {
		this.isFill = !this.isFill;
		this.repaint();
	}

	/**
	 * 操作マニュアルの表示を切り替える
	 */
	public void switchHelp() {
		this.isHelp = !this.isHelp;
		this.repaint();
	}

	/**
	 * ラベル表示を切り替える
	 * @param n 
	 */
	public void switchLabel(int n) {
		if ((this.isLabel & n) == 0) {
			this.isLabel += n;
		} else {
			this.isLabel -= n;
		}
		this.repaint();
	}

	/**
	 * 頂点表示を切り替える
	 */
	public void switchNodeView() {
		this.isNodeView = !this.isNodeView;
		this.repaint();
	}

	/**
	 * アンチエイリアスを切り替える
	 */
	public void switchRendering() {
		this.isAntialias = !this.isAntialias;
		this.repaint();
	}

	/**
	 * 最短経路を求めるアルゴリズムを切り替える
	 * @param algorithm 切り替えるための文字列
	 */
	public void switchShortestPathAlgorithm(String algorithm) {
		if(algorithm.endsWith("_a*")) {
			this.algorithm = new UnidirectShortestPathSearch(new DirectDistance(), this.maps);
		} else if (algorithm.endsWith("_dijkstra")) {
			this.algorithm = new UnidirectShortestPathSearch(null, this.maps);
		}
		this.reroute();
	}

	/**
	 * 拡大縮小を行う
	 * 
	 * @param x
	 * @param y
	 * @param d
	 */
	public void zoom(int x, int y, int d) {
		float newScale = this.scale * (1 + d * this.SCALE_SENSE);
		if (newScale > this.SCALE_MAX) {
			newScale = this.SCALE_MAX;
		} else if (newScale < this.SCALE_MIN) {
			newScale = this.SCALE_MIN;
		}
		y = this.getHeight() - y;
		this.screen.x = (int) (this.screen.x + x / this.scale - x / newScale);
		this.screen.y = (int) (this.screen.y + y / this.scale - y / newScale);

		this.scale = newScale;

		this.repaint();
	}
}