# -*- coding: utf-8 -*-

#########################################################################
## - Application Name: Machikane-Red
## - Version: 3.1904
## - Date: 2019-05-07
## - Copyright: (c) 2018-2019 Mitsuhiro Tsuda.
## - License: Machikane-Red (version 3.1904) is released
##            under the GNU General Public License (GPL).
##            See the copyright notice LICENSE.
#########################################################################

from gluon.html import *
import re
import json
import urllib


def d__elementset(dbm, classid=1):
    """
    要素セット辞書を返す
    {
        '*class_id': {
            'group': [
                *class_element_id
            ]
            ,'keyset': [
                *class_element_mr_name
            ]
            ,'nameset': {
                '*class_element_mr_name': {
                    'id':     *class_element_id
                    'title':   *class_element_name (title)
                }
            }
        }
    }
    """

    # 要素セット辞書
    d_elements = {}
    for row in dbm(dbm.mr_classes).select(orderby=[dbm.mr_classes.mr_order,dbm.mr_classes.id]):
        # クラス要素
        rows = dbm(
                dbm.mr_class_elements.mr_class_id == row.id
            ).select(
                orderby=[
                    dbm.mr_class_elements.mr_order
                    ,dbm.mr_class_elements.id
                ]
            )

        # 定義名をキーとする要素名と要素IDの辞書
        d_meta = {}
        for r in rows:
            if r.mr_name is not None and not r.mr_name in d_meta:
                # set first element
                d_meta[r.mr_name] = dict(id = r.id, title = r.mr_title)

        d_elements[row.id] = dict(
            target = row.mr_target
            ,title = row.mr_title
            ,description = row.mr_description
            ,group = [r.id for r in rows] #クラス要素IDの配列
            ,keyset = [r.mr_name for r in rows if r.mr_open and r.mr_rank < 3]
            ,fullkeyset = [r.mr_name for r in rows if r.mr_open]
            ,nameset = d_meta
        )
    try:
        return d_elements if classid == 0 else d_elements[classid]
    except:
        return None




def __get_from_external_iiif_manifest(link, version='2'):
    """
    Directoey of external iiif manifest
    """
    d_d = None
    if link and re.fullmatch('^https?:\/\/[a-zA-Z0-9\.\/\-_]+(\.json)*$',link.strip()):
        req = urllib.request.Request(link)
        if True: #try:
            res = urllib.request.urlopen(req).read()
            manifest = json.loads(res.decode('utf-8'))
            if '@context' in manifest and manifest['@context'] == 'http://iiif.io/api/presentation/2/context.json':

                d_d = {}
                d_d['manifest'] = link
                d_d['content'] = manifest
                d_d['title'] = [manifest.get('label','')]
                d_d['description'] = [manifest.get('description','')]
                d_d['license'] = [manifest.get('license','')]
                d_d['attribution'] = manifest.get('attribution')
                d_d['logo'] = manifest.get('logo','')

                if 'metadata' in manifest:
                    for d1 in manifest['metadata']:
                        if isinstance(d1['value'],list):
                            for d2 in d1['value']:
                                if not d1['label'] in d_d:
                                    d_d[d1['label']] = [d2['value']['@value']]
                                else:
                                    d_d[d1['label']].append(d2['value']['@value'])
                        else:
                            # string
                            if not d1['label'] in d_d:
                                d_d[d1['label']] = d1['value']
                            else:
                                d_d[d1['label']].append(d1['value'])

        #except:
        #    pass

    return d_d




def d_d_page(dbm, pageid, a_offset=0, a_limit=20):
    """
    指定ページidのページセクション一覧をdict形式で得る
    """
    return d_d_sections(dbm, pageid)


def d_d_sections(dbm, pageid, a_offset=0, a_limit=20):
    """
    指定ページidのページセクション一覧をdict形式で得る
    """

    title = dbm.mr_pages[pageid]['mr_title'] or None

    # コンテナ
    l_d = []

    rows = dbm((dbm.mr_sections.mr_open == True)
                & (dbm.mr_sections.mr_page_id == pageid)
            ).select(
                orderby=[dbm.mr_sections.mr_order,~dbm.mr_sections.id]
            )

    for row in rows:

        # 各リソースが依存するクラス要素セットの取得
        d_elements = d__elementset(dbm, row.mr_class_id)

        # 各リソースのアイテムの取得
        rows2 = dbm(
                    (dbm.mr_section_items.mr_section_id == row.id)
                    & dbm.mr_section_items.mr_class_element_id.belongs(d_elements['group'])
                    #& (dbm.mr_section_items.mr_open == True)
                ).select(
                    #left=dbm.mr_class_elements.on(dbm.mr_section_items.mr_class_element_id == dbm.mr_class_elements.id)
                )

        # 定義名の取得
        d_d = {}
        d_d['subid'] = row.id
        d_d['ids'] = (0, 0, 0, 0)
        d_d['keyset'] = d_elements['keyset']

        for r2 in rows2:
            for ky in d_elements['keyset']:
                if r2.mr_class_element_id == d_elements['nameset'][ky]['id']:
                    if ky in ('creator','date'):
                        d_d[ky] = [dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=r2.mr_text
                                )]
                    #elif ky in ('created','updated'):
                    #    d_d[ky] = dict(
                    #                label=d_elements['nameset'][ky]['title']
                    #                ,value=r2.mr_text
                    #            )
                    else:
                        if ky not in d_d:
                            d_d[ky] = [dict(
                                        label=d_elements['nameset'][ky]['title']
                                        ,value=r2.mr_text
                                    )]
                        else:
                            d_d[ky].append(
                                        dict(
                                            label=d_elements['nameset'][ky]['title']
                                            ,value=r2.mr_text
                                        )
                                    )
                    break

        for ky in d_elements['keyset']:
            if ky not in d_d:
                d_d[ky] = [dict(label=ky.capitalize(), value='')]

        l_d.append(d_d)

    return dict(title=title, array=l_d)




def d_d_collection(dbm, collid, q=[], a_offset=0, a_limit=20):
    """
    指定コレクションidのリソース一覧をdict形式で得る
    """
    return d_d_resources(dbm, collid)


def d_d_resources(dbm, collid, q=[], a_offset=0, a_limit=20):
    """
    指定コレクションidのリソース一覧をdict形式で得る
    """
    title = None

    if type(q) is not list:
        q = []

    # コンテナ
    l_d = []

    if collid > 1:
        rows = dbm((dbm.mr_resources.mr_open==True)
                    & dbm.mr_resources.mr_collection_ids.contains(collid)
                ).select(
                    limitby=(a_offset,a_limit)
                    ,orderby_on_limitby=False
                    ,orderby=[dbm.mr_resources.mr_order,dbm.mr_resources.id]
                )
        title = dbm.mr_collections[collid]['mr_title'] or None

    else:
        rows = dbm((dbm.mr_resources.mr_open==True)
                    & (dbm.mr_resources.id>0)
                ).select(
                    limitby=(a_offset,a_limit)
                    ,orderby_on_limitby=False
                    ,orderby=[dbm.mr_resources.mr_order,~dbm.mr_resources.id]
                )

    for row in rows:
        # row is a resource

        # リソースのクラスに属する要素セットを得る
        d_elements = d__elementset(dbm, row.mr_class_id)

        # リソースのメタデータを得る
        count_items = dbm((dbm.mr_resource_items.mr_open == True)
                    & (dbm.mr_resource_items.mr_resource_id == row.id)
                    & dbm.mr_resource_items.mr_text.contains(q, all=True)
                ).count()

        if count_items == 0: continue

        rows_items = dbm((dbm.mr_resource_items.mr_open == True)
                    & (dbm.mr_resource_items.mr_resource_id == row.id)
                    & dbm.mr_resource_items.mr_class_element_id.belongs(d_elements['group'])
                    & dbm.mr_resource_items.mr_text.contains(q, all=True)
                ).select(
                    left=dbm.mr_class_elements.on(dbm.mr_resource_items.mr_class_element_id == dbm.mr_class_elements.id)
                )

        # 外部マニフェストの解析
        rows_external = __get_from_external_iiif_manifest(row.mr_manifest_link)

        # 定義名の取得
        d_d = {}
        d_d['subid'] = row.id
        d_d['ids'] = (row.id, 0, 0, 0)

        d_d['keyset'] = d_elements['keyset']
        d_d['airkeyvalue'] = {}
        d_d['contains'] = set([])

        d_d['title'] = dict(label=None, value=None)
        d_d['description'] = dict(label=None, value=None)

        if rows_external:

            # 外部リソース
            for ky in d_elements['keyset']:

                _keycomp = False
                d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
                for r_item in rows_items:
                    if r_item.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=r_item.mr_resource_items.mr_text
                                )
                        _keycomp = True
                        break
                    #elif ky in rows:
                    #    d_d[ky] = dict(
                    #                label=d_elements['nameset'][ky]['title']
                    #                ,value=rows[ky]
                    #            )
                if ky in rows_external:
                    if (_keycomp and (ky not in ('title','description')) and ((ky in d_d) and (len(d_d[ky]['value'])==0))) or (not _keycomp):
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=rows_external[ky]
                                )

            for ky in rows_external:
                #if ky not in ['content','title','description'] and ky not in d_d['airkeyvalue']:
                if ky not in d_d['airkeyvalue']:
                    d_d['airkeyvalue'][ky] = rows_external[ky]

            d_d['role'] = 'airresource'
            d_d['folders'] = __get_l_d_folders_air(rows_external['content'], row.id, d_d['contains'])

        else:

            for ky in d_elements['keyset']:
                d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
                for r_item in rows_items:
                    if r_item.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=r_item.mr_resource_items.mr_text
                                )
                        break

            d_d['role'] = 'resource'
            d_d['folders'] = __get_l_d_folders(dbm, row.id, d_d['contains'])

        # areaアノテーション表示のために必要
        d_d['annotes'] = __get_l_d_annotes(dbm, d_d['ids'])

        l_d.append(d_d)

    return dict(title=title, array=l_d)




def d_d_resource(dbm, resourceid, a_offset=0, a_limit=20):
    """
    指定リソースidの内容をdict形式で得る
    """
    return d_d_items(dbm, resourceid)


def d_d_items(dbm, resourceid, a_offset=0, a_limit=20):
    """
    指定リソースidの内容をdict形式で得る
    """

    # コンテナ
    l_d = []

    d_d = {}

    # このリソース
    row_resource = dbm.mr_resources[resourceid]

    if row_resource:

        d_elements = d__elementset(dbm, row_resource.mr_class_id)

        # このリソースのアイテムの取得
        rows_items = dbm((dbm.mr_resource_items.mr_open == True)
                    & (dbm.mr_resource_items.mr_resource_id == row_resource.id)
                    & dbm.mr_resource_items.mr_class_element_id.belongs(d_elements['group'])
                ).select(
                    left=dbm.mr_class_elements.on(dbm.mr_resource_items.mr_class_element_id == dbm.mr_class_elements.id)
                )

        # 外部マニフェストの解析
        rows_external = __get_from_external_iiif_manifest(row_resource.mr_manifest_link)

        d_d['subid'] = row_resource.id
        d_d['ids'] = (resourceid, 0, 0, 0)

        d_d['keyset'] = d_elements['fullkeyset']
        d_d['airkeyvalue'] = {}
        d_d['contains'] = set([])

        d_d['title'] = dict(label=None, value=None)
        d_d['description'] = dict(label=None, value=None)

        if rows_external:

            # 外部リソース
            for ky in d_elements['keyset']:

                _keycomp = False
                d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
                for r_item in rows_items:
                    if r_item.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=r_item.mr_resource_items.mr_text
                                )
                        _keycomp = True
                        break
                    #elif ky in rows:
                    #    d_d[ky] = dict(
                    #                label=d_elements['nameset'][ky]['title']
                    #                ,value=rows[ky]
                    #            )
                if ky in rows_external:
                    if (_keycomp and (ky not in ('title','description')) and ((ky in d_d) and (len(d_d[ky]['value'])==0))) or (not _keycomp):
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=rows_external[ky]
                                )

            for ky in rows_external:
                #if ky not in ['content','title','description'] and ky not in d_d['airkeyvalue']:
                if ky not in d_d['airkeyvalue']:
                    d_d['airkeyvalue'][ky] = rows_external[ky]

            d_d['role'] = 'airresource'
            d_d['folders'] = __get_l_d_folders_air(rows_external['content'], row_resource.id, d_d['contains'])

        else:

            for ky in d_elements['fullkeyset']:
                d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
                for r_item in rows_items:
                    if r_item.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                        d_d[ky] = dict(
                                        label=d_elements['nameset'][ky]['title']
                                        ,value=r_item.mr_resource_items.mr_text
                                    )
                        break

            d_d['role'] = 'resource'
            d_d['folders'] = __get_l_d_folders(dbm, row_resource.id, d_d['contains'])

        # アノテーションはここでは不要
        #d_d['annotes'] = __get_l_d_annotes(dbm, d_d['ids'])

        l_d.append(d_d)

    return dict(title=d_d['title'].get('value',''), array=l_d)




def __get_d_resource(dbm, resourceid):
    """
    指定リソースidのメタデータのみを得る
    """
    d_d = {}

    row = dbm.mr_resources[resourceid]

    if row:

        d_elements = d__elementset(dbm, row.mr_class_id)

        rows = dbm((dbm.mr_resource_items.mr_open == True)
                    & (dbm.mr_resource_items.mr_resource_id == row.id)
                    & dbm.mr_resource_items.mr_class_element_id.belongs(d_elements['group'])
                ).select(
                    left=dbm.mr_class_elements.on(dbm.mr_resource_items.mr_class_element_id == dbm.mr_class_elements.id)
                )

        d_d['subid'] = resourceid
        d_d['ids'] = (resourceid, 1, 1, 1)

        d_d['keyset'] = d_elements['keyset']
        d_d['contains'] = set([])

        d_d['title'] = dict(label=None, value=None)
        d_d['description'] = dict(label=None, value=None)

        for ky in d_elements['keyset']:
            d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
            for r in rows:
                if r.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                    d_d[ky] = dict(
                                label=d_elements['nameset'][ky]['title']
                                ,value=r.mr_resource_items.mr_text
                            )
                    break

        d_d['role'] = 'resource'

    return d_d




def d_d_file(dbm, fileid, resourceid=1):
    """
    指定ファイルidの内容を得る
    """

    # コンテナ
    l_d = []

    # このファイル
    row = dbm.mr_files[fileid]

    if row:

        # 定義名の取得
        d_d = {}

        if row.mr_folder_id > 1:
            # フォルダにリンクしている場合（優先）

            # 所属フォルダのリソース（複数可）を取得し、最初のリソースのみに対応（暫定）
            if resourceid == 1:
                rf = dbm.mr_folders[row.mr_folder_id]
                if rf and len(rf.mr_resource_ids)>0:
                    resourceid = rf.mr_resource_ids[0]

            # 現状では外部リソースidを利用することになるため回避
            #da = d_d_folder(dbm, row.mr_folder_id, resourceid)
            #if len(da['array']) > 0:
            #    resourceid = da['array'][0]["ids"][0]

            # 外部指定のリソースidを使用
            d_meta = __get_d_resource(dbm, resourceid)

            for ky in d_meta:
                if ky in d_meta['keyset']:
                    d_d[ky] = d_meta[ky]

            # 定義名の取得
            d_d['subid'] = fileid
            d_d['ids'] = (resourceid, row.mr_folder_id, fileid, 1)

            d_d['keyset'] = []
            d_d['contains'] = set([])

            d_d['role'] = 'file'
            d_d['files'] = [__get_l_d_file(dbm, row, row.mr_folder_id, resourceid, d_d['contains'])]
            d_d['annotes'] = __get_l_d_annotes(dbm, d_d['ids'])

            l_d.append(d_d)

        elif len(row.mr_resource_ids)>0:
            # リソースにリンクしている場合

            # このファイルのリソース（複数可）を取得し、最初のリソースのみに対応（暫定）
            if resourceid == 1:
                resourceid = row.mr_resource_ids[0]

            # 外部指定のリソースidを使用
            d_meta = __get_d_resource(dbm, resourceid)

            for ky in d_meta:
                if ky in d_meta['keyset']:
                    d_d[ky] = d_meta[ky]

            d_d['subid'] = fileid
            d_d['ids'] = (resourceid, 1, fileid, 1)

            d_d['keyset'] = []
            d_d['contains'] = set([])

            d_d['role'] = 'file'
            d_d['files'] = [__get_l_d_file(dbm, row, 1, resourceid, d_d['contains'])]
            d_d['annotes'] = __get_l_d_annotes(dbm, d_d['ids'])

            l_d.append(d_d)


    #return dict(title=d_d['title']['value'] or '' ,array=l_d)
    return dict(array=l_d)


def d_d_annote(dbm, annoteid):
    """
    指定アノテーションidの内容を得る
    """

    # コンテナ
    l_d = []

    # このアノテーション
    row = dbm.mr_annotes[annoteid]

    if row:

        # このアノテーションのリソースの取得
        r_resource = dbm.mr_resources[row.mr_resource_id]

        # このアノテーションのファイルの取得
        r_file = dbm.mr_files[row.mr_file_id]

        # 定義名の取得
        d_d = {}
        d_d['subid'] = row.id
        d_d['ids'] = (row.mr_resource_id, 1, r_file.id, row.id)
        d_d['role'] = 'annote'

        d_d['keyset'] = []
        d_d['contains'] = set([])
        d_d['title'] = dict(label='slug', value=row.mr_name)
        d_d['description'] = dict(label='text', value=row.mr_text)

        d_d['files'] = [__get_l_d_file(dbm, r_file, 1, row.mr_resource_id, d_d['contains'])]
        d_d['annotes'] = []

        d_d['area'] = dict(label='area', value=row.mr_annote_area)
        d_d['data'] = dict(label='data', value=row.mr_annote_data)
        d_d['rotation'] = dict(label='rotation', value=row.mr_annote_rotation)

        l_d.append(d_d)

    return dict(title=d_d['title']['value'] or None ,array=l_d)




def __get_l_d_folder_files(dbm, folderid, resourceid, l_contains, a_offset=0, a_limit=20):
    """
    指定フォルダidのファイル一覧を得る
    """

    l_d_f = []

    if folderid > 1:

        frows = dbm((dbm.mr_files.mr_open == True)
                    & (dbm.mr_files.mr_folder_id == folderid)
                ).select(
                    orderby=[dbm.mr_files.mr_order,dbm.mr_files.id]
                )

        if frows:

            for fr in frows:

                d_f = __get_l_d_file(dbm, fr, folderid, resourceid, l_contains)

                l_d_f.append(d_f)
                l_contains.add(d_f['format'])

    return l_d_f


"""
def d_d_file(dbm, idset):

    resourceid, folderid, fileid, annoteid = idset

    l_contains = set([])

    l_d = __get_l_d_file(dbm, fileid, folderid, resourceid, l_contains):

    return dict(format=l_contains, array=l_d)
"""

def __get_l_d_file(dbm, r, folderid, resourceid, l_contains, a_offset=0, a_limit=20):
    """
    指定ファイルidの内容を得る
    """

    d_f = {}

    if r:

            d_f['subid'] = r.id
            d_f['ids'] = (resourceid, folderid, r.id, 1)
            d_f['role'] = 'file'

            # Metadata
            d_f['title'] = r.mr_title
            if len(r.mr_title) == 0:
                d_f['title'] = r.mr_original_file_name

            # Thumbnail
            d_f['thumbnail'] = None #XML('&nbsp;')
            if r.mr_ims is not None and '.jpg' in r.mr_ims:
                d_f['thumbnail'] = [r.mr_subfolder, r.mr_ims]

            # Snap image
            d_f['snapimage'] = None #XML('&nbsp;')
            if r.mr_imm is not None and '.jpg' in r.mr_imm:
                d_f['snapimage'] = [r.mr_subfolder, r.mr_imm]

            # Link
            d_f['href'] = [r.mr_subfolder,r.mr_iml,r.mr_public_file_format]
            if r.mr_public_file_format == 'pdf':
                d_f['href'] = [r.mr_subfolder, r.mr_iml, r.mr_public_file_format]

            # Other matadata
            d_f['identifier'] = r.mr_identifier
            d_f['width'] = r.mr_imlw
            d_f['height'] = r.mr_imlh
            d_f['format'] = r.mr_public_file_format
            #d_f['date'] = r.mr_create_date
            #d_f['note'] = r.mr_note
            #
            d_f["annotes"] = __get_l_d_annotes(dbm, d_f['ids'])

            l_contains.add(d_f['format'])

    return d_f




def __get_l_d_folders(dbm, resourceid, l_contains, a_offset=0, a_limit=20):
    """
    指定リソースidのフォルダ一覧を得る
    """
    # フォルダのコンテナ
    l_f = []

    rows = dbm((dbm.mr_folders.mr_open == True)
                & (dbm.mr_folders.mr_resource_ids.contains(resourceid))
            ).select(
                orderby=[dbm.mr_folders.mr_order,~dbm.mr_folders.id]
            )

    if rows:

        for r in rows:
            # r is a folder

            l_d_f = []
            l_d_f_tops = []

            d = {'subid':r.id,'ids':(resourceid, r.id, 0, 0),'role':'folder','files':l_d_f,'tops':l_d_f_tops}

            frows = dbm((dbm.mr_files.mr_open == True)
                        & (dbm.mr_files.mr_folder_id == r.id)
                    ).select(
                        orderby=[dbm.mr_files.mr_order,dbm.mr_files.id]
                    )

            if frows:

                for fr in frows:
                    # fr is a file in the folder

                    d_f = __get_l_d_file(dbm, fr, r.id, resourceid, l_contains)
                    d_f['ids'] = (resourceid, r.id, fr.id, 0)

                    l_d_f.append(d_f)

                    if r.mr_delegates:
                        if fr.id in r.mr_delegates:
                            l_d_f_tops.append(len(l_d_f)-1)
                    elif len(l_d_f_tops) < 1:
                        l_d_f_tops.append(0)

            l_f.append(d)


    # File

    res_l_f = get_l_d_files(dbm, resourceid, l_contains)

    if len(res_l_f) > 0:
        l_f.extend(res_l_f)

    return l_f




def d_d_folder(dbm, folderid, resourceid=1):
    """
    指定フォルダidの内容を得る
    """

    # コンテナ
    l_d = []

    # このフォルダ
    row = dbm.mr_folders[folderid]

    if row:

        # このフォルダのリソース（複数可）の取得
        if len(row.mr_resource_ids)>0:

            # このファイルのリソース（複数可）を取得し、最初のリソースのみに対応（暫定）
            if resourceid == 1:
                resourceid = row.mr_resource_ids[0]

            # 外部指定のリソースidを使用
            d_meta = __get_d_resource(dbm, resourceid)

            # 定義名の取得
            d_d = {}

            for ky in d_meta:
                if ky in d_meta['keyset']:
                    d_d[ky] = d_meta[ky]

            d_d['subid'] = folderid
            d_d['ids'] = (resourceid, folderid, 1, 1)

            d_d['keyset'] = []
            d_d['contains'] = set([])

            d_d['role'] = 'folder'
            d_d['files'] = __get_l_d_folder_files(dbm, folderid, resourceid, d_d['contains'])
            d_d['annotes'] = []
            for r in d_d['files']:
                d_d['annotes'].append(r["annotes"])

            l_d.append(d_d)

    #return dict(title=d_d['title']['value'] or '' ,array=l_d)
    return dict(array=l_d)


def get_l_d_files(dbm, resourceid, l_contains, a_offset=0, a_limit=20):
    """
    指定リソースidのファイル一覧を得る
    """

    # リソースに関連するファイルのコンテナ
    l_f = []

    frows = dbm((dbm.mr_files.mr_open == True)
                & (dbm.mr_files.mr_resource_ids.contains(resourceid))
                & (dbm.mr_files.mr_folder_id == 1)
            ).select(
                orderby=[dbm.mr_files.mr_order,~dbm.mr_files.id]
            )

    if frows:

        for rf in frows:

            d_f = __get_l_d_file(dbm, rf, 1, resourceid, l_contains)
            d_f['ids'] = (resourceid, 1, rf.id, 1)

            d = {'subid':rf.id,'ids':(resourceid, 0, rf.id, 0),'role':'file','files':[d_f],'tops':[0]}

            l_f.append(d)

    return l_f




def __get_l_d_folders_air(jsondata, resourceid, l_contains, a_offset=0, a_limit=20):
    """
    マニフェストからフォルダ一覧を得る
    """

    # リソースに関連するフォルダのコンテナ
    l_f = []

    # parse of Manifest
    if ('@type' in jsondata) and (jsondata['@type'] == 'sc:Manifest'):

        d_d = {}

        d_d['airkeyvalue'] = {}

        if 'label' in jsondata:
            d_d['title'] = jsondata.get('label','')
        if 'description' in jsondata:
            d_d['description'] = jsondata.get('description','')
        if 'license' in jsondata:
            d_d['license'] = jsondata.get('license',None)
        if 'attribution' in jsondata:
            d_d['attribution'] = jsondata.get('attribution',None)
        if 'logo' in jsondata:
            d_d['logo'] = jsondata.get('logo',None)

        if 'metadata' in jsondata:
            for d1 in jsondata['metadata']:
                if isinstance(d1['value'],list):
                    for d2 in d1['value']:
                        if not d1['label'] in d_d:
                            d_d[d1['label']] = [d2['value']['@value']]
                        else:
                            d_d[d1['label']].append(d2['value']['@value'])
                else:
                    # string
                    if not d1['label'] in d_d:
                        d_d[d1['label']] = d1['value']
                    else:
                        d_d[d1['label']].append(d1['value'])

                if d1['label'] not in d_d['airkeyvalue']:
                    d_d['airkeyvalue'][d1['label']] = d1['label']

        if 'sequences' in jsondata:
            for d1 in jsondata['sequences']:

                l_d_f = []

                d = {'subid':0,'ids':(resourceid, 0, 0, 0),'role':'air-folder','metadata':d_d,'files':l_d_f,'tops':[0]}

                if ('@type' in d1) and (d1['@type'] == 'sc:Sequence') and ('canvases' in d1):
                    for d2 in d1['canvases']:

                        if ('@type' in d2) and (d2['@type'] == 'sc:Canvas'):

                            d_f = {}

                            d_f['subid'] = resourceid
                            #d_f['ids'] = (0, r.id, 0, 0)
                            d_f['role'] = 'airfile'
                            #d_f['thumbnailpath'] = None

                            # Canvas
                            if 'label' in d2:
                                d_f['title'] = d2['label']
                            if 'width' in d2:
                                d_f['width'] = d2.get('width',0)
                            if 'height' in d2:
                                d_f['height'] = d2.get('height',0)
                            if 'thumbnail' in d2:
                                d3 = d2['thumbnail']
                                if ('@type' in d3) and (d3['@type'] == 'dctypes:Image'):
                                    if '@id' in d3:
                                        d_f['thumbnailpath'] = d3['@id']
                                    if 'format' in d3:
                                        d_f['thumbnailformat'] = d3['format']
                                    if 'width' in d3:
                                        d_f['thumbnailwidth'] = d3.get('width',0)
                                    if 'height' in d3:
                                        d_f['thumbnailheight'] = d3.get('height',0)

                            if 'images' in d2:
                                for d3 in d2['images'][:1]:
                                    if ('@type' in d3) and (d3['@type'] == 'oa:Annotation'):
                                        if 'resource' in d3:
                                            d4 = d3['resource']
                                            if ('@type' in d4) and (d4['@type'] == 'dctypes:Image'):
                                                if '@id' in d4:
                                                    d_f['path'] = d4['@id']
                                                if 'format' in d4:
                                                    d_f['format'] = d4['format']

                                                if 'service' in d4:
                                                    d5 = d4['service']
                                                    if '@id' in d5:
                                                        d_f['path'] = d5['@id']

                            #if 'otherContent' in d2:
                            #    for d3 in d2['otherContent']:
                            #        if ('@type' in d3) and (d3['@type'] == 'sc:AnnotationList'):
                            #            d_f['path'] = d3['@id']

                            # Snap image
                            d_f['thumbnail'] = [d_f['thumbnailpath']]
                            d_f['snapimage'] = [d_f['thumbnailpath']]

                            # Link
                            urls = urllib.parse.urlparse(jsondata['@id'])
                            d_f['href'] = [urls.scheme,urls.netloc,urls.path]

                            # Other matadata
                            d_f['identifier'] = None

                            l_contains.add(d_f['format'])

                            d_f['ids'] = (resourceid, 0, 0, 0)

                            l_d_f.append(d_f)

                l_f.append(d)

    return l_f




def d_d_annotes(dbm, idset, a_offset=0, a_limit=20):

    l_d_a = __get_l_d_annotes(dbm, idset)

    return dict(length=len(l_d_a),array=l_d_a)


def __get_l_d_annotes(dbm, idset, a_offset=0, a_limit=20):
    """
    List of annotations
    """

    # リソースに関連するアノテーションのコンテナ
    l_d_a = []

    resourceid, folderid, fileid, annoteid = idset

    rows = dbm((dbm.mr_annotes.mr_open == True)
                & (dbm.mr_annotes.mr_resource_id == resourceid)
                & (dbm.mr_annotes.mr_file_id == fileid)
            ).select(
                orderby=[dbm.mr_annotes.mr_order,~dbm.mr_annotes.id]
            )

    if rows:

        for r in rows:

            # アノテーションがあるファイル
            rf = dbm.mr_files[r.mr_file_id]

            d_a = {}

            d_a['subid'] = r.id

            d_a['ids'] = (resourceid, folderid, rf.id, r.id)
            d_a['role'] = 'annote'

            # Metadata

            if len(r.mr_text) > 0:
                d_a['title'] = r.mr_text[:]
            elif len(r.mr_name) > 0:
                d_a['title'] = r.mr_name[:]
            else:
                d_a['title'] = '(Undefined)'

            # Thumbnail
            d_a['thumbnail'] = XML('&nbsp;')
            if rf.mr_ims is not None and '.jpg' in rf.mr_ims:
                d_a['thumbnail'] = [rf.mr_subfolder, rf.mr_ims]

            # Snap image
            d_a['snapimage'] = XML('&nbsp;')
            if rf.mr_imm is not None and '.jpg' in rf.mr_imm:
                d_a['snapimage'] = [rf.mr_subfolder, rf.mr_imm]

            # Link
            d_a['href'] = [rf.mr_subfolder,rf.mr_iml,rf.mr_public_file_format]
            #if rf.mr_public_file_format == 'pdf':
            #    d_a['href'] = [rf.mr_subfolder, rf.mr_iml, rf.mr_public_file_format]

            # Other matadata
            d_a['identifier'] = rf.mr_identifier
            d_a['width'] = rf.mr_imlw
            d_a['height'] = rf.mr_imlh
            d_a['format'] = rf.mr_public_file_format

            d_a['annotearea'] = r.mr_annote_area    # list
            d_a['annotetype'] = r.mr_annote_type    # str
            d_a['annoterotation'] = r.mr_annote_rotation    # double
            d_a['annotedata'] = r.mr_annote_data    # list
            #d_a['date'] = rf.mr_create_date
            #d_a['note'] = rf.mr_note

            l_d_a.append(d_a)

    return l_d_a
