# -*- coding: utf-8 -*-

#########################################################################
## - Application Name: Machikane-Red
## - Version: 3.1906r2
## - Date: 2019-06-10
## - Copyright: (c) 2018-2019 Mitsuhiro Tsuda.
## - License: Machikane-Red (version 3.1906) is released
##            under the GNU General Public License (GPL).
##            See the copyright notice LICENSE.
#########################################################################

from gluon.html import *
import re
import json
import urllib


def d__elementset(dbm, classid=1, managed=False):
    """
    要素セット辞書を返す
    {
        '*class_id': {
            'group': [
                *class_element_id
            ]
            ,'keyset': [
                *class_element_mr_name
            ]
            ,'nameset': {
                '*class_element_mr_name': {
                    'id':     *class_element_id
                    'title':   *class_element_name (title)
                }
            }
        }
    }

    @managed: 管理用（編集者・管理者のみTrue）
    """

    # 要素セット辞書
    d_elements = {}
    for row in dbm(dbm.mr_classes).select(orderby=[dbm.mr_classes.mr_order,dbm.mr_classes.id]):
        # クラス要素
        rows = dbm(
                dbm.mr_class_elements.mr_class_id == row.id
            ).select(
                orderby=[
                    dbm.mr_class_elements.mr_order
                    ,dbm.mr_class_elements.id
                ]
            )

        # 定義名をキーとする要素名と要素IDの辞書
        d_meta = {}
        for r in rows:
            if r.mr_name is not None and not r.mr_name in d_meta:
                # set first element
                d_meta[r.mr_name] = dict(id = r.id, title = r.mr_title)

        if managed:
            l_keysets = [r.mr_name for r in rows if r.mr_open]
        else:
            l_keysets = [r.mr_name for r in rows if r.mr_open and (r.mr_rank < 3)]

        d_elements[row.id] = dict(
            target = row.mr_target
            ,title = row.mr_title
            ,description = row.mr_description
            ,group = [r.id for r in rows] #クラス要素IDの配列
            ,smkeyset = [r.mr_name for r in rows if r.mr_open and (r.mr_rank < 2)]
            ,keyset = l_keysets
            ,nameset = d_meta
        )
    try:
        return d_elements if classid == 0 else d_elements[classid]
    except:
        return None


def __get_d_external_iiif_manifest(uri, version='2'):
    """
    外部マニフェストを読み込み、オブジェクトを得る
    """

    # オブジェクト
    d_d = None

    if uri and re.fullmatch('^https?:\/\/[a-zA-Z0-9\.\/\-_]+(\.json)*$',uri.strip()):

        req = urllib.request.Request(uri)

        try:
            res = urllib.request.urlopen(req).read()
            manifest = json.loads(res.decode('utf-8'))
            if '@context' in manifest and manifest['@context'] == 'http://iiif.io/api/presentation/2/context.json':

                # オブジェクト
                d_d = {}

                d_d['manifest'] = uri
                d_d['content'] = manifest
                d_d['title'] = [manifest.get('label','')]
                d_d['description'] = [manifest.get('description','')]
                d_d['license'] = manifest.get('license','')
                d_d['attribution'] = manifest.get('attribution','')
                #d_d['logo'] = manifest.get('logo',None)

                if 'metadata' in manifest:
                    for d1 in manifest['metadata']:
                        if isinstance(d1['value'],list):
                            for d2 in d1['value']:
                                if not d1['label'] in d_d:
                                    d_d[d1['label']] = [d2['@value']]
                                else:
                                    d_d[d1['label']].append(d2['@value'])
                        else:
                            # string
                            if not d1['label'] in d_d:
                                d_d[d1['label']] = [d1['value']]
                            else:
                                d_d[d1['label']].append(d1['value'])

        except:
            pass

    return d_d


def d_d_page(dbm, pageid, managed=False, a_offset=0, a_limit=20):
    """
    指定ページidのページセクション一覧をdict形式で得る
    """

    title = dbm.mr_pages[pageid]['mr_title'] or None

    # コンテナ
    l_d = []

    rows = dbm((dbm.mr_sections.mr_open == True)
                & (dbm.mr_sections.mr_page_id == pageid)
            ).select(
                orderby=[dbm.mr_sections.mr_order,~dbm.mr_sections.id]
            )

    for row in rows:

        # 各リソースが依存するクラス要素セットの取得
        d_elements = d__elementset(dbm, row.mr_class_id, managed)

        # 各リソースのアイテムの取得
        rows2 = dbm(
                    (dbm.mr_section_items.mr_section_id == row.id)
                    & dbm.mr_section_items.mr_class_element_id.belongs(d_elements['group'])
                    #& (dbm.mr_section_items.mr_open == True)
                ).select(
                    #left=dbm.mr_class_elements.on(dbm.mr_section_items.mr_class_element_id == dbm.mr_class_elements.id)
                )

        # 定義名の取得
        d_d = {}
        d_d['subid'] = row.id
        d_d['ids'] = (1, 1, 1, 1)
        d_d['keyset'] = d_elements['keyset']

        for r2 in rows2:
            for ky in d_elements['keyset']:
                if r2.mr_class_element_id == d_elements['nameset'][ky]['id']:
                    if ky in ('creator','date'):
                        d_d[ky] = [dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=r2.mr_text
                                )]
                    #elif ky in ('created','updated'):
                    #    d_d[ky] = dict(
                    #                label=d_elements['nameset'][ky]['title']
                    #                ,value=r2.mr_text
                    #            )
                    else:
                        if ky not in d_d:
                            d_d[ky] = [dict(
                                        label=d_elements['nameset'][ky]['title']
                                        ,value=r2.mr_text
                                    )]
                        else:
                            d_d[ky].append(
                                        dict(
                                            label=d_elements['nameset'][ky]['title']
                                            ,value=r2.mr_text
                                        )
                                    )
                    break

        for ky in d_elements['keyset']:
            if ky not in d_d:
                d_d[ky] = [dict(label=ky.capitalize(), value='')]

        l_d.append(d_d)

    return dict(title=title, array=l_d)


def d_d_finder(dbm, collectionid, q=[], managed=False, a_offset=0, a_limit=20):
    """
    指定コレクションid および/あるいは サーチ結果idのリソース一覧をdict形式で得る
    サーチ対応, d_d_collection に代わる
    """

    # 検索ランク
    a_search_rank = 2  # 1: 可(デフォルト)
    if managed:
        a_search_rank = 3  # 1,2: 管理者の検索範囲

    if type(q) is not list:
        q = []

    # 検索コンテナ
    f_s = set([])

    # リソースのメタデータを検索する
    finds_resource_items = dbm((dbm.mr_resource_items.mr_open == True)
                & (dbm.mr_class_elements.mr_search_rank < a_search_rank)
                & dbm.mr_resource_items.mr_text.contains(q, all=True)
            ).select(
                left=dbm.mr_class_elements.on(dbm.mr_resource_items.mr_class_element_id == dbm.mr_class_elements.id)
            )

    if finds_resource_items:
        for find_resource in finds_resource_items:
            f_s.add(find_resource.mr_resource_items.mr_resource_id)
            pass

    # アノテーションのメタデータを検索する
    finds_annotes = dbm((dbm.mr_annotes.mr_open == True)
                & dbm.mr_annotes.mr_text.contains(q, all=True)
            ).select()

    if finds_annotes:
        for find_annote in finds_annotes:
            f_s.add(find_annote.mr_resource_id)

    #f_l = [v for v in f_s]

    # コンテナ
    l_d = []

    if collectionid > 1:
        rows_resources = dbm((dbm.mr_resources.mr_open==True)
                    & dbm.mr_resources.mr_collection_ids.contains(collectionid)
                    & dbm.mr_resources.id.belongs(f_s)
                ).select(
                    #limitby=(a_offset,a_limit)
                    #,orderby_on_limitby=False
                    #orderby=[dbm.mr_resources.mr_order,dbm.mr_resources.id]
                )
    else:
        rows_resources = dbm((dbm.mr_resources.mr_open==True)
                    & dbm.mr_resources.id.belongs(f_s)
                ).select(
                    #limitby=(a_offset,a_limit)
                    #,orderby_on_limitby=False
                    #,orderby=[dbm.mr_resources.mr_order,~dbm.mr_resources.id]
                )

    for row_resource in rows_resources:
        # row is a resource

        # リソースのクラスに属する要素セットを得る
        d_elements = d__elementset(dbm, row_resource.mr_class_id)

        # リソースのメタデータを得る
        count_items = dbm((dbm.mr_resource_items.mr_open == True)
                    & (dbm.mr_resource_items.mr_resource_id == row_resource.id)
                    #& dbm.mr_resource_items.mr_text.contains(q, all=True)
                ).count()

        if count_items == 0: continue

        rows_items = dbm((dbm.mr_resource_items.mr_open == True)
                    & (dbm.mr_resource_items.mr_resource_id == row_resource.id)
                    & dbm.mr_resource_items.mr_class_element_id.belongs(d_elements['group'])
                    #& dbm.mr_resource_items.mr_text.contains(q, all=True)
                ).select(
                    left=dbm.mr_class_elements.on(dbm.mr_resource_items.mr_class_element_id == dbm.mr_class_elements.id)
                    ,limitby=(a_offset,a_limit)
                    ,orderby_on_limitby=False
                    ,orderby=[dbm.mr_resource_items.mr_order,dbm.mr_resource_items.id]
                )

        # オブジェクト
        d_d = {}

        d_d['subid'] = row_resource.id
        d_d['ids'] = (row_resource.id, 1, 1, 1)
        d_d['keyset'] = d_elements['smkeyset']
        d_d['contains'] = set([])
        d_d['title'] = dict(label=None, value=None)
        d_d['description'] = dict(label=None, value=None)

        d_d['externalkeyvalue'] = {}

        # 外部マニフェストの解析
        if row_resource.mr_manifest_id:
            var_manifest = row_resource.mr_manifest_id
        else:
            # Deprecated
            var_manifest = row_resource.mr_manifest_link
        external_metad = __get_d_external_iiif_manifest(var_manifest)

        if external_metad:

            # 外部リソース
            for ky in d_elements['smkeyset']:

                _keycomp = False
                d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
                for r_item in rows_items:
                    if r_item.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=r_item.mr_resource_items.mr_text
                                )
                        _keycomp = True
                        break

                if ky in external_metad:
                    if (_keycomp and (ky not in ('title','description')) and ((ky in d_d) and (len(d_d[ky]['value'])==0))) or (not _keycomp):
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=external_metad[ky]
                                )

            for ky in external_metad:
                if ky not in d_d['externalkeyvalue']:
                    if ky in ('content',):
                        pass
                    else:
                        d_d['externalkeyvalue'][ky] = external_metad[ky]

            d_d['role'] = 'externalresource'
            d_d['folders'] = __get_l_d_folders_external(external_metad['content'], row_resource.id, d_d['contains'])

        else:

            for ky in d_elements['smkeyset']:
                d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
                for r_item in rows_items:
                    if r_item.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=r_item.mr_resource_items.mr_text
                                )
                        break

            d_d['role'] = 'resource'
            d_d['folders'] = __get_l_d_folders(dbm, row_resource.id, d_d['contains'])

        # areaアノテーション表示のために必要
        d_d['annotes'] = __get_l_d_annotes(dbm, d_d['ids'])

        l_d.append(d_d)

    return dict(title=None, array=l_d)


def d_d_collection(dbm, collectionid, q=[], a_offset=0, a_limit=20):
    """
    指定コレクションidのリソース一覧をdict形式で得る
    Deprecated(3.1906)
    """

    if type(q) is not list:
        q = []

    # コンテナ
    l_d = []

    if collectionid > 1:
        rows_resources = dbm((dbm.mr_resources.mr_open==True)
                    & dbm.mr_resources.mr_collection_ids.contains(collectionid)
                ).select(
                    limitby=(a_offset,a_limit)
                    ,orderby_on_limitby=False
                    ,orderby=[dbm.mr_resources.mr_order,dbm.mr_resources.id]
                )
        title = dbm.mr_collections[collectionid]['mr_title'] or None

    else:
        rows_resources = dbm((dbm.mr_resources.mr_open==True)
                    & (dbm.mr_resources.id>0)
                ).select(
                    limitby=(a_offset,a_limit)
                    ,orderby_on_limitby=False
                    ,orderby=[dbm.mr_resources.mr_order,~dbm.mr_resources.id]
                )

    for row_resource in rows_resources:
        # row is a resource

        # リソースのクラスに属する要素セットを得る
        d_elements = d__elementset(dbm, row_resource.mr_class_id)

        # リソースのメタデータを得る
        count_items = dbm((dbm.mr_resource_items.mr_open == True)
                    & (dbm.mr_resource_items.mr_resource_id == row_resource.id)
                    & dbm.mr_resource_items.mr_text.contains(q, all=True)
                ).count()

        if count_items == 0: continue

        rows_items = dbm((dbm.mr_resource_items.mr_open == True)
                    & (dbm.mr_resource_items.mr_resource_id == row_resource.id)
                    & dbm.mr_resource_items.mr_class_element_id.belongs(d_elements['group'])
                    & dbm.mr_resource_items.mr_text.contains(q, all=True)
                ).select(
                    left=dbm.mr_class_elements.on(dbm.mr_resource_items.mr_class_element_id == dbm.mr_class_elements.id)
                    ,limitby=(a_offset,a_limit)
                    ,orderby_on_limitby=False
                    ,orderby=[dbm.mr_resource_items.mr_order,dbm.mr_resource_items.id]
                )

        # オブジェクト
        d_d = {}

        d_d['subid'] = row_resource.id
        d_d['ids'] = (row_resource.id, 1, 1, 1)
        d_d['keyset'] = d_elements['smkeyset']
        d_d['contains'] = set([])
        d_d['title'] = dict(label=None, value=None)
        d_d['description'] = dict(label=None, value=None)

        d_d['externalkeyvalue'] = {}

        # 外部マニフェストの解析
        if row_resource.mr_manifest_id:
            var_manifest = row_resource.mr_manifest_id
        else:
            # Deprecated
            var_manifest = row_resource.mr_manifest_link
        external_metad = __get_d_external_iiif_manifest(var_manifest)

        if external_metad:

            # 外部リソース
            for ky in d_elements['smkeyset']:

                _keycomp = False
                d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
                for r_item in rows_items:
                    if r_item.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=r_item.mr_resource_items.mr_text
                                )
                        _keycomp = True
                        break

                if ky in external_metad:
                    if (_keycomp and (ky not in ('title','description')) and ((ky in d_d) and (len(d_d[ky]['value'])==0))) or (not _keycomp):
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=external_metad[ky]
                                )

            for ky in external_metad:
                if ky not in d_d['externalkeyvalue']:
                    if ky in ('content',):
                        pass
                    else:
                        d_d['externalkeyvalue'][ky] = external_metad[ky]

            d_d['role'] = 'externalresource'
            d_d['folders'] = __get_l_d_folders_external(external_metad['content'], row_resource.id, d_d['contains'])

        else:

            for ky in d_elements['smkeyset']:
                d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
                for r_item in rows_items:
                    if r_item.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=r_item.mr_resource_items.mr_text
                                )
                        break

            d_d['role'] = 'resource'
            d_d['folders'] = __get_l_d_folders(dbm, row_resource.id, d_d['contains'])

        # areaアノテーション表示のために必要
        d_d['annotes'] = __get_l_d_annotes(dbm, d_d['ids'])

        l_d.append(d_d)

    return dict(title=title, array=l_d)


def d_d_resource(dbm, resourceid, managed=False, a_offset=0, a_limit=20):
    """
    指定リソースidの内容をdict形式で得る
    """

    # コンテナ
    l_d = []

    # このリソース
    row_resource = dbm.mr_resources[resourceid]

    if row_resource:

        d_elements = d__elementset(dbm, row_resource.mr_class_id, managed)

        # このリソースのアイテムの取得
        rows_items = dbm((dbm.mr_resource_items.mr_open == True)
                    & (dbm.mr_resource_items.mr_resource_id == row_resource.id)
                    & dbm.mr_resource_items.mr_class_element_id.belongs(d_elements['group'])
                ).select(
                    left=dbm.mr_class_elements.on(dbm.mr_resource_items.mr_class_element_id == dbm.mr_class_elements.id)
                    ,limitby=(a_offset,a_limit)
                    ,orderby_on_limitby=False
                    ,orderby=[dbm.mr_resource_items.mr_order,dbm.mr_resource_items.id]
                )

        # オブジェクト
        d_d = {}

        d_d['subid'] = resourceid
        d_d['ids'] = (resourceid, 1, 1, 1)
        d_d['keyset'] = d_elements['keyset']
        d_d['contains'] = set([])
        d_d['title'] = dict(label=None, value=None)
        d_d['description'] = dict(label=None, value=None)

        d_d['externalkeyvalue'] = {}

        # 外部マニフェストの解析
        if row_resource.mr_manifest_id:
            var_manifest = row_resource.mr_manifest_id
        else:
            # Deprecated
            var_manifest = row_resource.mr_manifest_link
        external_metad = __get_d_external_iiif_manifest(var_manifest)

        if external_metad:

            # 外部リソース
            for ky in d_elements['keyset']:

                _keycomp = False
                d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
                for r_item in rows_items:
                    if r_item.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                        d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=r_item.mr_resource_items.mr_text
                                )
                        _keycomp = True
                        break

                if ky in external_metad:
                    #if (_keycomp and (ky not in ('title','description')) and ((ky in d_d) and (d_d[ky]['value'] is not None) and (len(d_d[ky]['value'])==0))) or (not _keycomp):
                    if (ky not in ('title','description')) and (ky in d_d):
                        if (external_metad[ky] is not None) and len(''.join(external_metad[ky]).strip())>0:
                            # 外部Manifestに値が存在しないか空白の場合、DB内の記載を維持する
                            # 外部Manifestに値が存在する場合は、強制的に書き換える
                            d_d[ky] = dict(
                                    label=d_elements['nameset'][ky]['title']
                                    ,value=external_metad[ky]   #list
                                )

            for ky in external_metad:
                if ky not in d_d['externalkeyvalue']:
                    if ky in ('content',):
                        pass
                    else:
                        d_d['externalkeyvalue'][ky] = external_metad[ky]

            d_d['role'] = 'externalresource'
            d_d['folders'] = __get_l_d_folders_external(external_metad['content'], row_resource.id, d_d['contains'])

        else:

            for ky in d_elements['keyset']:
                d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
                for r_item in rows_items:
                    if r_item.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                        d_d[ky] = dict(
                                        label=d_elements['nameset'][ky]['title']
                                        ,value=r_item.mr_resource_items.mr_text
                                    )
                        break

            d_d['role'] = 'resource'
            d_d['folders'] = __get_l_d_folders(dbm, row_resource.id, d_d['contains'])

        # アノテーションはここでは不要
        #d_d['annotes'] = __get_l_d_annotes(dbm, d_d['ids'])

        l_d.append(d_d)

    return dict(title=d_d['title'].get('value',''), array=l_d)


def __get_d_resource_metadata(dbm, resourceid, managed=False):
    """
    指定リソースidのメタデータを得る
    """

    # オブジェクト
    d_d = {}

    row_resource = dbm.mr_resources[resourceid]

    if row_resource:

        d_elements = d__elementset(dbm, row_resource.mr_class_id, managed)

        rows_items = dbm((dbm.mr_resource_items.mr_open == True)
                    & (dbm.mr_resource_items.mr_resource_id == row_resource.id)
                    & dbm.mr_resource_items.mr_class_element_id.belongs(d_elements['group'])
                ).select(
                    left=dbm.mr_class_elements.on(dbm.mr_resource_items.mr_class_element_id == dbm.mr_class_elements.id)
                )

        d_d['subid'] = resourceid
        d_d['ids'] = (resourceid, 1, 1, 1)

        d_d['keyset'] = d_elements['keyset']
        d_d['contains'] = set([])

        d_d['title'] = dict(label=None, value=None)
        d_d['description'] = dict(label=None, value=None)

        d_d['license'] = dict(label=None, value=None)
        d_d['attribution'] = dict(label=None, value=None)

        for ky in d_elements['keyset']:
            d_d[ky] = dict(label=d_elements['nameset'][ky]['title'], value=None)
            for r in rows_items:
                if r.mr_resource_items.mr_class_element_id == d_elements['nameset'][ky]['id']:
                    d_d[ky] = dict(
                                label=d_elements['nameset'][ky]['title']
                                ,value=r.mr_resource_items.mr_text
                            )
                    break

        d_d['role'] = 'resource'

    return d_d


def d_d_folder(dbm, folderid, resourceid=1):
    """
    指定フォルダidの内容を得る
    """

    # コンテナ
    l_d = []

    # このフォルダ
    row_folder = dbm.mr_folders[folderid]

    if row_folder:

        # このフォルダのリソース（複数可）の取得
        if len(row_folder.mr_resource_ids)>0:

            # このファイルのリソース（複数可）を取得し、最初のリソースのみに対応（暫定）
            if resourceid == 1:
                resourceid = row_folder.mr_resource_ids[0]

            # 外部指定のリソースidを使用
            d_meta = __get_d_resource_metadata(dbm, resourceid)

            # オブジェクト
            d_d = {}

            for ky in d_meta:
                if ky in d_meta['keyset']:
                    d_d[ky] = d_meta[ky]

            d_d['subid'] = folderid
            d_d['ids'] = (resourceid, folderid, 1, 1)
            d_d['keyset'] = []
            d_d['contains'] = set([])

            d_d['role'] = 'folder'

            d_d['files'] = []
            #d_d['annotes'] = []

            if folderid > 1:
                rows_files = dbm((dbm.mr_files.mr_open == True)
                                & (dbm.mr_files.mr_folder_id == folderid)
                            ).select(
                                orderby=[dbm.mr_files.mr_order,dbm.mr_files.id]
                            )
                if rows_files:
                    for row_file in rows_files:
                        d_f = __get_d_file(dbm, row_file, folderid, resourceid, d_d['contains'])
                        d_d['files'].append(d_f)
                        #d_d['annotes'].append(d_f["annotes"])
                        #d_d['contains'].add(d_f['format'])

            l_d.append(d_d)

    return dict(array=l_d)


def d_d_file(dbm, fileid, resourceid=1):
    """
    指定ファイルidの内容を得る
    """

    # コンテナ
    l_d = []

    # このファイル
    row_file = dbm.mr_files[fileid]

    if row_file:

        # オブジェクト
        d_d = {}

        if row_file.mr_folder_id > 1:
            # フォルダにリンクしている場合（優先）

            # 所属フォルダのリソース（複数可）を取得し、最初のリソースのみに対応（暫定）
            if resourceid == 1:
                row_folder = dbm.mr_folders[row_file.mr_folder_id]
                if row_folder and len(row_folder.mr_resource_ids)>0:
                    resourceid = row_folder.mr_resource_ids[0]

            # 現状では外部リソースidを利用することになるため回避
            #da = d_d_folder(dbm, row.mr_folder_id, resourceid)
            #if len(da['array']) > 0:
            #    resourceid = da['array'][0]["ids"][0]

            # 外部指定のリソースidを使用
            d_meta = __get_d_resource_metadata(dbm, resourceid)

            for ky in d_meta:
                if ky in d_meta['keyset']:
                    d_d[ky] = d_meta[ky]

            d_d['subid'] = fileid
            d_d['ids'] = (resourceid, row_file.mr_folder_id, fileid, 1)
            d_d['keyset'] = []
            d_d['contains'] = set([])

            d_d['role'] = 'file'

            d_file = __get_d_file(dbm, row_file, row_file.mr_folder_id, resourceid, d_d['contains'])
            d_d['files'] = [d_file]
            d_d['annotes'] = d_file['annotes']

        elif len(row_file.mr_resource_ids)>0:
            # リソースにリンクしている場合

            # このファイルのリソース（複数可）を取得し、最初のリソースのみに対応（暫定）
            if resourceid == 1:
                resourceid = row_file.mr_resource_ids[0]

            # 外部指定のリソースidを使用
            d_meta = __get_d_resource_metadata(dbm, resourceid)

            for ky in d_meta:
                if ky in d_meta['keyset']:
                    d_d[ky] = d_meta[ky]

            d_d['subid'] = fileid
            d_d['ids'] = (resourceid, 1, fileid, 1)
            d_d['keyset'] = []
            d_d['contains'] = set([])

            d_d['role'] = 'file'

            d_file = __get_d_file(dbm, row_file, 1, resourceid, d_d['contains'])
            d_d['files'] = [d_file]
            d_d['annotes'] = d_file['annotes']

        l_d.append(d_d)

    return dict(array=l_d)


def d_d_canvas(dbm, canvasid, resourceid=1):
    """
    指定キャンバスの内容を得る
    """

    # コンテナ
    l_d = []

    # オブジェクト
    d_d = {}

    d_d['subid'] = resourceid
    d_d['ids'] = (resourceid, 1, 1, 1)
    d_d['keyset'] = []
    d_d['contains'] = set([])

    d_d['role'] = 'resource'    # 注意

    # 外部指定のリソースidを使用
    d_meta = __get_d_resource_metadata(dbm, resourceid)

    for ky in d_meta:
        if ky in d_meta['keyset']:
            d_d[ky] = d_meta[ky]

    d_d['files'] = []
    d_d['annotes'] = []

    d_d['canvasid'] = canvasid

    if canvasid:
        rows_annotes = dbm((dbm.mr_annotes.mr_open == True)
                    & (dbm.mr_annotes.mr_resource_id == resourceid)
                    & (dbm.mr_annotes.mr_manifest_canvas_id==canvasid)
                ).select(
                    orderby=[dbm.mr_annotes.mr_order,dbm.mr_annotes.id]
                )
    else:
        # all annotations of the resource
        rows_annotes = dbm((dbm.mr_annotes.mr_open == True)
                    & (dbm.mr_annotes.mr_resource_id == resourceid)
                ).select(
                    orderby=[dbm.mr_annotes.mr_order,dbm.mr_annotes.id]
                )

    if rows_annotes:
        for row_annote in rows_annotes:
            d_d['annotes'].append(__get_d_annote(dbm, row_annote))

    l_d.append(d_d)

    return dict(array=l_d)


def d_d_annote(dbm, annoteid):
    """
    指定アノテーションidの内容を得る
    マニフェスト生成のみ
    """

    # コンテナ
    l_d = []

    # このアノテーション
    row_annote = dbm.mr_annotes[annoteid]

    if row_annote:

        # オブジェクト
        d_d = {}

        # 外部マニフェストの解析
        var_manifest = None
        if row_annote.mr_manifest_id and len(row_annote.mr_manifest_id.strip())>0:
            var_manifest = row_annote.mr_manifest_id
        external_metad = __get_d_external_iiif_manifest(var_manifest)

        d_d['subid'] = row_annote.id
        d_d['ids'] = (row_annote.mr_resource_id, 1, row_annote.mr_file_id, row_annote.id)
        d_d['keyset'] = []
        d_d['contains'] = set([])

        d_d['role'] = 'annote'

        d_d['name'] = row_annote.mr_name or '(Undefined)'
        d_d['text'] = row_annote.mr_text or ''

        text = row_annote.mr_text
        if text and len(text) > 0:
            texts = text.split('\r\n')
            if len(texts[0])>16:
                d_d['title'] = dict(label='title', value='%s...' % texts[0][:16])
            else:
                d_d['title'] = dict(label='title', value=texts[0])
        else:
            d_d['title'] = dict(label='title', value='(Undefined)')

        d_d['description'] = dict(label='text', value=d_d['text'])

        if external_metad:
            d_d['files'] = [__get_d_file_external(external_metad['content'], row_annote.mr_resource_id, row_annote.mr_manifest_canvas_id, d_d['contains'])]

        else:
            d_d['files'] = [__get_d_file(dbm, dbm.mr_files[row_annote.mr_file_id], 1, row_annote.mr_resource_id, d_d['contains'])]
            del d_d['files'][0]['annotes']

        d_d['area'] = row_annote.mr_annote_area
        d_d['data'] = row_annote.mr_annote_data
        d_d['rotation'] = row_annote.mr_annote_rotation

        l_d.append(d_d)

    return dict(title=None,array=l_d)


def __get_l_d_folders(dbm, resourceid, l_contains, a_offset=0, a_limit=20):
    """
    指定リソースidのフォルダ一覧（およびファイル一覧）を得る
    """

    # コンテナ（フォルダ、ファイル）、フォルダはファイルのグループ
    l_d = []

    # リソースにリンクしているフォルダを探す
    rows_folders = dbm((dbm.mr_folders.mr_open == True)
                & (dbm.mr_folders.mr_resource_ids.contains(resourceid))
            ).select(
                orderby=[dbm.mr_folders.mr_order,~dbm.mr_folders.id]
            )

    if rows_folders:

        for row_folder in rows_folders:

            # オブジェクト
            d = {}

            d['subid'] = row_folder.id
            d['ids'] = (resourceid, row_folder.id, 1, 1)

            d['role'] = 'folder'

            d['files'] = []
            d['tops'] = [0]

            rows_files = dbm((dbm.mr_files.mr_open == True)
                        & (dbm.mr_files.mr_folder_id == row_folder.id)
                    ).select(
                        orderby=[dbm.mr_files.mr_order,~dbm.mr_files.id]
                    )

            if rows_files:

                for row_file in rows_files:

                    # フォルダ下のファイル
                    d_f = __get_d_file(dbm, row_file, row_folder.id, resourceid, l_contains)
                    #d_f['ids'] = (resourceid, row_folder.id, row_file.id, 0)

                    d['files'].append(d_f)

                    if row_folder.mr_delegates:
                        if row_file.id in row_folder.mr_delegates:
                            d['tops'][0] = len(d['files'])-1

            l_d.append(d)


    # リソースにリンクしている（フォルダ下に無い）ファイルを探す
    rows_files = dbm((dbm.mr_files.mr_open == True)
                & (dbm.mr_files.mr_resource_ids.contains(resourceid))
                & (dbm.mr_files.mr_folder_id == 1)
            ).select(
                orderby=[dbm.mr_files.mr_order,~dbm.mr_files.id]
            )

    if rows_files:

        for row_file in rows_files:

            # オブジェクト
            d = {}

            d['subid'] = row_file.id
            d['ids'] = (resourceid, 1, row_file.id, 1)

            d['role'] = 'file'

            # フォルダ下に無いファイル
            d_f = __get_d_file(dbm, row_file, 1, resourceid, l_contains)
            d_f['ids'] = (resourceid, 1, row_file.id, 1)

            d['files'] = [d_f]
            d['tops'] = [0]

            l_d.append(d)

    return l_d


def __get_l_d_folders_external(jsondata, resourceid, l_contains, a_offset=0, a_limit=20):
    """
    マニフェストからフォルダ一覧を得る
    """

    # コンテナ
    l_d = []

    # マニフェストのパース
    if ('@type' in jsondata) and (jsondata['@type'] == 'sc:Manifest'):

        # メタデータのパース

        # メタデータ用オブジェクト
        d_m = {}

        d_m['externalkeyvalue'] = {}

        if 'label' in jsondata:
            d_m['title'] = [jsondata.get('label','')]
        if 'description' in jsondata:
            d_m['description'] = [jsondata.get('description','')]
        if 'license' in jsondata:
            d_m['license'] = jsondata.get('license',None)
        if 'attribution' in jsondata:
            d_m['attribution'] = jsondata.get('attribution',None)
        #if 'logo' in jsondata:
        #    d_m['logo'] = jsondata.get('logo',None)

        if 'metadata' in jsondata:
            for d1 in jsondata['metadata']:
                if isinstance(d1['value'],list):
                    for d2 in d1['value']:
                        if not d1['label'] in d_m:
                            d_m[d1['label']] = [d2['@value']]
                        else:
                            d_m[d1['label']].append(d2['@value'])
                else:
                    # string
                    if not d1['label'] in d_m:
                        d_m[d1['label']] = [d1['value']]
                    else:
                        d_m[d1['label']].append(d1['value'])

                if d1['label'] not in d_m['externalkeyvalue']:
                    d_m['externalkeyvalue'][d1['label']] = d_m[d1['label']]

        # イメージのパース

        if 'sequences' in jsondata:

            for d1 in jsondata['sequences']:

                # Sequenceのパース

                # オブジェクト
                d = {}

                d['subid'] = 1
                d['ids'] = (resourceid, 1, 1, 1)

                d['role'] = 'externalfolder'

                d['metdata'] = d_m
                d['files'] = []
                d['tops'] = [0]

                if ('@type' in d1) and (d1['@type'] == 'sc:Sequence') and ('canvases' in d1):

                    for d2 in d1['canvases']:

                        if ('@type' in d2) and (d2['@type'] == 'sc:Canvas'):

                            # Canvasのパース

                            # オブジェクト
                            d_c = {}

                            d_c['subid'] = resourceid   #注意
                            d_c['canvasid'] = d2['@id']

                            d_c['role'] = 'externalfile'

                            if 'label' in d2:
                                d_c['title'] = d2['label']
                            if 'width' in d2:
                                d_c['width'] = d2.get('width',0)
                            if 'height' in d2:
                                d_c['height'] = d2.get('height',0)

                            if 'images' in d2:
                                for d3 in d2['images'][:1]:
                                    # 制約（最初のみ）
                                    if ('@type' in d3) and (d3['@type'] == 'oa:Annotation'):
                                        if 'resource' in d3:
                                            d4 = d3['resource']
                                            if ('@type' in d4) and (d4['@type'] == 'dctypes:Image'):
                                                if '@id' in d4:
                                                    d_c['imagepath'] = d4['@id']
                                                if 'format' in d4:
                                                    d_c['format'] = d4['format']

                                                if 'service' in d4:
                                                    d5 = d4['service']
                                                    if '@id' in d5:
                                                        d_c['service'] = d5['@id']

                            if 'thumbnail' in d2:
                                d3 = d2['thumbnail']
                                if ('@type' in d3) and (d3['@type'] == 'dctypes:Image'):
                                    if '@id' in d3:
                                        d_c['thumbnailuri'] = d3['@id']
                                    if 'format' in d3:
                                        d_c['thumbnailformat'] = d3.get('format')
                                    if 'width' in d3:
                                        d_c['thumbnailwidth'] = d3.get('width',0)
                                    if 'height' in d3:
                                        d_c['thumbnailheight'] = d3.get('height',0)
                            else:
                                if 'service' in d_c:
                                    d_c['thumbnailuri'] = '%s/full/160,/0/default.jpg' % d_c['service']


                            # Snap image
                            if 'thumbnailuri' in d_c:
                                d_c['thumbnail'] = [d_c['thumbnailuri']]
                                d_c['snapimage'] = [d_c['thumbnailuri']]
                            else:
                                d_c['thumbnail'] = []
                                d_c['snapimage'] = []

                            #if 'otherContent' in d2:
                            #    for d3 in d2['otherContent']:
                            #        if ('@type' in d3) and (d3['@type'] == 'sc:AnnotationList'):
                            #            d_c['path'] = d3['@id']

                            # Link
                            urls = urllib.parse.urlparse(jsondata['@id'])
                            d_c['href'] = [urls.scheme,urls.netloc,urls.path]

                            # Other matadata
                            d_c['identifier'] = None

                            l_contains.add(d_c['format'])

                            d_c['ids'] = (resourceid, 1, 1, 1)

                            d['files'].append(d_c)

                l_d.append(d)

    return l_d


def __get_l_d_annotes(dbm, idset, a_offset=0, a_limit=20):
    """
    List of annotations
    """

    # リソースに関連するアノテーションのコンテナ
    l_d = []

    resourceid, folderid, fileid, annoteid = idset

    rows_annotes = dbm((dbm.mr_annotes.mr_open == True)
                & (dbm.mr_annotes.mr_resource_id == resourceid)
                & (dbm.mr_annotes.mr_file_id == fileid)
            ).select(
                orderby=[dbm.mr_annotes.mr_order,~dbm.mr_annotes.id]
            )

    if rows_annotes:

        for row_annote in rows_annotes:

            l_d.append(__get_d_annote(dbm, row_annote))

    return l_d


def __get_d_file(dbm, row_file, folderid, resourceid, l_contains, a_offset=0, a_limit=20):
    """
    指定ファイルidの内容を得る
    """

    # オブジェクト
    d_f = {}

    if row_file:

        d_f['subid'] = row_file.id
        d_f['ids'] = (resourceid, folderid, row_file.id, 1)

        d_f['role'] = 'file'

        # Metadata
        d_f['title'] = row_file.mr_title
        if len(row_file.mr_title) == 0:
            d_f['title'] = row_file.mr_original_file_name

        # Thumbnail
        d_f['thumbnail'] = None #XML('&nbsp;')
        if row_file.mr_ims is not None and '.jpg' in row_file.mr_ims:
            d_f['thumbnail'] = [row_file.mr_subfolder, row_file.mr_ims]

        # Snap image
        d_f['snapimage'] = None #XML('&nbsp;')
        if row_file.mr_imm is not None and '.jpg' in row_file.mr_imm:
            d_f['snapimage'] = [row_file.mr_subfolder, row_file.mr_imm]

        # Link
        d_f['href'] = [row_file.mr_subfolder,row_file.mr_iml,row_file.mr_public_file_format]
        if row_file.mr_public_file_format == 'pdf':
            d_f['href'] = [row_file.mr_subfolder, row_file.mr_iml, row_file.mr_public_file_format]

        # Other matadata
        d_f['identifier'] = row_file.mr_identifier
        d_f['width'] = row_file.mr_imlw
        d_f['height'] = row_file.mr_imlh
        d_f['format'] = row_file.mr_public_file_format
        #d_f['date'] = row_file.mr_create_date
        #d_f['note'] = row_file.mr_note

        d_f["annotes"] = __get_l_d_annotes(dbm, d_f['ids'])

        l_contains.add(d_f['format'])

    return d_f


def __get_d_file_external(jsondata, resourceid, canvasid, l_contains, a_offset=0, a_limit=20):
    """
    マニフェストからcanvasidファイルを得る
    """

    # オブジェクト
    d_c = {}

    # マニフェストのパース
    if ('@type' in jsondata) and (jsondata['@type'] == 'sc:Manifest'):

        # イメージのパース

        if 'sequences' in jsondata:

            for d1 in jsondata['sequences']:

                if ('@type' in d1) and (d1['@type'] == 'sc:Sequence') and ('canvases' in d1):

                    for d2 in d1['canvases']:

                        if ('@type' in d2) and (d2['@type'] == 'sc:Canvas'):

                            # Canvasのパース

                            if d2['@id'] and d2['@id'] == canvasid:

                                d_c['subid'] = resourceid   #注意
                                d_c['canvasid'] = d2['@id']

                                d_c['role'] = 'externalfile'

                                if 'label' in d2:
                                    d_c['title'] = d2['label']
                                if 'width' in d2:
                                    d_c['width'] = d2.get('width',0)
                                if 'height' in d2:
                                    d_c['height'] = d2.get('height',0)

                                if 'images' in d2:
                                    for d3 in d2['images'][:1]:
                                        # 制約（最初のみ）
                                        if ('@type' in d3) and (d3['@type'] == 'oa:Annotation'):
                                            if 'resource' in d3:
                                                d4 = d3['resource']
                                                if ('@type' in d4) and (d4['@type'] == 'dctypes:Image'):
                                                    if '@id' in d4:
                                                        d_c['imagepath'] = d4['@id']
                                                    if 'format' in d4:
                                                        d_c['format'] = d4['format']

                                                    if 'service' in d4:
                                                        d5 = d4['service']
                                                        if '@id' in d5:
                                                            d_c['service'] = d5['@id']

                                if 'thumbnail' in d2:
                                    d3 = d2['thumbnail']
                                    if ('@type' in d3) and (d3['@type'] == 'dctypes:Image'):
                                        if '@id' in d3:
                                            d_c['thumbnailuri'] = d3['@id']
                                        if 'format' in d3:
                                            d_c['thumbnailformat'] = d3.get('format')
                                        if 'width' in d3:
                                            d_c['thumbnailwidth'] = d3.get('width',0)
                                        if 'height' in d3:
                                            d_c['thumbnailheight'] = d3.get('height',0)
                                else:
                                    if 'service' in d_c:
                                        d_c['thumbnailuri'] = '%s/full/160,/0/default.jpg' % d_c['service']


                                # Snap image
                                if 'thumbnailuri' in d_c:
                                    d_c['thumbnail'] = [d_c['thumbnailuri']]
                                    d_c['snapimage'] = [d_c['thumbnailuri']]
                                else:
                                    d_c['thumbnail'] = []
                                    d_c['snapimage'] = []

                                #if 'otherContent' in d2:
                                #    for d3 in d2['otherContent']:
                                #        if ('@type' in d3) and (d3['@type'] == 'sc:AnnotationList'):
                                #            d_c['path'] = d3['@id']

                                # Link
                                urls = urllib.parse.urlparse(jsondata['@id'])
                                d_c['href'] = [urls.scheme,urls.netloc,urls.path]

                                # Other matadata
                                d_c['identifier'] = None

                                l_contains.add(d_c['format'])

                                d_c['ids'] = (resourceid, 1, 1, 1)

    return d_c


def __get_d_annote(dbm, row_annote):
    """
    Dict of annotations
    """

    # オブジェクト
    d_a = {}

    d_a['subid'] = row_annote.mr_file_id
    d_a['ids'] = (row_annote.mr_resource_id, 1, row_annote.mr_file_id, row_annote.id)

    if row_annote.mr_manifest_canvas_id and len(row_annote.mr_manifest_canvas_id)>0:

        d_a['subid'] = row_annote.mr_manifest_canvas_id
        d_a['ids'] = (row_annote.mr_resource_id, 1, 1, row_annote.id)

    elif row_annote.mr_file_id > 1:

        # アノテーションがあるファイル
        row_file = dbm.mr_files[row_annote.mr_file_id]

        d_a['ids'] = (row_annote.mr_resource_id, row_file.mr_folder_id, row_annote.mr_file_id, row_annote.id)

        # Thumbnail
        d_a['thumbnail'] = XML('&nbsp;')
        if row_file.mr_ims is not None and '.jpg' in row_file.mr_ims:
            d_a['thumbnail'] = [row_file.mr_subfolder, row_file.mr_ims]

        # Snap image
        d_a['snapimage'] = XML('&nbsp;')
        if row_file.mr_imm is not None and '.jpg' in row_file.mr_imm:
            d_a['snapimage'] = [row_file.mr_subfolder, row_file.mr_imm]

        # Link
        d_a['href'] = [row_file.mr_subfolder,row_file.mr_iml,row_file.mr_public_file_format]
        #if rf.mr_public_file_format == 'pdf':
        #    d_a['href'] = [rf.mr_subfolder, rf.mr_iml, rf.mr_public_file_format]

        # Other matadata
        d_a['identifier'] = row_file.mr_identifier
        d_a['width'] = row_file.mr_imlw
        d_a['height'] = row_file.mr_imlh
        d_a['format'] = row_file.mr_public_file_format

        d_a['subid'] = row_annote.id    #注意


    d_a['role'] = 'annote'

    d_a['name'] = row_annote.mr_name or '(Undefined)'
    d_a['text'] = row_annote.mr_text or ''

    # Metadata

    d_a['title'] = dict(label=None, value=None)
    d_a['description'] = dict(label=None, value=None)

    if row_annote.mr_text and len(row_annote.mr_text) > 0:
        texts = row_annote.mr_text.split('\r\n')
        if len(texts[0])>16:
            d_a['title'] = dict(label='title', value='%s...' % texts[0][:16])
        else:
            d_a['title'] = dict(label='title', value=texts[0])
    else:
        d_a['title'] = dict(label='title', value='(Undefined)')

    d_a['description'] = dict(label='text', value=row_annote.mr_text)

    d_a['annotearea'] = row_annote.mr_annote_area    # list
    d_a['annotetype'] = row_annote.mr_annote_type    # str
    d_a['annoterotation'] = row_annote.mr_annote_rotation    # double
    d_a['annotedata'] = row_annote.mr_annote_data    # list

    return d_a
