<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2008 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id$
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class reserve_mainDb extends BaseDb
{
	/**
	 * 予約定義値を取得
	 *
	 * @param string $type	定義タイプ
	 * @param string $key    キーとなる項目値
	 * @return string		値
	 */
	function getReserveConfig($type, $key)
	{
		$retValue = '';
		$queryStr = 'SELECT rc_value FROM reserve_config ';
		$queryStr .=  'WHERE rc_type  = ? ';
		$queryStr .=    'AND rc_id = ? ';
		$ret = $this->selectRecord($queryStr, array($type, $key), $row);
		if ($ret) $retValue = $row['rc_value'];
		return $retValue;
	}
	/**
	 * 予約定義値を更新
	 *
	 * @param string $type	定義タイプ
	 * @param string $key	キーとなる項目値
	 * @param string $value	設定値
	 * @return bool			true=更新成功、false=更新失敗
	 */
	function updateReserveConfig($type, $key, $value)
	{
		$queryStr = 'UPDATE reserve_config SET rc_value = ? ';
		$queryStr .=  'WHERE rc_type  = ? ';
		$queryStr .=    'AND rc_id = ? ';
		$params = array($value, $type, $key);
		$ret = $this->execStatement($queryStr, $params);
		return $ret;
	}
	/**
	 * リソース総数取得
	 *
	 * @param string		$type	定義タイプ
	 * @return int					総数
	 */
	function getAllResourceListCount($type)
	{
		$queryStr = 'SELECT * FROM reserve_resource ';
		$queryStr .=  'WHERE rr_deleted = false ';// 削除されていない
		$queryStr .=    'AND rr_type = ? ';
		return $this->selectRecordCount($queryStr, array($type));
	}
	/**
	 * リソースリスト取得
	 *
	 * @param string	$type		定義タイプ
	 * @param int		$limit		取得する項目数
	 * @param int		$page		取得するページ(1～)
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getAllResourceList($type, $limit, $page, $callback)
	{
		$offset = $limit * ($page -1);
		if ($offset < 0) $offset = 0;
		
		$queryStr = 'SELECT * FROM reserve_resource ';
		$queryStr .=  'WHERE rr_deleted = false ';// 削除されていない
		$queryStr .=    'AND rr_type = ? ';
		$queryStr .=  'ORDER BY rr_sort_order limit ' . $limit . ' offset ' . $offset;
		$this->selectLoop($queryStr, array($type), $callback);
	}
	/**
	 * リソース最大表示順を取得
	 *
	 * @param string		$type	定義タイプ
	 * @return int					最大インデックス
	 */
	function getMaxResourceIndex($type)
	{
		$index = 0;
		$queryStr = 'SELECT * FROM reserve_resource ';
		$queryStr .=  'WHERE rr_deleted = false ';// 削除されていない
		$queryStr .=    'AND rr_type = ? ';
		$queryStr .=  'ORDER BY rr_sort_order desc';
		$ret = $this->selectRecord($queryStr, array($type), $row);
		if ($ret) $index = $row['rr_sort_order'];
		return $index;
	}
	/**
	 * リソース情報を更新、または追加
	 *
	 * @param string	$type		定義タイプ
	 * @param int $serial			シリアル番号(0のとき追加)
	 * @param string	$name		名前
	 * @param string	$desc		説明
	 * @param bool $visible			表示状態
	 * @param int $sortOrder		表示順
	 * @return						true=成功、false=失敗
	 */
	function updateResource($type, $serial, $name, $desc, $visible, $sortOrder)
	{
		global $gEnvManager;

		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();

		if (empty($serial)){
			// インデックス番号を取得
			$index = 1;
			$queryStr = 'SELECT max(rr_index) as m FROM reserve_resource ';
			$queryStr .=  'WHERE rr_type = ? ';
			$ret = $this->selectRecord($queryStr, array($type), $row);
			if ($ret) $index = $row['m'] + 1;
		
			$queryStr = 'INSERT INTO reserve_resource (';
			$queryStr .=  'rr_type, ';
			$queryStr .=  'rr_index, ';
			$queryStr .=  'rr_name, ';
			$queryStr .=  'rr_description, ';
			$queryStr .=  'rr_visible, ';
			$queryStr .=  'rr_sort_order, ';
			$queryStr .=  'rr_update_user_id, ';
			$queryStr .=  'rr_update_dt ';
			$queryStr .=  ') VALUES (';
			$queryStr .=  '?, ?, ?, ?, ?, ?, ?, ?';
			$queryStr .=  ')';
			$this->execStatement($queryStr, array($type, $index, $name, $desc, $visible, $sortOrder, $userId, $now));
		} else {
			$queryStr  = 'UPDATE reserve_resource ';
			$queryStr .=   'SET rr_name = ?, ';
			$queryStr .=     'rr_description = ?, ';
			$queryStr .=     'rr_visible = ?, ';
			$queryStr .=     'rr_sort_order = ?, ';
			$queryStr .=     'rr_update_user_id = ?, ';
			$queryStr .=     'rr_update_dt = ? ';
			$queryStr .=   'WHERE rr_serial = ? ';
			$this->execStatement($queryStr, array($name, $desc, $visible, $sortOrder, $userId, $now, $serial));
		}
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * リソースリスト取得
	 *
	 * @param int		$serial		シリアル番号
	 * @param int		$row		取得するページ(1～)
	 * @return						true=成功、false=失敗
	 */
	function getResourceBySerial($serial, &$row)
	{
		$queryStr = 'SELECT * FROM reserve_resource ';
		$queryStr .=  'WHERE rr_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * リソース情報をシリアル番号で削除
	 *
	 * @param array   $serial		シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function deleteResourceBySerial($serial)
	{
		global $gEnvManager;
		
		// 引数のエラーチェック
		if (!is_array($serial)) return false;
		if (count($serial) <= 0) return true;
		
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM reserve_resource ';
			$queryStr .=   'WHERE rr_deleted = false ';		// 未削除
			$queryStr .=     'AND rr_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE reserve_resource ';
		$queryStr .=   'SET rr_deleted = true, ';	// 削除
		$queryStr .=     'rr_update_user_id = ?, ';
		$queryStr .=     'rr_update_dt = ? ';
		$queryStr .=   'WHERE rr_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($userId, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * 予約カレンダーの通常データをすべて削除
	 *
	 * @param string	$type		定義タイプ
	 * @return						true=成功、false=失敗
	 */
	function deleteAllCalendar($type)
	{
		// 全商品を削除
		$queryStr  = 'DELETE FROM reserve_calendar ';
		$queryStr .=   'WHERE ra_type = ? ';
		$queryStr .=   'AND ra_usual = true ';
		$ret = $this->execStatement($queryStr, array($type));
		return $ret;
	}
	/**
	 * 予約カレンダー情報を追加
	 *
	 * @param string	$type		定義タイプ
	 * @param bool $usual			通常日あるいは特定日
	 * @param int $specifyType		指定方法
	 * @param int $attr				日にち属性
	 * @param date $date			日付
	 * @param int $startTime		開始時間
	 * @param int $endTime			終了時間
	 * @param bool $available		利用可能あるいは利用不可
	 * @return						true=成功、false=失敗
	 */
	function addCalendar($type, $usual, $specifyType, $attr, $date, $startTime, $endTime, $available)
	{
		$params = array();
		$replace = array();
		$queryStr = 'INSERT INTO reserve_calendar (';
		$queryStr .=  'ra_type, ';						$params[] = $type;	$replace[] = '?';
		$queryStr .=  'ra_usual, ';						$params[] = $usual;	$replace[] = '?';
		$queryStr .=  'ra_specify_type, ';				$params[] = $specifyType;	$replace[] = '?';
		$queryStr .=  'ra_day_attribute, ';				$params[] = $attr;	$replace[] = '?';
		if (!empty($date)){
			$queryStr .=  'ra_date, ';					$params[] = $date;	$replace[] = '?';
		}
		$queryStr .=  'ra_start_time, ';				$params[] = $startTime;	$replace[] = '?';
		$queryStr .=  'ra_end_time, ';					$params[] = $endTime;	$replace[] = '?';
		$queryStr .=  'ra_available ';					$params[] = $available;	$replace[] = '?';
		$queryStr .=  ') VALUES (';
		$queryStr .=  implode(",", $replace);
		$queryStr .=  ')';
		$ret = $this->execStatement($queryStr, $params);
		return $ret;
	}
	/**
	 * 予約カレンダーの曜日設定値を取得
	 *
	 * @param string	$type		定義タイプ
	 * @param array	$rows			取得データ
	 * @return						true=成功、false=失敗
	 */
	function getCalendarByWeek($type, &$rows)
	{
		$queryStr = 'SELECT * FROM reserve_calendar ';
		$queryStr .=  'WHERE ra_type = ? AND ra_usual = true ';
		$queryStr .=    'AND (ra_specify_type = 0 OR ra_specify_type = 1)';		// デフォルトあるいは曜日指定
		$queryStr .=  'ORDER BY ra_specify_type, ra_start_time ';
		return $this->selectRecords($queryStr, array($type), $rows);
	}
}
?>
