<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2008 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id$
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class ec_customerDb extends BaseDb
{
	/**
	 * 取引先一覧を取得
	 *
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllCustomer($callback)
	{
		$queryStr = 'SELECT * FROM customer LEFT JOIN _login_user ON cc_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE cc_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY cc_sort_order, cc_id';
		$this->selectLoop($queryStr, array(), $callback, null);
	}
	/**
	 * 在庫予定情報用の取引先一覧を取得
	 *
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getStockCustomer($callback)
	{
		$queryStr = 'SELECT * FROM customer LEFT JOIN _login_user ON cc_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE cc_deleted = false ';		// 削除されていない
		$queryStr .=  'AND (cc_is_custmer = true OR cc_is_supplier = true OR cc_is_delivery = true) ';		// 得意先または仕入先または出庫先
		$queryStr .=  'ORDER BY cc_sort_order, cc_id';
		$this->selectLoop($queryStr, array(), $callback, null);
	}
	/**
	 * 取引先の新規追加
	 *
	 * @param int	  $id			取引先ID
	 * @param string  $name			名前
	 * @param int     $index		表示順
	 * @param bool    $isCustmer	得意先かどうか
	 * @param bool    $isSupplier	仕入先かどうか
	 * @param bool    $isDelivery	出荷先かどうか
	 * @param bool    $isPayment	支払先かどうか
	 * @param bool    $isBilling	請求先かどうか
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function addCustomer($id, $name, $index, $isCustmer, $isSupplier, $isDelivery, $isPayment, $isBilling, &$newSerial)
	{
		global $gEnvManager;

		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		if ($id == 0){		// IDが0のときは、取引先IDを新規取得
			// 取引先IDを決定する
			$queryStr = 'select max(cc_id) as mid from customer ';
			$ret = $this->selectRecord($queryStr, array(), $row);
			if ($ret){
				$cId = $row['mid'] + 1;
			} else {
				$cId = 1;
			}
		} else {
			$cId = $id;
		}
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$queryStr = 'SELECT * FROM customer ';
		$queryStr .=  'WHERE cc_id = ? ';
		$queryStr .=  'ORDER BY cc_history_index DESC ';
		$ret = $this->selectRecord($queryStr, array($cId), $row);
		if ($ret){
			if (!$row['cc_deleted']){		// レコード存在していれば終了
				$this->endTransaction();
				return false;
			}
			$historyIndex = $row['cc_history_index'] + 1;
		}
		
		// データを追加
		$queryStr = 'INSERT INTO customer ';
		$queryStr .=  '(cc_id, cc_history_index, cc_name, cc_sort_order, cc_is_custmer, cc_is_supplier, cc_is_delivery, cc_is_payment, cc_is_billing, cc_create_user_id, cc_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($cId, $historyIndex, $name, $index, $isCustmer, $isSupplier, $isDelivery, $isPayment, $isBilling, $userId, $now));
		
		// 新規のシリアル番号取得
		$queryStr = 'select max(cc_serial) as ns from customer ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
			
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * 取引先をシリアル番号で取得
	 *
	 * @param int		$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getCustomerBySerial($serial, &$row)
	{
		$queryStr  = 'select * from customer LEFT JOIN _login_user ON cc_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE cc_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}	
	/**
	 * 取引先の更新
	 *
	 * @param int     $serial		シリアル番号
	 * @param string  $name			名前
	 * @param int     $index		表示順
	 * @param bool    $isCustmer	得意先かどうか
	 * @param bool    $isSupplier	仕入先かどうか
	 * @param bool    $isDelivery	出荷先かどうか
	 * @param bool    $isPayment	支払先かどうか
	 * @param bool    $isBilling	請求先かどうか
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateCustomer($serial, $name, $index, $isCustmer, $isSupplier, $isDelivery, $isPayment, $isBilling, &$newSerial)
	{	
		global $gEnvManager;

		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
				
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$historyIndex = 0;		// 履歴番号
		$queryStr  = 'select * from customer ';
		$queryStr .=   'where cc_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if ($ret){		// 既に登録レコードがあるとき
			if ($row['cc_deleted']){		// レコードが削除されていれば終了
				$this->endTransaction();
				return false;
			}
			$historyIndex = $row['cc_history_index'] + 1;
		} else {		// 存在しない場合は終了
			$this->endTransaction();
			return false;
		}
		
		// 古いレコードを削除
		$queryStr  = 'UPDATE customer ';
		$queryStr .=   'SET cc_deleted = true, ';	// 削除
		$queryStr .=     'cc_update_user_id = ?, ';
		$queryStr .=     'cc_update_dt = ? ';
		$queryStr .=   'WHERE cc_serial = ?';
		$this->execStatement($queryStr, array($userId, $now, $serial));
		
		// 新規レコード追加
		$queryStr = 'INSERT INTO customer ';
		$queryStr .=  '(cc_id, cc_history_index, cc_name, cc_sort_order, cc_is_custmer, cc_is_supplier, cc_is_delivery, cc_is_payment, cc_is_billing, cc_create_user_id, cc_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($row['cc_id'], $historyIndex, $name, $index, $isCustmer, $isSupplier, $isDelivery, $isPayment, $isBilling, $userId, $now));

		// 新規のシリアル番号取得
		$queryStr = 'select max(cc_serial) as ns from customer ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * 取引先をシリアル番号で削除
	 *
	 * @param array   $serial		シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delCustomerBySerial($serial)
	{
		global $gEnvManager;
		
		// 引数のエラーチェック
		if (!is_array($serial)) return false;
		if (count($serial) <= 0) return true;
		
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM customer ';
			$queryStr .=   'WHERE cc_deleted = false ';		// 未削除
			$queryStr .=     'AND cc_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE customer ';
		$queryStr .=   'SET cc_deleted = true, ';	// 削除
		$queryStr .=     'cc_update_user_id = ?, ';
		$queryStr .=     'cc_update_dt = ? ';
		$queryStr .=   'WHERE cc_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($userId, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * 最大表示順を取得
	 *
	 * @return int					最大表示順
	 */
	function getMaxIndex()
	{
		$queryStr = 'SELECT max(cc_sort_order) as mi FROM customer ';
		$queryStr .=  'WHERE cc_deleted = false ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret){
			$index = $row['mi'];
		} else {
			$index = 0;
		}
		return $index;
	}
}
?>
