<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2008 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: wiki_mainDb.php 1051 2008-09-30 12:12:11Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class wiki_mainDb extends BaseDb
{
	/**
	 * ページが存在するかチェック
	 *
	 * @param string $name	ページ名
	 * @param string $type	ページタイプ
	 * @return				true=存在する、false=存在しない
	 */
	function isExistsPage($name, $type='')
	{
		$queryStr = 'SELECT * from wiki_content ';
		$queryStr .=  'WHERE wc_type = ? ';
		$queryStr .=    'AND wc_id = ? ';
		$queryStr .=    'AND wc_deleted = false';
		return $this->isRecordExists($queryStr, array($type, $name));
	}
	/**
	 * Wikiページデータの取得
	 *
	 * @param string  $name			ウィキページ名
	 * @param array   $row			レコード
	 * @param string  $type			ページタイプ
	 * @return bool					取得 = true, 取得なし= false
	 */
	function getPage($name, &$row, $type='')
	{
		$queryStr  = 'SELECT * FROM wiki_content ';
		$queryStr .=   'WHERE wc_deleted = false ';	// 削除されていない
		$queryStr .=    'AND wc_type = ? ';
		$queryStr .=   'AND wc_id = ? ';
		$ret = $this->selectRecord($queryStr, array($type, $name), $row);
		return $ret;
	}
	/**
	 * 取得可能なWikiページ名の取得
	 *
	 * @param string  $type			ページタイプ
	 * @return array				ページ名
	 */
	function getAvailablePages($type='')
	{
		$retValue = array();
		$queryStr  = 'SELECT wc_id FROM wiki_content ';
		$queryStr .=   'WHERE wc_deleted = false ';	// 削除されていない
		$queryStr .=    'AND wc_type = ? ';
		$queryStr .=    'ORDER BY wc_id';
		$ret = $this->selectRecords($queryStr, array($type), $rows);
		if ($ret){
			for ($i = 0; $i < count($rows); $i++){
				$retValue[] = $rows[$i]['wc_id'];
			}
		}
		return $retValue;
	}
	/**
	 * Wikiページの更新
	 *
	 * @param string  $name			ウィキページ名
	 * @param string  $data			データ
	 * @param string  $type			ページタイプ
	 * @return bool					true = 成功、false = 失敗
	 */
	function updatePage($name, $data, $type='')
	{
		global $gEnvManager;
		
		$historyIndex = 0;		// 履歴番号
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクションスタート
		$this->startTransaction();
				
		// 前レコードの削除状態チェック
		$queryStr = 'SELECT * FROM wiki_content ';
		$queryStr .=  'WHERE wc_type = ? ';
		$queryStr .=    'AND wc_id = ? ';
		$queryStr .=  'ORDER BY wc_history_index DESC ';
		$ret = $this->selectRecord($queryStr, array($type, $name), $row);
		if ($ret){
			if (!$row['wc_deleted']){		// レコードが削除されていなければ削除
				// 古いレコードを削除
				$queryStr  = 'UPDATE wiki_content ';
				$queryStr .=   'SET wc_deleted = true, ';	// 削除
				$queryStr .=     'wc_update_user_id = ?, ';
				$queryStr .=     'wc_update_dt = ? ';
				$queryStr .=   'WHERE wc_serial = ?';
				$this->execStatement($queryStr, array($userId, $now, $row['wc_serial']));
			}
			$historyIndex = $row['wc_history_index'] + 1;
		}

		// 新規レコード追加
		$queryStr = 'INSERT INTO wiki_content ';
		$queryStr .=  '(wc_type, wc_id, wc_history_index, wc_data, wc_create_user_id, wc_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($type, $name, $historyIndex, $data, $userId, $now));
		
		// トランザクション終了
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツ項目の削除
	 *
	 * @param string  $contentType	コンテンツタイプ
	 * @param string  $contentId	コンテンツID(0のとき新規)
	 * @param string  $lang			言語ID
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delContentItem($contentType, $contentId, $lang, $userId)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		
		// トランザクションスタート
		$this->startTransaction();
		
		// 前レコードの削除状態チェック
		$queryStr = 'SELECT * FROM content ';
		$queryStr .=  'WHERE cn_type = ? ';
		$queryStr .=    'AND cn_id = ? ';
		$queryStr .=    'AND cn_language_id = ? ';
		$queryStr .=  'ORDER BY cn_history_index DESC ';
		$ret = $this->selectRecord($queryStr, array($contentType, $contentId, $lang), $row);
		if ($ret){
			if ($row['cn_deleted']){		// レコードが削除されていれば終了
				return false;
			}
		} else {
			return false;
		}
			
		// 古いレコードを削除
		$queryStr  = 'UPDATE content ';
		$queryStr .=   'SET cn_deleted = true, ';	// 削除
		$queryStr .=     'cn_update_user_id = ?, ';
		$queryStr .=     'cn_update_dt = ? ';
		$queryStr .=   'WHERE cn_serial = ?';
		$ret = $this->execStatement($queryStr, array($userId, $now, $row['cn_serial']));
		
		// トランザクション終了
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * 設定値が存在するかチェック
	 *
	 * @param string $id	定義ID
	 * @return				true=存在する、false=存在しない
	 */
	function isExistsConfig($id)
	{
		$queryStr = 'SELECT * from wiki_config ';
		$queryStr .=  'WHERE wg_id = ? ';
		return $this->isRecordExists($queryStr, array($id));
	}
	/**
	 * 設定値を定義IDで取得
	 *
	 * @param string $id		定義ID
	 * @return string			定義値
	 */
	function getConfig($id)
	{
		$queryStr  = 'SELECT * FROM wiki_config ';
		$queryStr .=   'WHERE wg_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id), $row);
		if ($ret){
			return $row['wg_value'];
		} else {
			return '';
		}
	}
	/**
	 * 設定値の更新
	 *
	 * @param string $id			定義ID
	 * @param string  $data			データ
	 * @param string  $name			名前
	 * @param string  $desc			説明
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateConfig($id, $data, $name='', $desc='')
	{
		// トランザクション開始
		$this->startTransaction();

		$queryStr  = 'SELECT * FROM wiki_config ';
		$queryStr .=   'WHERE wg_id = ? ';	
		$ret = $this->selectRecord($queryStr, array($id), $row);
		if ($ret){
			$queryStr  = 'UPDATE wiki_config ';
			$queryStr .=   'SET wg_value = ?, ';
			$queryStr .=     'wg_name = ?, ';
			$queryStr .=     'wg_description = ? ';
			$queryStr .=   'WHERE wg_id = ?';
			$this->execStatement($queryStr, array($data, $row['wg_name'], $row['wg_description'], $id));			
		} else {
			$queryStr = 'INSERT INTO wiki_config (';
			$queryStr .=  'wg_id, ';
			$queryStr .=  'wg_value, ';
			$queryStr .=  'wg_name, ';
			$queryStr .=  'wg_description ';
			$queryStr .=  ') VALUES (';
			$queryStr .=  '?, ?, ?, ?';
			$queryStr .=  ')';
			$this->execStatement($queryStr, array($id, $data, $name, $desc));	
		}
				
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
}
?>
