<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2008 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: admin_mainDb.php 1184 2008-11-07 05:01:11Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');
require_once($gEnvManager->getIncludePath() . '/common/userInfo.php');		// ユーザ情報クラス

class admin_mainDb extends BaseDb
{	
	private $now;		// 現在日時
	private $userId;		// ログイン中のユーザ
	private $maxNo;		// 最大管理番号
	const CONFIG_KEY_DEFAULT_TEMPLATE			= 'default_template';			// システム定義値取得用キー(PC用デフォルトテンプレート)
	const CONFIG_KEY_DEFAULT_TEMPLATE_MOBILE	= 'mobile_default_template';	// システム定義値取得用キー(携帯用デフォルトテンプレート)
	
	// 取得値
	const CAN_DETAIL_CONFIG = 'permit_detail_config';				// 詳細設定が可能かどうか
		
	/**
	 * システム定義値を取得
	 *
	 * @param string $key		キーとなる項目値
	 * @return string $value	値
	 */
	function getSystemConfig($key)
	{
		$retValue = '';
		$queryStr = 'SELECT sc_value FROM _system_config ';
		$queryStr .=  'WHERE sc_id  = ?';
		$ret = $this->selectRecord($queryStr, array($key), $row);
		if ($ret) $retValue = $row['sc_value'];
		return $retValue;
	}
	/**
	 * システム定義値を更新
	 *
	 * @param string $key		キーとなる項目値
	 * @param string $value		値
	 * @return					true = 正常、false=異常
	 */
	function updateSystemConfig($key, $value)
	{
		$sql = "UPDATE _system_config SET sc_value = ? WHERE sc_id = ?";
		$params = array($value, $key);
		return $this->execStatement($sql, $params);
	}
	/**
	 * サイト定義値を更新
	 *
	 * @param string $lang		言語
	 * @param string $key		キーとなる項目値
	 * @param string $value		値
	 * @param int $user			ユーザID
	 * @return					true = 正常、false=異常
	 */
	function updateSiteDef($lang, $key, $value)
	{
		global $gEnvManager;
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のレコードの履歴インデックス取得
		$historyIndex = 0;		// 履歴番号
		$queryStr  = 'SELECT * FROM _site_def ';
		$queryStr .=   'WHERE sd_id = ? ';
		$queryStr .=     'AND sd_language_id = ? ';
		$queryStr .=  'ORDER BY sd_history_index desc ';
		$queryStr .=    'LIMIT 1';
		$ret = $this->selectRecord($queryStr, array($key, $lang), $row);
		if ($ret){
			$historyIndex = $row['sd_history_index'] + 1;
		
			// レコードが削除されていない場合は削除
			if (!$row['sd_deleted']){
				// 古いレコードを削除
				$queryStr  = 'UPDATE _site_def ';
				$queryStr .=   'SET sd_deleted = true, ';	// 削除
				$queryStr .=     'sd_update_user_id = ?, ';
				$queryStr .=     'sd_update_dt = ? ';
				$queryStr .=   'WHERE sd_serial = ?';
				$ret = $this->execStatement($queryStr, array($user, $now, $row['sd_serial']));
				if (!$ret) return false;
			}
		}
		
		// 新規レコード追加
		$queryStr = 'INSERT INTO _site_def ';
		$queryStr .=  '(';
		$queryStr .=  'sd_id, ';
		$queryStr .=  'sd_language_id, ';
		$queryStr .=  'sd_history_index, ';
		$queryStr .=  'sd_value, ';
		$queryStr .=  'sd_create_user_id, ';
		$queryStr .=  'sd_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?)';
		$ret =$this->execStatement($queryStr, array($key, $lang, $historyIndex, $value, $user, $now));
			
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * サイト定義値を取得
	 *
	 * @param string $lang		言語
	 * @param string $key		キーとなる項目値
	 * @return string			値
	 */
	function getSiteDef($lang, $key)
	{
		$queryStr  = 'SELECT * FROM _site_def ';
		$queryStr .=   'WHERE sd_deleted = false ';
		$queryStr .=     'AND sd_id = ? ';
		$queryStr .=     'AND sd_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($key, $lang), $row);
		if ($ret){
			return $row['sd_value'];
		} else {
			return '';
		}
	}
	/**
	 * システムの詳細設定が可能かどうか
	 *
	 * @return bool					true=可能、false=不可
	 */
	function canDetailConfig()
	{
		$retValue = $this->getSystemConfig(self::CAN_DETAIL_CONFIG);
		return $retValue;
	}
	/**
	 * ウィジェットリスト取得
	 *
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getAllWidgetList($callback)
	{
		$queryStr = 'select * from _widgets ';
		$queryStr .=  'where wd_deleted = false ';// 削除されていない
		$queryStr .=  'order by wd_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * 管理メニュー項目用ウィジェットリスト(管理画面ありでPC携帯両方)取得
	 *
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getAvailableWidgetListForEditMenu($callback)
	{
		$queryStr  = 'select * from _widgets ';
		$queryStr .=   'where wd_deleted = false ';// 削除されていない
		$queryStr .=     'and wd_available = true ';		// メニューから選択可能なもの
		$queryStr .=     'and wd_has_admin = true ';		// 管理機能あり
		$queryStr .=   'order by wd_sort_order,wd_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * ウィジェットリスト(メニューから選択可能なもの)取得
	 *
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getAvailableWidgetList($callback)
	{
		$queryStr  = 'select * from _widgets ';
		$queryStr .=   'where wd_deleted = false ';// 削除されていない
		$queryStr .=     'and wd_available = true ';		// メニューから選択可能なもの
		$queryStr .=     'and wd_mobile = false ';		// 携帯用ウィジェット以外
		$queryStr .=   'order by wd_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * 携帯用のウィジェットリスト(メニューから選択可能なもの)取得
	 *
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getAvailableMobileWidgetList($callback)
	{
		$queryStr  = 'select * from _widgets ';
		$queryStr .=   'where wd_deleted = false ';// 削除されていない
		$queryStr .=     'and wd_available = true ';		// メニューから選択可能なもの
		$queryStr .=     'and wd_mobile = true ';		// 携帯用ウィジェット
		$queryStr .=   'order by wd_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * ウィジェットIDリスト取得
	 *
	 * @param array  $rows			取得レコード
	 * @return						true=取得、false=取得せず
	 */
	function getAllWidgetIdList(&$rows)
	{
		$queryStr = 'select * from _widgets ';
		$queryStr .=  'where wd_deleted = false ';// 削除されていない
		$queryStr .=  'order by wd_id';
		
		$retValue = $this->selectRecords($queryStr, array(), $rows);
		return $retValue;
	}
	/**
	 * ウィジェットの追加
	 *
	 * @param string  $id			ウィジェットID
	 * @param string  $name			ウィジェット名
	 * @param int     $userId		更新者ユーザID
	 * @return						なし
	 */
	function addNewWidget($id, $name, $userId)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$historyIndex = 0;
		
		// トランザクション開始
		$this->startTransaction();
		
		// 同じIDが登録済みかどうかチェック
		$queryStr = 'select * from _widgets ';
		$queryStr .=  'where wd_id = ? ';
		$queryStr .=  'order by wd_history_index desc ';
		$queryStr .=    'limit 1';
		$ret = $this->selectRecord($queryStr, array($id), $row);
		if ($ret){
			if (!$row['wd_deleted']){		// レコードが削除されていなければ、削除
				// 古いレコードを削除
				$queryStr  = 'UPDATE _widgets ';
				$queryStr .=   'SET wd_deleted = true, ';	// 削除
				$queryStr .=     'wd_update_user_id = ?, ';
				$queryStr .=     'wd_update_dt = ? ';
				$queryStr .=   'WHERE wd_serial = ?';
				$this->execStatement($queryStr, array($userId, $now, $row['wd_serial']));
			}
			$historyIndex = $row['wd_history_index'] + 1;
		}
		
		$queryStr = 'INSERT INTO _widgets ';
		$queryStr .=  '(wd_id, wd_history_index, wd_name, wd_create_user_id, wd_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, now())';
		$this->execStatement($queryStr, array($id, $historyIndex, $name, $userId));
		
		// トランザクション確定
		$ret = $this->endTransaction();
	}
	/**
	 * ウィジェットの更新
	 *
	 * @param int $serial			シリアル番号
	 * @param string  $name			ウィジェット名
	 * @param bool $hasScript		スクリプトがあるかどうか
	 * @param bool $hasCss			CSSがあるかどうか
	 * @param bool $hasAdmin		管理画面があるかどうか
	 * @param bool $enableOperation	単体実行可能かどうか
	 * @param int $userId			ユーザID
	 * @return bool					true=成功、false=失敗
	 */
	function updateWidget($serial, $name, $hasScript, $hasCss, $hasAdmin, $enableOperation, $userId)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
				
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$historyIndex = 0;		// 履歴番号
		$queryStr  = 'select * from _widgets ';
		$queryStr .=   'where wd_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if ($ret){		// 既に登録レコードがあるとき
			if ($row['wd_deleted']){		// レコードが削除されていれば終了
				$this->endTransaction();
				return false;
			}
			$historyIndex = $row['wd_history_index'] + 1;
		} else {		// 存在しない場合は終了
			$this->endTransaction();
			return false;
		}
		
		// 古いレコードを削除
		$queryStr  = 'UPDATE _widgets ';
		$queryStr .=   'SET wd_deleted = true, ';	// 削除
		$queryStr .=     'wd_update_user_id = ?, ';
		$queryStr .=     'wd_update_dt = ? ';
		$queryStr .=   'WHERE wd_serial = ?';
		$this->execStatement($queryStr, array($userId, $now, $serial));
		
		// 新規レコード追加
		$queryStr  = 'INSERT INTO _widgets (';
		$queryStr .=   'wd_id, ';
		$queryStr .=   'wd_history_index, ';
		$queryStr .=   'wd_name, ';
		$queryStr .=   'wd_read_scripts, ';
		$queryStr .=   'wd_read_css, ';
		$queryStr .=   'wd_has_admin, ';
		$queryStr .=   'wd_enable_operation, ';
		$queryStr .=   'wd_create_user_id, ';
		$queryStr .=   'wd_create_dt ';
		$queryStr .= ') VALUES (';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?)';
		$this->execStatement($queryStr, array($row['wd_id'], $historyIndex, $name, $hasScript, $hasCss, $hasAdmin, $enableOperation, $userId, $now));

		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ウィジェットの削除
	 *
	 * @param string  $serial		シリアル番号
	 * @param int $userId			ユーザID
	 * @return						true=成功、false=失敗
	 */
	function deleteWidget($serial, $userId)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$queryStr  = 'select * from _widgets ';
		$queryStr .=   'WHERE wd_serial = ? ';
		$queryStr .=    'and wd_deleted = false';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if (!$ret){		// 登録レコードがないとき
			$this->endTransaction();
			return false;
		}
		
		// 古いレコードを削除
		$queryStr  = 'UPDATE _widgets ';
		$queryStr .=   'SET wd_deleted = true, ';	// 削除
		$queryStr .=     'wd_update_user_id = ?, ';
		$queryStr .=     'wd_update_dt = ? ';
		$queryStr .=   'WHERE wd_serial = ?';
		$this->execStatement($queryStr, array($userId, $now, $serial));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;		
	}
	/**
	 * ウィジェットの取得
	 *
	 * @param string  $serial		シリアル番号
	 * @return						true=正常、false=異常
	 */
	function getWidget($serial, &$row)
	{
		$queryStr  = 'select * from _widgets ';
		$queryStr .=   'where wd_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * ページIDのリストを取得
	 *
	 * @param function $callback	コールバック関数
	 * @param int $type				リストの種別
	 * @param bool $mobile			携帯対応かどうか
	 * @return						なし
	 */
	function getPageIdList($callback, $type, $mobile = false)
	{
		$queryStr = 'SELECT * FROM _page_id ';
		$queryStr .=  'WHERE pg_type = ? ';
		$queryStr .=  'AND pg_mobile = ? ';
		$queryStr .=  'ORDER BY pg_priority';
		$this->selectLoop($queryStr, array($type, (int)$mobile), $callback);// PostgreSQL対応(true,falseのままではDBに入らないため)
	}
	/**
	 * ページIDのリストを取得
	 *
	 * @param int $type				リストの種別
	 * @param array $row			取得データ
	 * @return bool					true=成功、false=失敗
	 */
	function getPageIdRecords($type, &$row)
	{
		$queryStr = 'select * from _page_id ';
		$queryStr .=  'where pg_type = ? ';
		$queryStr .=  'order by pg_priority';
		return $this->selectRecords($queryStr, array($type), $row);
	}
	/**
	 * ページIDの追加
	 *
	 * @param int $type				ページタイプ(0=ページID,1=ページサブID)
	 * @param string  $id			ID
	 * @param string  $name			名前
	 * @param string  $subid		デフォルトのサブページID
	 * @param string  $path			パス
	 * @param int  $priority		優先度
	 * @param string  $desc			説明
	 * @param bool  $isPublic		公開状況
	 * @return						true=成功、false=失敗
	 */
	function addPageId($type, $id, $name, $subid, $path, $priority, $desc, $isPublic)
	{
		// トランザクション開始
		$this->startTransaction();
		
		// 新規レコードを追加
		$queryStr  = 'INSERT INTO _page_id (';
		$queryStr .=   'pg_id, ';
		$queryStr .=   'pg_type, ';
		$queryStr .=   'pg_default_sub_id, ';
		$queryStr .=   'pg_path, ';
		$queryStr .=   'pg_name, ';
		$queryStr .=   'pg_description, ';
		$queryStr .=   'pg_priority, ';
		$queryStr .=   'pg_is_public ';
		$queryStr .= ') VALUES (';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?) ';
		$this->execStatement($queryStr, array($id, $type, $subid, $path, $name, $desc, $priority, $isPublic));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ページIDの存在チェック
	 *
	 * @param int $type				リストの種別
	 * @param string  $id			ID
	 * @return bool					true=存在する、false=存在しない
	 */
	function isExistsPageId($type, $id)
	{
		$queryStr = 'SELECT * FROM _page_id ';
		$queryStr .=  'WHERE pg_type = ? ';
		$queryStr .=  'AND pg_id = ? ';
		return $this->isRecordExists($queryStr, array($type, $id));
	}
	/**
	 * ページIDの削除
	 *
	 * @param int $type				リストの種別
	 * @param string  $id			ID
	 * @return bool					true=存在する、false=存在しない
	 */
	function deletePageId($type, $id)
	{
		// トランザクション開始
		$this->startTransaction();
		
		$queryStr = 'DELETE FROM _page_id ';
		$queryStr .=  'WHERE pg_type = ? ';
		$queryStr .=  'AND pg_id = ? ';
		$this->execStatement($queryStr, array($type, $id));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ページIDの更新
	 *
	 * @param int $type				リストの種別
	 * @param array  $rows			更新データ
	 * @return bool					true=成功、false=失敗
	 */
	function updatePageId($type, $rows)
	{
		// トランザクション開始
		$this->startTransaction();
		
		for ($i = 0; $i < count($rows); $i++){
			// 既存項目を更新
			$queryStr  = 'UPDATE _page_id ';
			$queryStr .=   'SET ';
			$queryStr .=     'pg_default_sub_id = ?, ';
			$queryStr .=     'pg_path = ?, ';
			$queryStr .=     'pg_name = ?, ';
			$queryStr .=     'pg_description = ?, ';
			$queryStr .=     'pg_priority = ?, ';
			$queryStr .=     'pg_is_public = ? ';
			$queryStr .=   'WHERE pg_id = ? ';
			$queryStr .=     'AND pg_type = ? ';
			$queryStr .=     'AND pg_editable = true ';		// 編集可能項目のみ更新
			$this->execStatement($queryStr, array($rows[$i]['pg_default_sub_id'], $rows[$i]['pg_path'], $rows[$i]['pg_name'], 
								$rows[$i]['pg_description'], $rows[$i]['pg_priority'], $rows[$i]['pg_is_public'], $rows[$i]['pg_id'], $type));
		}
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ページポジションのリストを取得
	 *
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getPagePositionList($callback)
	{
		$queryStr = 'select * from _template_position ';
		$queryStr .=  'order by tp_sort_order';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * ページ定義のリスト取得
	 *
	 * @param function $callback	コールバック関数
	 * @param string $pageId		ページID
	 * @param string $pageSubId		ページサブID
	 * @param string $position		表示位置。空文字列のときはすべて取得。
	 * @param int    $setId			定義セットID
	 * @return						なし
	 */
	function getPageDefList($callback, $pageId, $pageSubId, $position = '', $setId = 0)
	{
		$queryStr  = 'select * from _page_def left join _widgets on pd_widget_id = wd_id and wd_deleted = false ';
		$queryStr .=   'where pd_id = ? ';
		$queryStr .=     'and (pd_sub_id = ? or pd_sub_id = \'\') ';	// 空の場合は共通項目
		$queryStr .=     'and pd_set_id = ? ';
		if (empty($position)){
			$queryStr .=   'order by pd_position_id, pd_index';
			$this->selectLoop($queryStr, array($pageId, $pageSubId, $setId), $callback);
		} else {
			$queryStr .=     'and pd_position_id = ? ';
			$queryStr .=   'order by pd_position_id, pd_index';
			$this->selectLoop($queryStr, array($pageId, $pageSubId, $setId, $position), $callback);
		}
	}
	/**
	 * ページ定義があるか確認
	 *
	 * @param string $pageId		ページID
	 * @param string $pageSubId		ページサブID
	 * @param string $position	表示位置
	 * @param int $index		表示インデックス
	 * @param int    $setId			定義セットID
	 * @return bool				true=存在する、false=存在しない
	 */
	 /*
	function isPageDefExists($pageId, $pageSubId, $position, $index, $setId = 0)
	{
		$queryStr  = 'select * from _page_def ';
		$queryStr .=   'where pd_id = ? ';
		$queryStr .=     'and pd_sub_id = ? ';
		$queryStr .=     'and pd_position_id = ? ';
		$queryStr .=     'and pd_index = ? ';
		return $this->isRecordExists($queryStr, array($pageId, $pageSubId, $position, $index));
	}*/
	/**
	 * ページ定義を取得
	 *
	 * @param int $serialNo			シリアルNo
	 * @param array  $rows			更新データ
	 * @return bool				true=存在する、false=存在しない
	 */
	function getPageDef($serialNo, &$row)
	{
		$queryStr  = 'select * from _page_def ';
		$queryStr .=   'where pd_serial = ?';
		return $this->selectRecord($queryStr, array($serialNo), $row);
	}
	/**
	 * ページ定義項目の更新
	 *
	 * @param int $serialNo			シリアルNo(0のとき新規追加)
	 * @param string $pageId		ページID
	 * @param string $pageSubId		ページサブID
	 * @param string $position		表示位置
	 * @param int $index			表示インデックスNo
	 * @param string $widgetId		ウィジェットID
	 * @param int $configId			定義ID
	 * @param string $suffix		サフィックス
	 * @param string $style			css
	 * @param bool $visible			表示状態
	 * @param int $userId			ユーザID(データ更新者)
	 * @param int    $setId			定義セットID
	 * @return						true=成功、false=失敗
	 */
	function updatePageDef($serialNo, $pageId, $pageSubId, $position, $index, $widgetId, $configId, $suffix, $style, $visible, $userId, $setId = 0)
	{
		// 更新ユーザ、日時設定
		$this->now = date("Y/m/d H:i:s");	// 現在日時
		$this->userId = $userId;
		$editable = 1;		// 編集可能
		
		// トランザクション開始
		$this->startTransaction();

		if ($serialNo == 0){		// 新規追加
			// 新規データを追加
			$queryStr  = 'INSERT INTO _page_def (';
			$queryStr .=   'pd_id, ';
			$queryStr .=   'pd_sub_id, ';
			$queryStr .=   'pd_set_id, ';
			$queryStr .=   'pd_position_id, ';
			$queryStr .=   'pd_index, ';
			$queryStr .=   'pd_widget_id, ';
			$queryStr .=   'pd_config_id, ';
			$queryStr .=   'pd_suffix, ';
			$queryStr .=   'pd_style, ';
			$queryStr .=   'pd_visible, ';
			$queryStr .=   'pd_editable, ';
			$queryStr .=   'pd_update_user_id, ';
			$queryStr .=   'pd_update_dt) ';
			$queryStr .= 'VALUES (';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?, ';
			$queryStr .=   '?)';
			$this->execStatement($queryStr, array($pageId, $pageSubId, $setId, $position, $index, 
								$widgetId, $configId, $suffix, $style, $visible, $editable, $this->userId, $this->now));
		} else {			// 更新
			$queryStr  = 'select * from _page_def ';
			$queryStr .=   'where pd_serial = ? ';
			$ret = $this->selectRecord($queryStr, array($serialNo), $row);
			if ($ret){
				if ($row['pd_id'] != $pageId || $row['pd_sub_id'] != $pageSubId){			// 表示ページが変更された
					// 新規に追加して、古いレコードを削除
					// 新規データを追加
					$queryStr  = 'INSERT INTO _page_def (';
					$queryStr .=   'pd_id, ';
					$queryStr .=   'pd_sub_id, ';
					$queryStr .=   'pd_set_id, ';
					$queryStr .=   'pd_position_id, ';
					$queryStr .=   'pd_index, ';
					$queryStr .=   'pd_widget_id, ';
					$queryStr .=   'pd_config_id, ';
					$queryStr .=   'pd_suffix, ';
					$queryStr .=   'pd_style, ';
					$queryStr .=   'pd_visible, ';
					$queryStr .=   'pd_editable, ';
					$queryStr .=   'pd_update_user_id, ';
					$queryStr .=   'pd_update_dt) ';
					$queryStr .= 'VALUES (';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?, ';
					$queryStr .=   '?)';
					$this->execStatement($queryStr, array($pageId, $pageSubId, $row['pd_set_id'], $position, $index, 
										$widgetId, $configId, $suffix, $style, $visible, $editable, $this->userId, $this->now));
										
					// 旧データ削除
					$queryStr  = 'DELETE FROM _page_def WHERE pd_serial = ?';
					$this->execStatement($queryStr, array($serialNo));
				} else {
					// 既存項目を更新
					$queryStr  = 'UPDATE _page_def ';
					$queryStr .=   'SET ';
					$queryStr .=     'pd_position_id = ?, ';
					$queryStr .=     'pd_index = ?, ';
					$queryStr .=     'pd_widget_id = ?, ';
					$queryStr .=     'pd_config_id = ?, ';
					$queryStr .=     'pd_suffix = ?, ';
					$queryStr .=     'pd_style = ?, ';
					$queryStr .=     'pd_visible = ?, ';
					$queryStr .=     'pd_editable = ?, ';
					$queryStr .=     'pd_update_user_id = ?, ';
					$queryStr .=     'pd_update_dt = ? ';
					$queryStr .=   'WHERE pd_serial = ? ';
					$this->execStatement($queryStr, array($position, $index, 
										$widgetId, $configId, $suffix, $style, $visible, $editable, $this->userId, $this->now, $serialNo));
				}
			}
		}

		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ページ定義項目の削除
	 *
	 * @param int $serialNo			シリアルNo
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delPageDef($serialNo, $userId)
	{
		// 更新ユーザ、日時設定
		$this->now = date("Y/m/d H:i:s");	// 現在日時
		$this->userId = $userId;
		
		// トランザクション開始
		$this->startTransaction();
		
		$queryStr  = 'DELETE FROM _page_def WHERE pd_serial = ?';
		$this->execStatement($queryStr, array($serialNo));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	
	/**
	 * ページ定義項目をすべて削除
	 *
	 * @param string  $pageId			ページID
	 * @param string  $pageSubId		ページサブID
	 * @param string  $position			表示ポジション
	 * @param bool $withCommon			共通項目も削除するかどうか
	 * @param int    $setId			定義セットID
	 * @return						true=成功、false=失敗
	 */
	function delPageDefAll($pageId, $pageSubId, $position, $withCommon, $setId = 0)
	{
		// トランザクション開始
		$this->startTransaction();
		
		if (empty($position)){
			$queryStr  = 'DELETE FROM _page_def ';
			$queryStr .=   'WHERE pd_id = ? ';
			if ($withCommon){
				$queryStr .=     'and (pd_sub_id = ? or pd_sub_id = \'\') ';	// 空の場合は共通項目
			} else {
				$queryStr .=     'and pd_sub_id = ? ';	// 空の場合は共通項目
			}
			$queryStr .=     'and pd_set_id = ? ';
			$this->execStatement($queryStr, array($pageId, $pageSubId, $setId));
		} else {
			$queryStr  = 'DELETE FROM _page_def ';
			$queryStr .=   'WHERE pd_id = ? ';
			if ($withCommon){
				$queryStr .=     'and (pd_sub_id = ? or pd_sub_id = \'\') ';	// 空の場合は共通項目
			} else {
				$queryStr .=     'and pd_sub_id = ? ';	// 空の場合は共通項目
			}
			$queryStr .=     'and pd_position_id = ? ';
			$queryStr .=     'and pd_set_id = ? ';
			$this->execStatement($queryStr, array($pageId, $pageSubId, $position, $setId));
		}
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}

	/**
	 * ユーザリスト取得
	 *
	 * @param int		$limit				取得する項目数
	 * @param int		$page				取得するページ(1～)
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getAllUserList($limit, $page, $callback)
	{
		$offset = $limit * ($page -1);
		if ($offset < 0) $offset = 0;
		
		$queryStr = 'SELECT * FROM _login_user ';
		$queryStr .=  'WHERE lu_deleted = false ';// 削除されていない
		$queryStr .=  'ORDER BY lu_account limit ' . $limit . ' offset ' . $offset;
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * ユーザ総数取得
	 *
	 * @return int					総数
	 */
	function getAllUserListCount()
	{
		$queryStr = 'select * from _login_user ';
		$queryStr .=  'where lu_deleted = false ';// 削除されていない
		return $this->selectRecordCount($queryStr, array());
	}
	/**
	 * ログイン状況取得
	 *
	 * @param int		$limit				取得する項目数
	 * @param int		$page				取得するページ(1～)
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getUserLoginStatusList($limit, $page, $callback)
	{
		$offset = $limit * ($page -1);
		if ($offset < 0) $offset = 0;
		
		$queryStr = 'SELECT lu_id,lu_account,lu_name,lu_user_type,lu_user_status,ll_login_count,ll_access_log_serial,ll_pre_login_dt,ll_last_login_dt, ';
		$queryStr .=    'CASE WHEN ll_last_login_dt IS NULL THEN 1 ELSE 0 ';
		$queryStr .=    'END AS ord ';
		$queryStr .=    'FROM _login_user LEFT JOIN _login_log on lu_id = ll_user_id ';
		$queryStr .=  'WHERE lu_deleted = false ';// 削除されていない
		$queryStr .=  'ORDER BY ord, ll_last_login_dt desc limit ' . $limit . ' offset ' . $offset;
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * ログイン状況数取得
	 *
	 * @return int					総数
	 */
	function getUserLoginStatusListCount()
	{
		$queryStr = 'SELECT * FROM _login_user LEFT JOIN _login_log on lu_id = ll_user_id ';
		$queryStr .=  'WHERE lu_deleted = false ';// 削除されていない
		return $this->selectRecordCount($queryStr, array());
	}
	/**
	 * 新規ユーザの追加
	 *
	 * @param string  $name			名前
	 * @param string  $account		アカウント
	 * @param string  $password		パスワード
	 * @param bool    $isAdmin		管理者権限があるかどうか
	 * @param bool    $canLogin		ログインできるかどうか
	 * @param int     $userId		更新者ユーザID
	 * @param int     $newSerial	新規シリアル番号
	 * @return						true=成功、false=失敗
	 */
	function addNewUser($name, $account, $password, $isAdmin, $canLogin, $userId, &$newSerial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
				
		// トランザクション開始
		$this->startTransaction();
		
		// 新規IDを作成
		$newId = 1;
		$queryStr = 'select max(lu_id) as ms from _login_user ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newId = $row['ms'] + 1;
		
		// ユーザ種別を設定
		$userType = UserInfo::USER_TYPE_NORMAL;		// 一般ユーザ
		if ($isAdmin) $userType = UserInfo::USER_TYPE_SYS_ADMIN;		// 管理ユーザ
		
		// 新規レコードを追加
		$queryStr  = 'INSERT INTO _login_user (';
		$queryStr .=   'lu_id, ';
		$queryStr .=   'lu_history_index, ';
		$queryStr .=   'lu_name, ';
		$queryStr .=   'lu_account, ';
		$queryStr .=   'lu_password, ';
		$queryStr .=   'lu_user_type, ';
		$queryStr .=   'lu_enable_login, ';
		$queryStr .=   'lu_create_user_id, ';
		$queryStr .=   'lu_create_dt ';
		$queryStr .= ') VALUES (';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?) ';
		$this->execStatement($queryStr, array($newId, 0, $name, $account, $password, $userType, $canLogin, $userId, $now));
		
		// 新規のシリアル番号取得
		$queryStr = 'select max(lu_serial) as ns from _login_user ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ユーザの削除
	 *
	 * @param int $serialNo			シリアルNo
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delUser($serialNo, $userId)
	{
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$queryStr  = 'select * from _login_user ';
		$queryStr .=   'where lu_deleted = false ';		// 未削除
		$queryStr .=     'and lu_serial = ? ';
		$ret = $this->isRecordExists($queryStr, array($serialNo));
		// 存在しない場合は、既に削除されたとして終了
		if (!$ret){
			$this->endTransaction();
			return false;
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE _login_user ';
		$queryStr .=   'SET lu_deleted = true, ';	// 削除
		$queryStr .=     'lu_update_user_id = ?, ';
		$queryStr .=     'lu_update_dt = now() ';
		$queryStr .=   'WHERE lu_serial = ?';
		$this->execStatement($queryStr, array($userId, $serialNo));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ユーザの削除
	 *
	 * @param array $serial			シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delUserBySerial($serial)
	{
		global $gEnvManager;
			
		// 引数のエラーチェック
		if (!is_array($serial)) return false;
		if (count($serial) <= 0) return true;
		
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM _login_user ';
			$queryStr .=   'WHERE lu_deleted = false ';		// 未削除
			$queryStr .=     'AND lu_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE _login_user ';
		$queryStr .=   'SET lu_deleted = true, ';	// 削除
		$queryStr .=     'lu_update_user_id = ?, ';
		$queryStr .=     'lu_update_dt = ? ';
		$queryStr .=   'WHERE lu_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($userId, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ユーザの更新
	 *
	 * @param int $serial			シリアル番号
	 * @param string  $name			ユーザ名
	 * @param string  $account		アカウント
	 * @param string  $password		パスワード(空のときは更新しない)
	 * @param bool $isAdmin			管理者権限があるかどうか
	 * @param string $canLogin		ログイン可能かどうか
	 * @param int $userId			ユーザID
	 * @param int     $newSerial	新規シリアル番号
	 * @return						true=成功、false=失敗
	 */
	function updateUser($serial, $name, $account, $password, $isAdmin, $canLogin, $userId, &$newSerial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
				
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$historyIndex = 0;		// 履歴番号
		$queryStr  = 'select * from _login_user ';
		$queryStr .=   'where lu_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if ($ret){		// 既に登録レコードがあるとき
			if ($row['lu_deleted']){		// レコードが削除されていれば終了
				$this->endTransaction();
				return false;
			}
			$historyIndex = $row['lu_history_index'] + 1;
		} else {		// 存在しない場合は終了
			$this->endTransaction();
			return false;
		}

		// ユーザ種別を設定
		$userType = UserInfo::USER_TYPE_NORMAL;		// 一般ユーザ
		if ($isAdmin) $userType = UserInfo::USER_TYPE_SYS_ADMIN;		// 管理ユーザ
		
		// パスワードが設定されているときは更新
		$pwd = $row['lu_password'];
		if (!empty($password)) $pwd = $password;
		
		// 古いレコードを削除
		$queryStr  = 'UPDATE _login_user ';
		$queryStr .=   'SET lu_deleted = true, ';	// 削除
		$queryStr .=     'lu_update_user_id = ?, ';
		$queryStr .=     'lu_update_dt = ? ';
		$queryStr .=   'WHERE lu_serial = ?';
		$this->execStatement($queryStr, array($userId, $now, $serial));

		// 新規レコード追加
		$queryStr  = 'INSERT INTO _login_user (';
		$queryStr .=   'lu_id, ';
		$queryStr .=   'lu_history_index, ';
		$queryStr .=   'lu_name, ';
		$queryStr .=   'lu_account, ';
		$queryStr .=   'lu_password, ';
		$queryStr .=   'lu_user_type, ';
		$queryStr .=   'lu_assign, ';
		$queryStr .=   'lu_widget_id, ';
		$queryStr .=   'lu_enable_login, ';
		$queryStr .=   'lu_create_user_id, ';
		$queryStr .=   'lu_create_dt ';
		$queryStr .= ') VALUES (';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?) ';
		$this->execStatement($queryStr, array($row['lu_id'], $historyIndex, $name, $account, $pwd, $userType, $row['lu_assign'], $row['lu_widget_id'], $canLogin, $userId, $now));

		// 新規のシリアル番号取得
		$queryStr = 'select max(lu_serial) as ns from _login_user ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ユーザ情報をシリアル番号で取得
	 *
	 * @param string	$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getUserBySerial($serial, &$row)
	{
		$queryStr  = 'select * from _login_user ';
		$queryStr .=   'WHERE lu_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * ユーザアカウントが存在するかチェック
	 *
	 * @param string $account	アカウント
	 * @return					true=存在する、false=存在しない
	 */
	 /*
	function isExistsAccount($account)
	{
		$queryStr = 'SELECT * from _login_user ';
		$queryStr .=  'WHERE lu_account = ? ';
		$queryStr .=    'AND lu_deleted = false';
		return $this->isRecordExists($queryStr, array($account));
	}*/
	/**
	 * テンプレートリスト(携帯用以外)取得
	 *
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getAllTemplateList($callback)
	{
		$queryStr = 'SELECT * FROM _templates ';
		$queryStr .=  'WHERE tm_deleted = false ';// 削除されていない
		$queryStr .=    'AND tm_mobile = false ';		// 携帯用以外
		$queryStr .=  'ORDER BY tm_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * テンプレートリスト(携帯用)取得
	 *
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getAllMobileTemplateList($callback)
	{
		$queryStr = 'SELECT * FROM _templates ';
		$queryStr .=  'WHERE tm_deleted = false ';// 削除されていない
		$queryStr .=    'AND tm_mobile = true ';		// 携帯用
		$queryStr .=  'ORDER BY tm_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * テンプレートIDリスト取得
	 *
	 * @param array  $rows			取得レコード
	 * @return						true=取得、false=取得せず
	 */
	function getAllTemplateIdList(&$rows)
	{
		$queryStr = 'SELECT * FROM _templates ';
		$queryStr .=  'WHERE tm_deleted = false ';// 削除されていない
		$queryStr .=  'AND tm_mobile = false ';// 携帯以外
		$queryStr .=  'ORDER BY tm_id';
		$retValue = $this->selectRecords($queryStr, array(), $rows);
		return $retValue;
	}
	/**
	 * 携帯用テンプレートIDリスト取得
	 *
	 * @param array  $rows			取得レコード
	 * @return						true=取得、false=取得せず
	 */
	function getAllMobileTemplateIdList(&$rows)
	{
		$queryStr = 'SELECT * FROM _templates ';
		$queryStr .=  'WHERE tm_deleted = false ';// 削除されていない
		$queryStr .=  'AND tm_mobile = true ';// 携帯
		$queryStr .=  'ORDER BY tm_id';
		$retValue = $this->selectRecords($queryStr, array(), $rows);
		return $retValue;
	}
	/**
	 * テンプレートの追加
	 *
	 * @param string  $id			テンプレートID
	 * @param string  $name			テンプレート名
	 * @param int     $userId		更新者ユーザID
	 * @return						なし
	 */
	function addNewTemplate($id, $name, $userId)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$historyIndex = 0;
		
		// トランザクション開始
		$this->startTransaction();
		
		// 同じIDが登録済みかどうかチェック
		$queryStr = 'select * from _templates ';
		$queryStr .=  'where tm_id = ? ';
		$queryStr .=  'order by tm_history_index desc ';
		$queryStr .=    'limit 1';
		$ret = $this->selectRecord($queryStr, array($id), $row);
		if ($ret){
			if (!$row['tm_deleted']){		// レコードが削除されていなければ、削除
				// 古いレコードを削除
				$queryStr  = 'UPDATE _templates ';
				$queryStr .=   'SET tm_deleted = true, ';	// 削除
				$queryStr .=     'tm_update_user_id = ?, ';
				$queryStr .=     'tm_update_dt = ? ';
				$queryStr .=   'WHERE tm_serial = ?';
				$this->execStatement($queryStr, array($userId, $now, $row['tm_serial']));			
			}
			$historyIndex = $row['tm_history_index'] + 1;
		}
		
		$queryStr = 'INSERT INTO _templates ';
		$queryStr .=  '(tm_id, tm_name, tm_history_index, tm_create_dt, tm_create_user_id) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, now(), ?)';
		$this->execStatement($queryStr, array($id, $name, $historyIndex, $userId));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * テンプレートの更新
	 *
	 * @param string $templateId	テンプレートID
	 * @param string  $name			ウィジェット名
	 * @return						true=成功、false=失敗
	 */
	function updateTemplate($templateId, $name)
	{
		global $gEnvManager;
		
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
				
		// トランザクション開始
		$this->startTransaction();
		
		// 既存データを取得
		$historyIndex = 0;		// 履歴番号
		$queryStr = 'SELECT * FROM _templates ';
		$queryStr .=  'WHERE tm_id = ? ';
		$queryStr .=  'ORDER BY tm_history_index DESC ';
		$ret = $this->selectRecord($queryStr, array($templateId), $row);
		if ($ret){
			if ($row['tm_deleted']){		// レコードが削除されていれば終了
				$this->endTransaction();
				return false;
			} else {		// レコードが削除されていなければ、削除
				// 古いレコードを削除
				$queryStr  = 'UPDATE _templates ';
				$queryStr .=   'SET tm_deleted = true, ';	// 削除
				$queryStr .=     'tm_update_user_id = ?, ';
				$queryStr .=     'tm_update_dt = ? ';
				$queryStr .=   'WHERE tm_serial = ?';
				$this->execStatement($queryStr, array($userId, $now, $row['tm_serial']));			
			}
			$historyIndex = $row['tm_history_index'] + 1;
		} else {
			$this->endTransaction();
			return false;
		}
		
		// 新規レコード追加
		$queryStr  = 'INSERT INTO _templates (';
		$queryStr .=   'tm_id, ';
		$queryStr .=   'tm_history_index, ';
		$queryStr .=   'tm_name, ';
		$queryStr .=   'tm_create_user_id, ';
		$queryStr .=   'tm_create_dt ';
		$queryStr .= ') VALUES (';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?)';
		$this->execStatement($queryStr, array($row['tm_id'], $historyIndex, $name, $userId, $now));

		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * テンプレートの削除
	 *
	 * @param string $templateId	テンプレートID
	 * @return						true=成功、false=失敗
	 */
	function deleteTemplate($templateId)
	{
		global $gEnvManager;
		
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// 既存データを取得
		$queryStr = 'SELECT * FROM _templates ';
		$queryStr .=  'WHERE tm_deleted = false ';
		$queryStr .=  'AND tm_id = ? ';
		$ret = $this->selectRecord($queryStr, array($templateId), $row);
		if (!$ret){
			$this->endTransaction();
			return false;
		}
		// レコードを削除
		$queryStr  = 'UPDATE _templates ';
		$queryStr .=   'SET tm_deleted = true, ';	// 削除
		$queryStr .=     'tm_update_user_id = ?, ';
		$queryStr .=     'tm_update_dt = ? ';
		$queryStr .=   'WHERE tm_serial = ?';
		$this->execStatement($queryStr, array($userId, $now, $row['tm_serial']));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;		
	}
	/**
	 * デフォルトテンプレートの変更
	 *
	 * @param string  $templateId	デフォルトテンプレートID
	 * @return						true=成功、false=失敗
	 */
	function changeDefaultTemplate($templateId)
	{
		return $this->updateSystemConfig(self::CONFIG_KEY_DEFAULT_TEMPLATE, $templateId);
	}
	/**
	 * 携帯用デフォルトテンプレートの変更
	 *
	 * @param string  $templateId	デフォルトテンプレートID
	 * @return						true=成功、false=失敗
	 */
	function changeDefaultMobileTemplate($templateId)
	{
		return $this->updateSystemConfig(self::CONFIG_KEY_DEFAULT_TEMPLATE_MOBILE, $templateId);
	}
	/**
	 * 通貨一覧を取得
	 *
	 * @param string	$lang				言語
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	 /*
	function getCurrency($lang, $callback)
	{
		$queryStr = 'SELECT * FROM _currency ';
		$queryStr .=  'WHERE cu_language_id = ? ';
		$queryStr .=  'ORDER BY cu_index';
		$this->selectLoop($queryStr, array($lang), $callback, null);
	}*/
	/**
	 * すべての言語を取得
	 *
	 * @param function	$callback			コールバック関数
	 * @return			true=取得、false=取得せず
	 */
	function getAllLang($callback)
	{
		$queryStr = 'SELECT * FROM _language ORDER BY ln_priority';
		$this->selectLoop($queryStr, array(), $callback, null);
	}
	/**
	 * メニュー項目の表示状態を更新
	 *
	 * @param string $groupId	更新対象グループID
	 * @param bool $visible		表示状態
	 * @return					true = 正常、false=異常
	 */
	function updateMenuVisible($groupId, $visible)
	{
		$sql = "UPDATE _nav_item SET ni_visible = ? WHERE ni_group_id = ?";
		$params = array($visible, $groupId);
		return $this->execStatement($sql, $params);
	}
	/**
	 * トップ画面表示項目を取得
	 *
	 * @param string $navId			ナビゲーションバー識別ID
	 * @param array  $rows			取得レコード
	 * @return						true=取得、false=取得せず
	 */
	/*function getTopPageItems($navId, &$rows)
	{
		$queryStr  = 'SELECT * FROM _nav_item ';
		$queryStr .=   'WHERE ni_nav_id = ? ';
		$queryStr .=     'AND ni_top_page_index != 0 ';		// 0以外を表示
		$queryStr .=     'AND ni_visible = true ';
		$queryStr .=   'ORDER BY ni_top_page_index';
		$retValue = $this->selectRecords($queryStr, array($navId), $rows);
		return $retValue;
	}*/
	/**
	 * ナビゲーションバー項目を取得
	 *
	 * @param string $navId			ナビゲーションバー識別ID
	 * @param string $parentId		親項目ID
	 * @param array  $rows			取得レコード
	 * @return						true=取得、false=取得せず
	 */
	function getNavItems($navId, $parentId, &$rows)
	{
		$queryStr  = 'SELECT * FROM _nav_item ';
		$queryStr .=   'WHERE ni_nav_id = ? ';
		$queryStr .=     'AND ni_parent_id = ? ';
		$queryStr .=     'AND ni_visible = true ';
		$queryStr .=   'ORDER BY ni_index';
		
		$retValue = $this->selectRecords($queryStr, array($navId, $parentId), $rows);
		return $retValue;
	}
	/**
	 * ナビゲーションバー項目をすべて取得
	 *
	 * @param string 	$navId			ナビゲーションバー識別ID
	 * @param function	$callback		コールバック関数
	 * @return 			なし
	 */
	function getNavItemsAll($navId, $callback)
	{
		$queryStr  = 'SELECT * FROM _nav_item ';
		$queryStr .=   'WHERE ni_nav_id = ? ';
		$queryStr .=   'ORDER BY ni_id';
		$this->selectLoop($queryStr, array($navId), $callback, null);
	}
	/**
	 * ナビゲーションバー項目を削除
	 *
	 * @param string $menuId		メニュー識別ID
	 * @return						true=成功、false=失敗
	 */
	function delNavItems($menuId)
	{
		$sql = "DELETE FROM _nav_item WHERE ni_nav_id = ?";
		$params = array($menuId);
		$this->execStatement($sql, $params);
	}
	/**
	 * ナビゲーションバー項目の最大IDを取得
	 *
	 * @return int			最大ID
	 */
	function getNavItemsMaxId()
	{
		$max = 0;
		$queryStr = 'SELECT max(ni_id) as mid FROM _nav_item ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $max = $row['mid'];
		return $max;
	}
	/**
	 * ナビゲーションバー項目キー存在チェック
	 *
	 * @param string $menuId		メニュー識別ID
	 * @param string $taskId		タスクID
	 * @param string $param			追加パラメータ
	 * @return bool					true=存在する、false=存在しない
	 */
	function isExistsNavItemKey($menuId, $taskId, $param)
	{
		$queryStr = 'SELECT * FROM _nav_item ';
		$queryStr .=  'WHERE ni_nav_id = ? ';
		$queryStr .=  'AND ni_task_id = ? ';
		$queryStr .=  'AND ni_param = ? ';
		return $this->isRecordExists($queryStr, array($menuId, $taskId, $param));
	}
	/**
	 * ナビゲーションバー項目を更新
	 *
	 * @param string $navId			ナビゲーションバー識別ID
	 * @param int $id				項目ID
	 * @param int $parentId			項目親項目ID
	 * @param int $index			インデックス番号起動
	 * @param string $taskId		タスクID
	 * @param string $param			追加パラメータ
	 * @param int $control			改行指示(0=改行しない、1=改行)
	 * @param string $name			項目名
	 * @param string $helpTitle		ヘルプタイトル
	 * @param string $helpBody		ヘルプ本体
	 * @return						true=成功、false=失敗
	 */
	function addNavItems($navId, $id, $parentId, $index, $taskId, $param, $control, $name, $helpTitle, $helpBody)
	{
		// 新規レコード追加
		$groupId = '';
		$queryStr  = 'INSERT INTO _nav_item (';
		$queryStr .=   'ni_id, ';
		$queryStr .=   'ni_parent_id, ';
		$queryStr .=   'ni_index, ';
		$queryStr .=   'ni_nav_id, ';
		$queryStr .=   'ni_task_id, ';
		$queryStr .=   'ni_param, ';
		$queryStr .=   'ni_group_id, ';
		$queryStr .=   'ni_view_control, ';
		$queryStr .=   'ni_name, ';
		$queryStr .=   'ni_help_title, ';
		$queryStr .=   'ni_help_body ';
		$queryStr .= ') VALUES (';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?)';
		$ret = $this->execStatement($queryStr, array($id, $parentId, $index, $navId, $taskId, $param, $groupId, $control, $name, $helpTitle, $helpBody));
		return $ret;
	}
	/**
	 * 変換キーテーブルを取得
	 *
	 * @param string	$key				キー文字列
	 * @param string	$group				グループID
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllKey($key, $group, $callback)
	{
		$queryStr = 'SELECT * FROM _key_value ';
		$queryStr .=  'WHERE kv_deleted = false ';
		$queryStr .=    'AND kv_id LIKE \'' . $key . '%\' ';
		$queryStr .=    'AND kv_group_id = ? ';
		$queryStr .=  'ORDER BY kv_id';
		$this->selectLoop($queryStr, array($group), $callback, null);
	}
}
?>
