<?php
/**
 * HTTPリクエスト、レスポンス、セッション管理マネージャー
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2008 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: requestManager.php 1178 2008-11-05 06:44:00Z fishbone $
 * @link       http://www.magic3.org
 */
class RequestManager
{
	private $db;						// DBオブジェクト
	private $tmpCookie;		// クッキー送信前のクッキー格納データ
	private $magicQuote;	// バックスラッシュでの文字エスケープ処理
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		global $gInstanceManager;
		global $gEnvManager;
		
		// システムDBオブジェクト取得
		$this->db = $gInstanceManager->getSytemDbObject();
		
		// セッションハンドラ設定
		// セッションDBが使用可能な場合は、ハンドラを設定し、
		// 使用不可の場合は、デフォルトのセッション管理を使用する
		if (M3_SESSION_DB && $gEnvManager->canUseDbSession()){
			ini_set('session.save_handler', 'user');		// 追加(2008/7/7)
			session_set_save_handler(	array($this, '_sessionOpen'),
										array($this, '_sessionClose'),
										array($this, '_sessionRead'),
										array($this, '_sessionWrite'),
										array($this, '_sessionDestroy'),
										array($this, '_sessionGc'));
		}
		
		// その他パラメータ初期化
		$this->tmpCookie = array();		// クッキー送信前のクッキー格納データ
		
		// magic quotesが組み込まれているか
		if (get_magic_quotes_gpc() == 1) $this->magicQuote = true;
	}
	/**
	 * POST値が設定されているか判断
	 */
	public function isPost($name)
	{
		return isset($_POST[$name]);
	}
	/**
	 * GET値が設定されているか判断
	 */
	public function isGet($name)
	{
		return isset($_GET[$name]);
	}
	/**
	 * POST値取得
	 */
	public function valueOfPost($name)
	{
		return self::isPost($name) ? $_POST[$name] : '';
	}
	/**
	 * POST値取得(トリミング(前後の空白削除)あり)
	 */
	public function trimValueOfPost($name)
	{
		$value = self::isPost($name) ? $_POST[$name] : '';
		return $this->_trimValueOf($name, $value);
	}
	/**
	 * GET値取得
	 */
	public function valueOfGet($name)
	{
		return self::isGet($name) ? $_GET[$name] : '';
	}
	/**
	 * GET値取得(トリミング(前後の空白削除)あり)
	 */
	public function trimValueOfGet($name)
	{
		$value = self::isGet($name) ? $_GET[$name] : '';
		return $this->_trimValueOf($name, $value);
	}
	/**
	 * POST,GETから値を取得
	 *
	 * POSTまたはGETから値を取得
	 *
	 * @param string $name		キー値
	 * @param string $default  	値が存在しないときのデフォルト値
	 * @return string			取得値
	 */
	public function valueOf($name, $default = '')
	{
		//return isset($_POST[$name]) ? $_POST[$name] : (isset($_GET[$name]) ? $_GET[$name] : $default);
		//return self::gpc_stripslashes(isset($_POST[$name]) ? $_POST[$name] : (isset($_GET[$name]) ? $_GET[$name] : $default));
		return $this->gpc_stripslashes(isset($_POST[$name]) ? $_POST[$name] : (isset($_GET[$name]) ? $_GET[$name] : $default));
	}
	/**
	 * POST,GETから値を取得(トリミング(前後の空白削除)あり)
	 *
	 * POSTまたはGETから値を取得し、トリミング(前後の空白削除)する
	 * デフォルトでHTMLタグを取り除く
	 *
	 * @param string $name		キー値
	 * @param string $default  	値が存在しないときのデフォルト値
	 * @return string			取得値
	 */
	public function trimValueOf($name, $default = '')
	{
		$srcValue = isset($_POST[$name]) ? $_POST[$name] : (isset($_GET[$name]) ? $_GET[$name] : $default);
		return $this->_trimValueOf($name, $srcValue);
	}
	/**
	 * トリミング(前後の空白削除)処理
	 *
	 * デフォルトでHTMLタグを取り除く
	 *
	 * @param string $name		キー値
	 * @param string $value  	値
	 * @return string			取得値
	 */
	public function _trimValueOf($name, $value)
	{
		global $gLogManager;
		global $gEnvManager;
		
		// トリムをかける
		if (is_array($value)){
			$trimValue = array_map('trim', $value);

			$destValue = array();
			for ($i = 0; $i < count($trimValue); $i++){
				$stripValue = $this->gpc_stripslashes($trimValue[$i]);
				$stripValue = strip_tags($stripValue);
				if (strlen($stripValue) != strlen($trimValue[$i])){
					// ウィジェット内での処理の場合はウィジェットIDも出力
					$widgetId = $gEnvManager->getCurrentWidgetId();// ウィジェットID
					if (empty($widgetId)){
						$gLogManager->warn(__METHOD__, 'POSTまたはGET値の異常データを検出: name=' . $name . ', value=[' . $value[$i] . ']');
					} else {		// ウィジェットIDが設定されているとき
						$gLogManager->warn(__METHOD__, 'POSTまたはGET値の異常データを検出: name=' . $name . ', value=[' . $value[$i] . '], widgetid=' . $widgetId);
					}
				}
				$destValue[] = $stripValue;
			}
		} else {
			$trimValue = trim($value);
			
			// HTMLタグが含まれていた場合は、ログを残す
			$destValue = $this->gpc_stripslashes($trimValue);
			$destValue = strip_tags($destValue);
			if (strlen($trimValue) != strlen($destValue)){
				// ウィジェット内での処理の場合はウィジェットIDも出力
				$widgetId = $gEnvManager->getCurrentWidgetId();// ウィジェットID
				if (empty($widgetId)){
					$gLogManager->warn(__METHOD__, 'POSTまたはGET値の異常データを検出: name=' . $name . ', value=[' . $value . ']');
				} else {		// ウィジェットIDが設定されているとき
					$gLogManager->warn(__METHOD__, 'POSTまたはGET値の異常データを検出: name=' . $name . ', value=[' . $value . '], widgetid=' . $widgetId);
				}
			}
		}
		return $destValue;
	}
	/**
	 * (携帯用)POST,GETから値を取得(トリミング(前後の空白削除)あり)
	 *
	 * POSTまたはGETから値を取得し、トリミング(前後の空白削除)する
	 * デフォルトでHTMLタグを取り除く
	 * 携帯用の文字コード変換を行う
	 *
	 * @param string $name		キー値
	 * @param string $default  	値が存在しないときのデフォルト値
	 * @return string			取得値
	 */
	public function mobileTrimValueOf($name, $default = '')
	{
		global $gEnvManager;
		
		// 携帯用エンコーディングを取得
		$mobileEncoding = $gEnvManager->getMobileEncoding();
		
		// 入力データの文字コードをシステム内部コードに変換する
		$srcValue = $this->trimValueOf($name, $default);
		$destValue = '';
		if (function_exists('mb_convert_encoding')) $destValue = mb_convert_encoding($srcValue, M3_ENCODING, $mobileEncoding);
		return $destValue;
	}
	/**
	 * $_SERVERから値を取得(トリミング(前後の空白削除)あり)
	 *
	 * $_SERVERから値を取得し、トリミング(前後の空白削除)する
	 *
	 * @param string $name		キー値
	 * @param string $default  	値が存在しないときのデフォルト値
	 * @return string			取得値
	 */
	public function trimServerValueOf($name, $default = '')
	{
		$value = isset($_SERVER[$name]) ? $_SERVER[$name] : $default;
		
		// トリムをかける
		if (is_array($value)){
			$value = array_map('trim', $value);
		} else {
			$value = trim($value);
		}
		return $value;
	}
	/**
	 * セッションから値を取得
	 *
	 * $_SESSIONから値を取得する。
	 *
	 * @param string $name		キー値
	 * @param string $default  	値が存在しないときのデフォルト値
	 * @return string			取得値
	 */
	public function getSessionValue($name, $default = '')
	{
		$value = isset($_SESSION[$name]) ? $_SESSION[$name] : $default;
		return $value;
	}
	/**
	 * セッションに値を格納
	 *
	 * $_SESSIONに値を格納する。
	 *
	 * @param string $name		キー値
	 * @param string $value  	格納値
	 * @return 					なし
	 */
	public function setSessionValue($name, $value = '')
	{
		$_SESSION[$name] = $value;
	}
	/**
	 * セッション格納値を開放
	 *
	 * $_SESSIONの指定値を開放する
	 *
	 * @param string $name		キー値
	 * @return 					なし
	 */
	public function unsetSessionValue($name)
	{
		unset($_SESSION[$name]);
	}
	/**
	 * セッションからシリアライズされた値を取得
	 *
	 * @param string $name		キー値
	 * @return object			取得したオブジェクト、なしの場合はnull
	 */
	public function getSessionValueWithSerialize($name)
	{
		$value = isset($_SESSION[$name]) ? unserialize($_SESSION[$name]) : null;
		return $value;
	}
	/**
	 * セッションに値をシリアライズして格納
	 *
	 * @param string $name		キー値
	 * @param object $value  	格納するオブジェクト、nullをセットした場合はセッション値を削除
	 * @return 					なし
	 */
	public function setSessionValueWithSerialize($name, $value = null)
	{
		if ($value == null){
			unset($_SESSION[$name]);
		} else {
			$_SESSION[$name] = serialize($value);
		}
	}
	/**
	 * クッキーから値を取得
	 *
	 * @param string $name		キー値
	 * @param string $default  	値が存在しないときのデフォルト値
	 * @return string			取得値
	 */
	public function getCookieValue($name, $default = '')
	{
		// 設定したクッキー値があれば取得
		$value = isset($this->tmpCookie[$name]) ? $this->tmpCookie[$name] : '';		// クッキー送信前のクッキー格納データ
		
		if ($value == '') $value = isset($_COOKIE[$name]) ? $_COOKIE[$name] : $default;
		return $value;
	}
	/**
	 * クッキーに値を格納
	 *
	 * @param string $name		キー値
	 * @param string $value  	格納値
	 * @param int $expireDay  	クッキーの生存期間(日)
	 * @return 					なし
	 */
	public function setCookieValue($name, $value = '', $expireDay = 30)
	{
		$cookExpire = time() + 60 * 60 * 24 * $expireDay;
		setcookie($name, $value, $cookExpire);
		
		// 設定したクッキー値を保存
		$this->tmpCookie[$name] = $value;		// クッキー送信前のクッキー格納データ
	}
	/**
	 * クエリー文字列中のWikiページ名を取得
	 *
	 * @return string			Wikiページ名、空の時はWikiページ名なし
	 */
	public function getWikiPageFromQuery()
	{
		$wikiPage = '';
		$args = explode('&', $_SERVER['QUERY_STRING']);// 「&」で分割
		for ($i = 0; $i < count($args); $i++){
			$line = $args[$i];
			$pos = strpos($line, '=');
			if ($pos){		// 「=」が存在するとき
				//list($key, $value) = explode('=', $line);
			} else {
				$wikiPage = $line;
				break;
			}
		}
		return $wikiPage;
	}
	/**
	 * 文字のエスケープ処理
	 *
	 * '(シングルクオート)、" (ダブルクオート)、\(バックスラッシュ) 、NULLにバックスラッシュでエスケープ処理を行う
	 *
	 * @param string $str		変換元文字列
	 * @return string			変換後文字列
	 */
	function gpc_addslashes($str){
		if ($this->magicQuote){
			return $str;
		} else {
			return addslashes($str);
		}
	}
	/**
	 * 文字のエスケープ処理をはずす
	 *
	 * '(シングルクオート)、" (ダブルクオート)、\(バックスラッシュ) 、NULLのバックスラッシュでのエスケープ処理を削除
	 *
	 * @param string $str		変換元文字列
	 * @return string			変換後文字列
	 */
	function gpc_stripslashes($str){
		if ($this->magicQuote){  
			return stripslashes($str);
		} else {  
			return $str;  
		}  
	}
	//******************************************************
	// patTemplateのテンプレートに値を埋め込む。値を省略した場合は、POST,GETデータから取得したデータを再設定する。
	// $name: patTemplateのテンプレート名
	// $valueName: HTMLタグに設定した名前。patTemplateの値埋め込み用キーワードも同じものを使用する。
	// $default: 値埋め込み用キーワードに埋め込む値。省略の場合はPOST,GETデータから$valueNameの値を取得する。
	function addValueToTemplate($name, $valueName, $default = null)
	{
		if ($default == null){
			$this->tmpl->addVar($name, $valueName, $this->valueOf($valueName));
		} else {
			$this->tmpl->addVar($name, $valueName, $default);
		}
	}
	/**
	 * セッションを開く
	 */
	function _sessionOpen($save_path, $session_name)
	{
		return true;
	}
	/**
	 * セッションを閉じる
	 */
	function _sessionClose()
	{
		return true;
	}
	/**
	 * セッションデータを読み込む
	 */
	function _sessionRead($id)
	{
		return $this->db->readSession($id);
	}
	/**
	 * セッションデータを書き込む
	 *
	 * @param string $id  			セッションID
	 * @param string $sessData  	セッションデータ
	 */
	function _sessionWrite($id, $sessData)
	{
		return $this->db->writeSession($id, $sessData);
	}
	/**
	 * セッションを破棄する
	 *
	 * @param string $id  			セッションID
	 */
	function _sessionDestroy($id)
	{
		return $this->db->destroySession($id);
	}
	/**
	 * 時間経過したセッションを破棄する
	 *
	 * @param int $maxlifetime  	セッションの生存時間(秒)
	 */
	function _sessionGc($maxlifetime)
	{
		return $this->db->gcSession($maxlifetime);
	}
}
?>
