<?php
/**
 * 環境取得用マネージャー
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2009 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: envManager.php 1856 2009-05-06 05:11:28Z fishbone $
 * @link       http://www.magic3.org
 */
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/version.php');	// システムバージョンクラス
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/userInfo.php');		// ユーザ情報クラス
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/core.php');

class EnvManager extends Core
{
	public  $sysVersion;		// システムバージョンオブジェクト
	private $currentTemplateId;	// 現在のテンプレートID(ディレクトリ名)
	private $currentTemplateType;	// 現在のテンプレートのタイプ
	private $currentWidgetId;	// 現在作成中のウィジェットId
	private $currentWidgetConfigId;	// 現在作成中のウィジェットの定義ID
	private $currentIWidgetConfigId;	// 現在作成中のインナーウィジェットの定義ID
	private $currentIWidgetId;	// 現在作成中のインナーウィジェットId
	private $currentPageId;		// 現在のページId
	private $currentPageSubId;	// 現在のページサブId
	private $currentPageDefSerial;	// 現在処理を行っているページ定義のレコードのシリアル番号
	private $defaultPageSubId;	// デフォルトのページサブId
	private $currentWidgetPrefix;	// 現在作成中のウィジェットのプレフィックス文字列
	private $currentWidgetTitle;	// 現在作成中のウィジェットのタイトル文字列
	private $defaultLanguage;	// デフォルト言語(システムで固定)
	private $currentLanguage;	// 現在の言語(ユーザによって可変)
	private $multiLanguage;		// 多言語対応かどうか
 	private $db;				// DBオブジェクト
	private $canUseDbSession;	// DBセッションが使用できるかどうか
	private $canUseDb;			// DBが使用可能状態にあるかどうか
	private $canChangeLang;		// 言語変更可能かどうか
	private $siteName;			// サイト名称
	private $siteOwner;			// サイト所有者
	private $siteCopyRight;		// サイトコピーライト
	private $siteEmail;			// サイトEメール
	private $widgetLog;			// ウィジェット実行ログを残すかどうか
	private $isMobileSite;		// 携帯URLアクセスかどうか
	private $mobileEncoding;	// 携帯用の入力、出力エンコーディング
	private $workDir;			// 作業用ディレクトリ
	private $userAgent = array();	// アクセス端末の情報
	const DEFAULT_CSV_DELIM_CODE = 'csv_delim_code';		// デフォルトのCSV区切り文字コード
	const DEFAULT_CSV_NL_CODE = 'csv_nl_code';		// デフォルトのCSV改行コード
	const DEFAULT_CSV_FILE_SUFFIX = 'csv_file_suffix';		// デフォルトのCSVファイル拡張子
	const MULTI_LANGUAGE = 'multi_language';		// 多言語対応かどうか
	const MOBILE_ENCODING = 'mobile_encoding';		// 携帯用入出力エンコーディング
	const MOBILE_CHARSET = 'mobile_charset';		// 携帯用HTML上のエンコーディング表記
	const CONFIG_ID_WORK_DIR = 'work_dir';			// 作業用ディレクトリ
	const DEFAULT_PAGE_ID = 'index';					// デフォルトのページID
	const DEFAULT_MOBILE_PAGE_ID = 'm_index';					// 携帯用デフォルトのページID
	const DEFAULT_ADMIN_PAGE_ID = 'admin_index';		// デフォルトの管理機能用ページID
	const USER_AGENT_TYPE_PC = 'pc';					// アクセス端末の種類(PC)
	const USER_AGENT_TYPE_MOBILE = 'mobile';			// アクセス端末の種類(携帯)
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		// 親クラスを呼び出す
		parent::__construct();
		
		// データ初期化
		$this->workDir = M3_SYSTEM_WORK_DIR_PATH;// 作業用ディレクトリ
		
		// システムバージョンオブジェクト作成
		$this->sysVersion = new m3Version();
		
		// システムDBオブジェクト取得
		//$this->db = $gInstanceManager->getSytemDbObject();
		$this->db = $this->gInstance->getSytemDbObject();
		
		// ######## DBの接続チェック ########
		if (defined('M3_STATE_IN_INSTALL')){		// システムがインストールモードで起動のとき
			$this->canUseDb = false;			// DBは使用できない
		} else {
			// システム名称、バージョンを取得
			$status = $this->db->getDisplayErrMessage();	// 出力状態を取得
			$this->db->displayErrMessage(false);		// 画面へのエラー出力を抑止
			$value = $this->db->getSystemConfig(M3_TB_FIELD_SYSTEM_NAME);
			$this->db->displayErrMessage($status);		// 抑止解除
			// 値が取得できたときは、セッションDBテーブルも作成されているとする
			if ($value == ''){
				$this->canUseDbSession = false;
				$this->canUseDb = false;			// DBは使用できない
			} else {
				$this->canUseDbSession = true;
				$this->canUseDb = true;			// DBは使用可能
				
				// システム関係のパラメータを取得
				$this->loadSystemParams();
			}
		}
		// 日本語処理関係
		if (extension_loaded('mbstring')){	// mbstring使用可能
			ini_set('mbstring.encoding_translation',        'Off');		// ここでは設定を変更できない？
			ini_set('mbstring.http_input',                  'pass');
			ini_set('mbstring.http_output',                 'pass');
			ini_set('mbstring.substitute_character',		'none');	// 無効な文字の代替出力
			ini_set('mbstring.func_overload',               '0');
			
			if (function_exists('mb_language')) mb_language("Japanese");
			if (function_exists('mb_internal_encoding')) mb_internal_encoding("UTF-8");
		}
	}
	/**
	 * システム関係のパラメータを再取得
	 */
	public function loadSystemParams()
	{
		// デフォルト値取得
		$this->defaultLanguage = $this->db->getDefaultLanguage();
		$this->currentLanguage = $this->defaultLanguage;
		
		// 言語変更可能かどうか
		$this->canChangeLang = $this->db->getCanChangeLang();
		$this->multiLanguage = $this->db->getSystemConfig(self::MULTI_LANGUAGE);		// 多言語対応かどうか
		
		$this->mobileEncoding = $this->db->getSystemConfig(self::MOBILE_ENCODING);	// 携帯用の入力、出力エンコーディング
		$this->mobileCharset = $this->db->getSystemConfig(self::MOBILE_CHARSET);		// 携帯用HTML上のエンコーディング表記
		
		// 作業用ディレクトリ
		$value = $this->db->getSystemConfig(self::CONFIG_ID_WORK_DIR);
		if (!empty($value)) $this->workDir = $value;
	}
	/**
	 * デバッグ出力を行うかどうか
	 */
	public function getSystemDebugOut()
	{
		return M3_SYSTEM_DEBUG_OUT;
	}

	// ##################### システム全体のパス環境 #####################
	/**
	 * システムルートディレクトリへのパスを取得
	 */
	public function getSystemRootPath()
	{
		return M3_SYSTEM_ROOT_PATH;
	}
	/**
	 * 管理用ディレクトリへのパスを取得
	 */
	public function getAdminPath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . M3_DIR_NAME_ADMIN;
	}
	/**
	 * includeディレクトリへのパスを取得
	 */
	public function getIncludePath()
	{
		return M3_SYSTEM_INCLUDE_PATH;
	}
	/**
	 * インナーウィジェット用ディレクトリへのパスを取得
	 */
	public function getIWidgetsPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'iwidgets';
	}
	/**
	 * addonsディレクトリへのパスを取得
	 */
	public function getAddonsPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'addons';
	}
	/**
	 * コンテナクラス用ディレクトリへのパスを取得
	 */
	public function getContainerPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'container';
	}
	/**
	 * DBクラス用ディレクトリへのパスを取得
	 */
	public function getDbPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'db';
	}
	/**
	 * ライブラリ用ディレクトリへのパスを取得
	 */
	public function getLibPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'lib';
	}
	/**
	 * SQL格納用ディレクトリへのパスを取得
	 */
	public function getSqlPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'sql';
	}
	/**
	 * テーブル作成用SQL格納用ディレクトリへのパスを取得
	 */
	public function getTablesPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'tables';
	}
	/**
	 * Coreディレクトリへのパスを取得
	 */
	public function getCorePath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'core';
	}
	/**
	 * Commonディレクトリへのパスを取得
	 */
	public function getCommonPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'common';
	}
	/**
	 * dataディレクトリへのパスを取得
	 */
	public function getDataPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'data';
	}
	/**
	 * Joomla用ルートディレクトリへのパスを取得
	 */
	public function getJoomlaRootPath()
	{
		return M3_SYSTEM_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'mos';
	}
	/**
	 * スクリプトファイルディレクトリへのパスを取得
	 */
	public function getScriptsPath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'scripts';
	}
	/**
	 * テンプレート用ディレクトリへのパスを取得
	 */
	public function getTemplatesPath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'templates';
	}
	/**
	 * リソース用ディレクトリへのパスを取得
	 */
	public function getResourcePath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'resource';
	}
	/**
	 * widgetsディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets
	 */
	public function getWidgetsPath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'widgets';
	}
	/**
	 * システムのルートURLを取得
	 */
	public function getRootUrl()
	{
		return M3_SYSTEM_ROOT_URL;
	}
	/**
	 * widgetsディレクトリへのURLを取得
	 *
	 * 例) http://www.magic3.org/magic3/widgets
	 */
	public function getWidgetsUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/widgets';
	}
	/**
	 * リソース用ディレクトリへのパスを取得
	 */
	public function getResourceUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/resource';
	}
	/**
	 * 画像用ディレクトリへのパスを取得
	 */
	public function getImagesUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/images';
	}
	/**
	 * 絵文字画像用ディレクトリへのパスを取得
	 */
	public function getEmojiImagesUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/images/system/emoji';
	}
	/**
	 * scriptsディレクトリ(共通スクリプトディレクトリ)へのURLを取得
	 *
	 * 例) http://www.magic3.org/magic3/scripts
	 */
	public function getScriptsUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/scripts';
	}
	/**
	 * templatesディレクトリ(テンプレートディレクトリ)へのURLを取得
	 *
	 * 例) http://www.magic3.org/magic3/templates
	 */
	public function getTemplatesUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/templates';
	}
	/**
	 * 管理用ディレクトリへのパスを取得
	 */
	public function getAdminUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/' . M3_DIR_NAME_ADMIN;
	}
	/**
	 * システムのデフォルトindexのURLを取得
	 */
	public function getDefaultUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/index.php';
	}
	/**
	 * システムの携帯用デフォルトindexのURLを取得
	 *
	 * @param bool $withMobileParam		携帯用のパラメータを付加するかどうか
	 * @return string					携帯用デフォルトURL
	 */
	public function getDefaultMobileUrl($withMobileParam=false)
	{
		$url = M3_SYSTEM_ROOT_URL . '/' . M3_DIR_NAME_MOBILE . '/index.php';
		if ($withMobileParam){		// 携帯用のパラメータを付加するとき
			$url = $this->_createUrl($url, $this->_getMobileUrlParam());
		}
		return $url;
	}
	/**
	 * システムのデフォルトの管理用indexのURLを取得
	 */
	public function getDefaultAdminUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/' . M3_DIR_NAME_ADMIN . '/index.php';
	}
	/**
	 * 作業用ディレクトリへのパスを取得
	 */
	public function getWorkDirPath()
	{
		return $this->workDir;
	}
	/**
	 * 一時ディレクトリを取得
	 *
	 * 一時ディレクトリを作成してパスを取得
	 *
	 * @return string		ディレクトリパス(失敗のときは空を返す)
	 */
	public function getTempDir()
	{
		$path = $this->workDir . '/' . M3_SYSTEM_WORK_DIRNAME_HEAD . uniqid();
		if (!file_exists($path)){// ディレクトリがないとき
			if (!mkdir($path, M3_SYSTEM_DIR_PERMISSION)){
				$path = $this->workDir . '/' . M3_SYSTEM_WORK_DIRNAME_HEAD . uniqid(rand());
				if (!mkdir($path, M3_SYSTEM_DIR_PERMISSION)) return '';
			}
		}
		return $path;
	}
	// ##################### パス処理 #####################
	/**
	 * サーバのURLを取得
	 *
	 * 例) http://www.magic3.org, http://www.magic3.org:8080
	 */
	public function getServerUrl()
	{
		// クライアントからの情報を元にURLを取得
		//$url = 'http://' . $_SERVER['SERVER_NAME'];
		//if (!empty($_SERVER['SERVER_PORT'])) $url .= ':' . $_SERVER['SERVER_PORT'];
		$url = 'http://' . $_SERVER['HTTP_HOST'];
		return $url;
	}
	/**
	 * 現在実行中のスクリプトファイルのURLを取得
	 *
	 * 例) http://www.magic3.org/magic3/index.php
	 */
	public function getCurrentScriptUrl()
	{
		//return $_SERVER["SCRIPT_URI"];		// SCRIPT_URIはサーバによってはundefinedになる
		return self::getServerUrl() . $_SERVER["PHP_SELF"];
	}
	/**
	 * クライアントから要求されたURI(パラメータ付き)を取得
	 *
	 * 例) http://www.magic3.org/magic3/index.php?aaa=bbb
	 */
	public function getCurrentRequestUri()
	{
		return self::getServerUrl() . $_SERVER["REQUEST_URI"];
	}
	/**
	 * 現在実行中のスクリプトファイルのパスを取得
	 *
	 * 例) /var/www/html/magic3/index.php
	 */
	public function getCurrentScriptPath()
	{
		return realpath($_SERVER["SCRIPT_FILENAME"]);
	}	
	/**
	 * ドキュメントルートを取得
	 *
	 * 例) /var/www/html
	 */
	public function getDocumentRoot()
	{
		return $_SERVER["DOCUMENT_ROOT"];
	}
	/**
	 * ドキュメントルートURLを取得
	 *
	 * 例) http://www.magic3.org, http://www.magic3.org:8080
	 */
	public function getDocumentRootUrl()
	{
		$rootUrl = parse_url($this->getRootUrl());
		$url = 'http://' . $rootUrl['host'];
		if (!empty($rootUrl['port'])) $url .= ':' . $rootUrl['port'];
		return $url;
	}
	/**
	 * システムルートURLを求める
	 *
	 * @return string		システムのルートURL。算出できなかったときは空文字列を返す。
	 */
	public function calcSystemRootUrl()
	{	
		// 相対パスを得る
		$base = explode(DIRECTORY_SEPARATOR, self::getSystemRootPath());
		$target = explode(DIRECTORY_SEPARATOR, self::getCurrentScriptPath());
		
		for ($i = 0; $i < count($base); $i++)
		{
			if ($base[$i] != $target[$i]) break;
		}
		$relativePath = '';
		for ($j = $i; $j < count($target); $j++)
		{
			$relativePath .= '/' . $target[$j];
		}
		// システムルートディレクトリ取得
		$sytemRootUrl = '';
		$pos = strrpos(self::getCurrentScriptUrl(), $relativePath);
		if (!($pos === false)){
			$sytemRootUrl = substr(self::getCurrentScriptUrl(), 0, $pos);
		}
		return $sytemRootUrl;
	}
	/**
	 * 相対パスを求める
	 *
	 * @param string $basePath		基点となるディレクトリの絶対パス
	 * @param string $targetPath	対象となるディレクトリの絶対パス
	 * @return string				相対パス
	 */
	public function calcRelativePath($basePath, $targetPath)
	{
		// 相対パスを得る
		$base = explode('/', $basePath);
		$target = explode('/', $targetPath);
		
		for ($i = 0; $i < count($base); $i++)
		{
			if ($base[$i] != $target[$i]) break;
		}
		$relativePath = '';
		for ($j = $i; $j < count($target); $j++)
		{
			$relativePath .= '/' . $target[$j];
		}
		return $relativePath;
	}
	/**
	 * パーマネントリンク用の現在のページURLを取得
	 *
	 * @param bool $hasSubPage		サブページIDを必ず付加するかどうか。必ず付加しない場合はデフォルトページのとき省略
	 * @return string				パーマネントリンクURL
	 */
	/*public function getCurrentPermanentPageUrl($hasSubPage=false)
	{
		$url = $this->getDefaultUrl();
		if ($hasSubPage || $this->currentPageSubId != $this->defaultPageSubId) $url .= '?sub=' . $this->currentPageSubId;
		return $url;
	}*/
	/**
	 * ドキュメントルートからのリソース用ディレクトリへの相対パスを取得
	 */
	public function getRelativeResourcePathToDocumentRoot()
	{
		// 相対パスを得る
		$res = parse_url($this->getResourceUrl());
		return $res['path'];
	}
	/**
	 * アプリケーションルートから指定ディレクトリへの相対パスを取得
	 *
	 * @param string $url	指定URL
	 */
	public function getRelativePathToSystemRootUrl($url)
	{
		// URLから相対パスを得る
		$root = parse_url($this->getRootUrl());
		$target = parse_url($url);		
		//return $this->calcRelativePath($this->getSystemRootPath(), $path);
		return $this->calcRelativePath($root['path'], $target['path']);
	}
	/**
	 * アプリケーションルートから指定ディレクトリへの相対パスを取得
	 *
	 * @param string $path	指定パス
	 * @return string		相対パス
	 */
	public function getRelativePathToSystemRootPath($path)
	{
		return $this->calcRelativePath($this->getSystemRootPath(), $path);
	}
	/**
	 * マクロ変換したパスを取得
	 *
	 * @param string $path	変換元パス(絶対パス、相対パス)
	 * @return string		変換したパス
	 */
	public function getMacroPath($path)
	{
		$destPath = $path;
		if (strncmp($destPath, '/', 1) == 0){		// 相対パス表記のとき
			$destPath = M3_TAG_START . M3_TAG_MACRO_ROOT_URL . M3_TAG_END . $this->getRelativePathToSystemRootUrl($this->getDocumentRootUrl() . $destPath);
		} else if (strncmp($destPath, 'http', strlen('http')) == 0 || strncmp($destPath, 'https', strlen('https')) == 0){				// 絶対パス表記のとき
			$destPath = str_replace($this->getRootUrl(), M3_TAG_START . M3_TAG_MACRO_ROOT_URL . M3_TAG_END, $destPath);// マクロ変換
		}
		return $destPath;
	}
	/**
	 * 管理者用URLへのアクセスかどうか
	 *
	 * @return bool		管理者用ディレクトリへのアクセスのときは、true。それ以外の場合はfalse。
	 */
	public function isAdminDirAccess()
	{
		if (dirname($this->getCurrentScriptPath()) == $this->getAdminPath()){
			return true;
		} else {
			return false;
		}
	}
	// ##################### カレントのテンプレート関係 #####################
	/**
	 * 現在のテンプレート
	 */
	public function setCurrentTemplateId($name)
	{
		$this->currentTemplateId = $name;

		// テンプレートの属性を取得
		$this->currentTemplateType = 0;
		if ($this->canUseDb){		// DB使用可能なとき
			if ($this->db->getTemplate($name, $row)){
				$this->currentTemplateType = $row['tm_type'];		// テンプレートタイプ
			}
		}
	}
	/**
	 * 現在テンプレート
	 */
	public function getCurrentTemplateId()
	{
		return $this->currentTemplateId;
	}
	/**
	 * 現在のテンプレートタイプ
	 *
	 * @return int		0=デフォルトテンプレート(Joomla!v1.0)、1=Joomla!v1.5
	 */
	public function getCurrentTemplateType()
	{
		return $this->currentTemplateType;
	}
	/**
	 * 現在のテンプレートへのパスを取得
	 *
	 * 例) /var/www/html/magic3/templates/menu
	 */
	public function getCurrentTemplatePath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'templates' . DIRECTORY_SEPARATOR . $this->currentTemplateId;
	}
	/**
	 * 現在のテンプレートへのURLを取得
	 *
	 * 例) http://www.magic3.org/magic3/templates/menu
	 */
	public function getCurrentTemplateUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/templates/' . $this->currentTemplateId;
	}
	
	// ##################### カレントのウィジェット関係 #####################
	/**
	 * 現在処理中のウィジェットのルートディレクトリへのパスを取得
	 *
	 * 例) http://www.magic3.org/magic3/widgets/xxxxx
	 */
	public function getCurrentWidgetRootUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/widgets/' . $this->currentWidgetId;
	}
	/**
	 * 現在処理中のウィジェットのルートディレクトリへのパスを取得
	 */
	public function getCurrentWidgetRootPath()
	{
		return M3_SYSTEM_ROOT_PATH . DIRECTORY_SEPARATOR . 'widgets' . DIRECTORY_SEPARATOR . $this->currentWidgetId;
	}
	/**
	 * 現在処理中のウィジェットのincludeディレクトリへのパスを取得
	 */
	public function getCurrentWidgetIncludePath()
	{
		return M3_SYSTEM_ROOT_PATH . '/widgets/' . $this->currentWidgetId . '/include';
	}
	/**
	 * 現在処理中のウィジェットのdbディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/db
	 */
	public function getCurrentWidgetDbPath()
	{
		return $this->getWidgetsPath() . DIRECTORY_SEPARATOR . $this->currentWidgetId . DIRECTORY_SEPARATOR . 'include' . DIRECTORY_SEPARATOR . 'db';
	}
	/**
	 * 現在処理中のウィジェットのcontainerディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/container
	 */
	public function getCurrentWidgetContainerPath()
	{
		return $this->getWidgetsPath() . DIRECTORY_SEPARATOR . $this->currentWidgetId . DIRECTORY_SEPARATOR . 'include' . DIRECTORY_SEPARATOR . 'container';
	}
	/**
	 * 現在処理中のウィジェットのlibディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/lib
	 */
	public function getCurrentWidgetLibPath()
	{
		return $this->getWidgetsPath() . DIRECTORY_SEPARATOR . $this->currentWidgetId . DIRECTORY_SEPARATOR . 'include' . DIRECTORY_SEPARATOR . 'lib';
	}
	/**
	 * 現在処理中のウィジェットのtemplateディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/template
	 */
	public function getCurrentWidgetTemplatePath()
	{
		return $this->getWidgetsPath() . DIRECTORY_SEPARATOR . $this->currentWidgetId . DIRECTORY_SEPARATOR . 'include' . DIRECTORY_SEPARATOR . 'template';
	}
	/**
	 * 現在処理中のウィジェットのsqlディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/sql
	 */
	public function getCurrentWidgetSqlPath()
	{
		return $this->getWidgetsPath() . DIRECTORY_SEPARATOR . $this->currentWidgetId . DIRECTORY_SEPARATOR . 'include' . DIRECTORY_SEPARATOR . 'sql';
	}
	/**
	 * 現在処理中のウィジェットのscriptsディレクトリへURLを取得
	 */
	public function getCurrentWidgetScriptsUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/widgets/' . $this->currentWidgetId . '/scripts';
	}
	/**
	 * 現在処理中のウィジェットのimagesディレクトリへURLを取得
	 */
	public function getCurrentWidgetImagesUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/widgets/' . $this->currentWidgetId . '/images';
	}
	/**
	 * 現在処理中のウィジェットのscriptsディレクトリへURLを取得
	 */
	public function getCurrentWidgetCssUrl()
	{
		return M3_SYSTEM_ROOT_URL . '/widgets/' . $this->currentWidgetId . '/css';
	}
	/**
	 * 現在処理中のインナーウィジェットのルートディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/iwidgets/yyyyy
	 */
	public function getCurrentIWidgetRootPath()
	{
		// ウィジェットIDとインナーウィジェットIDを取り出す
		list($widgetId, $iWidgetId) = explode(M3_WIDGET_ID_SEPARATOR, $this->currentIWidgetId);

		return $this->getWidgetsPath() . '/' . $widgetId . '/include/iwidgets/' . $iWidgetId;
	}
	/**
	 * 現在処理中のインナーウィジェットのdbディレクトリへのパスを取得
	 *
	 * 例) /var/www/html/magic3/widgets/xxxxx/include/iwidgets/yyyyy/include/db
	 */
	public function getCurrentIWidgetDbPath()
	{
		// ウィジェットIDとインナーウィジェットIDを取り出す
		list($widgetId, $iWidgetId) = explode(M3_WIDGET_ID_SEPARATOR, $this->currentIWidgetId);

		return $this->getWidgetsPath() . '/' . $widgetId . '/include/iwidgets/' . $iWidgetId . DIRECTORY_SEPARATOR . 'include' . DIRECTORY_SEPARATOR . 'db';
	}
	/**
	 * 現在処理中のウィジェット
	 */
	public function setCurrentWidgetId($id)
	{
		$this->currentWidgetId = $id;
	}
	/**
	 * 現在処理中のウィジェット
	 */
	public function getCurrentWidgetId()
	{
		return $this->currentWidgetId;
	}
	/**
	 * 現在処理中のインナーウィジェット
	 */
	public function setCurrentIWidgetId($id)
	{
		$this->currentIWidgetId = $id;
	}
	/**
	 * 現在処理中のインナーウィジェット
	 */
	public function getCurrentIWidgetId()
	{
		return $this->currentIWidgetId;
	}
	/**
	 * 現在作成中のウィジェットの定義ID
	 *
	 * @param int $id		定義ID
	 */
	public function setCurrentWidgetConfigId($id)
	{
		$this->currentWidgetConfigId = $id;
	}
	/**
	 * 現在作成中のウィジェットの定義ID
	 *
	 * @return int			定義ID
	 */
	public function getCurrentWidgetConfigId()
	{
		return $this->currentWidgetConfigId;
	}
	/**
	 * 現在作成中のインナーウィジェットの定義ID
	 *
	 * @param int $id		定義ID
	 */
	public function setCurrentIWidgetConfigId($id)
	{
		$this->currentIWidgetConfigId = $id;
	}
	/**
	 * 現在作成中のインナーウィジェットの定義ID
	 *
	 * @return int			定義ID
	 */
	public function getCurrentIWidgetConfigId()
	{
		return $this->currentIWidgetConfigId;
	}
	/**
	 * 現在処理中のウィジェットのプレフィックス文字列
	 */
	public function setCurrentWidgetPrefix($val)
	{
		$this->currentWidgetPrefix = $val;
	}
	/**
	 * 現在処理中のウィジェットのプレフィックス文字列
	 */
	public function getCurrentWidgetPrefix()
	{
		return $this->currentWidgetPrefix;
	}
	/**
	 * 現在処理中のウィジェットのタイトル文字列
	 */
	public function setCurrentWidgetTitle($val)
	{
		$this->currentWidgetTitle = $val;
	}
	/**
	 * 現在処理中のウィジェットのタイトル文字列
	 */
	public function getCurrentWidgetTitle()
	{
		return $this->currentWidgetTitle;
	}
	/**
	 * 現在処理を行っているページ定義のレコードのシリアル番号
	 *
	 * @param int $serial		シリアル番号
	 */
	public function setCurrentPageDefSerial($serial)
	{
		$this->currentPageDefSerial = $serial;
	}
	/**
	 * 現在処理を行っているページ定義のレコードのシリアル番号
	 *
	 * @return int			シリアル番号
	 */
	public function getCurrentPageDefSerial()
	{
		return $this->currentPageDefSerial;
	}
	// ##################### ユーザ情報 #####################
	/**
	 * 現在アクセス中のユーザ情報取得
	 *
	 * @return UserInfo		ユーザ情報。設定されていない場合はnullを返す。
	 */
	public function getCurrentUserInfo()
	{
		global $gInstanceManager;

		return $gInstanceManager->getUserInfo();
	}
	/**
	 * 現在アクセス中のユーザID取得
	 *
	 * @return int		ユーザID,ユーザが確定できないときは0
	 */
	public function getCurrentUserId()
	{
		global $gInstanceManager;

		$userInfo = $gInstanceManager->getUserInfo();
		if (is_null($userInfo)){		// ログインしていない場合
			return 0;
		} else {
			return $userInfo->userId;
		}
	}
	/**
	 * 現在アクセス中のユーザの名前を取得
	 *
	 * @return string		ユーザ名,ユーザが確定できないときは空文字列
	 */
	public function getCurrentUserName()
	{
		global $gInstanceManager;

		$userInfo = $gInstanceManager->getUserInfo();
		if (is_null($userInfo)){		// ログインしていない場合
			return '';
		} else {
			return $userInfo->name;
		}
	}
	/**
	 * 管理者用一時キーを取得
	 *
	 * @return string		管理者キー、管理者でないときは空文字列
	 */
	public function getAdminKey()
	{
		global $gInstanceManager;

		$userInfo = $gInstanceManager->getUserInfo();
		if (is_null($userInfo)){		// ログインしていない場合
			return '';
		} else {
			return $userInfo->_adminKey;
		}
	}
	/**
	 * 現在アクセス中のユーザに管理者権限があるかどうかを返す
	 *
	 * @return bool		true=ログイン中かつ管理者権限あり、false=未ログインまたはログイン中であるが管理者権限なし
	 */
	public function isSystemAdmin()
	{
		$isAdmin = false;
		$userInfo = $this->getCurrentUserInfo();
		if (is_null($userInfo)){		// ログインしていない場合
		} else {
			if ($userInfo->isSystemAdmin()){	// システム管理者の場合
				$isAdmin = true;
			}
		}
		return $isAdmin;
	}
	/**
	 * 現在アクセス中のユーザがログインしているか確認
	 *
	 * @return bool		true=ログイン中、false=未ログイン
	 */
	public function isCurrentUserLogined()
	{
		$userInfo = $this->getCurrentUserInfo();
		if (is_null($userInfo)){		// ログインしていない場合
			return false;
		} else {
			return true;
		}
	}
	/**
	 * 指定のユーザに管理者権限があるかどうかを返す
	 *
	 * @param int $userId	ユーザID
	 * @return bool			true=管理者権限あり、false=管理者権限なし
	 */
	public function isSystemAdminUser($userId)
	{
		return $this->db->isSystemAdmin($userId);
	}
	// ##################### アクセスログ #####################
	/**
	 * 現在のアクセスログのシリアル番号を返す
	 *
	 * @return int			アクセスログシリアル番号
	 */
	public function getCurrentAccessLogSerial()
	{
		global $gAccessManager;
		return $gAccessManager->getAccessLogSerialNo();
	}
	// ##################### ページ制御 #####################
	/**
	 * デフォルトのページID取得
	 *
	 * @return string			デフォルトのページID
	 */
	public function getDefaultPageId()
	{
		return self::DEFAULT_PAGE_ID;
	}
	/**
	 * 携帯用デフォルトのページID取得
	 *
	 * @return string			携帯用デフォルトのページID
	 */
	public function getDefaultMobilePageId()
	{
		return self::DEFAULT_MOBILE_PAGE_ID;
	}
	/**
	 * デフォルトの管理機能用ページID取得
	 *
	 * @return string			デフォルトの管理機能用ページID
	 */
	public function getDefaultAdminPageId()
	{
		return self::DEFAULT_ADMIN_PAGE_ID;
	}
	/**
	 * 現在のページID
	 */
	public function setCurrentPageId($id)
	{
		$this->currentPageId = $id;
		
		// 現在のページIDが変更のときは、デフォルトのページサブIDを更新
		if ($this->canUseDb) $this->defaultPageSubId = $this->db->getDefaultPageSubId($this->currentPageId);
	}
	/**
	 * 現在のページID
	 */
	public function getCurrentPageId()
	{
		return $this->currentPageId;
	}
	/**
	 * 現在のページサブID
	 */
	public function setCurrentPageSubId($id)
	{
		$this->currentPageSubId = $id;
	}
	/**
	 * 現在のページサブID
	 */
	public function getCurrentPageSubId()
	{
		return $this->currentPageSubId;
	}
	/**
	 * 現在のページのデフォルトのページサブID
	 *
	 * @return string 		デフォルトのページサブID
	 */
	public function getDefaultPageSubId()
	{
		return $this->defaultPageSubId;
	}
	/**
	 * 指定ページのデフォルトのページサブID
	 *
	 * @param string $pageId	ページID
	 * @return string 			デフォルトのページサブID
	 */
	public function getDefaultPageSubIdByPageId($pageId)
	{
		return $this->db->getDefaultPageSubId($pageId);
	}
	/**
	 * 現在のページID、サブページIDのURLを作成
	 *
	 * @return string		作成したURL
	 */
	public function createCurrentPageUrl()
	{
		return $this->getCurrentScriptUrl() . '?sub=' . $this->getCurrentPageSubId();
	}
	/**
	 * 現在のページID、サブページIDのURLを作成(セッションID付き)
	 *
	 * @return string		作成したURL
	 */
	/*public function createCurrentPageUrlWithSessionId()
	{
		return $this->getCurrentScriptUrl() . '?sub=' . $this->getCurrentPageSubId() . '&' . session_name() . '=' . session_id();
	}*/
	/**
	 * 携帯用の現在のページID、サブページIDのURLを作成
	 *
	 * 携帯用URLには以下の情報を付加する
	 * ・セッションID
	 * ・ドコモ端末の場合はiモードID受信用のパラメータを付加
	 *
	 * @param string,array	$addParam 追加パラメータ
	 * @return string		作成したURL
	 */
	public function createCurrentPageUrlForMobile($addParam='')
	{
		// 携帯用パラメータ取得
		$param = $this->_getMobileUrlParam();
		
		// ページサブID付加
		$param['sub'] = $this->getCurrentPageSubId();

		// 追加パラメータがある場合
		if (!empty($addParam)){
			if (is_array($addParam)){		// 配列の場合
				$newParam = $addParam;
			} else {		// 文字列の場合
				$newParam = array();
				$addParamArray = explode('&', trim($addParam, "?&"));
				for ($i = 0; $i < count($addParamArray); $i++){
					list($key, $value) = explode('=', $addParamArray[$i]);
					$key = trim($key);
					$value = trim($value);
					$newParam[$key] = $value;
				}
			}
			$param = array_merge($param, $newParam);
		}
		$url = $this->_createUrl($this->getCurrentScriptUrl(), $param);
		return $url;
	}
	/**
	 * デフォルト言語取得
	 */
	public function getDefaultLanguage()
	{
		return $this->defaultLanguage;
	}
	/**
	 * デフォルトの言語名をカレントの言語で表したもの
	 */
	public function getDefaultLanguageNameByCurrentLanguage()
	{
		return $this->db->getLanguageNameByDispLanguageId($this->defaultLanguage, $this->currentLanguage);
	}
	/**
	 * カレント言語取得
	 */
	public function getCurrentLanguage()
	{
		return $this->currentLanguage;
	}
	/**
	 * カレント言語設定
	 */
	public function setCurrentLanguage($value)
	{
		$this->currentLanguage = $value;
	}
	/**
	 * カレント言語の変更可否を取得
	 */
	public function getCanChangeLang()
	{
		return $this->canChangeLang;
	}
	/**
	 * 多言語対応かどうかを取得
	 */
	public function getMultiLanguage()
	{
		return $this->multiLanguage;
	}
	/**
	 * 携帯用の入出力エンコーディングを設定
	 *
	 * @param string		エンコーディング
	 */
	public function setMobileEncoding($value)
	{
		$this->mobileEncoding = $value;
	}
	/**
	 * 携帯用の入出力エンコーディングを取得
	 *
	 * @return string		エンコーディング
	 */
	public function getMobileEncoding()
	{
		return $this->mobileEncoding;
	}
	/**
	 * 携帯用HTML上のエンコーディング表記を設定
	 *
	 * @param string		エンコーディング
	 */
	public function setMobileCharset($value)
	{
		$this->mobileCharset = $value;
	}
	/**
	 * 携帯用HTML上のエンコーディング表記を取得
	 *
	 * @return string		エンコーディング
	 */
	public function getMobileCharset()
	{
		return $this->mobileCharset;
	}
	/**
	 * DBセッションが使用できるかどうか
	 */
	public function canUseDbSession()
	{
		return $this->canUseDbSession;
	}
	/**
	 * DBが使用可能かどうか
	 */
	public function canUseDb()
	{
		return $this->canUseDb;
	}
	/**
	 * Timestamp型データの初期値を取得
	 *
	 * @param string Timestmp型初期データ文字列
	 */
	public function getInitValueOfTimestamp()
	{
		if ($this->db->getDbType() == M3_DB_TYPE_MYSQL){		// MySQLの場合
			return M3_TIMESTAMP_INIT_VALUE_MYSQL;
		} else if ($this->db->getDbType() == M3_DB_TYPE_PGSQL){
			return M3_TIMESTAMP_INIT_VALUE_PGSQL;
		} else {
			return '';
		}
	}
	/**
	 * Date型データの初期値を取得
	 *
	 * @param string Date型初期データ文字列
	 */
	public function getInitValueOfDate()
	{
		if ($this->db->getDbType() == M3_DB_TYPE_MYSQL){		// MySQLの場合
			return M3_DATE_INIT_VALUE_MYSQL;
		} else if ($this->db->getDbType() == M3_DB_TYPE_PGSQL){
			return M3_DATE_INIT_VALUE_PGSQL;
		} else {
			return '';
		}
	}
	/**
	 * サイトの名称を取得
	 *
	 * @param bool $reload	データを再取得するかどうか
	 * @return string		サイト名称
	 */
	public function getSiteName($reload = false)
	{
		// DBが使用不可のときは空文字列を返す
		if (!$this->canUseDb) return '';
		
		if ($reload || empty($this->siteName)){			// サイト名称
			$this->siteName = $this->db->getSiteDefValue(M3_TB_FIELD_SITE_NAME);
		}
		return $this->siteName;
	}
	/**
	 * サイトの所有者を取得
	 *
	 * @param bool $reload	データを再取得するかどうか
	 * @return string		サイト所有者
	 */
	public function getSiteOwner($reload = false)
	{
		// DBが使用不可のときは空文字列を返す
		if (!$this->canUseDb) return '';
		
		if ($reload || empty($this->siteOwner)){			// サイト所有者
			$this->siteOwner = $this->db->getSiteDefValue(M3_TB_FIELD_SITE_OWNER);
		}
		return $this->siteOwner;
	}
	/**
	 * サイトコピーライトを取得
	 *
	 * @param bool $reload	データを再取得するかどうか
	 * @return string		サイトコピーライト
	 */
	public function getSiteCopyRight($reload = false)
	{
		// DBが使用不可のときは空文字列を返す
		if (!$this->canUseDb) return '';
		
		if ($reload || empty($this->siteCopyRight)){			// サイトコピーライト
			$this->siteCopyRight = $this->db->getSiteDefValue(M3_TB_FIELD_SITE_COPYRIGHT);
		}
		return $this->siteCopyRight;
	}
	/**
	 * サイトEメールを取得
	 *
	 * @param bool $reload	データを再取得するかどうか
	 * @return string		サイトEメール
	 */
	public function getSiteEmail($reload = false)
	{
		// DBが使用不可のときは空文字列を返す
		if (!$this->canUseDb) return '';
		
		if ($reload || empty($this->siteEmail)){			// サイトEメール
			$this->siteEmail = $this->db->getSiteDefValue(M3_TB_FIELD_SITE_EMAIL);
		}
		return $this->siteEmail;
	}
	/**
	 * デフォルトCSV区切り文字コードを取得
	 *
	 * @return string		区切り文字
	 */
	public function getDefaultCsvDelimCode()
	{
		return $this->db->getSystemConfig(self::DEFAULT_CSV_DELIM_CODE);
	}
	/**
	 * デフォルトCSV改行コードを取得
	 *
	 * @return string		改行文字
	 */
	public function getDefaultCsvNLCode()
	{
		return $this->db->getSystemConfig(self::DEFAULT_CSV_NL_CODE);
	}
	/**
	 * デフォルトCSVファイル拡張子を取得
	 *
	 * @return string		改行文字
	 */
	public function getDefaultCsvFileSuffix()
	{
		return $this->db->getSystemConfig(self::DEFAULT_CSV_FILE_SUFFIX);
	}
	/**
	 * ウィジェット実行ログを残すかどうか
	 *
	 * @return bool		true=ログ出力、false=ログ出力しない
	 */
	public function getWidgetLog()
	{
		return $this->widgetLog;
	}
	
	/**
	 * 携帯アクセスかどうかを設定
	 *
	 * @param bool $status			true=携帯アクセス、false=通常アクセス
	 * @return 			なし
	 */
	public function setIsMobileSite($status)
	{
		$this->isMobileSite = $status;
		
		if ($status){
			// ##### 携帯用の設定 #####
			// セッションをURLに保存
			ini_set('session.use_cookies', 0);	// クッキーは使用しない
		}
	}
	/**
	 * 携帯アクセスかどうか
	 *
	 * @return bool		true=携帯アクセス、false=通常アクセス
	 */
	public function getIsMobileSite()
	{
		return $this->isMobileSite;
	}
	/**
	 * 携帯端末IDを取得
	 *
	 * @return string		携帯端末ID
	 */
	public function getMobileId()
	{
		global $gInstanceManager;
		global $gRequestManager;
		
		$agent = $gInstanceManager->getMobileAgent();
		if ($agent->isDoCoMo()){	// ドコモ端末のとき
			$mobileId = $gRequestManager->trimServerValueOf('HTTP_X_DCMGUID');
			if (!empty($mobileId)) $mobileId = 'DC-' . $mobileId;		// キャリアコードを付加
		} else if ($agent->isEZweb()){	// au端末のとき
			$mobileId = $gRequestManager->trimServerValueOf('HTTP_X_UP_SUBNO');
			// ドメイン名を消去
			$pos = strpos($mobileId, '.ezweb.ne.jp');
			if ($pos !== false) $mobileId = substr($mobileId, 0, $pos);
			if (!empty($mobileId)) $mobileId = 'AU-' . $mobileId;		// キャリアコードを付加
		} else if ($agent->isSoftBank()){	// ソフトバンク端末のとき
			$mobileId = $gRequestManager->trimServerValueOf('HTTP_X_JPHONE_UID');
			if (!empty($mobileId)) $mobileId = 'SB-' . $mobileId;		// キャリアコードを付加
		} else {		// その他の端末のとき(PC用)
			$mobileId = '';
		}
		return $mobileId;
	}
	/**
	 * 携帯端末でのアクセスかどうか
	 *
	 * @return bool		true=携帯端末アクセス、false=携帯端末以外からのアクセス
	 */
	public function isMobile()
	{
		global $gInstanceManager;
		
		$agent = $gInstanceManager->getMobileAgent();
		if (method_exists($agent, 'isNonMobile')){
			if (!$agent->isNonMobile()){			// 携帯端末でのアクセスの場合
				return true;
			}
		}
		return false;
	}
	/**
	 * 携帯端末でのアクセスかどうか
	 *
	 * @return bool		true=携帯端末アクセス、false=携帯端末以外からのアクセス
	 */
	public function isMobileAgent()
	{
		global $gRequestManager;
		
		if (empty($this->userAgent)){
			$agents = array(
				// pattern: A regular-expression that matches device(browser)'s name and version
				// profile: A group of browsers
				// Windows CE (Microsoft(R) Internet Explorer 5.5 for Windows(R) CE)
				// Sample: "Mozilla/4.0 (compatible; MSIE 5.5; Windows CE; sigmarion3)" (sigmarion, Hand-held PC)
				array('pattern'=>'#\b(?:MSIE [5-9]).*\b(Windows CE)\b#',	'type'	=> self::USER_AGENT_TYPE_PC),

				// ACCESS "NetFront" / "Compact NetFront" and thier OEM, expects to be "Mozilla/4.0"
				// Sample: "Mozilla/4.0 (PS2; PlayStation BB Navigator 1.0) NetFront/3.0" (PlayStation BB Navigator, for SONY PlayStation 2)
				// Sample: "Mozilla/4.0 (PDA; PalmOS/sony/model crdb/Revision:1.1.19) NetFront/3.0" (SONY Clie series)
				// Sample: "Mozilla/4.0 (PDA; SL-A300/1.0,Embedix/Qtopia/1.1.0) NetFront/3.0" (SHARP Zaurus)
				array('pattern'=>'#^(?:Mozilla/4).*\b(NetFront)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_PC),

				// Embedded browsers (Non-rich)

				// Windows CE (the others)
				// Sample: "Mozilla/2.0 (compatible; MSIE 3.02; Windows CE; 240x320 )" (GFORT, NTT DoCoMo)
				array('pattern'=>'#\b(Windows CE)\b#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// ACCESS "NetFront" / "Compact NetFront" and thier OEM
				// Sample: "Mozilla/3.0 (AveFront/2.6)" ("SUNTAC OnlineStation", USB-Modem for PlayStation 2)
				// Sample: "Mozilla/3.0(DDIPOCKET;JRC/AH-J3001V,AH-J3002V/1.0/0100/c50)CNF/2.0" (DDI Pocket: AirH" Phone by JRC)
				array('pattern'=>'#\b(NetFront)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),
				array('pattern'=>'#\b(CNF)/([0-9\.]+)#',		'type'	=> self::USER_AGENT_TYPE_MOBILE),
				array('pattern'=>'#\b(AveFront)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),
				array('pattern'=>'#\b(AVE-Front)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE), // The same?

				// NTT-DoCoMo, i-mode (embeded Compact NetFront) and FOMA (embedded NetFront) phones
				// Sample: "DoCoMo/1.0/F501i", "DoCoMo/1.0/N504i/c10/TB/serXXXX" // c以降は可変
				// Sample: "DoCoMo/2.0 MST_v_SH2101V(c100;TB;W22H12;serXXXX;iccxxxx)" // ()の中は可変
				array('pattern'=>'#^(DoCoMo)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// Vodafone's embedded browser
				// Sample: "J-PHONE/2.0/J-T03"	// 2.0は"ブラウザの"バージョン
				// Sample: "J-PHONE/4.0/J-SH51/SNxxxx SH/0001a Profile/MIDP-1.0 Configuration/CLDC-1.0 Ext-Profile/JSCL-1.1.0"
				array('pattern'=>'#^(J-PHONE)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// Openwave(R) Mobile Browser (EZweb, WAP phone, etc)
				// Sample: "OPWV-SDK/62K UP.Browser/6.2.0.5.136 (GUI) MMP/2.0"
				array('pattern'=>'#\b(UP\.Browser)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// Opera, dressing up as other embedded browsers
				// Sample: "Mozilla/3.0(DDIPOCKET;KYOCERA/AH-K3001V/1.4.1.67.000000/0.1/C100) Opera 7.0" (Like CNF at 'keitai'-mode)
				array('pattern'=>'#\b(?:DDIPOCKET|WILLCOM)\b.+\b(Opera) ([0-9\.]+)\b#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// Planetweb http://www.planetweb.com/
				// Sample: "Mozilla/3.0 (Planetweb/v1.07 Build 141; SPS JP)" ("EGBROWSER", Web browser for PlayStation 2)
				array('pattern'=>'#\b(Planetweb)/v([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// DreamPassport, Web browser for SEGA DreamCast
				// Sample: "Mozilla/3.0 (DreamPassport/3.0)"
				array('pattern'=>'#\b(DreamPassport)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// Palm "Web Pro" http://www.palmone.com/us/support/accessories/webpro/
				// Sample: "Mozilla/4.76 [en] (PalmOS; U; WebPro)"
				array('pattern'=>'#\b(WebPro)\b#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// ilinx "Palmscape" / "Xiino" http://www.ilinx.co.jp/
				// Sample: "Xiino/2.1SJ [ja] (v. 4.1; 153x130; c16/d)"
				array('pattern'=>'#^(Palmscape)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),
				array('pattern'=>'#^(Xiino)/([0-9\.]+)#',		'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// SHARP PDA Browser (SHARP Zaurus)
				// Sample: "sharp pda browser/6.1[ja](MI-E1/1.0) "
				array('pattern'=>'#^(sharp [a-z]+ browser)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// WebTV
				array('pattern'=>'#^(WebTV)/([0-9\.]+)#',	'type'	=> self::USER_AGENT_TYPE_MOBILE),

				// Desktop-PC browsers

				// Opera (for desktop PC, not embedded) -- See BugTrack/743 for detail
				// NOTE: Keep this pattern above MSIE and Mozilla
				// Sample: "Opera/7.0 (OS; U)" (not disguise)
				// Sample: "Mozilla/4.0 (compatible; MSIE 5.0; OS) Opera 6.0" (disguise)
				array('pattern'=>'#\b(Opera)[/ ]([0-9\.]+)\b#',	'type'	=> self::USER_AGENT_TYPE_PC),

				// MSIE: Microsoft Internet Explorer (or something disguised as MSIE)
				// Sample: "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0)"
				array('pattern'=>'#\b(MSIE) ([0-9\.]+)\b#',	'type'	=> self::USER_AGENT_TYPE_PC),

				// Mozilla Firefox
				// NOTE: Keep this pattern above Mozilla
				// Sample: "Mozilla/5.0 (Windows; U; Windows NT 5.0; ja-JP; rv:1.7) Gecko/20040803 Firefox/0.9.3"
				array('pattern'=>'#\b(Firefox)/([0-9\.]+)\b#',	'type'	=> self::USER_AGENT_TYPE_PC),

				// Loose default: Including something Mozilla
				array('pattern'=>'#^([a-zA-z0-9 ]+)/([0-9\.]+)\b#',	'type'	=> self::USER_AGENT_TYPE_PC),

				array('pattern'=>'#^#',	'type'	=> self::USER_AGENT_TYPE_PC),	// Sentinel
			);

			$userAgent = $gRequestManager->trimServerValueOf('HTTP_USER_AGENT');
			foreach ($agents as $agent) {
				if (preg_match($agent['pattern'], $userAgent, $matches)) {
					$this->userAgent['type'] = isset($agent['type']) ? $agent['type'] : '';
					$this->userAgent['name']    = isset($matches[1]) ? $matches[1] : '';	// device or browser name
					$this->userAgent['vers']    = isset($matches[2]) ? $matches[2] : ''; // 's version
					break;
				}
			}
		}
		if ($this->userAgent['type'] == self::USER_AGENT_TYPE_MOBILE){			// アクセス端末の種類(携帯)
			return true;
		} else {
			return false;
		}
	}
	/**
	 * URLを作成
	 *
	 * @param string		$baseUrl	URL
	 * @param string,array	$addParam 	追加パラメータ
	 * @return string					作成したURL
	 */
	function _createUrl($baseUrl, $addParam='')
	{
		//$pathArray = parse_url($baseUrl);
		$url = $baseUrl;
		
		// 追加パラメータがある場合
		if (!empty($addParam)){
			if (strEndsWith($url, '.php')) $url .= '?';
			
			if (is_array($addParam)){		// 配列の場合
				foreach (array_keys($addParam) as $key){
					if (!strEndsWith($url, '?')) $url .= '&';
					$url .= urlencode($key) . '=' . urlencode($addParam[$key]);
				}
			} else {
				if (!strEndsWith($url, '?')) $url .= '&';
				$url .= trim($addParam, '&');
			}
		}
		return $url;
	}
	/**
	 * 携帯用のURLパラメータを取得
	 *
	 * @return array			URLパラメータ
	 */
	function _getMobileUrlParam()
	{
		global $gInstanceManager;
		
		$param = array();
		$agent = $gInstanceManager->getMobileAgent();
		if (method_exists($agent, 'isNonMobile')){
			if (!$agent->isNonMobile()){			// 携帯端末でのアクセスの場合
				// ログインしている場合はセッションIDを付加
				if ($this->isCurrentUserLogined()) $param[session_name()] = session_id();
					
				// ドコモ端末の場合はiモードIDを送信させる
				if ($agent->isDoCoMo()) $param['guid'] = 'ON';
			}
		}
		return $param;
	}
}
?>
