<?php
/**
 * コンテナクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2009 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: admin_mainUserlistWidgetContainer.php 2018 2009-06-29 02:16:46Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getContainerPath() . '/baseWidgetContainer.php');
require_once($gEnvManager->getCurrentWidgetDbPath() . '/admin_mainDb.php');
require_once($gEnvManager->getIncludePath() . '/common/userInfo.php');		// ユーザ情報クラス

class admin_mainUserlistWidgetContainer extends BaseWidgetContainer
{
	private $db;	// DB接続オブジェクト
	private $sysDb;		// システムDBオブジェクト
	private $serialNo;	// シリアルNo
	private $serialArray = array();		// 表示されているコンテンツシリアル番号
	private $userTypeArray;		// ユーザ種別(-1=未承認ユーザ、0=仮ユーザ、10=一般ユーザ、50=システム運営者、100=システム管理者)
	private $changeUserTypeArray;// ユーザタイプ変更用
	private $userType;		// ユーザタイプ
	const DEFAULT_LIST_COUNT = 30;			// 最大リスト表示数
	const DEFAULT_PASSWORD = '********';	// 設定済みを示すパスワード
		
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		// 親クラスを呼び出す
		parent::__construct();
		
		// DB接続オブジェクト作成
		$this->db = new admin_mainDb();
		$this->sysDb = $this->gInstance->getSytemDbObject();
		
		// ユーザタイプメニュー項目
		$this->userTypeArray = array(	array(	'name' => '-- 未選択 --',		'value' => ''),
										array(	'name' => '一般ユーザ',			'value' => strval(UserInfo::USER_TYPE_NORMAL)),
										array(	'name' => 'システム運営者',		'value' => strval(UserInfo::USER_TYPE_MANAGER)),	// このレベル以上が管理機能が使用できる
										array(	'name' => 'システム管理者',		'value' => strval(UserInfo::USER_TYPE_SYS_ADMIN)));
		// ユーザタイプ変更用
		$this->changeUserTypeArray = array(	array(	'name' => '未承認ユーザ',		'value' => strval(UserInfo::USER_TYPE_NOT_AUTHENTICATED)),
											array(	'name' => '仮ユーザ',			'value' => strval(UserInfo::USER_TYPE_TMP)),
											array(	'name' => '一般ユーザ',			'value' => strval(UserInfo::USER_TYPE_NORMAL)));
	}
	/**
	 * テンプレートファイルを設定
	 *
	 * _assign()でデータを埋め込むテンプレートファイルのファイル名を返す。
	 * 読み込むディレクトリは、「自ウィジェットディレクトリ/include/template」に固定。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。そのまま_assign()に渡る
	 * @return string 						テンプレートファイル名。テンプレートライブラリを使用しない場合は空文字列「''」を返す。
	 */
	function _setTemplate($request, &$param)
	{
		$task = $request->trimValueOf('task');
		if ($task == 'userlist_detail'){		// 詳細画面
			return 'userlist_detail.tmpl.html';
		} else {			// 一覧画面
			return 'userlist.tmpl.html';
		}
	}
	/**
	 * テンプレートにデータ埋め込む
	 *
	 * _setTemplate()で指定したテンプレートファイルにデータを埋め込む。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。_setTemplate()と共有。
	 * @param								なし
	 */
	function _assign($request, &$param)
	{
		$task = $request->trimValueOf('task');
		if ($task == 'userlist_detail'){	// 詳細画面
			return $this->createDetail($request);
		} else {			// 一覧画面
			return $this->createList($request);
		}
	}
	/**
	 * 一覧画面作成
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param								なし
	 */
	function createList($request)
	{
		$act = $request->trimValueOf('act');
		
		if ($act == 'delete'){		// メニュー項目の削除
			$listedItem = explode(',', $request->trimValueOf('seriallist'));
			$delItems = array();
			for ($i = 0; $i < count($listedItem); $i++){
				// 項目がチェックされているかを取得
				$itemName = 'item' . $i . '_selected';
				$itemValue = ($request->trimValueOf($itemName) == 'on') ? 1 : 0;
				
				if ($itemValue){		// チェック項目
					$delItems[] = $listedItem[$i];
				}
			}
			if (count($delItems) > 0){
				$ret = $this->db->delUserBySerial($delItems);
				if ($ret){		// データ削除成功のとき
					$this->setGuidanceMsg('データを削除しました');
				} else {
					$this->setAppErrorMsg('データ削除に失敗しました');
				}
			}
		}
		
		$maxListCount = self::DEFAULT_LIST_COUNT;				// 表示項目数
		$pageNo = $request->trimValueOf('page');				// ページ番号
		if (empty($pageNo)) $pageNo = 1;
		
		// 総数を取得
		$totalCount = $this->db->getAllUserListCount();

		// 表示するページ番号の修正
		$pageCount = (int)(($totalCount -1) / $maxListCount) + 1;		// 総ページ数
		if ($pageNo < 1) $pageNo = 1;
		if ($pageNo > $pageCount) $pageNo = $pageCount;
		$this->firstNo = ($pageNo -1) * $maxListCount + 1;		// 先頭番号
		
		// ページング用リンク作成
		$pageLink = '';
		if ($pageCount > 1){	// ページが2ページ以上のときリンクを作成
			for ($i = 1; $i <= $pageCount; $i++){
				if ($i == $pageNo){
					$link = '&nbsp;' . $i;
				} else {
					$link = '&nbsp;<a href="#" onclick="selpage(\'' . $i . '\');return false;">' . $i . '</a>';
				}
				$pageLink .= $link;
			}
		}
		$this->tmpl->addVar("_widget", "page_link", $pageLink);
		$this->tmpl->addVar("_widget", "total_count", $totalCount);
		$this->tmpl->addVar("_widget", "page", $pageNo);	// ページ番号
		
		// ユーザリストを取得
		$this->db->getAllUserList($maxListCount, $pageNo, array($this, 'userListLoop'));
		$this->tmpl->addVar("_widget", "serial_list", implode($this->serialArray, ','));// 表示項目のシリアル番号を設定
	}
	/**
	 * 詳細画面作成
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param								なし
	 */
	function createDetail($request)
	{
		$userId = $this->gEnv->getCurrentUserId();
		
		$act = $request->trimValueOf('act');
		$this->serialNo = $request->trimValueOf('serial');		// 選択項目のシリアル番号
		$name = $request->trimValueOf('item_name');
		$account = $request->trimValueOf('item_account');
		$password = $request->trimValueOf('password');
		$this->userType = $request->trimValueOf('item_usertype');		// ユーザ種別
		$canLogin = ($request->trimValueOf('item_canlogin') == 'on') ? 1 : 0;		// ログインできるかどうか

		$isAdmin = false;			// 管理権限ありかどうか
		$limitedMenu = false;		// ユーザタイプメニューの項目を制限するかどうか
		$reloadData = false;		// データの再読み込み
		if ($act == 'update'){		// 行更新のとき
			// 入力チェック
			$this->checkInput($name, '名前');
			//$this->checkSingleByte($account, 'アカウント');
			$this->checkInput($account, 'アカウント');
			
			// アカウント重複チェック
			// 設定データを取得
			$ret = $this->db->getUserBySerial($this->serialNo, $row);
			if ($ret){
				if ($row['lu_account'] != $account && $this->sysDb->isExistsAccount($account)) $this->setMsg(self::MSG_USER_ERR, 'アカウントが重複しています');
			} else {
				$this->setMsg(self::MSG_APP_ERR, 'データ取得に失敗しました');
			}
			
			// エラーなしの場合は、データを更新
			if ($this->getMsgCount() == 0){
				// 管理権限ありのときは、ユーザタイプが変更できない
				if ($row['lu_user_type'] >= UserInfo::USER_TYPE_MANAGER) $this->userType = $row['lu_user_type']; 	// 管理画面が使用できるかどうか
				
				// システム管理者は常にログイン可能
				if ($this->userType == UserInfo::USER_TYPE_SYS_ADMIN) $canLogin = 1;
				
				// ユーザ種別が負のときはログイン不可
				if (intval($this->userType) < 0) $canLogin = 0;
				
				$ret = $this->db->updateUser($this->serialNo, $name, $account, $password, $this->userType, $canLogin, $newSerial);
				if ($ret){		// データ追加成功のとき
					$this->setMsg(self::MSG_GUIDANCE, 'データを更新しました');
					$this->serialNo = $newSerial;
					$reloadData = true;		// データの再読み込み
				} else {
					$this->setMsg(self::MSG_APP_ERR, 'データ更新に失敗しました');
				}
			}
		} else if ($act == 'add'){		// 新規追加のとき
			// 入力チェック
			$this->checkInput($name, '名前');
			//$this->checkSingleByte($account, 'アカウント');
			$this->checkInput($account, 'アカウント');
			if ($this->userType == '') $this->setUserErrorMsg('ユーザ種別が選択されていません');

			// アカウント重複チェック
			if ($this->sysDb->isExistsAccount($account)) $this->setMsg(self::MSG_USER_ERR, 'アカウントが重複しています');
						
			// エラーなしの場合は、データを登録
			if ($this->getMsgCount() == 0){
				// システム管理者は常にログイン可能
				if ($this->userType == UserInfo::USER_TYPE_SYS_ADMIN) $canLogin = 1;
				
				// ユーザ種別が負のときはログイン不可
				if (intval($this->userType) < 0) $canLogin = 0;
				
				$ret = $this->db->addNewUser($name, $account, $password, $this->userType, $canLogin, $newSerial);
				if ($ret){		// データ追加成功のとき
					$this->setMsg(self::MSG_GUIDANCE, 'データを追加しました');
					$this->serialNo = $newSerial;
					$reloadData = true;		// データの再読み込み
				} else {
					$this->setMsg(self::MSG_APP_ERR, 'データ追加に失敗しました');
				}
			}
		} else if ($act == 'delete'){		// 削除のとき
			$ret = $this->db->delUserBySerial(array($this->serialNo));
			if ($ret){		// データ削除成功のとき
				$this->setMsg(self::MSG_GUIDANCE, 'データを削除しました');
			} else {
				$this->setMsg(self::MSG_APP_ERR, 'データ削除に失敗しました');
			}
		} else {
			if (!empty($this->serialNo)) $reloadData = true;		// データの再読み込み
		}
		if ($reloadData){		// データの再読み込み
			// 設定データを取得
			$ret = $this->db->getUserBySerial($this->serialNo, $row);
			if ($ret){
				$name = $row['lu_name'];
				$account = $row['lu_account'];
				$this->userType = $row['lu_user_type'];			// ユーザ種別
				if ($this->userType < UserInfo::USER_TYPE_MANAGER){	// 管理権限なしの場合は、選択用メニューを表示
					$limitedMenu = true;
				} else {
					$isAdmin = true;			// 管理権限ありかどうか
				}
				$canLogin = $row['lu_enable_login'];		// ログインできるかどうか
				$loginUserId = $row['lu_id'];				// ユーザID
			}
		}
		// ユーザタイプ選択メニュー作成
		$this->createUserTypeMenu($limitedMenu);
		
		// 取得データを設定
		$this->tmpl->addVar("_widget", "serial", $this->serialNo);
		$this->tmpl->addVar("_widget", "name", $this->convertToDispString($name));
		$this->tmpl->addVar("_widget", "account", $this->convertToDispString($account));
		$canLoginCheck = '';
		if ($canLogin) $canLoginCheck = 'checked';
		$this->tmpl->addVar("_widget", "can_login", $canLoginCheck);
		$this->tmpl->addVar("_widget", "userid", $loginUserId);// ユーザID
				
		if (empty($this->serialNo)){		// ユーザIDが空のときは新規とする
			$this->tmpl->setAttribute('add_button', 'visibility', 'visible');// 新規登録ボタン表示
		} else {
			$this->tmpl->setAttribute('update_button', 'visibility', 'visible');// 更新ボタン表示
			$this->tmpl->addVar("_widget", "password", self::DEFAULT_PASSWORD);// 入力済みを示すパスワードの設定
			
			// 管理権限ありの場合は変更不可にする
			if ($isAdmin) $this->tmpl->addVar('_widget', 'usertype_disabled', 'disabled');
		}
		
		// ディレクトリを設定
		$this->tmpl->addVar("_widget", "script_url", $this->gEnv->getScriptsUrl());
	}
	/**
	 * ユーザリスト、取得したデータをテンプレートに設定する
	 *
	 * @param int $index			行番号(0～)
	 * @param array $fetchedRow		フェッチ取得した行
	 * @param object $param			未使用
	 * @return bool					true=処理続行の場合、false=処理終了の場合
	 */
	function userListLoop($index, $fetchedRow, $param)
	{
		$isAdmin = '';
		if ($fetchedRow['lu_user_type'] >= UserInfo::USER_TYPE_MANAGER){	// 管理画面が使用できるかどうか
			$isAdmin = 'checked';
		}
		$canLogin = '';
		if ($fetchedRow['lu_enable_login']){
			$canLogin = 'checked';
		}
		// ユーザ種別
		switch ($fetchedRow['lu_user_type']){
			case UserInfo::USER_TYPE_NOT_AUTHENTICATED:
				$userType = '<font color="darkgray">未承認</font>';
				break;
			case UserInfo::USER_TYPE_TMP:
				$userType = '<font color="black">仮登録</font>';
				break;
			case UserInfo::USER_TYPE_NORMAL:
				$userType = '<font color="green">一般ユーザ</font>';
				break;
			case UserInfo::USER_TYPE_MANAGER:
				$userType = '<font color="orange">運営者</font>';
				break;
			case UserInfo::USER_TYPE_SYS_ADMIN:
				$userType = '<font color="red">管理者</font>';
				break;
			default:
				$userType = '該当なし';
				break;
		}
		$row = array(
			'index' => $index,													// 行番号
			'serial' => $this->convertToDispString($fetchedRow['lu_serial']),			// シリアル番号
			'id' => $this->convertToDispString($fetchedRow['lu_id']),			// ID
			'name' => $this->convertToDispString($fetchedRow['lu_name']),		// 名前
			'account' => $this->convertToDispString($fetchedRow['lu_account']),		// アカウント
			'user_type' => $userType,		// ユーザ種別
			'update_dt' => $this->convertToDispDateTime($fetchedRow['lu_create_dt']),	// 更新日時
			'is_admin' => $isAdmin,													// 管理者権限があるかどうか
			'can_login' => $canLogin,												// ログイン可能かどうか
			'selected' => $selected												// 項目選択用ラジオボタン
		);
		$this->tmpl->addVars('userlist', $row);
		$this->tmpl->parseTemplate('userlist', 'a');
		
		// 表示中のコンテンツIDを保存
		$this->serialArray[] = $fetchedRow['lu_serial'];
		return true;
	}
	/**
	 * ユーザタイプ選択メニュー作成
	 *
	 * @param bool $limited			// メニュー項目を制限するかどうか
	 * @return なし
	 */
	function createUserTypeMenu($limited)
	{
		// 管理権限なしのユーザに設定されている場合は変更可能メニューを表示
		if ($limited){
			for ($i = 0; $i < count($this->changeUserTypeArray); $i++){
				$value = $this->changeUserTypeArray[$i]['value'];
				$name = $this->changeUserTypeArray[$i]['name'];
			
				$selected = '';
				if ($value == $this->userType) $selected = 'selected';
			
				$row = array(
					'value'    => $value,			// 値
					'name'     => $name,			// 名前
					'selected' => $selected														// 選択中かどうか
				);
				$this->tmpl->addVars('usertype_list', $row);
				$this->tmpl->parseTemplate('usertype_list', 'a');
			}
		} else {
			for ($i = 0; $i < count($this->userTypeArray); $i++){
				$value = $this->userTypeArray[$i]['value'];
				$name = $this->userTypeArray[$i]['name'];
			
				$selected = '';
				if ($value == $this->userType) $selected = 'selected';
			
				$row = array(
					'value'    => $value,			// 値
					'name'     => $name,			// 名前
					'selected' => $selected														// 選択中かどうか
				);
				$this->tmpl->addVars('usertype_list', $row);
				$this->tmpl->parseTemplate('usertype_list', 'a');
			}
		}
	}
}
?>
