<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    ユーザ作成コンテンツ
 * @author     株式会社 毎日メディアサービス
 * @copyright  Copyright 2010 株式会社 毎日メディアサービス.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: user_contentDb.php 2963 2010-03-23 12:19:29Z fishbone $
 * @link       http://www.m-media.co.jp
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class user_contentDb extends BaseDb
{
	/**
	 * タブ定義をすべて取得(一般用)
	 *
	 * @param string	$lang				言語
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllVisibleTabs($lang, $callback)
	{
		$queryStr = 'SELECT * FROM user_content_tab ';
		$queryStr .=  'WHERE ub_deleted = false ';		// 削除されていない
		$queryStr .=    'AND ub_visible = true ';		// 表示中
		$queryStr .=    'AND ub_language_id = ? ';
		$queryStr .=  'ORDER BY ub_index';
		$this->selectLoop($queryStr, array($lang), $callback);
	}
	/**
	 * タブ定義一覧を取得(管理用)
	 *
	 * @param string	$lang				言語
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllTabs($lang, $callback)
	{
		$queryStr = 'SELECT * FROM user_content_tab LEFT JOIN _login_user ON ub_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE ub_deleted = false ';		// 削除されていない
		$queryStr .=    'AND ub_language_id = ? ';
		$queryStr .=  'ORDER BY ub_index';
		$this->selectLoop($queryStr, array($lang), $callback);
	}
	/**
	 * タブ項目をシリアル番号で取得
	 *
	 * @param string	$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getTabBySerial($serial, &$row)
	{
		$queryStr  = 'SELECT * FROM user_content_tab LEFT JOIN _login_user ON ub_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE ub_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * タブ項目を識別IDで取得
	 *
	 * @param string	$lang				言語
	 * @param string	$id					識別ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getTabById($lang, $id, &$row)
	{
		$queryStr = 'SELECT * FROM user_content_tab ';
		$queryStr .=  'WHERE ub_deleted = false ';
		$queryStr .=  'AND ub_id = ? ';
		$queryStr .=  'AND ub_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id, $lang), $row);
		return $ret;
	}
	/**
	 * 最大表示順を取得
	 *
	 * @return int					最大表示順
	 */
	function getMaxTabIndex()
	{
		$queryStr = 'SELECT max(ub_index) as mindex FROM user_content_tab ';
		$queryStr .=  'WHERE ub_deleted = false ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret){
			$index = $row['mindex'];
		} else {
			$index = 0;
		}
		return $index;
	}
	/**
	 * タブ情報を更新
	 *
	 * @param string $serial	シリアル番号(0のときは新規登録)
	 * @param string $lang		言語ID
	 * @param string $id		タブ識別キー
	 * @param string $name		名前
	 * @param string $html		テンプレートHTML
	 * @param int $index		表示順
	 * @param bool $visible		表示制御
	 * @param string $useItemId	使用しているコンテンツ項目ID
	 * @param int $newSerial	新規シリアル番号
	 * @return					true = 正常、false=異常
	 */
	function updateTab($serial, $lang, $id, $name, $html, $index, $visible, $useItemId, &$newSerial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		$limited = false;		// ユーザ制限
		
		// トランザクション開始
		$this->startTransaction();
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$desc = '';
		if (empty($serial)){		// 新規登録のとき
			$queryStr = 'SELECT * FROM user_content_tab ';
			$queryStr .=  'WHERE ub_id = ? ';
			$queryStr .=     'AND ub_language_id = ? ';
			$queryStr .=  'ORDER BY ub_history_index DESC ';
			$ret = $this->selectRecord($queryStr, array($id, $lang), $row);
			if ($ret){
				if (!$row['ub_deleted']){		// レコード存在していれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ub_history_index'] + 1;
				$desc = $row['ub_description'];
				$limited = $row['ub_user_limited'];
			}
		} else {		// 更新のとき
			// 指定のシリアルNoのレコードが削除状態でないかチェック
			$queryStr  = 'SELECT * FROM user_content_tab ';
			$queryStr .=   'WHERE ub_serial = ? ';
			$ret = $this->selectRecord($queryStr, array($serial), $row);
			if ($ret){		// 既に登録レコードがあるとき
				if ($row['ub_deleted']){		// レコードが削除されていれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ub_history_index'] + 1;
				$desc = $row['ub_description'];
				$limited = $row['ub_user_limited'];
				
				// 識別IDと言語の変更は不可
				$id = $row['ub_id'];
				$lang = $row['ub_language_id'];
			} else {		// 存在しない場合は終了
				$this->endTransaction();
				return false;
			}
			
			// 古いレコードを削除
			$queryStr  = 'UPDATE user_content_tab ';
			$queryStr .=   'SET ub_deleted = true, ';	// 削除
			$queryStr .=     'ub_update_user_id = ?, ';
			$queryStr .=     'ub_update_dt = ? ';
			$queryStr .=   'WHERE ub_serial = ?';
			$ret = $this->execStatement($queryStr, array($userId, $now, $serial));
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// データを追加
		$queryStr = 'INSERT INTO user_content_tab ';
		$queryStr .=  '(ub_id, ub_language_id, ub_history_index, ub_name, ub_description, ub_template_html, ub_use_item_id, ub_index, ub_visible, ub_user_limited, ub_create_user_id, ub_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($id, $lang, $historyIndex, $name, $desc, $html, $useItemId, $index, intval($visible), intval($limited), $userId, $now));
		
		// 新規のシリアル番号取得
		$queryStr = 'SELECT MAX(ub_serial) AS ns FROM user_content_tab ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * タブ情報の削除
	 *
	 * @param array $serial			シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delTab($serial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		if (!is_array($serial) || count($serial) <= 0) return true;
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM user_content_tab ';
			$queryStr .=   'WHERE ub_deleted = false ';		// 未削除
			$queryStr .=     'AND ub_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE user_content_tab ';
		$queryStr .=   'SET ub_deleted = true, ';	// 削除
		$queryStr .=     'ub_update_user_id = ?, ';
		$queryStr .=     'ub_update_dt = ? ';
		$queryStr .=   'WHERE ub_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($user, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツ項目定義一覧を取得(管理用)
	 *
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllItems($callback)
	{
		$queryStr = 'SELECT * FROM user_content_item LEFT JOIN _login_user ON ui_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE ui_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY ui_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * 設定されたデータと共にコンテンツ項目定義一覧を取得(管理用)
	 *
	 * @param string  $roomId		ルームID
	 * @param string  $langId		言語ID
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllItemsWithData($roomId, $langId, $callback)
	{
	//$queryStr = 'SELECT * FROM _page_info RIGHT JOIN _page_id ON pn_sub_id = pg_id AND pg_type = 1 AND pn_deleted = false AND pn_id = ? AND pn_language_id = ? ';// 2010/2/23更新
		$queryStr = 'SELECT * FROM user_content_item LEFT JOIN _login_user ON ui_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE ui_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY ui_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * コンテンツ項目定義一覧を取得
	 *
	 * @param array  $rows			取得レコード
	 * @return						true=取得、false=取得せず
	 */
	function getAllContentItems(&$rows)
	{
		$queryStr = 'SELECT * FROM user_content_item ';
		$queryStr .=  'WHERE ui_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY ui_id';
		$retValue = $this->selectRecords($queryStr, array(), $rows);
		return $retValue;
	}
	/**
	 * コンテンツ項目を識別IDで取得
	 *
	 * @param string	$id					識別ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getItemById($id, &$row)
	{
		$queryStr = 'SELECT * FROM user_content_item ';
		$queryStr .=  'WHERE ui_deleted = false ';
		$queryStr .=  'AND ui_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id), $row);
		return $ret;
	}
	/**
	 * コンテンツ項目をシリアル番号で取得
	 *
	 * @param string	$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getItemBySerial($serial, &$row)
	{
		$queryStr  = 'SELECT * FROM user_content_item LEFT JOIN _login_user ON ui_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE ui_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * コンテンツ項目情報を更新
	 *
	 * @param string $serial	シリアル番号(0のときは新規登録)
	 * @param string $id		コンテンツ項目識別キー
	 * @param string $name		名前
	 * @param int $type		コンテンツ項目タイプ
	 * @param int $newSerial	新規シリアル番号
	 * @return					true = 正常、false=異常
	 */
	function updateItem($serial, $id, $name, $type, &$newSerial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$desc = '';
		if (empty($serial)){		// 新規登録のとき
			$queryStr = 'SELECT * FROM user_content_item ';
			$queryStr .=  'WHERE ui_id = ? ';
			$queryStr .=  'ORDER BY ui_history_index DESC ';
			$ret = $this->selectRecord($queryStr, array($id), $row);
			if ($ret){
				if (!$row['ui_deleted']){		// レコード存在していれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ui_history_index'] + 1;
			}
		} else {		// 更新のとき
			// 指定のシリアルNoのレコードが削除状態でないかチェック
			$queryStr  = 'SELECT * FROM user_content_item ';
			$queryStr .=   'WHERE ui_serial = ? ';
			$ret = $this->selectRecord($queryStr, array($serial), $row);
			if ($ret){		// 既に登録レコードがあるとき
				if ($row['ui_deleted']){		// レコードが削除されていれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ui_history_index'] + 1;
				
				// 識別IDと言語の変更は不可
				$id = $row['ui_id'];
			} else {		// 存在しない場合は終了
				$this->endTransaction();
				return false;
			}
			
			// 古いレコードを削除
			$queryStr  = 'UPDATE user_content_item ';
			$queryStr .=   'SET ui_deleted = true, ';	// 削除
			$queryStr .=     'ui_update_user_id = ?, ';
			$queryStr .=     'ui_update_dt = ? ';
			$queryStr .=   'WHERE ui_serial = ?';
			$ret = $this->execStatement($queryStr, array($userId, $now, $serial));
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// データを追加
		$queryStr = 'INSERT INTO user_content_item ';
		$queryStr .=  '(ui_id, ui_history_index, ui_name, ui_type, ui_create_user_id, ui_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($id, $historyIndex, $name, $type, $userId, $now));
		
		// 新規のシリアル番号取得
		$queryStr = 'SELECT MAX(ui_serial) AS ns FROM user_content_item ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツ項目情報の削除
	 *
	 * @param array $serial			シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delItem($serial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		if (!is_array($serial) || count($serial) <= 0) return true;
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM user_content_item ';
			$queryStr .=   'WHERE ui_deleted = false ';		// 未削除
			$queryStr .=     'AND ui_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE user_content_item ';
		$queryStr .=   'SET ui_deleted = true, ';	// 削除
		$queryStr .=     'ui_update_user_id = ?, ';
		$queryStr .=     'ui_update_dt = ? ';
		$queryStr .=   'WHERE ui_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($user, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ルーム一覧を取得(管理用)
	 *
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllRooms($callback)
	{
		$queryStr = 'SELECT * FROM user_content_room LEFT JOIN _login_user ON ur_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE ur_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY ur_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * ルーム情報を識別IDで取得
	 *
	 * @param string	$id					識別ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getRoomById($id, &$row)
	{
		$queryStr = 'SELECT * FROM user_content_room ';
		$queryStr .=  'WHERE ur_deleted = false ';
		$queryStr .=  'AND ur_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id), $row);
		return $ret;
	}
	/**
	 * ルーム情報をシリアル番号で取得
	 *
	 * @param string	$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getRoomBySerial($serial, &$row)
	{
		$queryStr  = 'SELECT * FROM user_content_room LEFT JOIN _login_user ON ur_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE ur_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * ルーム情報を更新
	 *
	 * @param string $serial	シリアル番号(0のときは新規登録)
	 * @param string $id		コンテンツ項目識別キー
	 * @param string $name		名前
	 * @param bool $visible		表示制御
	 * @param int $newSerial	新規シリアル番号
	 * @return					true = 正常、false=異常
	 */
	function updateRoom($serial, $id, $name, $visible, &$newSerial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$desc = '';
		if (empty($serial)){		// 新規登録のとき
			$queryStr = 'SELECT * FROM user_content_room ';
			$queryStr .=  'WHERE ur_id = ? ';
			$queryStr .=  'ORDER BY ur_history_index DESC ';
			$ret = $this->selectRecord($queryStr, array($id), $row);
			if ($ret){
				if (!$row['ur_deleted']){		// レコード存在していれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ur_history_index'] + 1;
			}
		} else {		// 更新のとき
			// 指定のシリアルNoのレコードが削除状態でないかチェック
			$queryStr  = 'SELECT * FROM user_content_room ';
			$queryStr .=   'WHERE ur_serial = ? ';
			$ret = $this->selectRecord($queryStr, array($serial), $row);
			if ($ret){		// 既に登録レコードがあるとき
				if ($row['ur_deleted']){		// レコードが削除されていれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ur_history_index'] + 1;
				
				// 識別IDと言語の変更は不可
				$id = $row['ur_id'];
			} else {		// 存在しない場合は終了
				$this->endTransaction();
				return false;
			}
			
			// 古いレコードを削除
			$queryStr  = 'UPDATE user_content_room ';
			$queryStr .=   'SET ur_deleted = true, ';	// 削除
			$queryStr .=     'ur_update_user_id = ?, ';
			$queryStr .=     'ur_update_dt = ? ';
			$queryStr .=   'WHERE ur_serial = ?';
			$ret = $this->execStatement($queryStr, array($userId, $now, $serial));
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// データを追加
		$queryStr = 'INSERT INTO user_content_room ';
		$queryStr .=  '(ur_id, ur_history_index, ur_name, ur_visible, ur_create_user_id, ur_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($id, $historyIndex, $name, intval($visible), $userId, $now));
		
		// 新規のシリアル番号取得
		$queryStr = 'SELECT MAX(ur_serial) AS ns FROM user_content_room ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ルーム情報の削除
	 *
	 * @param array $serial			シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delRoom($serial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		if (!is_array($serial) || count($serial) <= 0) return true;
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM user_content_room ';
			$queryStr .=   'WHERE ur_deleted = false ';		// 未削除
			$queryStr .=     'AND ur_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE user_content_room ';
		$queryStr .=   'SET ur_deleted = true, ';	// 削除
		$queryStr .=     'ur_update_user_id = ?, ';
		$queryStr .=     'ur_update_dt = ? ';
		$queryStr .=   'WHERE ur_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($user, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツの更新
	 *
	 * @param string  $id			コンテンツID
	 * @param string  $roomId		ルームID
	 * @param string  $langId		言語ID
	 * @param string  $html			設定データ
	 * @param float   $number		数値データ
	 * @param bool    $visible		表示状態
	 * @param timestamp	$startDt	期間(開始日)
	 * @param timestamp	$endDt		期間(終了日)
	 * @param bool    $limited		ユーザ制限するかどうか
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateContent($id, $roomId, $langId, $html, $number, $visible, $startDt, $endDt, $limited)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		$contentType = '';
				
		// トランザクション開始
		$this->startTransaction();
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$queryStr = 'SELECT * FROM user_content ';
		$queryStr .=  'WHERE uc_id = ? ';
		$queryStr .=    'AND uc_room_id = ? ';
		$queryStr .=    'AND uc_language_id = ? ';
		$queryStr .=  'ORDER BY uc_history_index DESC ';
		$ret = $this->selectRecord($queryStr, array($id, $roomId, $langId), $row);
		if ($ret){
			$historyIndex = $row['uc_history_index'] + 1;
			
			if (!$row['uc_deleted']){		// レコード存在していれば削除
				// 古いレコードを削除
				$queryStr  = 'UPDATE user_content ';
				$queryStr .=   'SET uc_deleted = true, ';	// 削除
				$queryStr .=     'uc_update_user_id = ?, ';
				$queryStr .=     'uc_update_dt = ? ';
				$queryStr .=   'WHERE uc_serial = ?';
				$this->execStatement($queryStr, array($user, $now, $row['uc_serial']));
			}
		}

		// 新規レコード追加
		$queryStr  = 'INSERT INTO user_content ';
		$queryStr .=   '(uc_id, ';
		$queryStr .=   'uc_room_id, ';
		$queryStr .=   'uc_language_id, ';
		$queryStr .=   'uc_history_index, ';
		$queryStr .=   'uc_data, ';
		$queryStr .=   'uc_data_search_num, ';
		$queryStr .=   'uc_visible, ';
		$queryStr .=   'uc_active_start_dt, ';
		$queryStr .=   'uc_active_end_dt, ';
		$queryStr .=   'uc_user_limited, ';
		$queryStr .=   'uc_create_user_id, ';
		$queryStr .=   'uc_create_dt) ';
		$queryStr .=   'VALUES ';
		$queryStr .=   '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($id, $roomId, $langId, $historyIndex, $html, $number, 
							intval($visible), $startDt, $endDt, intval($limited), $user, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツをコンテンツIDで取得
	 *
	 * @param string  $id			コンテンツID
	 * @param string  $roomId		ルームID
	 * @param string  $langId		言語ID
	 * @param array   $row			レコード
	 * @return bool					取得 = true, 取得なし= false
	 */
	function getContent($id, $roomId, $langId, &$row)
	{
		$queryStr  = 'SELECT * FROM user_content LEFT JOIN _login_user ON uc_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE uc_deleted = false ';	// 削除されていない
		$queryStr .=     'AND uc_id = ? ';
		$queryStr .=     'AND uc_room_id = ? ';
		$queryStr .=     'AND uc_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id, $roomId, $langId), $row);
		return $ret;
	}
	
	
	
	
	



	/**
	 * コンテンツ項目を取得
	 *
	 * @param function	$callback			コールバック関数
	 * @param array		$contentIdArray		コンテンツID
	 * @param string	$lang				言語
	 * @param bool		$all				すべてのデータを取得するか、ユーザ制限のないデータを取得するかを指定
	 * @param string	$now				現在日時
	 * @param bool		$preview			プレビューモードかどうか
	 * @return 			なし
	 */
	function getContentItems($callback, $contentIdArray, $lang, $all, $now, $preview)
	{
		$params = array();
		$contentType = '';
		$initDt = $this->gEnv->getInitValueOfTimestamp();
		
		// コンテンツIDの指定がない場合は、デフォルト値を取得
		if (empty($contentIdArray)){
			$queryStr = 'SELECT * FROM content ';
			$queryStr .=  'WHERE cn_deleted = false ';		// 削除されていない
			$queryStr .=    'AND cn_default = true ';
			$queryStr .=    'AND cn_type = ? ';
			$queryStr .=    'AND cn_language_id = ? ';
			$params[] = $contentType;
			$params[] = $lang;
			
			if (!$preview){		// プレビューモードでないときは取得制限
				$queryStr .=    'AND cn_visible = true ';
				if (!$all) $queryStr .=    'AND cn_user_limited = false ';		// ユーザ制限のないデータ
			
				// 公開期間を指定
				$queryStr .=    'AND (cn_active_start_dt = ? OR (cn_active_start_dt != ? AND cn_active_start_dt <= ?)) ';
				$queryStr .=    'AND (cn_active_end_dt = ? OR (cn_active_end_dt != ? AND cn_active_end_dt > ?))';
				$params[] = $initDt;
				$params[] = $initDt;
				$params[] = $now;
				$params[] = $initDt;
				$params[] = $initDt;
				$params[] = $now;
			}
			$queryStr .=  'ORDER BY cn_serial';
			$this->selectLoop($queryStr, $params, $callback, null);
		} else {
			$contentId = implode(',', $contentIdArray);
			
			// CASE文作成
			$caseStr = 'CASE cn_id ';
			for ($i = 0; $i < count($contentIdArray); $i++){
				$caseStr .= 'WHEN ' . $contentIdArray[$i] . ' THEN ' . $i . ' ';
			}
			$caseStr .= 'END AS no';

			$queryStr = 'SELECT *, ' . $caseStr . ' FROM content ';
			$queryStr .=  'WHERE cn_deleted = false ';		// 削除されていない
			$queryStr .=    'AND cn_type = ? ';
			$queryStr .=    'AND cn_id in (' . $contentId . ') ';
			$queryStr .=    'AND cn_language_id = ? ';
			$params[] = $contentType;
			$params[] = $lang;
			
			if (!$preview){		// プレビューモードでないときは取得制限
				$queryStr .=    'AND cn_visible = true ';
				if (!$all) $queryStr .=    'AND cn_user_limited = false ';		// ユーザ制限のないデータ
			
				// 公開期間を指定
				$queryStr .=    'AND (cn_active_start_dt = ? OR (cn_active_start_dt != ? AND cn_active_start_dt <= ?)) ';
				$queryStr .=    'AND (cn_active_end_dt = ? OR (cn_active_end_dt != ? AND cn_active_end_dt > ?))';
				$params[] = $initDt;
				$params[] = $initDt;
				$params[] = $now;
				$params[] = $initDt;
				$params[] = $initDt;
				$params[] = $now;
			}
			$queryStr .=  'ORDER BY no';
			$this->selectLoop($queryStr, $params, $callback, null);
		}
	}
	/**
	 * コンテンツ項目一覧を取得(管理用)
	 *
	 * @param function	$callback			コールバック関数
	 * @param string	$lang				言語
	 * @return 			なし
	 */
	function getAllContents($callback, $lang)
	{
		$contentType = '';
		$queryStr = 'SELECT * FROM content LEFT JOIN _login_user ON cn_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE cn_type = ? ';
		$queryStr .=    'AND cn_language_id = ? ';
		$queryStr .=    'AND cn_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY cn_id';
		$this->selectLoop($queryStr, array($contentType, $lang), $callback, null);
	}
	/**
	 * コンテンツの対応言語を取得(管理用)
	 *
	 * @param string	$contentId			コンテンツID
	 * @return			true=取得、false=取得せず
	 */
	function getLangByContentId($contentId, &$rows)
	{
		$contentType = '';
		$queryStr = 'SELECT ln_id, ln_name, ln_name_en FROM content LEFT JOIN _language ON cn_language_id = ln_id ';
		$queryStr .=  'WHERE cn_deleted = false ';	// 削除されていない
		$queryStr .=    'AND cn_type = ? ';
		$queryStr .=    'AND cn_id = ? ';
		$queryStr .=  'ORDER BY cn_id, ln_priority';
		$retValue = $this->selectRecords($queryStr, array($contentType, $contentId), $rows);
		return $retValue;
	}
	/**
	 * コンテンツの対応言語を取得(管理用)
	 *
	 * @param function	$callback			コールバック関数
	 * @param string	$contentId			コンテンツID
	 * @return			なし
	 */
	function getLangLoopByContentId($callback, $contentId)
	{
		// コンテンツIDがないときは終了
		if (empty($contentId)) return;
		
		$contentType = '';
		$queryStr = 'SELECT ln_id, ln_name, ln_name_en FROM content LEFT JOIN _language ON cn_language_id = ln_id ';
		$queryStr .=  'WHERE cn_deleted = false ';	// 削除されていない
		$queryStr .=    'AND cn_type = ? ';
		$queryStr .=    'AND cn_id = ? ';
		$queryStr .=  'ORDER BY cn_id, ln_priority';
		$this->selectLoop($queryStr, array($contentType, $contentId), $callback, null);
	}
	/**
	 * コンテンツ項目をシリアル番号で取得
	 *
	 * @param string	$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getContentBySerial($serial, &$row)
	{
		$queryStr  = 'select * from content LEFT JOIN _login_user ON cn_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE cn_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	
	/**
	 * コンテンツ項目の新規追加
	 *
	 * @param string  $contentid	コンテンツID(0のときはコンテンツIDを新規取得)
	 * @param string  $lang			言語ID
	 * @param string  $name			コンテンツ名
	 * @param string  $desc			説明
	 * @param string  $html			HTML
	 * @param bool    $visible		表示状態
	 * @param bool    $default		デフォルトで使用するかどうか
	 * @param bool    $limited		ユーザ制限するかどうか
	 * @param string  $key			外部参照用キー
	 * @param string  $metaTitle	METAタグ、タイトル
	 * @param string  $metaDesc		METAタグ、ページ要約
	 * @param string  $metaKeyword	METAタグ、検索用キーワード
	 * @param timestamp	$startDt	期間(開始日)
	 * @param timestamp	$endDt		期間(終了日)
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function addContentItem($contentid, $lang, $name, $desc, $html, $visible, $default, $limited, $key, $metaTitle, $metaDesc, $metaKeyword, $startDt, $endDt, &$newSerial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		$contentType = '';
				
		// トランザクション開始
		$this->startTransaction();
		
		if ($contentid == 0){		// コンテンツIDが0のときは、コンテンツIDを新規取得
			// コンテンツIDを決定する
			$queryStr = 'select max(cn_id) as mid from content ';
			$queryStr .=  'WHERE cn_type = ? ';
			$ret = $this->selectRecord($queryStr, array($contentType), $row);
			if ($ret){
				$contId = $row['mid'] + 1;
			} else {
				$contId = 1;
			}
		} else {
			$contId = $contentid;
		}
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$queryStr = 'SELECT * FROM content ';
		$queryStr .=  'WHERE cn_type = ? ';
		$queryStr .=    'AND cn_id = ? ';
		$queryStr .=    'AND cn_language_id = ? ';
		$queryStr .=  'ORDER BY cn_history_index DESC ';
		$ret = $this->selectRecord($queryStr, array($contentType, $contId, $lang), $row);
		if ($ret){
			if (!$row['cn_deleted']){		// レコード存在していれば終了
				$this->endTransaction();
				return false;
			}
			$historyIndex = $row['cn_history_index'] + 1;
		}
		// デフォルトを設定のときは他のデフォルトを解除
		if ($default){
			$queryStr  = 'UPDATE content ';
			$queryStr .=   'SET cn_default = false, ';		// デフォルトをクリア
			$queryStr .=     'cn_update_user_id = ?, ';
			$queryStr .=     'cn_update_dt = ? ';
			$queryStr .=   'WHERE cn_deleted = false ';
			$queryStr .=     'AND cn_type = ? ';
			$queryStr .=     'AND cn_language_id = ? ';
			$this->execStatement($queryStr, array($user, $now, $contentType, $lang));
		}
		// データを追加
		$queryStr  = 'INSERT INTO content ';
		$queryStr .=   '(';
		$queryStr .=   'cn_type, ';
		$queryStr .=   'cn_id, ';
		$queryStr .=   'cn_language_id, ';
		$queryStr .=   'cn_history_index, ';
		$queryStr .=   'cn_name, ';
		$queryStr .=   'cn_description, ';
		$queryStr .=   'cn_html, ';
		$queryStr .=   'cn_visible, ';
		$queryStr .=   'cn_default, ';
		$queryStr .=   'cn_user_limited, ';
		$queryStr .=   'cn_key, ';
		$queryStr .=   'cn_meta_title, ';
		$queryStr .=   'cn_meta_description, ';
		$queryStr .=   'cn_meta_keywords, ';
		$queryStr .=   'cn_active_start_dt, ';
		$queryStr .=   'cn_active_end_dt, ';
		$queryStr .=   'cn_create_user_id, ';
		$queryStr .=   'cn_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($contentType, $contId, $lang, $historyIndex, $name, $desc, $html,
								intval($visible), intval($default), intval($limited), $key, $metaTitle, $metaDesc, $metaKeyword, $startDt, $endDt, $user, $now));
		
		// 新規のシリアル番号取得
		$queryStr = 'select max(cn_serial) as ns from content ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
			
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツ項目の削除
	 *
	 * @param array $serial			シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delContentItem($serial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		if (!is_array($serial) || count($serial) <= 0) return true;
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM content ';
			$queryStr .=   'WHERE cn_deleted = false ';		// 未削除
			$queryStr .=     'AND cn_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE content ';
		$queryStr .=   'SET cn_deleted = true, ';	// 削除
		$queryStr .=     'cn_update_user_id = ?, ';
		$queryStr .=     'cn_update_dt = now() ';
		$queryStr .=   'WHERE cn_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($user));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツIDでコンテンツ項目を削除
	 *
	 * @param int $serial			シリアルNo
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delContentItemById($serial, $userId)
	{
		$contentType = '';
				
		// トランザクション開始
		$this->startTransaction();
		
		// コンテンツIDを取得
		$queryStr  = 'select * from content ';
		$queryStr .=   'where cn_deleted = false ';		// 未削除
		$queryStr .=     'and cn_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if ($ret){		// 既に登録レコードがあるとき
			if ($row['cn_deleted']){		// レコードが削除されていれば終了
				$this->endTransaction();
				return false;
			}
		} else {		// 存在しない場合は終了
			$this->endTransaction();
			return false;
		}
		$contId = $row['cn_id'];
		
		// レコードを削除
		$queryStr  = 'UPDATE content ';
		$queryStr .=   'SET cn_deleted = true, ';	// 削除
		$queryStr .=     'cn_update_user_id = ?, ';
		$queryStr .=     'cn_update_dt = now() ';
		$queryStr .=   'WHERE cn_type = ? ';
		$queryStr .=     'AND cn_id = ? ';
		$this->execStatement($queryStr, array($userId, $contentType, $contId));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツ項目をすべて削除
	 *
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delAllContentItems($userId)
	{
		$contentType = '';
			
		// トランザクション開始
		$this->startTransaction();
		
		// レコードを削除
		$queryStr  = 'UPDATE content ';
		$queryStr .=   'SET cn_deleted = true, ';	// 削除
		$queryStr .=     'cn_update_user_id = ?, ';
		$queryStr .=     'cn_update_dt = now() ';
		$queryStr .=   'WHERE cn_type = ? AND cn_deleted = false';
		$this->execStatement($queryStr, array($userId, $contentType));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツ項目を検索
	 *
	 * @param int		$limit				取得する項目数
	 * @param int		$page				取得するページ(1～)
	 * @param string	$keyword			検索キーワード
	 * @param string	$langId				言語
	 * @param bool		$all				すべてのデータを取得するか、ユーザ制限のないデータを取得するかを指定
	 * @param string	$now				現在日時
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function searchContentByKeyword($limit, $page, $keyword, $langId, $all, $now, $callback)
	{
		$contentType = '';
		$initDt = $this->gEnv->getInitValueOfTimestamp();
		$offset = $limit * ($page -1);
		if ($offset < 0) $offset = 0;
		
		$params = array();
		$queryStr = 'SELECT * FROM content ';
		$queryStr .=  'WHERE cn_visible = true ';
		$queryStr .=    'AND cn_deleted = false ';		// 削除されていない
		$queryStr .=    'AND cn_type = ? ';$params[] = $contentType;
		$queryStr .=    'AND cn_language_id = ? ';$params[] = $langId;
		if (!$all) $queryStr .=    'AND cn_user_limited = false ';		// ユーザ制限のないデータ

		// タイトルと記事を検索
		if (!empty($keyword)){
			// 「'"\」文字をエスケープ
			$keyword = addslashes($keyword);
			$queryStr .=    'AND (cn_name LIKE \'%' . $keyword . '%\' ';
			$queryStr .=    'OR cn_html LIKE \'%' . $keyword . '%\' ';
			$queryStr .=    'OR cn_description LIKE \'%' . $keyword . '%\') ';
		}
		
		// 公開期間を指定
		$queryStr .=    'AND (cn_active_start_dt = ? OR (cn_active_start_dt != ? AND cn_active_start_dt <= ?)) ';
		$queryStr .=    'AND (cn_active_end_dt = ? OR (cn_active_end_dt != ? AND cn_active_end_dt > ?)) ';
		$params[] = $initDt;
		$params[] = $initDt;
		$params[] = $now;
		$params[] = $initDt;
		$params[] = $initDt;
		$params[] = $now;
		
		$queryStr .=  'ORDER BY cn_create_dt desc limit ' . $limit . ' offset ' . $offset;
		$this->selectLoop($queryStr, $params, $callback, null);
	}
	/**
	 * すべての言語を取得
	 *
	 * @param function	$callback			コールバック関数
	 * @return			true=取得、false=取得せず
	 */
	function getAllLang($callback)
	{
		$queryStr = 'SELECT * FROM _language ORDER BY ln_priority';
		$this->selectLoop($queryStr, array(), $callback, null);
	}
	/**
	 * メニュー項目の追加
	 *
	 * @param string  $menuId		メニューID
	 * @param string  $name			メニュー名
	 * @param string  $url			URL
	 * @return bool					true = 成功、false = 失敗
	 */
	function addMenuItem($menuId, $name, $url)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// IDを求める
		$id = 1;
		$queryStr = 'SELECT max(md_id) as ms FROM _menu_def ';
		$ret = $this->selectRecord($queryStr, array(), $maxRow);
		if ($ret) $id = $maxRow['ms'] + 1;
			
		// 最大値インデックスを取得
		$index = 1;
		$queryStr = 'SELECT max(md_index) as ms FROM _menu_def ';
		$queryStr .=  'WHERE md_menu_id = ? ';
		$ret = $this->selectRecord($queryStr, array($menuId), $maxRow);
		if ($ret) $index = $maxRow['ms'] + 1;
			
		$queryStr = 'INSERT INTO _menu_def ';
		$queryStr .=  '(md_id, md_index, md_menu_id, md_name, md_link_url, md_update_user_id, md_update_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($id, $index, $menuId, $name, $url, $userId, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * メニューIDのリストを取得
	 *
	 * @param function $callback	コールバック関数
	 * @return						なし
	 */
	function getMenuIdList($callback)
	{
		$queryStr = 'SELECT * FROM _menu_id ';
		$queryStr .=  'ORDER BY mn_sort_order';
		$this->selectLoop($queryStr, array(), $callback);
	}
}
?>
