<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    ユーザ作成コンテンツ
 * @author     株式会社 毎日メディアサービス
 * @copyright  Copyright 2010 株式会社 毎日メディアサービス.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: user_contentDb.php 2971 2010-03-25 05:06:34Z fishbone $
 * @link       http://www.m-media.co.jp
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class user_contentDb extends BaseDb
{
	/**
	 * タブ定義をすべて取得(一般用)
	 *
	 * @param string	$lang				言語
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllVisibleTabs($lang, $callback)
	{
		$queryStr = 'SELECT * FROM user_content_tab ';
		$queryStr .=  'WHERE ub_deleted = false ';		// 削除されていない
		$queryStr .=    'AND ub_visible = true ';		// 表示中
		$queryStr .=    'AND ub_language_id = ? ';
		$queryStr .=  'ORDER BY ub_index';
		$this->selectLoop($queryStr, array($lang), $callback);
	}
	/**
	 * タブ定義一覧を取得(管理用)
	 *
	 * @param string	$lang				言語
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllTabs($lang, $callback)
	{
		$queryStr = 'SELECT * FROM user_content_tab LEFT JOIN _login_user ON ub_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE ub_deleted = false ';		// 削除されていない
		$queryStr .=    'AND ub_language_id = ? ';
		$queryStr .=  'ORDER BY ub_index';
		$this->selectLoop($queryStr, array($lang), $callback);
	}
	/**
	 * タブ項目をシリアル番号で取得
	 *
	 * @param string	$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getTabBySerial($serial, &$row)
	{
		$queryStr  = 'SELECT * FROM user_content_tab LEFT JOIN _login_user ON ub_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE ub_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * タブ項目を識別IDで取得
	 *
	 * @param string	$lang				言語
	 * @param string	$id					識別ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getTabById($lang, $id, &$row)
	{
		$queryStr = 'SELECT * FROM user_content_tab ';
		$queryStr .=  'WHERE ub_deleted = false ';
		$queryStr .=  'AND ub_id = ? ';
		$queryStr .=  'AND ub_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id, $lang), $row);
		return $ret;
	}
	/**
	 * 最大表示順を取得
	 *
	 * @return int					最大表示順
	 */
	function getMaxTabIndex()
	{
		$queryStr = 'SELECT max(ub_index) as mindex FROM user_content_tab ';
		$queryStr .=  'WHERE ub_deleted = false ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret){
			$index = $row['mindex'];
		} else {
			$index = 0;
		}
		return $index;
	}
	/**
	 * タブ情報を更新
	 *
	 * @param string $serial	シリアル番号(0のときは新規登録)
	 * @param string $lang		言語ID
	 * @param string $id		タブ識別キー
	 * @param string $name		名前
	 * @param string $html		テンプレートHTML
	 * @param int $index		表示順
	 * @param bool $visible		表示制御
	 * @param string $useItemId	使用しているコンテンツ項目ID
	 * @param int $newSerial	新規シリアル番号
	 * @return					true = 正常、false=異常
	 */
	function updateTab($serial, $lang, $id, $name, $html, $index, $visible, $useItemId, &$newSerial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		$limited = false;		// ユーザ制限
		
		// トランザクション開始
		$this->startTransaction();
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$desc = '';
		if (empty($serial)){		// 新規登録のとき
			$queryStr = 'SELECT * FROM user_content_tab ';
			$queryStr .=  'WHERE ub_id = ? ';
			$queryStr .=     'AND ub_language_id = ? ';
			$queryStr .=  'ORDER BY ub_history_index DESC ';
			$ret = $this->selectRecord($queryStr, array($id, $lang), $row);
			if ($ret){
				if (!$row['ub_deleted']){		// レコード存在していれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ub_history_index'] + 1;
				$desc = $row['ub_description'];
				$limited = $row['ub_user_limited'];
			}
		} else {		// 更新のとき
			// 指定のシリアルNoのレコードが削除状態でないかチェック
			$queryStr  = 'SELECT * FROM user_content_tab ';
			$queryStr .=   'WHERE ub_serial = ? ';
			$ret = $this->selectRecord($queryStr, array($serial), $row);
			if ($ret){		// 既に登録レコードがあるとき
				if ($row['ub_deleted']){		// レコードが削除されていれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ub_history_index'] + 1;
				$desc = $row['ub_description'];
				$limited = $row['ub_user_limited'];
				
				// 識別IDと言語の変更は不可
				$id = $row['ub_id'];
				$lang = $row['ub_language_id'];
			} else {		// 存在しない場合は終了
				$this->endTransaction();
				return false;
			}
			
			// 古いレコードを削除
			$queryStr  = 'UPDATE user_content_tab ';
			$queryStr .=   'SET ub_deleted = true, ';	// 削除
			$queryStr .=     'ub_update_user_id = ?, ';
			$queryStr .=     'ub_update_dt = ? ';
			$queryStr .=   'WHERE ub_serial = ?';
			$ret = $this->execStatement($queryStr, array($userId, $now, $serial));
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// データを追加
		$queryStr = 'INSERT INTO user_content_tab ';
		$queryStr .=  '(ub_id, ub_language_id, ub_history_index, ub_name, ub_description, ub_template_html, ub_use_item_id, ub_index, ub_visible, ub_user_limited, ub_create_user_id, ub_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($id, $lang, $historyIndex, $name, $desc, $html, $useItemId, $index, intval($visible), intval($limited), $userId, $now));
		
		// 新規のシリアル番号取得
		$queryStr = 'SELECT MAX(ub_serial) AS ns FROM user_content_tab ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * タブ情報の削除
	 *
	 * @param array $serial			シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delTab($serial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		if (!is_array($serial) || count($serial) <= 0) return true;
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM user_content_tab ';
			$queryStr .=   'WHERE ub_deleted = false ';		// 未削除
			$queryStr .=     'AND ub_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE user_content_tab ';
		$queryStr .=   'SET ub_deleted = true, ';	// 削除
		$queryStr .=     'ub_update_user_id = ?, ';
		$queryStr .=     'ub_update_dt = ? ';
		$queryStr .=   'WHERE ub_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($user, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツ項目定義一覧を取得(管理用)
	 *
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllItems($callback)
	{
		$queryStr = 'SELECT * FROM user_content_item LEFT JOIN _login_user ON ui_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE ui_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY ui_index, ui_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * コンテンツ項目定義一覧を取得
	 *
	 * @param array  $rows			取得レコード
	 * @return						true=取得、false=取得せず
	 */
	function getAllContentItems(&$rows)
	{
		$queryStr = 'SELECT * FROM user_content_item ';
		$queryStr .=  'WHERE ui_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY ui_index, ui_id';
		$retValue = $this->selectRecords($queryStr, array(), $rows);
		return $retValue;
	}
	/**
	 * コンテンツ項目を識別IDで取得
	 *
	 * @param string	$id					識別ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getItemById($id, &$row)
	{
		$queryStr = 'SELECT * FROM user_content_item ';
		$queryStr .=  'WHERE ui_deleted = false ';
		$queryStr .=  'AND ui_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id), $row);
		return $ret;
	}
	/**
	 * コンテンツ項目をシリアル番号で取得
	 *
	 * @param string	$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getItemBySerial($serial, &$row)
	{
		$queryStr  = 'SELECT * FROM user_content_item LEFT JOIN _login_user ON ui_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE ui_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * コンテンツ項目情報を更新
	 *
	 * @param string $serial	シリアル番号(0のときは新規登録)
	 * @param string $id		コンテンツ項目識別キー
	 * @param string $name		名前
	 * @param int $type		コンテンツ項目タイプ
	 * @param int $newSerial	新規シリアル番号
	 * @return					true = 正常、false=異常
	 */
	function updateItem($serial, $id, $name, $type, &$newSerial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$desc = '';
		if (empty($serial)){		// 新規登録のとき
			$queryStr = 'SELECT * FROM user_content_item ';
			$queryStr .=  'WHERE ui_id = ? ';
			$queryStr .=  'ORDER BY ui_history_index DESC ';
			$ret = $this->selectRecord($queryStr, array($id), $row);
			if ($ret){
				if (!$row['ui_deleted']){		// レコード存在していれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ui_history_index'] + 1;
			}
		} else {		// 更新のとき
			// 指定のシリアルNoのレコードが削除状態でないかチェック
			$queryStr  = 'SELECT * FROM user_content_item ';
			$queryStr .=   'WHERE ui_serial = ? ';
			$ret = $this->selectRecord($queryStr, array($serial), $row);
			if ($ret){		// 既に登録レコードがあるとき
				if ($row['ui_deleted']){		// レコードが削除されていれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ui_history_index'] + 1;
				
				// 識別IDと言語の変更は不可
				$id = $row['ui_id'];
			} else {		// 存在しない場合は終了
				$this->endTransaction();
				return false;
			}
			
			// 古いレコードを削除
			$queryStr  = 'UPDATE user_content_item ';
			$queryStr .=   'SET ui_deleted = true, ';	// 削除
			$queryStr .=     'ui_update_user_id = ?, ';
			$queryStr .=     'ui_update_dt = ? ';
			$queryStr .=   'WHERE ui_serial = ?';
			$ret = $this->execStatement($queryStr, array($userId, $now, $serial));
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// データを追加
		$queryStr = 'INSERT INTO user_content_item ';
		$queryStr .=  '(ui_id, ui_history_index, ui_name, ui_type, ui_create_user_id, ui_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($id, $historyIndex, $name, $type, $userId, $now));
		
		// 新規のシリアル番号取得
		$queryStr = 'SELECT MAX(ui_serial) AS ns FROM user_content_item ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツ項目情報の削除
	 *
	 * @param array $serial			シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delItem($serial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		if (!is_array($serial) || count($serial) <= 0) return true;
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM user_content_item ';
			$queryStr .=   'WHERE ui_deleted = false ';		// 未削除
			$queryStr .=     'AND ui_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE user_content_item ';
		$queryStr .=   'SET ui_deleted = true, ';	// 削除
		$queryStr .=     'ui_update_user_id = ?, ';
		$queryStr .=     'ui_update_dt = ? ';
		$queryStr .=   'WHERE ui_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($user, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ルーム一覧を取得(管理用)
	 *
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllRooms($callback)
	{
		$queryStr = 'SELECT * FROM user_content_room LEFT JOIN _login_user ON ur_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE ur_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY ur_id';
		$this->selectLoop($queryStr, array(), $callback);
	}
	/**
	 * ルーム情報を識別IDで取得(管理用)
	 *
	 * @param string	$id					識別ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getRoomById($id, &$row)
	{
		$queryStr = 'SELECT * FROM user_content_room ';
		$queryStr .=  'WHERE ur_deleted = false ';
		$queryStr .=  'AND ur_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id), $row);
		return $ret;
	}
	/**
	 * 公開可能なルームかどうか
	 *
	 * @param string	$id			識別ID
	 * @return bool					true=存在する、false=存在しない
	 */
	function isActiveRoom($id)
	{
		$queryStr = 'SELECT * FROM user_content_room ';
		$queryStr .=  'WHERE ur_deleted = false ';
		$queryStr .=  'AND ur_visible = true ';
		$queryStr .=  'AND ur_id = ? ';
		return $this->isRecordExists($queryStr, array($id));
	}
	/**
	 * ルーム情報をシリアル番号で取得
	 *
	 * @param string	$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getRoomBySerial($serial, &$row)
	{
		$queryStr  = 'SELECT * FROM user_content_room LEFT JOIN _login_user ON ur_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE ur_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * ルーム情報を更新
	 *
	 * @param string $serial	シリアル番号(0のときは新規登録)
	 * @param string $id		コンテンツ項目識別キー
	 * @param string $name		名前
	 * @param bool $visible		表示制御
	 * @param int $newSerial	新規シリアル番号
	 * @return					true = 正常、false=異常
	 */
	function updateRoom($serial, $id, $name, $visible, &$newSerial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$desc = '';
		if (empty($serial)){		// 新規登録のとき
			$queryStr = 'SELECT * FROM user_content_room ';
			$queryStr .=  'WHERE ur_id = ? ';
			$queryStr .=  'ORDER BY ur_history_index DESC ';
			$ret = $this->selectRecord($queryStr, array($id), $row);
			if ($ret){
				if (!$row['ur_deleted']){		// レコード存在していれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ur_history_index'] + 1;
			}
		} else {		// 更新のとき
			// 指定のシリアルNoのレコードが削除状態でないかチェック
			$queryStr  = 'SELECT * FROM user_content_room ';
			$queryStr .=   'WHERE ur_serial = ? ';
			$ret = $this->selectRecord($queryStr, array($serial), $row);
			if ($ret){		// 既に登録レコードがあるとき
				if ($row['ur_deleted']){		// レコードが削除されていれば終了
					$this->endTransaction();
					return false;
				}
				$historyIndex = $row['ur_history_index'] + 1;
				
				// 識別IDと言語の変更は不可
				$id = $row['ur_id'];
			} else {		// 存在しない場合は終了
				$this->endTransaction();
				return false;
			}
			
			// 古いレコードを削除
			$queryStr  = 'UPDATE user_content_room ';
			$queryStr .=   'SET ur_deleted = true, ';	// 削除
			$queryStr .=     'ur_update_user_id = ?, ';
			$queryStr .=     'ur_update_dt = ? ';
			$queryStr .=   'WHERE ur_serial = ?';
			$ret = $this->execStatement($queryStr, array($userId, $now, $serial));
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// データを追加
		$queryStr = 'INSERT INTO user_content_room ';
		$queryStr .=  '(ur_id, ur_history_index, ur_name, ur_visible, ur_create_user_id, ur_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($id, $historyIndex, $name, intval($visible), $userId, $now));
		
		// 新規のシリアル番号取得
		$queryStr = 'SELECT MAX(ur_serial) AS ns FROM user_content_room ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ルーム情報の削除
	 *
	 * @param array $serial			シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delRoom($serial)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		
		if (!is_array($serial) || count($serial) <= 0) return true;
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM user_content_room ';
			$queryStr .=   'WHERE ur_deleted = false ';		// 未削除
			$queryStr .=     'AND ur_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE user_content_room ';
		$queryStr .=   'SET ur_deleted = true, ';	// 削除
		$queryStr .=     'ur_update_user_id = ?, ';
		$queryStr .=     'ur_update_dt = ? ';
		$queryStr .=   'WHERE ur_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($user, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツの更新
	 *
	 * @param string  $id			コンテンツID
	 * @param string  $roomId		ルームID
	 * @param string  $langId		言語ID
	 * @param string  $html			設定データ
	 * @param float   $number		数値データ
	 * @param bool    $visible		表示状態
	 * @param timestamp	$startDt	期間(開始日)
	 * @param timestamp	$endDt		期間(終了日)
	 * @param bool    $limited		ユーザ制限するかどうか
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateContent($id, $roomId, $langId, $html, $number, $visible, $startDt, $endDt, $limited)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$user = $this->gEnv->getCurrentUserId();	// 現在のユーザ
		$contentType = '';
				
		// トランザクション開始
		$this->startTransaction();
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$queryStr = 'SELECT * FROM user_content ';
		$queryStr .=  'WHERE uc_id = ? ';
		$queryStr .=    'AND uc_room_id = ? ';
		$queryStr .=    'AND uc_language_id = ? ';
		$queryStr .=  'ORDER BY uc_history_index DESC ';
		$ret = $this->selectRecord($queryStr, array($id, $roomId, $langId), $row);
		if ($ret){
			$historyIndex = $row['uc_history_index'] + 1;
			
			if (!$row['uc_deleted']){		// レコード存在していれば削除
				// 古いレコードを削除
				$queryStr  = 'UPDATE user_content ';
				$queryStr .=   'SET uc_deleted = true, ';	// 削除
				$queryStr .=     'uc_update_user_id = ?, ';
				$queryStr .=     'uc_update_dt = ? ';
				$queryStr .=   'WHERE uc_serial = ?';
				$this->execStatement($queryStr, array($user, $now, $row['uc_serial']));
			}
		}

		// 新規レコード追加
		$queryStr  = 'INSERT INTO user_content ';
		$queryStr .=   '(uc_id, ';
		$queryStr .=   'uc_room_id, ';
		$queryStr .=   'uc_language_id, ';
		$queryStr .=   'uc_history_index, ';
		$queryStr .=   'uc_data, ';
		$queryStr .=   'uc_data_search_num, ';
		$queryStr .=   'uc_visible, ';
		$queryStr .=   'uc_active_start_dt, ';
		$queryStr .=   'uc_active_end_dt, ';
		$queryStr .=   'uc_user_limited, ';
		$queryStr .=   'uc_create_user_id, ';
		$queryStr .=   'uc_create_dt) ';
		$queryStr .=   'VALUES ';
		$queryStr .=   '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($id, $roomId, $langId, $historyIndex, $html, $number, 
							intval($visible), $startDt, $endDt, intval($limited), $user, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * コンテンツをコンテンツIDで取得
	 *
	 * @param string  $id			コンテンツID
	 * @param string  $roomId		ルームID
	 * @param string  $langId		言語ID
	 * @param array   $row			レコード
	 * @return bool					取得 = true, 取得なし= false
	 */
	function getContent($id, $roomId, $langId, &$row)
	{
		$queryStr  = 'SELECT * FROM user_content LEFT JOIN _login_user ON uc_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE uc_deleted = false ';	// 削除されていない
		$queryStr .=     'AND uc_id = ? ';
		$queryStr .=     'AND uc_room_id = ? ';
		$queryStr .=     'AND uc_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id, $roomId, $langId), $row);
		return $ret;
	}
	/**
	 * コンテンツ一覧を取得(管理用)
	 *
	 * @param string  $roomId		ルームID
	 * @param string  $langId		言語ID
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllContents($roomId, $langId, $callback)
	{
		if (empty($roomId) || empty($langId)) return;
		
		$queryStr  = 'SELECT * FROM user_content RIGHT JOIN user_content_item ON uc_id = ui_id AND ui_deleted = false ';
		$queryStr .=   'LEFT JOIN _login_user ON uc_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE (uc_deleted IS NULL ';
		$queryStr .=     'AND ui_deleted = false) ';
		$queryStr .=     'OR (uc_deleted = false ';	// 削除されていない
		$queryStr .=     'AND uc_room_id = ? ';
		$queryStr .=     'AND uc_language_id = ?) ';
		$queryStr .=  'ORDER BY ui_index, ui_id';
		$this->selectLoop($queryStr, array($roomId, $langId), $callback);
	}
	/**
	 * ルームIDですべてのコンテンツを取得
	 *
	 * @param string  $roomId		ルームID
	 * @param string  $langId		言語ID
	 * @param array   $rows			取得レコード
	 * @return bool					取得 = true, 取得なし= false
	 */
	function getAllContentsByRoomId($roomId, $langId, &$rows)
	{
		if (empty($roomId) || empty($langId)) return;
		
		$queryStr  = 'SELECT * FROM user_content LEFT JOIN user_content_item ON uc_id = ui_id AND ui_deleted = false ';
		$queryStr .=   'WHERE uc_deleted = false ';	// 削除されていない
		$queryStr .=     'AND uc_room_id = ? ';
		$queryStr .=     'AND uc_language_id = ? ';
		$retValue = $this->selectRecords($queryStr, array($roomId, $langId), $rows);
		return $retValue;
	}
}
?>
