<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    カスタム検索
 * @author     株式会社 毎日メディアサービス
 * @copyright  Copyright 2010 株式会社 毎日メディアサービス.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: custom_searchDb.php 3045 2010-04-20 08:37:35Z fishbone $
 * @link       http://www.m-media.co.jp
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class custom_searchDb extends BaseDb
{
	/**
	 * すべてのカテゴリ種別を取得
	 *
	 * @param string  $langId		言語ID
	 * @param array  $rows			取得レコード
	 * @return						true=取得、false=取得せず
	 */
	function getAllCategory($langId, &$rows)
	{
		// カテゴリ情報を取得
		$queryStr  = 'SELECT * FROM user_content_category ';
		$queryStr .=   'WHERE ua_deleted = false ';		// 削除されていない
		$queryStr .=     'AND ua_item_id = ? ';
		$queryStr .=     'AND ua_language_id = ? ';
		$queryStr .=   'ORDER BY ua_index';
		$retValue = $this->selectRecords($queryStr, array(''/*カテゴリ情報のみ*/, $langId), $rows);
		return $retValue;
	}
	/**
	 * メニュー作成用のカテゴリ一覧を取得
	 *
	 * @param string  $langId			言語ID
	 * @param array   $categoryArray	取得するカテゴリID
	 * @param array  $rows				取得レコード
	 * @return							true=取得、false=取得せず
	 */
	function getAllCategoryForMenu($langId, $categoryArray, &$rows)
	{
		if (count($categoryArray) <= 0) return false;
		
		// CASE文作成
		$categoryId = '';
		$caseStr = 'CASE ua_id ';
		for ($i = 0; $i < count($categoryArray); $i++){
			$id = '\'' . addslashes($categoryArray[$i]) . '\'';
			$caseStr .= 'WHEN ' . $id . ' THEN ' . $i . ' ';
			$categoryId .= $id . ',';
		}
		$caseStr .= 'END AS no,';
		$categoryId = rtrim($categoryId, ',');
		// タイトルを最後にする
		$caseStr .=   'CASE ua_item_id ';
		$caseStr .=     'WHEN \'\' THEN 1 ';
		$caseStr .=     'ELSE 0 ';
		$caseStr .=   'END AS type ';
		
		$queryStr  = 'SELECT *, ' . $caseStr . ' FROM user_content_category ';
		$queryStr .=   'WHERE ua_deleted = false ';		// 削除されていない
		$queryStr .=     'AND ua_language_id = ? ';
		$queryStr .=     'AND ua_id in (' . $categoryId . ') ';
		$queryStr .=   'ORDER BY no, type, ua_index';
		$retValue = $this->selectRecords($queryStr, array($langId), $rows);
		return $retValue;
	}
	/**
	 * 検索キーワードログ書き込み
	 *
	 * @param string $cid			クッキー値のクライアントID
	 * @param string $widgetId		実行ウィジェット
	 * @param string $keyword		検索キーワード
	 * @param string $basicWord		比較用基本ワード
	 * @return bool					true=ログ出力完了、false=失敗
	 */
	function writeKeywordLog($cid, $widgetId, $keyword, $basicWord)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		$logSerial = $this->gEnv->getCurrentAccessLogSerial();	// 現在のアクセスログシリアル番号
		
		// トランザクション開始
		$this->startTransaction();
		
		// データを追加
		$queryStr  = 'INSERT INTO _search_word ';
		$queryStr .=   '(sw_client_id, sw_widget_id, sw_word, sw_basic_word, sw_access_log_serial, sw_dt) ';
		$queryStr .= 'VALUES ';
		$queryStr .=   '(?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($cid, $widgetId, $keyword, $basicWord, $logSerial, $now));

		// トランザクション終了
		return $this->endTransaction();
	}
	/**
	 * コンテンツを検索(表示用)
	 *
	 * @param int		$limit				取得する項目数
	 * @param int		$page				取得するページ(1～)
	 * @param string	$keyword			検索キーワード
	 * @param array     $categoryInfo		絞り込み用カテゴリ
	 * @param string	$langId				言語
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function searchContentsByKeyword($limit, $page, $keyword, $categoryInfo, $langId, $callback)
	{
		$offset = $limit * ($page -1);
		if ($offset < 0) $offset = 0;
		$initDt = $this->gEnv->getInitValueOfTimestamp();		// 日時初期化値
		$now = date("Y/m/d H:i:s");	// 現在日時
		$roomArray = array();
		
		// カテゴリ条件から検索対象のルームを取得
		if (!empty($categoryInfo)){
			$allParams = array();
			$allQueryStr = '';
			$keys = array_keys($categoryInfo);
			$categoryCount = count($keys);
			for ($i = 0; $i < $categoryCount; $i++){
				$queryStr  = 'SELECT um_room_id FROM user_content_room_category ';
				$params = array();
				
				$key = $keys[$i];
				$item = $categoryInfo[$key];
				if (is_array($item)){		// 複数項目の場合
					$queryStr .=    'WHERE um_category_id = ? AND (';
					$params[] = $key;
					$itemCount = count($item);
					for ($j = 0; $j < $itemCount; $j++){
						$queryStr .=    'um_category_item_id = ?';
						if ($j < $itemCount -1) $queryStr .=    ' OR ';
						$params[] = $item[$j];
					}
					$queryStr .=    ') ';
				} else {
					$queryStr .=    'WHERE um_category_id = ? AND um_category_item_id = ? ';
					$params[] = $key;
					$params[] = $item;
				}
				
				$allParams = array_merge($params, $allParams);
				if ($i == 0){
					$allQueryStr = $queryStr;
				} else {
					$allQueryStr = $queryStr . ' AND um_room_id IN (' . $allQueryStr . ')';
				}
			}
			$retValue = $this->selectRecords($allQueryStr, $allParams, $rows);
			if ($retValue){
				$rowCount = count($rows);
				for ($i = 0; $i < $rowCount; $i++){
					$roomArray[] = $rows[$i]['um_room_id'];
				}
			}
		}

		$params = array();		// パラメータ初期化
		$queryStr  = 'SELECT DISTINCT \'uc\' AS type, uc_room_id AS id, ur_name AS name, ur_content_update_dt AS dt ';
		$queryStr .= 'FROM user_content LEFT JOIN user_content_room ON uc_room_id = ur_id AND ur_deleted = false ';
		$queryStr .= 'WHERE uc_language_id = ? ';	$params[] = $langId;
		$queryStr .=   'AND uc_deleted = false ';		// 削除されていない
		$queryStr .=   'AND uc_visible = true ';		// 公開中

		// 検索対象のルームを制限
		if (!empty($categoryInfo)){		// カテゴリ制限のとき
			if (count($roomArray) == 0){
				$idStr = '\'/dummy/\'';				// ダミーデータを設定
			} else {
				$idStr = '';
				for ($i = 0; $i < count($roomArray); $i++){
					$idStr = '\'' . addslashes($roomArray[$i]) . '\',';
				}
				$idStr = rtrim($idStr, ',');
			}
			$queryStr .=  'AND uc_room_id in (' . $idStr . ') ';
		}
		
		// コンテンツを検索
		if (!empty($keyword)){
			// 「'"\」文字をエスケープ
			$keyword = addslashes($keyword);
			
			$queryStr .=    'AND (uc_data LIKE \'%' . $keyword . '%\' ';		// コンテンツ実データ
			$queryStr .=    'OR uc_room_id LIKE \'%' . $keyword . '%\' ';			// ルームID
			$queryStr .=    'OR ur_name LIKE \'%' . $keyword . '%\') ';			// ルーム名
		}
		
		// 公開期間を指定
		$queryStr .=    'AND (uc_active_start_dt = ? OR (uc_active_start_dt != ? AND uc_active_start_dt <= ?)) ';
		$queryStr .=    'AND (uc_active_end_dt = ? OR (uc_active_end_dt != ? AND uc_active_end_dt > ?)) ';
		$params[] = $initDt;
		$params[] = $initDt;
		$params[] = $now;
		$params[] = $initDt;
		$params[] = $initDt;
		$params[] = $now;
		
		// コンテンツ更新の最新のデータから取得
		$queryStr .=  'ORDER BY ur_content_update_dt desc limit ' . $limit . ' offset ' . $offset;
		$this->selectLoop($queryStr, $params, $callback);
	}
	/**
	 * デフォルトのタブ定義を取得
	 *
	 * @param string	$lang				言語
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getDefaultTab($lang, &$row)
	{
		$queryStr = 'SELECT * FROM user_content_tab ';
		$queryStr .=  'WHERE ub_deleted = false ';		// 削除されていない
		$queryStr .=    'AND ub_visible = true ';		// 表示中
		$queryStr .=    'AND ub_language_id = ? ';
		$queryStr .=  'ORDER BY ub_index';
		$ret = $this->selectRecord($queryStr, array($lang), $row);
		return $ret;
	}
	/**
	 * ルームIDですべてのコンテンツを取得
	 *
	 * @param string  $roomId		ルームID
	 * @param string  $langId		言語ID
	 * @param array   $rows			取得レコード
	 * @return bool					取得 = true, 取得なし= false
	 */
	function getAllContentsByRoomId($roomId, $langId, &$rows)
	{
		if (empty($roomId) || empty($langId)) return false;
		
		$queryStr  = 'SELECT * FROM user_content LEFT JOIN user_content_item ON uc_id = ui_id AND ui_deleted = false ';
		$queryStr .=   'WHERE uc_deleted = false ';	// 削除されていない
		$queryStr .=     'AND uc_room_id = ? ';
		$queryStr .=     'AND uc_language_id = ? ';
		$retValue = $this->selectRecords($queryStr, array($roomId, $langId), $rows);
		return $retValue;
	}
}
?>
