<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2008 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: ec_stockDb.php 597 2008-05-08 09:14:44Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class ec_stockDb extends BaseDb
{
	/**
	 * 在庫状況を日付範囲で取得
	 *
	 * @param int $id				商品ID
	 * @param timestamp $startDate	データを取得する先頭日付
	 * @param timestamp $endDate	データを取得する終了日付(結果にこの日付のデータは含まれない)
	 * @param int $planType			取得する区分(0=すべて、1=入庫、2=出庫、3=棚卸)
	 * @param function	$callback			コールバック関数
	 * @return なし
	 */
	function getStockRecordByDate($id, $startDate, $endDate, $planType, $callback)
	{
		$params = array();
		$queryStr = 'SELECT * FROM stock_plan LEFT JOIN customer ON sp_customer_id = cc_id AND cc_deleted = false ';
		$queryStr .=  'WHERE sp_deleted = false ';		// データが削除されていない
		$queryStr .=    'AND sp_product_id = ? '; $params[] = $id;		// 商品ID
		if (!empty($startDate)){
			$queryStr .=    'AND ? <= sp_scheduled_dt ';
			$params[] = $startDate;
		}
		if (!empty($endDate)){
			$queryStr .=    'AND sp_scheduled_dt < ? ';
			$params[] = $endDate;
		}
		if (!empty($planType)){
			$queryStr .=    'AND sp_plan_type = ? ';
			$params[] = $planType;		// 区分
		}
		$queryStr .=  'ORDER BY sp_scheduled_dt, sp_index ';
		//return $this->selectRecords($queryStr, $params, $rows);
		$this->selectLoop($queryStr, $params, $callback, null);
	}
	/**
	 * 在庫状況を日付範囲で取得
	 *
	 * @param int $id				商品ID
	 * @param timestamp $startDate	データを取得する先頭日付
	 * @param timestamp $endDate	データを取得する終了日付(結果にこの日付のデータは含まれない)
	 * @param int $planType			取得する区分(0=すべて、1=入庫、2=出庫、3=棚卸)
	 * @param array     $rows		取得レコード
	 * @return bool					取得 = true, 取得なし= false
	 */
	function getStockRecordsByDate($id, $startDate, $endDate, $planType, &$rows)
	{
		$params = array();
		$queryStr = 'SELECT * FROM stock_plan LEFT JOIN customer ON sp_customer_id = cc_id AND cc_deleted = false ';
		$queryStr .=  'WHERE sp_deleted = false ';		// データが削除されていない
		$queryStr .=    'AND sp_product_id = ? '; $params[] = $id;		// 商品ID
		if (!empty($startDate)){
			$queryStr .=    'AND ? <= sp_scheduled_dt ';
			$params[] = $startDate;
		}
		if (!empty($endDate)){
			$queryStr .=    'AND sp_scheduled_dt < ? ';
			$params[] = $endDate;
		}
		if (!empty($planType)){
			$queryStr .=    'AND sp_plan_type = ? ';
			$params[] = $planType;		// 区分
		}
		$queryStr .=  'ORDER BY sp_scheduled_dt, sp_index ';
		return $this->selectRecords($queryStr, $params, $rows);
	}
	/**
	 * 最近の棚卸日付を取得
	 *
	 * @param int $id				商品ID
	 * @param timestamp $date		この日付以前で検索
	 * @param timestamp $startDate	取得した日付
	 * @return bool					取得 = true, 取得なし= false
	 */
	function getStockStartDate($id, $date, &$startDate)
	{
		$planType = 3;		// 棚卸
		$params = array();
		$queryStr = 'SELECT * FROM stock_plan ';
		$queryStr .=  'WHERE sp_deleted = false ';		// データが削除されていない
		$queryStr .=    'AND sp_product_id = ? '; $params[] = $id;		// 商品ID
		if (!empty($date)){
			$queryStr .=    'AND sp_scheduled_dt <= ? ';
			$params[] = $date;
		}
		$queryStr .=    'AND sp_plan_type = ? ';
		$params[] = $planType;		// 区分
		$queryStr .=  'ORDER BY sp_scheduled_dt DESC ';
		$ret = $this->selectRecord($queryStr, $params, $row);
		if ($ret){
			$startDate = $row['sp_scheduled_dt'];
		} else {		// 見つからないときは先頭の日付
			$params = array();
			$queryStr = 'SELECT * FROM stock_plan ';
			$queryStr .=  'WHERE sp_deleted = false ';		// データが削除されていない
			$queryStr .=    'AND sp_product_id = ? '; $params[] = $id;		// 商品ID
			$queryStr .=  'ORDER BY sp_scheduled_dt ';
			$ret = $this->selectRecord($queryStr, $params, $row);
			if ($ret){
				$startDate = $row['sp_scheduled_dt'];
			} else {
				$startDate = '';
			}
		}
		return $ret;
	}
	/**
	 * 在庫予定の新規追加
	 *
	 * @param int	  $id			商品ID
	 * @param timestamp  $date		日付
	 * @param int     $type			処理区分
	 * @param int     $customerId	取引先ID
	 * @param int     $quantity		数量
	 * @param string  $note			備考
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function addStockPlan($id, $date, $type, $customerId, $quantity, $note, &$newSerial)
	{
		global $gEnvManager;

		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// インデックス番号を決定する
		$queryStr = 'SELECT * FROM stock_plan ';
		$queryStr .=  'WHERE sp_deleted = false ';		// 削除されていない
		$queryStr .=    'AND sp_product_id = ? ';
		$queryStr .=    'AND sp_scheduled_dt = ? ';
		$queryStr .=  'ORDER BY sp_index desc';
		$ret = $this->selectRecord($queryStr, array($id, $date), $row);
		if ($ret){
			$index = $row['sp_index'] + 1;
		} else {
			$index = 0;
		}
		// 履歴番号を決定する
		$queryStr = 'SELECT * FROM stock_plan ';
		$queryStr .=  'WHERE sp_product_id = ? ';
		$queryStr .=    'AND sp_scheduled_dt = ? ';
		$queryStr .=    'AND sp_index = ? ';
		$queryStr .=  'ORDER BY sp_history_index desc';
		$ret = $this->selectRecord($queryStr, array($id, $date, $index), $row);
		if ($ret){
			$historyIndex = $row['sp_history_index'] + 1;
		} else {
			$historyIndex = 0;
		}		
		
		// データを追加
		$queryStr = 'INSERT INTO stock_plan ';
		$queryStr .=  '(sp_product_id, sp_scheduled_dt, sp_index, sp_history_index, sp_plan_type, sp_customer_id, sp_quantity, sp_note, sp_create_user_id, sp_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($id, $date, $index, $historyIndex, $type, $customerId, $quantity, $note, $userId, $now));
		
		// 新規のシリアル番号取得
		$queryStr = 'select max(sp_serial) as ns from stock_plan ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
			
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * 在庫予定をシリアル番号で取得
	 *
	 * @param int		$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getStockPlanBySerial($serial, &$row)
	{
		$queryStr  = 'select * from stock_plan LEFT JOIN _login_user ON sp_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE sp_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}	
	/**
	 * 在庫予定の更新
	 *
	 * @param int     $serial		シリアル番号
	 * @param int     $type			処理区分
	 * @param int     $customerId	取引先ID
	 * @param int     $quantity		数量
	 * @param string  $note			備考
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateStockPlan($serial, $type, $customerId, $quantity, $note, &$newSerial)
	{	
		global $gEnvManager;

		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
				
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$historyIndex = 0;		// 履歴番号
		$queryStr  = 'select * from stock_plan ';
		$queryStr .=   'where sp_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if ($ret){		// 既に登録レコードがあるとき
			if ($row['sp_deleted']){		// レコードが削除されていれば終了
				$this->endTransaction();
				return false;
			}
			$historyIndex = $row['sp_history_index'] + 1;
		} else {		// 存在しない場合は終了
			$this->endTransaction();
			return false;
		}
		
		// 古いレコードを削除
		$queryStr  = 'UPDATE stock_plan ';
		$queryStr .=   'SET sp_deleted = true, ';	// 削除
		$queryStr .=     'sp_update_user_id = ?, ';
		$queryStr .=     'sp_update_dt = ? ';
		$queryStr .=   'WHERE sp_serial = ?';
		$this->execStatement($queryStr, array($userId, $now, $serial));
		
		// 新規レコード追加
		$queryStr = 'INSERT INTO stock_plan ';
		$queryStr .=  '(sp_product_id, sp_scheduled_dt, sp_index, sp_history_index, sp_plan_type, sp_customer_id, sp_quantity, sp_note, sp_create_user_id, sp_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($row['sp_product_id'], $row['sp_scheduled_dt'], $row['sp_index'], $historyIndex, $type, $customerId, $quantity, $note, $userId, $now));

		// 新規のシリアル番号取得
		$queryStr = 'select max(sp_serial) as ns from stock_plan ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * 在庫予定をシリアル番号で削除
	 *
	 * @param array   $serial		シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function deleteStockPlanBySerial($serial)
	{
		global $gEnvManager;
		
		// 引数のエラーチェック
		if (!is_array($serial)) return false;
		if (count($serial) <= 0) return true;
		
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
		
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		for ($i = 0; $i < count($serial); $i++){
			$queryStr  = 'SELECT * FROM stock_plan ';
			$queryStr .=   'WHERE sp_deleted = false ';		// 未削除
			$queryStr .=     'AND sp_serial = ? ';
			$ret = $this->isRecordExists($queryStr, array($serial[$i]));
			// 存在しない場合は、既に削除されたとして終了
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE stock_plan ';
		$queryStr .=   'SET sp_deleted = true, ';	// 削除
		$queryStr .=     'sp_update_user_id = ?, ';
		$queryStr .=     'sp_update_dt = ? ';
		$queryStr .=   'WHERE sp_serial in (' . implode($serial, ',') . ') ';
		$this->execStatement($queryStr, array($userId, $now));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
}
?>
