<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2009 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: ec_dispDb.php 2048 2009-07-07 06:12:22Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class ec_dispDb extends BaseDb
{
	/**
	 * Eコマース定義値を取得
	 *
	 * @param string $key		キーとなる項目値
	 * @return string $value	値
	 */
	function getConfig($key)
	{
		$retValue = '';
		$queryStr = 'SELECT cg_value FROM commerce_config ';
		$queryStr .=  'WHERE cg_id  = ?';
		$ret = $this->selectRecord($queryStr, array($key), $row);
		if ($ret) $retValue = $row['cg_value'];
		return $retValue;
	}
	/**
	 * Eコマース定義値を更新
	 *
	 * @param string $key		キーとなる項目値
	 * @param string $value		値
	 * @return					true = 正常、false=異常
	 */
	function updateConfig($key, $value)
	{
		// データの確認
		$queryStr = 'SELECT cg_value FROM commerce_config ';
		$queryStr .=  'WHERE cg_id  = ?';
		$ret = $this->isRecordExists($queryStr, array($key));
		if ($ret){
			$queryStr = "UPDATE commerce_config SET cg_value = ? WHERE cg_id = ?";
			return $this->execStatement($queryStr, array($value, $key));
		} else {
			$queryStr = "INSERT INTO commerce_config (cg_id, cg_value) VALUES (?, ?)";
			return $this->execStatement($queryStr, array($key, $value));
		}
	}
	/**
	 * 通貨一覧を取得
	 *
	 * @param string	$lang				言語
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getCurrency($lang, $callback)
	{
		$queryStr = 'SELECT * FROM currency ';
		$queryStr .=  'WHERE cu_language_id = ? ';
		$queryStr .=  'ORDER BY cu_index';
		$this->selectLoop($queryStr, array($lang), $callback, null);
	}
	/**
	 * コンテンツ項目を外部用キーで取得
	 *
	 * @param string	$key				外部用キー
	 * @param string	$langId				言語ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getContentByKey($key, $langId, &$row)
	{
		$queryStr  = 'SELECT * FROM content ';
		$queryStr .=   'WHERE cn_deleted = false ';	// 削除されていない
		$queryStr .=   'AND cn_type = ? ';
		$queryStr .=   'AND cn_key = ? ';
		$queryStr .=   'AND cn_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array(''/*デフォルトコンテンツ*/, $key, $langId), $row);
		return $ret;
	}
	/**
	 * 変換キーテーブルを取得
	 *
	 * @param string	$key				キー文字列
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
/*	function getAllKey($key, $callback)
	{
		$queryStr = 'SELECT * FROM _key_value ';
		$queryStr .=  'WHERE kv_deleted = false ';
		$queryStr .=    'AND kv_id LIKE \'' . $key . '%\' ';
		$queryStr .=  'ORDER BY kv_id';
		$this->selectLoop($queryStr, array(), $callback, null);
	}*/
	/**
	 * 変換キーテーブルを取得
	 *
	 * @param string	$key				キー文字列
	 * @param string	$group				グループID
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getAllKey($key, $group, $callback)
	{
		$queryStr = 'SELECT * FROM _key_value ';
		$queryStr .=  'WHERE kv_deleted = false ';
		$queryStr .=    'AND kv_id LIKE \'' . $key . '%\' ';
		$queryStr .=    'AND kv_group_id = ? ';
		$queryStr .=  'ORDER BY kv_id';
		$this->selectLoop($queryStr, array($group), $callback, null);
	}
	/**
	 * 商品カテゴリー一覧を取得
	 *
	 * @param string	$category			親カテゴリーID
	 * @param string	$lang				言語
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getCategoryList($category, $lang, $callback)
	{
		$queryStr = 'SELECT * FROM product_category ';
		$queryStr .=  'WHERE pc_deleted = false ';		// 削除されていない
		$queryStr .=    'AND pc_visible = true ';		// 表示可能なカテゴリー
		$queryStr .=    'AND pc_parent_id = ? ';		
		$queryStr .=    'AND pc_language_id = ? ';		
		$queryStr .=  'ORDER BY pc_sort_order';
		$this->selectLoop($queryStr, array($category, $lang), $callback, null);
	}
	/**
	 * 商品一覧を取得
	 *
	 * @param string	$categoryId			商品カテゴリーID
	 * @param string	$lang				言語
	 * @param int		$limit				取得する項目数
	 * @param int		$offset				取得する先頭位置(0～)
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getProductByCategoryId($categoryId, $lang, $limit, $offset, $callback)
	{
		$queryStr = 'SELECT distinct(pt_serial) ';
		$queryStr .=  'FROM product RIGHT JOIN product_with_category ON pt_serial = pw_product_serial ';
		$queryStr .=  'WHERE pt_deleted = false ';// 削除されていない
		$queryStr .=    'AND pt_visible = true ';		// 表示可能な商品	
		$queryStr .=    'AND pt_language_id = ? ';
		if (!empty($categoryId)) $queryStr .=    'AND pw_category_id in (' . $categoryId . ') ';
		
		$serialArray = array();
		$ret = $this->selectRecords($queryStr, array($lang), $serialRows);
		if ($ret){
			for ($i = 0; $i < count($serialRows); $i++){
				$serialArray[] = $serialRows[$i]['pt_serial'];
			}
		}
		$serialStr = implode(',', $serialArray);

		$queryStr = 'SELECT * FROM product ';
		if (!empty($serialStr)) $queryStr .=  'WHERE pt_serial in (' . $serialStr . ') ';
		$queryStr .=  'ORDER BY pt_sort_order limit ' . $limit . ' offset ' . $offset;
		$this->selectLoop($queryStr, array(), $callback, null);
	}
	/**
	 * 商品総数を取得
	 *
	 * @param string	$categoryId			商品カテゴリーID
	 * @param string	$lang				言語
	 * @return int							商品総数
	 */
	function getProductCountByCategoryId($categoryId, $lang)
	{
		$queryStr = 'SELECT distinct(pt_serial) ';
		$queryStr .=  'FROM product RIGHT JOIN product_with_category ON pt_serial = pw_product_serial ';
		$queryStr .=  'WHERE pt_deleted = false ';// 削除されていない
		$queryStr .=    'AND pt_visible = true ';		// 表示可能な商品
		$queryStr .=    'AND pt_language_id = ? ';
		if (!empty($categoryId)) $queryStr .=    'AND pw_category_id in (' . $categoryId . ') ';
		return $this->selectRecordCount($queryStr, array($lang));
	}
	/**
	 * 商品一覧を取得
	 *
	 * @param string	$keyword			検索キーワード
	 * @param string	$lang				言語
	 * @param int		$limit				取得する項目数
	 * @param int		$offset				取得する先頭位置(0～)
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getProductByKeyword($keyword, $lang, $limit, $offset, $callback)
	{
		$params = array();
		$queryStr = 'SELECT * FROM product ';
		$queryStr .=  'WHERE pt_language_id = ? '; $params[] = $lang;
		$queryStr .=    'AND pt_deleted = false ';		// 削除されていない
		$queryStr .=    'AND pt_visible = true ';		// 表示可能な商品
		
		// 商品名、商品コード、説明を検索
		if (!empty($keyword)){
			// 「'"\」文字をエスケープ
			$keyword = addslashes($keyword);
			
			$queryStr .=    'AND (pt_name LIKE \'%' . $keyword . '%\' ';
			$queryStr .=    'OR pt_code LIKE \'%' . $keyword . '%\' ';
			$queryStr .=    'OR pt_description LIKE \'%' . $keyword . '%\' ';
			$queryStr .=    'OR pt_description_short LIKE \'%' . $keyword . '%\') ';
		}
		$queryStr .=  'ORDER BY pt_sort_order limit ' . $limit . ' offset ' . $offset;
		$this->selectLoop($queryStr, $params, $callback, null);
	}
	/**
	 * 商品総数を取得
	 *
	 * @param string	$keyword			検索キーワード
	 * @param string	$lang				言語
	 * @return int							商品総数
	 */
	function getProductCountByKeyword($keyword, $lang)
	{
		$params = array();
		$queryStr = 'SELECT * FROM product ';
		$queryStr .=  'WHERE pt_language_id = ? '; $params[] = $lang;
		$queryStr .=    'AND pt_deleted = false ';		// 削除されていない
		$queryStr .=    'AND pt_visible = true ';		// 表示可能な商品

		// 商品名、商品コード、説明を検索
		if (!empty($keyword)){
			// 「'"\」文字をエスケープ
			$keyword = addslashes($keyword);
			
			$queryStr .=    'AND (pt_name LIKE \'%' . $keyword . '%\' ';
			$queryStr .=    'OR pt_code LIKE \'%' . $keyword . '%\' ';
			$queryStr .=    'OR pt_description LIKE \'%' . $keyword . '%\' ';
			$queryStr .=    'OR pt_description_short LIKE \'%' . $keyword . '%\') ';
		}
		return $this->selectRecordCount($queryStr, $params);
	}
	/**
	 * 商品をシリアル番号で取得
	 *
	 * @param int		$serial				シリアル番号
	 * @param array     $row				レコード
	 * @param array     $row2				商品価格
	 * @param array     $row3				商品画像
	 * @param array     $row4				商品ステータス
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getProductBySerial($serial, &$row, &$row2, &$row3, &$row4)
	{
		$queryStr  = 'select * from product LEFT JOIN _login_user ON pt_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE pt_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if ($ret){
			$queryStr  = 'SELECT * FROM product_price ';
			$queryStr .=   'WHERE pp_deleted = false ';// 削除されていない
			$queryStr .=     'AND pp_product_id = ? ';
			$queryStr .=     'AND pp_language_id = ? ';
			$this->selectRecords($queryStr, array($row['pt_id'], $row['pt_language_id']), $row2);
			
			$queryStr  = 'SELECT * FROM product_image ';
			$queryStr .=   'WHERE im_deleted = false ';// 削除されていない
			$queryStr .=     'AND im_type = 2 ';		// 商品画像
			$queryStr .=     'AND im_id = ? ';
			$queryStr .=     'AND im_language_id = ? ';
			$this->selectRecords($queryStr, array($row['pt_id'], $row['pt_language_id']), $row3);
			
			// 商品ステータス
			$queryStr  = 'SELECT * FROM product_status ';
			$queryStr .=   'WHERE ps_deleted = false ';// 削除されていない
			$queryStr .=     'AND ps_id = ? ';
			$queryStr .=     'AND ps_language_id = ? ';
			$this->selectRecords($queryStr, array($row['pt_id'], $row['pt_language_id']), $row4);
		}
		return $ret;
	}
	/**
	 * 商品を商品ID、言語IDで取得
	 *
	 * @param int		$id					商品ID
	 * @param string	$langId				言語ID
	 * @param array     $row				レコード
	 * @param array     $row2				商品価格
	 * @param array     $row3				商品画像
	 * @param array     $row4				商品ステータス
	 * @param array     $row5				商品カテゴリー
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getProductByProductId($id, $langId, &$row, &$row2, &$row3, &$row4, &$row5)
	{
		$queryStr  = 'SELECT * FROM product LEFT JOIN _login_user ON pt_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE pt_deleted = false ';	// 削除されていない
		$queryStr .=    'AND pt_visible = true ';		// 表示可能な商品
		$queryStr .=    'AND pt_id = ? ';
		$queryStr .=    'AND pt_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id, $langId), $row);
		if ($ret){
			$queryStr  = 'SELECT * FROM product_price ';
			$queryStr .=   'WHERE pp_deleted = false ';// 削除されていない
			$queryStr .=     'AND pp_product_id = ? ';
			$queryStr .=     'AND pp_language_id = ? ';
			$this->selectRecords($queryStr, array($row['pt_id'], $row['pt_language_id']), $row2);
			
			$queryStr  = 'SELECT * FROM product_image ';
			$queryStr .=   'WHERE im_deleted = false ';// 削除されていない
			$queryStr .=     'AND im_type = 2 ';		// 商品画像
			$queryStr .=     'AND im_id = ? ';
			$queryStr .=     'AND im_language_id = ? ';
			$this->selectRecords($queryStr, array($row['pt_id'], $row['pt_language_id']), $row3);
			
			// 商品ステータス
			$queryStr  = 'SELECT * FROM product_status ';
			$queryStr .=   'WHERE ps_deleted = false ';// 削除されていない
			$queryStr .=     'AND ps_id = ? ';
			$queryStr .=     'AND ps_language_id = ? ';
			$this->selectRecords($queryStr, array($row['pt_id'], $row['pt_language_id']), $row4);
			
			// 商品カテゴリー
			$queryStr  = 'SELECT * FROM product_with_category ';
			$queryStr .=   'WHERE pw_product_serial = ? ';
			$queryStr .=  'ORDER BY pw_index ';
			$this->selectRecords($queryStr, array($row['pt_serial']), $row5);
		}
		return $ret;
	}
	/**
	 * 画像情報を取得
	 *
	 * @param string	$type			画像タイプ
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getProductImageInfo($type, &$row)
	{
		$queryStr  = 'SELECT * FROM image_size ';
		$queryStr .=   'WHERE is_id = ? ';
		$ret = $this->selectRecord($queryStr, array($type), $row);
		return $ret;
	}
	/**
	 * 単位情報を取得
	 *
	 * @param string	$type				単位タイプ
	 * @param string	$lang				言語
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getUnitTypeRecord($type, $lang, &$row)
	{
		$queryStr = 'SELECT * FROM unit_type ';
		$queryStr .=  'WHERE ut_id = ? AND ut_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($type, $lang), $row);
		return $ret;
	}
	
	/**
	 * カートヘッダ情報を取得
	 *
	 * @param string	$cartId				カートID
	 * @param string	$lang				言語
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getCartHead($cartId, $lang, &$row)
	{
		$queryStr = 'SELECT * FROM shop_cart ';
		$queryStr .=  'WHERE sh_id = ? AND sh_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($cartId, $lang), $row);
		return $ret;
	}
	/**
	 * カート商品情報を取得
	 *
	 * @param int		$serial				カートシリアル番号
	 * @param string	$productId			商品ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getCartItem($serial, $productId, &$row)
	{
		$queryStr = 'SELECT * FROM shop_cart_item ';
		$queryStr .=  'WHERE si_head_serial = ? AND si_product_id = ? ';
		$ret = $this->selectRecord($queryStr, array($serial, $productId), $row);
		return $ret;
	}
	/**
	 * カートに商品を追加
	 *
	 * @param int		$serial				カートシリアル番号
	 * @param int		$item_serial		カート商品シリアル番号
	 * @param string	$cartId				カートID
	 * @param string	$productId			商品ID
	 * @param string	$lang				言語
	 * @param string	$currency			通貨
	 * @param float	    $priceWithTax		税込み価格(単位個数あたり)
	 * @param int		$quantity			数量
	 * @return bool							取得 = true, 取得なし= false
	 */
	function addProductToCart($serial, $item_serial, $cartId, $productId, $lang, $currency, $priceWithTax, $quantity, $userId)
	{
		$now = date("Y/m/d H:i:s");	// 現在日時
		
		// トランザクション開始
		$this->startTransaction();
		
		if ($serial == 0){		// カートがまだ存在しないとき
			// 新規作成
			$queryStr = "INSERT INTO shop_cart (sh_id, sh_language_id, sh_user_id, sh_dt) VALUES (?, ?, ?, ?)";
			$params = array($cartId, $lang, $userId, $now);
			$this->execStatement($queryStr, $params);
			
			$queryStr = 'select max(sh_serial) as m from shop_cart';
			$ret = $this->selectRecord($queryStr, array(), $row);
			if ($ret){
				$maxSerial = $row['m'];
			} else {
				$this->endTransaction();
				return false;
			}
			
			// 商品の登録
			$price = $priceWithTax * $quantity;
			$queryStr = "INSERT INTO shop_cart_item (si_head_serial, si_product_id, si_currency_id, si_quantity, si_subtotal) VALUES (?, ?, ?, ?, ?)";
			$params = array($maxSerial, $productId, $currency, $quantity, $price);
			$this->execStatement($queryStr, $params);			
		} else if ($item_serial == 0){		// カート商品が存在しないとき
			// カートヘッダの日付を更新
			$queryStr  = 'UPDATE shop_cart ';
			$queryStr .=   'SET sh_dt = ? ';
			$queryStr .=   'WHERE sh_serial = ? ';
			$this->execStatement($queryStr, array($now, $serial));
		
			// 商品の登録
			$price = $priceWithTax * $quantity;
			$queryStr = "INSERT INTO shop_cart_item (si_head_serial, si_product_id, si_currency_id, si_quantity, si_subtotal) VALUES (?, ?, ?, ?, ?)";
			$params = array($serial, $productId, $currency, $quantity, $price);
			$this->execStatement($queryStr, $params);			
		} else {		// カートに商品が存在するときは更新
			// カートヘッダの日付を更新
			$queryStr  = 'UPDATE shop_cart ';
			$queryStr .=   'SET sh_dt = ? ';
			$queryStr .=   'WHERE sh_serial = ? ';
			$this->execStatement($queryStr, array($now, $serial));
					
			$queryStr = 'SELECT * FROM shop_cart_item ';
			$queryStr .=  'WHERE si_serial = ?';
			$ret = $this->selectRecord($queryStr, array($item_serial), $row);
			if (!$ret){
				$this->endTransaction();
				return false;
			}
		
			// カート商品を更新
			$newQuantity = $quantity + $row['si_quantity'];
			$newPrice = $priceWithTax * $quantity + $row['si_subtotal'];
			$queryStr  = 'UPDATE shop_cart_item ';
			$queryStr .=   'SET si_quantity = ?, ';
			$queryStr .=   'si_subtotal = ? ';
			$queryStr .=   'WHERE si_serial = ? ';
			$this->execStatement($queryStr, array($newQuantity, $newPrice, $item_serial));
		}
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * カート商品情報を無効化
	 *
	 * @param int		$serial				カートシリアル番号
	 * @param int		$item_serial		カート商品シリアル番号
	 * @return bool							取得 = true, 取得なし= false
	 */
	function voidProductInCart($serial, $item_serial)
	{
		if ($serial == 0) return true;		// カートがまだ存在しないとき
			
		$now = date("Y/m/d H:i:s");	// 現在日時
				
		// トランザクション開始
		$this->startTransaction();
		
		// カートヘッダの日付を更新
		$queryStr  = 'UPDATE shop_cart ';
		$queryStr .=   'SET sh_dt = ? ';
		$queryStr .=   'WHERE sh_serial = ? ';
		$this->execStatement($queryStr, array($now, $serial));
		
		// カート商品を無効化
		$queryStr  = 'UPDATE shop_cart_item ';
		$queryStr .=   'SET si_available = ? ';
		$queryStr .=   'WHERE si_serial = ? ';
		$this->execStatement($queryStr, array(0, $item_serial));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
}
?>
