<?php
/**
 * コンテナクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2011 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: admin_photo_mainImagebrowseWidgetContainer.php 4314 2011-09-12 12:27:21Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getCurrentWidgetContainerPath() . '/admin_photo_mainBaseWidgetContainer.php');

class admin_photo_mainImagebrowseWidgetContainer extends admin_photo_mainBaseWidgetContainer
{
	private $serialArray = array();		// 表示されているファイルのインデックス番号
	private $photoBasePath;				// アクセス可能な画像格納ディレクトリ
	private $sortOrderByDateAsc;		// 日付でソート
	private $masterMimeType;			// マスター画像のMIMEタイプ
	private $permitMimeType;			// アップロードを許可する画像タイプ
	private $userId;					// 現在のユーザ
	const DEFAULT_LIST_COUNT = 30;			// 最大リスト表示数
	const FILE_ICON_FILE = '/images/system/tree/file.png';			// ファイルアイコン
	const FOLDER_ICON_FILE = '/images/system/tree/folder.png';		// ディレクトリアイコン
	const PARENT_ICON_FILE = '/images/system/tree/parent.png';		// 親ディレクトリアイコン
	const ICON_SIZE = 16;		// アイコンのサイズ
	const CSS_FILE = '/swfupload2.5/css/default.css';		// CSSファイルのパス
	const PHOTO_DIR = '/etc/photo';		// マスター画像格納ディレクトリ
	const PHOTO_HOME_DIR = '/etc/photo/home';		// マスター画像格納ディレクトリ（ユーザ別)
	const DEFAULT_THUMBNAIL_SIZE = 128;		// サムネール画像サイズ
	const DEFAULT_IMAGE_EXT = 'jpg';			// 画像ファイルのデフォルト拡張子
	const DEFAULT_IMAGE_TYPE = IMAGETYPE_JPEG;
	const DEFAULT_PUBLIC_IMAGE_SIZE = 450;		// 一般表示用画像(ウォータマーク入り画像)の縦または横の最大サイズ
	const THUMBNAIL_DIR = '/widgets/photo/image';		// 画像格納ディレクトリ
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		// 親クラスを呼び出す
		parent::__construct();
		
		$this->masterMimeType = image_type_to_mime_type(IMAGETYPE_JPEG);			// マスター画像のMIMEタイプ
		$this->permitMimeType = array(	image_type_to_mime_type(IMAGETYPE_GIF),
										image_type_to_mime_type(IMAGETYPE_JPEG),
										image_type_to_mime_type(IMAGETYPE_PNG),
										image_type_to_mime_type(IMAGETYPE_BMP));			// アップロードを許可する画像タイプ
	}
	/**
	 * テンプレートファイルを設定
	 *
	 * _assign()でデータを埋め込むテンプレートファイルのファイル名を返す。
	 * 読み込むディレクトリは、「自ウィジェットディレクトリ/include/template」に固定。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。そのまま_assign()に渡る
	 * @return string 						テンプレートファイル名。テンプレートライブラリを使用しない場合は空文字列「''」を返す。
	 */
	function _setTemplate($request, &$param)
	{	
		return 'imagebrowse.tmpl.html';
	}
	/**
	 * テンプレートにデータ埋め込む
	 *
	 * _setTemplate()で指定したテンプレートファイルにデータを埋め込む。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。_setTemplate()と共有。
	 * @param								なし
	 */
	function _assign($request, &$param)
	{
		$this->langId = $this->gEnv->getCurrentLanguage();
		$account = $this->gEnv->getCurrentUserAccount();
		$this->userId		= $this->gEnv->getCurrentUserId();			// 画像アップロードユーザ
		
		// パラメータエラーチェック
		if (empty($account)){
			$this->SetMsg(self::MSG_APP_ERR, $this->_('Error parameters.'));// パラメータにエラーがあります。
			return;
		}
		
		// 移動可能なディレクトリ範囲を取得
		if ($this->_isLimitedUser){		// 使用限定ユーザの場合
			$this->photoBasePath = $this->gEnv->getIncludePath() . self::PHOTO_HOME_DIR . DIRECTORY_SEPARATOR . $account;				// 画像格納ディレクトリ
		} else {
			$this->photoBasePath = $this->gEnv->getIncludePath() . self::PHOTO_DIR;				// 画像格納ディレクトリ
		}

		// パスを取得
		$path = trim($request->valueOf('path'));		// 現在のパス
		if (empty($path)){
			$path = $this->photoBasePath;
		} else {
			if (!strStartsWith($path, '/')) $path = '/' . $path;
			$path = $this->photoBasePath . $path;
		}
		// パスのエラーチェック
		if (isDescendantPath($this->photoBasePath, $path)){
			if (!is_dir($path)){
				$this->setUserErrorMsg($this->_('Failed in accessing.'));			// アクセスできません
				return;
			}
		} else {
			$this->setUserErrorMsg($this->_('Failed in accessing.'));			// アクセスできません
			return;
		}
		
		// ページ番号
		$viewCount = self::DEFAULT_LIST_COUNT;				// 表示項目数
		$pageNo = $request->trimValueOf('page');				// ページ番号
		if (empty($pageNo)) $pageNo = 1;
		
		$act = $request->trimValueOf('act');
		if ($act == 'uploadfile'){		// ファイルアップロードのとき
			// ページ作成処理中断
			$this->gPage->abortPage();

			// ファイルのアップロード処理
			if (isset($_FILES["Filedata"]) && is_uploaded_file($_FILES['Filedata']['tmp_name'])){		// アップロードファイルがある場合
				// ファイル名を作成
				$code = $account . '-' . $this->_mainDb->getNewPhotoNo($this->userId);		// 画像コード
				$ret = $this->_mainDb->isExistsPhotoCode($code);		// 画像コードの重複確認
				if (!$ret){		// 存在しない場合
					$filename = md5($code);
					$originalFilename = $_FILES['Filedata']['name'];		// 元のファイル名
					$filePath = $path . DIRECTORY_SEPARATOR . $filename;
					if (file_exists($filePath)) $this->writeError(__METHOD__, 'アップロード画像がすでに存在しています。新しい画像で置き換えます。', 1100, 'オリジナルファイル名=' . $originalFilename);

					// アップされたファイルをコピー
					$ret = move_uploaded_file($_FILES['Filedata']['tmp_name'], $filePath);
					if ($ret){
						$isErr = false;		// エラーが発生したかどうか

						// 画像情報を取得
						$imageSize = @getimagesize($filePath);
						if ($imageSize){
							$imageWidth = $imageSize[0];
							$imageHeight = $imageSize[1];
							$imageType = $imageSize[2];
							$imageMimeType = $imageSize['mime'];	// ファイルタイプを取得

							// 受付可能なファイルタイプかどうか
							if (!in_array($imageMimeType, $this->permitMimeType)){
								$isErr = true;	
								$this->writeUserError(__METHOD__, 'アップロード画像のタイプが不正です。アカウント: ' . $account, 2200, 'オリジナルファイル名=' . $originalFilename);
							}
						} else {
							$isErr = true;
							$this->writeUserError(__METHOD__, 'アップロード画像が不正です。アカウント: ' . $account, 2200, 'オリジナルファイル名=' . $originalFilename);
						}

						if (!$isErr){
							// サムネール画像作成
							$thumbnailPath = $this->getThumbnailPath($filename);
							$ret = $this->createThumbImage($filePath, $imageType, $thumbnailPath, self::DEFAULT_IMAGE_TYPE, self::DEFAULT_THUMBNAIL_SIZE);
							if (!$ret){
								$isErr = true;		// エラー発生
								$this->writeError(__METHOD__, '画像ファイル作成に失敗しました。', 1100,
													'オリジナルファイル名=' . $originalFilename . ', 画像ファイル=' . $thumbnailPath);// 運用ログに記録
							}

							// ウォータマーク画像作成
							if (!$isErr){
								$watermarkPath = $this->gEnv->getCurrentWidgetRootPath() . '/images/default_mark.png';
								$watermarkSize = @getimagesize($watermarkPath);
								if ($watermarkSize) $watermarkType = $watermarkSize[2];	// ファイルタイプを取得
								$ret = $this->createPublicImage($filePath, $imageType, $watermarkPath, $watermarkType,
													$this->getPublicImagePath($filename), self::DEFAULT_IMAGE_TYPE, self::DEFAULT_PUBLIC_IMAGE_SIZE);
								if (!$ret){
									$isErr = true;		// エラー発生
									$this->writeError(__METHOD__, '画像ファイル作成に失敗しました。', 1100,
														'オリジナルファイル名=' . $originalFilename . ', 画像ファイル=' . $watermarkPath);// 運用ログに記録
								}
							}
							
							if (!$isErr){
								//$relativePath = strtr(str_replace($this->gEnv->getIncludePath() . self::PHOTO_DIR, '', $path), '\\', '/');		// 画像格納ディレクトリ
								$relativePath = strtr(substr($path, strlen($this->gEnv->getIncludePath() . self::PHOTO_DIR)), '\\', '/');		// 画像格納ディレクトリ
								$imageSize = $imageWidth . 'x' . $imageHeight;		// 画像の縦横サイズ
								$filesize = filesize($filePath);;			// ファイルサイズ(バイト数)
								$name = removeExtension($originalFilename);		// 画像名
								$name = strtr($name, '_', ' ');
								$camera = '';	// カメラ
								$location = '';	// 場所
								$date = $this->gEnv->getInitValueOfDate();		// 撮影日
								$note = '';			// その他情報
								$license = '';		// ライセンス
								$keyword = '';		// 検索情報
								$visible = true;	// 表示
								$ret = $this->_mainDb->updatePhotoInfo($this->_isLimitedUser, 0/*新規追加*/, $this->langId, $filename, $relativePath, $code, $imageMimeType,
												$imageSize, $originalFilename, $filesize, $name, $camera, $location, $date, $note, $license, $this->userId, $keyword, $visible, $newSerial);
								if (!$ret){
									$isErr = true;
									$this->writeError(__METHOD__, 'アップロード画像のデータ登録に失敗しました。', 1100, 'オリジナルファイル名=' . $originalFilename);
								}
							}
						}
						
						// エラー処理
						if ($isErr){
							header("HTTP/1.1 595 File Upload Error");		// エラーコードはブラウザ画面に表示される
							
							// アップロードファイル削除
							unlink($filePath);
						}
					} else {
						header("HTTP/1.1 596 File Upload Error");		// エラーコードはブラウザ画面に表示される
					}
				} else {
					header("HTTP/1.1 597 File Upload Error");		// エラーコードはブラウザ画面に表示される
					$this->writeError(__METHOD__, '画像コードが重複しています。', 1100, '画像コード=' . $code);
				}
			} else {			// アップロードファイルがないとき
				header("HTTP/1.1 598 File Upload Error");			// エラーコードはブラウザ画面に表示される
				if (isset($_FILES["Filedata"])) echo $_FILES["Filedata"]["error"];
			}
			
			// システム強制終了
			$this->gPage->exitSystem();
		} else if ($act == 'delete'){			// ファイル削除のとき
			$listedItem = explode(',', $request->trimValueOf('seriallist'));
			$delItems = array();	// シリアル番号
			$delPhotos = array();	// 写真ID
			$delFiles = array();	// ファイル名
			for ($i = 0; $i < count($listedItem); $i++){
				// 項目がチェックされているかを取得
				$itemName = 'item' . $i . '_selected';
				$itemValue = ($request->trimValueOf($itemName) == 'on') ? 1 : 0;
				
				if ($itemValue){		// チェック項目
					// 削除可能かチェック
					$filename = $request->trimValueOf('item' . $i . '_name');
					$filePath = $path . DIRECTORY_SEPARATOR . $filename;
					if (is_writable($filePath) && strStartsWith($filePath, $this->photoBasePath . DIRECTORY_SEPARATOR)){
						$serial = $listedItem[$i];
						if ($serial > 0){
							// 写真情報のアクセス権をチェック
							$ret = $this->_mainDb->getPhotoInfoBySerial($serial, $row);
							if ($ret){
								if (!$this->_isLimitedUser || ($this->_isLimitedUser && $this->userId == $row['ht_owner_id'])){
									$delItems[] = $serial;
									$delPhotos[] = $filename;
								}
							}
						}
						$delFiles[] = $filePath;		// 削除するファイルのパス
					} else {
						//$this->setMsg(self::MSG_USER_ERR, '削除できないファイルが含まれています。ファイル名=' . $this->convertToDispString($filename));
						$this->setMsg(self::MSG_USER_ERR, sprintf($this->_('Include files not allowed to delete. (filename: %s)'), $this->convertToDispString($filename)));		// 削除できないファイルが含まれています。(ファイル名： %s)
						break;
					}
				}
			}
			if ($this->getMsgCount() == 0 && count($delFiles) > 0){
				$ret = $this->_mainDb->delPhotoInfo($delItems);
				if ($ret){
					// ファイル、ディレクトリ削除
					for ($i = 0; $i < count($delFiles); $i++){
						$file = $delFiles[$i];
						if (is_dir($file)){
							// ファイル、ディレクトリがある場合は削除しない
							$files = getFileList($file);
							if (count($files) == 0){
								if (!rmDirectory($file)) $ret = false;
							} else {
								$ret = false;
							}
						} else {
							if (!@unlink($file)) $ret = false;
						}
					}
					// 公開画像、サムネールを削除
					for ($i = 0; $i < count($delPhotos); $i++){
						$thumbnailPath = $this->getThumbnailPath($delPhotos[$i]);
						if (!@unlink($thumbnailPath)) $ret = false;
						$publicImagePath = $this->getPublicImagePath($delPhotos[$i]);
						if (!@unlink($publicImagePath)) $ret = false;
					}
				}
				if ($ret){		// ファイル削除成功のとき
					//$this->setGuidanceMsg('ファイルを削除しました');
					$this->setGuidanceMsg($this->_('Files deleted.'));			// ファイルを削除しました
				} else {
					//$this->setAppErrorMsg('ファイル削除に失敗しました');
					$this->setAppErrorMsg($this->_('Failed in deleting files.'));			// ファイル削除に失敗しました
				}
			}
		} else if ($act == 'createdir'){			// ディレクトリ作成のとき
			$dirName = $request->trimValueOf('directory_name');
			$this->checkSingleByte($dirName, $this->_('Directory Name'));		// ディレクトリ名
			
			if ($this->getMsgCount() == 0){
				$dirPath = $path . DIRECTORY_SEPARATOR . $dirName;
				$ret = @mkdir($dirPath, M3_SYSTEM_DIR_PERMISSION);
				if ($ret){		// ファイル削除成功のとき
					$this->setGuidanceMsg($this->_('Directory created.'));			// ディレクトリを作成しました
					$dirName = '';
				} else {
					$this->setAppErrorMsg($this->_('Failed in creating directory.'));			// ディレクトリ作成に失敗しました
				}
			}
		}
		
		// カレントディレクトリのパスを作成
		$photoParentPath = dirname($this->photoBasePath);
		$pathLink = '';
		//$relativePath = str_replace($photoParentPath, '', $path);
		$relativePath = substr($path, strlen($photoParentPath));
		$relativePath = trim($relativePath, DIRECTORY_SEPARATOR);
		if (!empty($relativePath)){
			//$absPath = $this->gEnv->getSystemRootPath();
			$absPath = $photoParentPath;
			$pathArray = explode(DIRECTORY_SEPARATOR, $relativePath);
			for ($i = 0; $i < count($pathArray); $i++){
				if ($i == count($pathArray) -1){
					$pathLink .= '&nbsp;' . DIRECTORY_SEPARATOR . '&nbsp;' . $this->convertToDispString($pathArray[$i]);
				} else {
					$absPath .= DIRECTORY_SEPARATOR . $pathArray[$i];
					$relativeFilePath = substr($absPath, strlen($this->photoBasePath));
					$pathLink .= '&nbsp;' . DIRECTORY_SEPARATOR . '&nbsp;';
					//$pathLink .= '<a href="#" onclick="selDir(\'' . $this->adaptWindowsPath($this->convertToDispString($absPath)) . '\');return false;">' . $this->convertToDispString($pathArray[$i]) . '</a>';
					//$pathLink .= '<a href="#" onclick="selDir(\'' . $this->adaptWindowsPath($this->convertToDispString($relativeFilePath)) . '\');return false;">' . $this->convertToDispString($pathArray[$i]) . '</a>';
					$pageUrl = $this->_baseUrl . '&task=imagebrowse&path=' . $relativeFilePath;
					$pathLink .= '<a href="' . $this->convertUrlToHtmlEntity($this->getUrl($pageUrl)) . '">' . $this->convertToDispString($pathArray[$i]) . '</a>';
				}
			}
		}
	
/*		$this->canDeleteFile = false;
		if ($relativePath == M3_DIR_NAME_RESOURCE || strStartsWith($relativePath, M3_DIR_NAME_RESOURCE . DIRECTORY_SEPARATOR)){
			$this->canDeleteFile = true;
		} else {
			// 削除ボタンを使用不可にする
			$this->tmpl->addVar("_widget", "del_disabled", 'disabled ');
		}*/
		
		// 総数を取得
		$fileList = $this->getFileList($path);
		$totalCount = count($fileList);

		// 表示するページ番号の修正
		$pageCount = (int)(($totalCount -1) / $viewCount) + 1;		// 総ページ数
		if ($pageNo < 1) $pageNo = 1;
		if ($pageNo > $pageCount) $pageNo = $pageCount;
		$this->firstNo = ($pageNo -1) * $viewCount + 1;		// 先頭番号
		$startNo = ($pageNo -1) * $viewCount +1;		// 先頭の行番号
		$endNo = $pageNo * $viewCount > $totalCount ? $totalCount : $pageNo * $viewCount;// 最後の行番号
		
		// ページング用リンク作成
		$pageLink = '';
		if ($pageCount > 1){	// ページが2ページ以上のときリンクを作成
			for ($i = 1; $i <= $pageCount; $i++){
				if ($i == $pageNo){
					$link = '&nbsp;' . $i;
				} else {
					//$link = '&nbsp;<a href="#" onclick="selpage(\'' . $i . '\');return false;">' . $i . '</a>';
					$relativePath = substr($path, strlen($this->photoBasePath));
					$pageUrl = $this->_baseUrl . '&task=imagebrowse&path=' . $relativePath;
					if ($i > 1) $pageUrl .= '&page=' . $i;
					$link = '&nbsp;<a href="' . $this->convertUrlToHtmlEntity($this->getUrl($pageUrl)) . '">' . $i . '</a>';
				}
				$pageLink .= $link;
			}
		}
		$this->tmpl->addVar("_widget", "page_link", $pageLink);
		$this->tmpl->addVar("_widget", "total_count", sprintf($this->_('%d Total'), $totalCount));// 全 x件
		$this->tmpl->addVar("_widget", "page", $pageNo);	// ページ番号
		$this->tmpl->addVar("search_range", "start_no", $startNo);
		$this->tmpl->addVar("search_range", "end_no", $endNo);
		if ($totalCount > 0) $this->tmpl->setAttribute('search_range', 'visibility', 'visible');// 検出範囲を表示
		
		// ファイル一覧を作成
		$this->createFileList($path, $fileList, $startNo, $endNo);
		
		$this->tmpl->addVar('_widget', 'path', substr($path, strlen($this->photoBasePath)));// 現在のディレクトリ
		$this->tmpl->addVar('_widget', 'path_link', $pathLink);// 現在のディレクトリ
		$this->tmpl->addVar("_widget", "serial_list", implode($this->serialArray, ','));// 表示項目のシリアル番号を設定
		$this->tmpl->addVar('_widget', 'directory_name', $this->convertToDispString($dirName));// ディレクトリ作成用
		
		// アップロード実行用URL
		$uploadUrl = $this->gEnv->getDefaultAdminUrl() . '?' . M3_REQUEST_PARAM_OPERATION_COMMAND . '=' . M3_REQUEST_CMD_CONFIG_WIDGET;	// ウィジェット設定画面
		$uploadUrl .= '&' . M3_REQUEST_PARAM_WIDGET_ID . '=' . $this->gEnv->getCurrentWidgetId();	// ウィジェットID
		$uploadUrl .= '&' . M3_REQUEST_PARAM_OPERATION_TASK . '=' . 'imagebrowse';
		$uploadUrl .= '&' . M3_REQUEST_PARAM_OPERATION_ACT . '=' . 'uploadfile';
		$uploadUrl .= '&' . M3_REQUEST_PARAM_ADMIN_KEY . '=' . $this->gEnv->getAdminKey();	// 管理者キー
		//$uploadUrl .= '&path=' . $this->adaptWindowsPath($path);
		$uploadUrl .= '&path=' . $this->adaptWindowsPath(substr($path, strlen($this->photoBasePath)));					// アップロードディレクトリ
		$this->tmpl->addVar("_widget", "upload_url", $this->getUrl($uploadUrl));
		
		// テキストをローカライズ
		$localeText = array();
		$localeText['msg_file_upload'] = $this->_('Files uploaded. Refresh file list?');		// アップロード終了しました。ファイル一覧を更新しますか?
		$localeText['msg_select_file'] = $this->_('Select files to delete.');		// 削除する項目を選択してください
		$localeText['msg_delete_file'] = $this->_('Delete selected files?');		// 選択項目を削除しますか?
		$localeText['msg_create_directory'] = $this->_('Create directory?');		// ディレクトリを作成しますか?
		$localeText['label_path'] = $this->_('Path:');		// パス
		$localeText['label_delete'] = $this->_('Delete');		// 削除
		$localeText['label_check'] = $this->_('Select');		// 選択
		$localeText['label_filename'] = $this->_('Filename');		// ファイル名
		$localeText['label_code'] = $this->_('Code');		// コード
		$localeText['label_size'] = $this->_('Size');		// サイズ
		$localeText['label_date'] = $this->_('Update Date');		// 更新日時
		$localeText['label_upload_status'] = $this->_('Upload Status');		// アップロード状況
		$localeText['label_filesize'] = $this->_('Max Filesize');		// 1ファイル最大サイズ
		$localeText['label_select_file'] = $this->_('Select Files');		// ファイルを選択
		$localeText['label_create_directory'] = $this->_('Create Directory');		// ディレクトリ作成
		$localeText['label_directory_name'] = $this->_('Directory Name');// ディレクトリ名
		$localeText['label_create'] = $this->_('Create');// 作成
		$localeText['label_cancel'] = $this->_('Cancel');// キャンセル
		$localeText['label_range'] = $this->_('Range:');		// 範囲：
		$this->setLocaleText($localeText);
	}
	/**
	 * ファイル一覧を作成
	 *
	 * @param string $path		パス
	 * @param array $fileList	ファイル一覧
	 * @param int $startNo		開始番号
	 * @param int $endNo		終了番号
	 * @return なし
	 */
	function createFileList($path, $fileList, $startNo, $endNo)
	{
		// 親ディレクトリを追加
		if ($path != $this->photoBasePath){
			$file = '..';
			$relativeFilePath = substr(dirname($path), strlen($this->photoBasePath));
			//$fileLink = '<a href="#" onclick="selDir(\'' . $this->adaptWindowsPath($this->convertToDispString(dirname($path))) . '\');return false;">' . $this->convertToDispString($file) . '</a>';
			//$fileLink = '<a href="#" onclick="selDir(\'' . $this->adaptWindowsPath($this->convertToDispString($relativeFilePath)) . '\');return false;">' . $this->convertToDispString($file) . '</a>';
			$pageUrl = $this->_baseUrl . '&task=imagebrowse&path=' . $relativeFilePath;
			$fileLink = '<a href="' . $this->convertUrlToHtmlEntity($this->getUrl($pageUrl)) . '">' . $this->convertToDispString($file) . '</a>';
					
			// アイコン作成
			$iconTitle = $file;
			$iconUrl = $this->gEnv->getRootUrl() . self::PARENT_ICON_FILE;
			//$iconTag = '<a href="#" onclick="selDir(\'' . $this->adaptWindowsPath($this->convertToDispString(dirname($path))) . '\');return false;">';
			//$iconTag = '<a href="#" onclick="selDir(\'' . $this->adaptWindowsPath($this->convertToDispString($relativeFilePath)) . '\');return false;">';
			$iconTag = '<a href="' . $this->convertUrlToHtmlEntity($this->getUrl($pageUrl)) . '">';
			$iconTag .= '<img src="' . $this->getUrl($iconUrl) . '" width="' . self::ICON_SIZE . '" height="' . self::ICON_SIZE . '" border="0" alt="' . $iconTitle . '" title="' . $iconTitle . '" />';
			$iconTag .= '</a>';
			
			$checkDisabled = 'disabled ';
			$row = array(
				'icon'		=> $iconTag,		// アイコン
				'name'		=> $this->convertToDispString($file),			// ファイル名
				'filename'    => $fileLink,			// ファイル名
				'size'     => $size,			// ファイルサイズ
				'check_disabled' => $checkDisabled,		// チェックボックス使用制御
				'selected' => $selected														// 選択中かどうか
			);
			$this->tmpl->addVars('file_list', $row);
			$this->tmpl->parseTemplate('file_list', 'a');
		}
			
		$index = 0;			// インデックス番号
		//for ($i = 0; $i < count($fileList); $i++){
		for ($i = $startNo -1; $i < $endNo; $i++){
			$filePath = $fileList[$i];
			$relativeFilePath = substr($filePath, strlen($this->photoBasePath));

			//$pathParts = pathinfo($filePath);
			////$file = basename($filePath);
			//$file = $pathParts['basename'];
			$file = end(explode('/', $filePath));			// pathinfo,basenameは日本語処理できないので日本語対応
			$size = '';
			$fileLink = '';
			$filenameOption = '';			// ファイル名オプション
			$code = '';			// 画像コード
			$checkDisabled = '';		// チェックボックス使用制御
			if (is_dir($filePath)){			// ディレクトリのとき
				// アイコン作成
				$iconUrl = $this->gEnv->getRootUrl() . self::FOLDER_ICON_FILE;
				$iconTitle = $this->convertToDispString($file);
				//$iconTag = '<a href="#" onclick="selDir(\'' . $this->adaptWindowsPath($this->convertToDispString($filePath)) . '\');return false;">';
				//$iconTag = '<a href="#" onclick="selDir(\'' . $this->adaptWindowsPath($this->convertToDispString($relativeFilePath)) . '\');return false;">';
				$pageUrl = $this->_baseUrl . '&task=imagebrowse&path=' . $relativeFilePath;
				$iconTag = '<a href="' . $this->convertUrlToHtmlEntity($this->getUrl($pageUrl)) . '">';
				$iconTag .= '<img src="' . $this->getUrl($iconUrl) . '" width="' . self::ICON_SIZE . '" height="' . self::ICON_SIZE . '" border="0" alt="' . $iconTitle . '" title="' . $iconTitle . '" />';
				$iconTag .= '</a>';
			
				//$fileLink = '<a href="#" onclick="selDir(\'' . $this->adaptWindowsPath($this->convertToDispString($filePath)) . '\');return false;">' . $this->convertToDispString($file) . '</a>';
				//$fileLink = '<a href="#" onclick="selDir(\'' . $this->adaptWindowsPath($this->convertToDispString($relativeFilePath)) . '\');return false;">' . $this->convertToDispString($file) . '</a>';
				$fileLink = '<a href="' . $this->convertUrlToHtmlEntity($this->getUrl($pageUrl)) . '">' . $this->convertToDispString($file) . '</a>';
				
				// ファイルまたはディレクトリがないときは削除可能
				$files = getFileList($filePath);
				if (count($files) > 0){
					$checkDisabled = 'disabled ';		// チェックボックス使用制御
				}
				$serial = -1;
			} else {		// 画像ファイルのとき
				$thumbnailPath = $this->getThumbnailPath($file);
				
				// アイコン作成
				$iconTitle = $this->convertToDispString($file);
				if (file_exists($thumbnailPath)){	// サムネールが存在する場合
					$iconUrl = $this->getThumbnailUrl($file);
					$iconTag = '<img src="' . $this->getUrl($iconUrl) . '" width="' . self::DEFAULT_THUMBNAIL_SIZE . '" height="' . self::DEFAULT_THUMBNAIL_SIZE . '" border="0" alt="' . $iconTitle . '" title="' . $iconTitle . '" />';
				} else {
					$iconUrl = $this->gEnv->getRootUrl() . self::FILE_ICON_FILE;
					$iconTag = '<img src="' . $this->getUrl($iconUrl) . '" width="' . self::ICON_SIZE . '" height="' . self::ICON_SIZE . '" border="0" alt="' . $iconTitle . '" title="' . $iconTitle . '" />';
				}
				
				// 画像情報取得
				$ret = $this->_mainDb->getPhotoInfo($file, $row);
				if ($ret){
					$serial = $row['ht_serial'];
					$code = $row['ht_code'];			// 画像コード
					$filenameOption = '<br />オリジナルファイル名： ' . $this->convertToDispString($row['ht_original_filename']);		// ファイル名オプション
					$filenameOption .= '<br />タイトル名： ' . $this->convertToDispString($row['ht_name']);		// タイトル名
					
					// 使用限定ユーザの場合は、所有者でなければ削除できない
					if ($this->_isLimitedUser && $this->userId != $row['ht_owner_id']){
						$checkDisabled = 'disabled ';		// チェックボックス使用制御
						$serial = -1;
					}
				} else {
					$checkDisabled = 'disabled ';		// チェックボックス使用制御
					$serial = -1;
				}
				// ファイル削除用チェックボックス
				//if (!$this->canDeleteFile || !is_writable($filePath)) $checkDisabled = 'disabled ';		// チェックボックス使用制御
				
				$fileLink = $this->convertToDispString($file);
				$size = filesize($filePath);
				

				

			}
	
			// ファイル更新日時
			$updateDate = date('Y/m/d H:i:s', filemtime($filePath));
			
			$row = array(
				'index'		=> $index,			// インデックス番号(チェックボックス識別)
				'icon'		=> $iconTag,		// アイコン
				'name'		=> $this->convertToDispString($file),			// ファイル名
				'filename'    	=> $fileLink,			// ファイル名
				'filename_option'    	=> $filenameOption,			// ファイル名オプション
				'code'		=> $this->convertToDispString($code),			// 画像コード
				'size'     		=> $size,			// ファイルサイズ
				'date'    => $updateDate,			// 更新日時
				'check_disabled' => $checkDisabled,		// チェックボックス使用制御
			);
			$this->tmpl->addVars('file_list', $row);
			$this->tmpl->parseTemplate('file_list', 'a');
			
			// インデックス番号を保存
			$this->serialArray[] = $serial;
			$index++;
		}
	}
	/**
	 * ファイル一覧を作成
	 *
	 * @param string $path	ディレクトリパス
	 * @return array		ファイルパスのリスト
	 */
	function getFileList($path)
	{
		$fileList = array();
		
		// 引数エラーチェック
		if (!is_dir($path)) return $fileList;
		
		$dir = dir($path);
		while (($file = $dir->read()) !== false){
			$filePath = $path . DIRECTORY_SEPARATOR . $file;
			// カレントディレクトリかどうかチェック
			if ($file != '.' && $file != '..') $fileList[] = $filePath;
		}
		$dir->close();
		
		// 一覧を日付でソート
		$this->sortOrderByDateAsc = false;	// 日付で降順にソート
		usort($fileList, array($this, 'sortOrderByDate'));
		
		return $fileList;
	}
	/**
	 * ファイルパスを日付で昇順にソートする。ディレクトリは先頭。
	 *
	 * @param string  	$path1			比較するパス1
	 * @param string	$path2			比較するパス2
	 * @return int						同じとき0、$path1が$path2より大きいとき1,$path1が$path2より小さいとき-1を返す
	 */
	function sortOrderByDate($path1, $path2)
	{
		// ディレクトリは常に先頭に表示
		if (is_dir($path1)){			// ディレクトリのとき
			if (!is_dir($path2)) return -1; // ファイルのとき
		} else {
			if (is_dir($path2)) return 1;	// ディレクトリのとき
		}
		$fileTime1 = filemtime($path1);
		$fileTime2 = filemtime($path2);
		
		if ($fileTime1 == $fileTime2) return 0;
		if ($this->sortOrderByDateAsc){	// 日付で昇順にソート
			return ($fileTime1 < $fileTime2) ? -1 : 1;
		} else {
			return ($fileTime1 > $fileTime2) ? -1 : 1;
		}
	}
	/**
	 * CSSファイルをHTMLヘッダ部に設定
	 *
	 * CSSファイルをHTMLのheadタグ内に追加出力する。
	 * _assign()よりも後に実行される。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。
	 * @return string 						CSS文字列。出力しない場合は空文字列を設定。
	 */
	function _addCssFileToHead($request, &$param)
	{
		return $this->getUrl($this->gEnv->getScriptsUrl() . self::CSS_FILE);
	}
	/**
	 * windowパスを使用する場合は、Javascriptに認識させる
	 *
	 * @param string         $src			パス文字列
	 * @return string 						作成文字列
	 */
	function adaptWindowsPath($src)
	{
		// 「\」を有効にする
		if (DIRECTORY_SEPARATOR == '\\'){		// windowsパスのとき
			return addslashes($src);
		} else {
			return $src;
		}
	}
	/**
	 * サムネール画像のパスを取得
	 *
	 * @param string $photoId		画像ID
	 * @return string				画像パス
	 */
	function getThumbnailPath($photoId)
	{
		return $this->gEnv->getResourcePath() . self::THUMBNAIL_DIR . '/' . $photoId . '_' . self::DEFAULT_THUMBNAIL_SIZE . '.' . self::DEFAULT_IMAGE_EXT;
	}
	/**
	 * サムネール画像のURLを取得
	 *
	 * @param string $photoId		画像ID
	 * @return string				画像パス
	 */
	function getThumbnailUrl($photoId)
	{
		return $this->gEnv->getResourceUrl() . self::THUMBNAIL_DIR . '/' . $photoId . '_' . self::DEFAULT_THUMBNAIL_SIZE . '.' . self::DEFAULT_IMAGE_EXT;
	}
	/**
	 * 公開画像のパスを取得
	 *
	 * @param string $photoId		画像ID
	 * @return string				画像パス
	 */
	function getPublicImagePath($photoId)
	{
		return $this->gEnv->getResourcePath() . self::THUMBNAIL_DIR . '/' . $photoId . '.' . self::DEFAULT_IMAGE_EXT;
	}
	/**
	 * サムネールを作成
	 *
	 * @param string $path		ファイルパス
	 * @param int $type			画像タイプ
	 * @param string $destPath	出力ファイル保存のパス
	 * @param int $destType		出力画像タイプ
	 * @param int $size			サムネールの縦横サイズ
	 * @return bool				true=成功、false=失敗
	 */
	function createThumbImage($path, $type, $destPath, $destType, $size)
	{
		// 画像作成
		switch ($type){
			case IMAGETYPE_GIF:
				$image = @imagecreatefromgif($path);
				break;
			case IMAGETYPE_JPEG:
				$image = @imagecreatefromjpeg($path);
				break;
			case IMAGETYPE_PNG:
				$image = @imagecreatefrompng($path);
				break;
			default:
				return false;
		}

		// 画像サイズ取得
		$width = imagesx($image);
		$height = imagesy($image);

		if ($width > $height){
			$n_height = $height * ($size / $width);
			$n_width = $size;
		} else {
			$n_width = $width * ($size / $height);
			$n_height = $size;
		}
		
		$x = 0;
		$y = 0;
		if ($n_width < $size) $x = round(($size - $n_width) / 2);
		if ($n_height < $size) $y = round(($size - $n_height) / 2);
		
		// TrueColorイメージを作成
		$thumb = imagecreatetruecolor($size, $size);
		$bgcolor = imagecolorallocate($thumb, 255, 255, 255);		// 背景色設定
		imagefill($thumb, 0, 0, $bgcolor);
		
		// 画像リサイズ
		if (function_exists("imagecopyresampled")){
			if (!imagecopyresampled($thumb, $image, $x, $y, 0, 0, $n_width, $n_height, $width, $height)){
				if (!imagecopyresized($thumb, $image, $x, $y, 0, 0, $n_width, $n_height, $width, $height)) return false;
			}
		} else {
			if (!imagecopyresized($thumb, $image, $x, $y, 0, 0, $n_width, $n_height, $width, $height)) return false;
		}

		// 画像出力
		switch ($destType){
			case IMAGETYPE_GIF:
				$ret = @imagegif($thumb, $destPath);
				break;
			case IMAGETYPE_JPEG:
				$ret = @imagejpeg($thumb, $destPath);
				break;
			case IMAGETYPE_PNG:
				$ret = @imagepng($thumb, $destPath);
				break;
		}
		// イメージを破棄
		$ret = imagedestroy($image);
		$ret = imagedestroy($thumb);
		return $ret;
	}
	/**
	 * ウォータマーク入り公開画像を作成
	 *
	 * @param string $path		元画像ファイルパス
	 * @param int $type			元画像の画像タイプ
	 * @param string $wPath		ウォータマーク画像ファイルパス
	 * @param int $wType		ウォータマーク画像の画像タイプ
	 * @param string $destPath	出力画像のパス
	 * @param int $destType		出力画像の画像タイプ
	 * @param int $size			出力画像の縦または横の最大サイズ
	 * @return bool				true=成功、false=失敗
	 */
	function createPublicImage($path, $type, $wPath, $wType, $destPath, $destType, $size)
	{
		// 画像作成
		switch ($type){
			case IMAGETYPE_GIF:
				$image = @imagecreatefromgif($path);
				break;
			case IMAGETYPE_JPEG:
				$image = @imagecreatefromjpeg($path);
				break;
			case IMAGETYPE_PNG:
				$image = @imagecreatefrompng($path);
				break;
			default:
				return false;
		}
		
		// 画像サイズ取得
		$width = imagesx($image);
		$height = imagesy($image);
		
		// 出力サイズ取得
		$destWidth = $size;
		$destHeight = $size;
		$imageRatio = $width / $height;
		if ($destWidth / $destHeight > $imageRatio){
			$destWidth = $destHeight * $imageRatio;
		} else {
			$destHeight = $destWidth / $imageRatio;
		}
		// 出力画像作成
		$destImage = imagecreatetruecolor($destWidth, $destHeight);
		imagecopyresampled($destImage, $image, 0, 0, 0, 0, $destWidth, $destHeight, $width, $height);
		
		// 画像作成
		switch ($wType){
			case IMAGETYPE_GIF:
				$wImage = @imagecreatefromgif($wPath);
				break;
			case IMAGETYPE_JPEG:
				$wImage = @imagecreatefromjpeg($wPath);
				break;
			case IMAGETYPE_PNG:
				$wImage = @imagecreatefrompng($wPath);
				break;
			default:
				return false;
		}
		
		// 画像サイズ取得
		$wWidth = imagesx($wImage);
		$wHeight = imagesy($wImage);
		
//		$startwidth = ($width - $wWidth) / 2; 
//		$startheight = ($height - $wHeight) / 2;
		$startwidth = ($destWidth - $wWidth) / 2; 
		$startheight = ($destHeight - $wHeight) / 2;
	//	imagecopymerge($image, $wImage, $startwidth, $startheight, 0, 0, $wWidth, $wHeight, 10);
		imagecopymerge($destImage, $wImage, $startwidth, $startheight, 0, 0, $wWidth, $wHeight, 10);
		
		// 画像出力
		switch ($destType){
			case IMAGETYPE_GIF:
				//$ret = @imagegif($image, $destPath);
				$ret = @imagegif($destImage, $destPath);
				break;
			case IMAGETYPE_JPEG:
				//$ret = @imagejpeg($image, $destPath);
				$ret = @imagejpeg($destImage, $destPath);
				break;
			case IMAGETYPE_PNG:
				//$ret = @imagepng($image, $destPath);
				$ret = @imagepng($destImage, $destPath);
				break;
		}
		// イメージを破棄
		$ret = imagedestroy($destImage);
		$ret = imagedestroy($image);
		$ret = imagedestroy($wImage);
		return $ret;
	}
}
?>
