<?php
/**
 * index.php用コンテナクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2010 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: s_contentWidgetContainer.php 4555 2012-01-02 04:54:39Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getContainerPath() . '/baseWidgetContainer.php');
require_once($gEnvManager->getCurrentWidgetDbPath() . '/s_contentDb.php');
require_once($gEnvManager->getCommonPath() . '/valueCheck.php');

class s_contentWidgetContainer extends BaseWidgetContainer
{
	private $db;			// DB接続オブジェクト
	private $_contentCreated;	// コンテンツが取得できたかどうか
	private $currentDay;		// 現在日
	private $currentHour;		// 現在時間
	private $headTitle;		// METAタグタイトル
	private $headDesc;		// METAタグ要約
	private $headKeyword;	// METAタグキーワード
	private $showTitle;			// コンテンツタイトルの表示制御
	private $showEdit;			// 編集ボタンの表示
	private $isSystemManageUser;	// システム運用可能ユーザかどうか
	private $pageTitle;				// 画面タイトル、パンくずリスト用タイトル
	const CONTENT_TYPE = 'smartphone';			// コンテンツタイプ
	const VIEW_CONTENT_TYPE = 'sc';		// 参照数カウント用
	const DEFAULT_SEARCH_LIST_COUNT = 20;			// 最大リスト表示数
	const MESSAGE_NO_CONTENT		= 'コンテンツが見つかりません';
	const CONTENT_SIZE = 200;			// 検索結果コンテンツの文字列最大長
	const DEFAULT_MESSAGE_DENY = 'コンテンツを表示できません';		// アクセス不可の場合のメッセージ
	const ICON_SIZE = 16;		// アイコンのサイズ
	const EDIT_ICON_FILE = '/images/system/page_edit.png';		// 編集アイコン
	const NEW_ICON_FILE = '/images/system/page_add.png';		// 新規アイコン
	const DEFAULT_TITLE_SEARCH = 'コンテンツ検索';		// 検索時のデフォルトタイトル
	const DEFAULT_TITLE_SEARCH_RESULTS = 'コンテンツ検索結果';	// 検索実行後のデフォルトタイトル
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		// 親クラスを呼び出す
		parent::__construct();
		
		// HTMLメタタグを初期化
		$this->headTitle = '';
		$this->headDesc = '';
		$this->headKeyword = '';
		
		// DBオブジェクト作成
		$this->db = new s_contentDb();
	}
	/**
	 * テンプレートファイルを設定
	 *
	 * _assign()でデータを埋め込むテンプレートファイルのファイル名を返す。
	 * 読み込むディレクトリは、「自ウィジェットディレクトリ/include/template」に固定。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。そのまま_assign()に渡る
	 * @return string 						テンプレートファイル名。テンプレートライブラリを使用しない場合は空文字列「''」を返す。
	 */
	function _setTemplate($request, &$param)
	{
		$act = $request->trimValueOf('act');
		if ($act == 'search'){
			return 'search.tmpl.html';
		} else {
			return 'main.tmpl.html';
		}
	}
	/**
	 * テンプレートにデータ埋め込む
	 *
	 * _setTemplate()で指定したテンプレートファイルにデータを埋め込む。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。_setTemplate()と共有。
	 * @param								なし
	 */
	function _assign($request, &$param)
	{
		// 現在日時を取得
		$now = date("Y/m/d H:i:s");	// 現在日時
		$this->currentDay = date("Y/m/d");		// 日
		$this->currentHour = (int)date("H");		// 時間
		$this->currentPageUrl = $this->gEnv->createCurrentPageUrl();// 現在のページURL
		$this->isSystemManageUser = $this->gEnv->isSystemManageUser();	// システム運用可能ユーザかどうか
		$cmd = $request->trimValueOf(M3_REQUEST_PARAM_OPERATION_COMMAND);
				
		// 管理者でプレビューモードのときは表示制限しない
		$preview = false;
		if ($this->isSystemManageUser && $cmd == M3_REQUEST_CMD_PREVIEW){		// システム運用者以上
			$preview = true;
		}
		
		// ログインユーザでないときは、ユーザ制限のない項目だけ表示
		$all = false;
		if ($this->gEnv->isCurrentUserLogined()) $all = true;
		
		// ウィジェットパラメータ取得
		$this->showTitle = 1;			// コンテンツタイトルを表示するかどうか
		$showMessageDeny = 1;					// アクセス不可の場合にメッセージを表示するかどうか
		$messageDeny = self::DEFAULT_MESSAGE_DENY;							// アクセス不可の場合のメッセージ
		$this->showEdit = 1;			// 編集ボタンを表示するかどうか
		$jqueryMobileFormat = 1;			// jQueryMobile用のフォーマットで出力するかどうか
		
		$paramObj = $this->getWidgetParamObj();
		if (!empty($paramObj)){
			$this->showTitle = $paramObj->showTitle;			// コンテンツタイトルを表示するかどうか
			$showMessageDeny = $paramObj->showMessageDeny;					// アクセス不可の場合にメッセージを表示するかどうか
			$messageDeny = $paramObj->messageDeny;							// アクセス不可の場合のメッセージ
			$this->showEdit = $paramObj->showEdit;			// 編集ボタンを表示するかどうか
			$jqueryMobileFormat = $paramObj->jqueryMobileFormat;			// jQueryMobile用のフォーマットで出力するかどうか
		}

		$this->pageTitle = '';		// 画面タイトル、パンくずリスト用タイトル
		$act = $request->trimValueOf('act');
		$contentid = $request->trimValueOf(M3_REQUEST_PARAM_CONTENT_ID);
		if (empty($contentid)) $contentid = $request->trimValueOf(M3_REQUEST_PARAM_CONTENT_ID_SHORT);		// 略式コンテンツID
		$keyword = $request->trimValueOf('keyword');// 検索キーワード
		if ($act == 'search'){			// 検索
			$itemCount = self::DEFAULT_SEARCH_LIST_COUNT;		// 取得数
			
			// キーワード検索のとき
			if (empty($keyword)){
				$keyword = '検索キーワードが入力されていません';
				$this->headTitle = self::DEFAULT_TITLE_SEARCH;
			} else {
				$this->db->searchContentByKeyword(self::CONTENT_TYPE, $itemCount, 1, $keyword, $this->gEnv->getCurrentLanguage(), $all, $now, array($this, 'searchItemsLoop'));
				$this->headTitle = self::DEFAULT_TITLE_SEARCH_RESULTS . '[' . $keyword . ']';
				if (!$this->isExistsViewData) $keyword .= '&nbsp;&nbsp;' . self::MESSAGE_NO_CONTENT;
			}
			$this->pageTitle = $this->headTitle;// 画面タイトル、パンくずリスト用タイトル
			$this->tmpl->addVar("_widget", "keyword", $keyword);
			
			// コンテンツタイトルの設定
			if (!empty($this->showTitle)){		// タイトルを表示するとき
				$this->tmpl->setAttribute('show_title', 'visibility', 'visible');// タイトル表示
				$headClassStr = $this->gDesign->getDefaultContentHeadClassString();			// コンテンツヘッダ用CSSクラス
				$this->tmpl->addVar("show_title", "class", $headClassStr);
				$this->tmpl->addVar("show_title", "title", $this->pageTitle);
			}
		} else if (empty($contentid)){	// コンテンツIDがないときはデフォルトデータを取得
			// 定義IDが0以外のときは、定義IDをコンテンツIDとする
			$contentIdArray = array();
			
			// 定義ID取得
			$configId = $this->gEnv->getCurrentWidgetConfigId();
			if (!empty($configId)) $contentIdArray[] = $configId;
		
			$this->db->getContentItems(self::CONTENT_TYPE, array($this, 'itemsLoop'), $contentIdArray, $this->gEnv->getCurrentLanguage(), $all, $now, $preview);
			if (!$this->_contentCreated){		// コンテンツが取得できなかったときはデフォルト言語で取得
				$this->db->getContentItems(self::CONTENT_TYPE, array($this, 'itemsLoop'), $contentIdArray, $this->gEnv->getDefaultLanguage(), $all, $now, $preview);
			}
		} else {		// コンテンツIDで指定
			// データエラーチェック
			$contentIdArray = explode(',', $contentid);
			if (ValueCheck::isNumeric($contentIdArray)){		// すべて数値であるかチェック
				$this->db->getContentItems(self::CONTENT_TYPE, array($this, 'itemsLoop'), $contentIdArray, $this->gEnv->getCurrentLanguage(), $all, $now, $preview);
				if (!$this->_contentCreated){		// コンテンツが取得できなかったときはデフォルト言語で取得
					$this->db->getContentItems(self::CONTENT_TYPE, array($this, 'itemsLoop'), $contentIdArray, $this->gEnv->getDefaultLanguage(), $all, $now, $preview);
				}
				
				// コンテンツアクセス不可のときはアクセス不可メッセージを出力
				if ($showMessageDeny && !$this->_contentCreated) $this->setAppErrorMsg($messageDeny);
			} else {
				$this->setAppErrorMsg('IDにエラー値があります');
			}
		}
		// HTMLサブタイトルを設定
		// ヘッダタグのタイトル名が設定されている場合は、タイトル名を優先する
		if (empty($this->headTitle)) $this->gPage->setHeadSubTitle($this->pageTitle);
		
		// jQuery Mobile対応
		if ($jqueryMobileFormat) $this->tmpl->addVar("_widget", "data_role", 'data-role="content"');
		
		// 運用可能ユーザの場合は編集用ボタンを表示
		if ($this->isSystemManageUser && $act != 'search'){		// 検索画面以外
			// 設定画面表示用のスクリプトを埋め込む
			$editUrl = $this->getConfigAdminUrl('openby=simple&task=content_detail');
			$this->tmpl->setAttribute('admin_script', 'visibility', 'visible');
			$this->tmpl->addVar("admin_script", "edit_url", $editUrl);
		}
	}
	/**
	 * ヘッダ部メタタグの設定
	 *
	 * HTMLのheadタグ内に出力する。
	 * _assign()よりも後に実行される。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ
	 * @return array 						設定データ。連想配列で「title」「description」「keywords」を設定。
	 */
	function _setHeadMeta($request, &$param)
	{
		$headData = array(	'title' => $this->headTitle,
							'description' => $this->headDesc,
							'keywords' => $this->headKeyword);
		return $headData;
	}
	/**
	 * ウィジェットのタイトルを設定
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。そのまま_assign()に渡る
	 * @return string 						ウィジェットのタイトル名
	 */
	function _setTitle($request, &$param)
	{
		return $this->pageTitle;
	}
	/**
	 * 取得したコンテンツ項目をテンプレートに設定する
	 *
	 * @param int		$index			行番号
	 * @param array		$fetchedRow		取得行
	 * @param object	$param			任意使用パラメータ
	 * @return bool						trueを返すとループ続行。falseを返すとその時点で終了。
	 */
	function itemsLoop($index, $fetchedRow)
	{
		// ページタイトルの設定
		if (empty($this->pageTitle)) $this->pageTitle = $fetchedRow['cn_name'];		// 画面タイトル、パンくずリスト用タイトル
		
		// コンテンツタイトルの出力設定
		if (empty($this->showTitle)) $this->tmpl->addVar('contentlist', 'type', 'hide_title');
		
		// ビューカウントを更新
		if (!$this->isSystemManageUser){		// システム運用者以上の場合はカウントしない
			$this->gInstance->getAnalyzeManager()->updateContentViewCount(self::VIEW_CONTENT_TYPE, $fetchedRow['cn_serial'], $this->currentDay, $this->currentHour);
		}

		// METAタグを設定
		if (!empty($this->headTitle) && !strEndsWith($this->headTitle, ',')) $this->headTitle .= ',';
		if (!empty($this->headDesc) && !strEndsWith($this->headDesc, ',')) $this->headDesc .= ',';
		if (!empty($this->headKeyword) && !strEndsWith($this->headKeyword, ',')) $this->headKeyword .= ',';
		$this->headTitle .= $fetchedRow['cn_meta_title'];
		$this->headDesc .= $fetchedRow['cn_meta_description'];
		$this->headKeyword .= $fetchedRow['cn_meta_keywords'];
		
		// HTMLを出力
		// 出力内容は特にエラーチェックしない
		//$contentText = $this->convertToDispString($fetchedRow['cn_html']);
		$rootUrl = $this->getUrl($this->gEnv->getRootUrl());
		$contentText = $fetchedRow['cn_html'];
		$contentText = str_replace(M3_TAG_START . M3_TAG_MACRO_ROOT_URL . M3_TAG_END, $rootUrl, $contentText);// アプリケーションルートを変換
		
		// 登録したキーワードを変換
		$this->gInstance->getTextConvManager()->convByKeyValue($contentText, $contentText, true/*改行コーをbrタグに変換*/);
		
		$headClassStr = $this->gDesign->getDefaultContentHeadClassString();			// コンテンツヘッダ用CSSクラス

		// コンテンツ編集権限がある場合はボタンを表示
		$buttonList = '';
		if (!empty($this->showEdit) && $this->isSystemManageUser){
			$iconUrl = $this->gEnv->getRootUrl() . self::EDIT_ICON_FILE;		// 編集アイコン
			$iconTitle = '編集';
			$editImg = '<img src="' . $this->getUrl($iconUrl) . '" width="' . self::ICON_SIZE . '" height="' . self::ICON_SIZE . '" border="0" alt="' . $iconTitle . '" title="' . $iconTitle . '" />';
			$buttonList = '<span style="line-height:0;"><a href="#" onclick="editContent(' . $fetchedRow['cn_id'] . ');">' . $editImg . '</a></span><br />';
			$buttonList = '<div style="text-align:right;position:absolute;top:10px;z-index:10;width:100%;">' . $buttonList . '</div>';
		}
		
		$row = array(
			'class' => $headClassStr,		// コンテンツヘッダ用CSSクラス
			'title' => $this->convertToDispString($fetchedRow['cn_name']),
			'content' => $contentText,	// コンテンツ
			'button_list' => $buttonList	// 記事編集ボタン
		);
		$this->tmpl->addVars('contentlist', $row);
		$this->tmpl->parseTemplate('contentlist', 'a');
		
		// コンテンツが取得できた
		$this->_contentCreated = true;
		return true;
	}
	/**
	 * 取得したコンテンツ項目をテンプレートに設定する
	 *
	 * @param int		$index			行番号
	 * @param array		$fetchedRow		取得行
	 * @param object	$param			任意使用パラメータ
	 * @return bool						trueを返すとループ続行。falseを返すとその時点で終了。
	 */
	function searchItemsLoop($index, $fetchedRow)
	{
		// タイトルを設定
		$title = $fetchedRow['cn_name'];

		// 記事へのリンクを生成
		$linkUrl = $this->getUrl($this->currentPageUrl . '&' . M3_REQUEST_PARAM_CONTENT_ID . '=' . $fetchedRow['cn_id']);
		$link = '<a href="' . $this->convertUrlToHtmlEntity($linkUrl) . '" >' . $title . '</a>';

		// テキストに変換
		//$contentText = strip_tags($fetchedRow['cn_html']);
		$contentText = $this->gInstance->getTextConvManager()->htmlToText($fetchedRow['cn_html']);
		
		// アプリケーションルートを変換
		$rootUrl = $this->getUrl($this->gEnv->getRootUrl());
		$contentText = str_replace(M3_TAG_START . M3_TAG_MACRO_ROOT_URL . M3_TAG_END, $rootUrl, $contentText);
		
		// 登録したキーワードを変換
		$this->gInstance->getTextConvManager()->convByKeyValue($contentText, $contentText);
		
		// 検索結果用にテキストを詰める。改行、タブ、スペース削除。
		$contentText = str_replace(array("\r", "\n", "\t", " "), '', $contentText);
		
		// 文字列長を修正
		if (function_exists('mb_strimwidth')){
			$contentText = mb_strimwidth($contentText, 0, self::CONTENT_SIZE, '…');
		} else {
			$contentText = substr($contentText, 0, self::CONTENT_SIZE) . '...';
		}

		$headClassStr = $this->gDesign->getDefaultContentHeadClassString();			// コンテンツヘッダ用CSSクラス
		
		$row = array(
			'class' => $headClassStr,		// コンテンツヘッダ用CSSクラス
			'title' => $link,			// リンク付きタイトル
			'content' => $this->convertToDispString($contentText)	// コンテンツ
		);
		$this->tmpl->addVars('contentlist', $row);
		$this->tmpl->parseTemplate('contentlist', 'a');
		$this->isExistsViewData = true;				// 表示データがあるかどうか
		return true;
	}
	/**
	 * 期間から公開可能かチェック
	 *
	 * @param timestamp	$startDt		公開開始日時
	 * @param timestamp	$endDt			公開終了日時
	 * @param timestamp	$now			基準日時
	 * @return bool						true=公開可能、false=公開不可
	 */
	function isActive($startDt, $endDt, $now)
	{
		$isActive = false;		// 公開状態

		if ($startDt == $this->gEnv->getInitValueOfTimestamp() && $endDt == $this->gEnv->getInitValueOfTimestamp()){
			$isActive = true;		// 公開状態
		} else if ($startDt == $this->gEnv->getInitValueOfTimestamp()){
			if (strtotime($now) < strtotime($endDt)) $isActive = true;		// 公開状態
		} else if ($endDt == $this->gEnv->getInitValueOfTimestamp()){
			if (strtotime($now) >= strtotime($startDt)) $isActive = true;		// 公開状態
		} else {
			if (strtotime($startDt) <= strtotime($now) && strtotime($now) < strtotime($endDt)) $isActive = true;		// 公開状態
		}
		return $isActive;
	}
}
?>
