<?php
/**
 * 画像処理マネージャー
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2012 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: imageManager.php 4660 2012-02-05 07:02:55Z fishbone $
 * @link       http://www.magic3.org
 */
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/core.php');

class ImageManager extends Core
{
	const CONTENT_DIR = '/etc/';
	const THUMBNAIL_DIR = '/thumb/';
	const DEFAULT_THUMBNAIL_EXT = 'png';
	const DEFAULT_THUMBNAIL_SIZE = 72;
	const DEFAULT_THUMBNAIL_TYPE = '72c';
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		// 親クラスを呼び出す
		parent::__construct();
	}
	/**
	 * デフォルトのサムネールを作成
	 *
	 * @param string $contentType	コンテンツタイプ
	 * @param string $contentId		コンテンツID
	 * @param string $path			元の画像ファイル
	 * @return bool				true=作成,false=作成失敗
	 */
	function createDefaultThumb($contentType, $contentId, $path)
	{
		global $gEnvManager;
		
		// サムネール画像のパス
		$destPath = $gEnvManager->getResourcePath() . self::CONTENT_DIR . $contentType . self::THUMBNAIL_DIR . $contentId . '_' . self::DEFAULT_THUMBNAIL_TYPE . '.' . self::DEFAULT_THUMBNAIL_EXT;

		// 画像格納用のディレクトリ作成
		$destDir = dirname($destPath);
		if (!file_exists($destDir)) mkdir($destDir, M3_SYSTEM_DIR_PERMISSION, true/*再帰的*/);
					
		$ret = $this->createThumb($path, $destPath, self::DEFAULT_THUMBNAIL_SIZE, IMAGETYPE_PNG);
		return $ret;
	}
	/**
	 * サムネール画像のURLを取得
	 *
	 * @param string $contentType	コンテンツタイプ
	 * @param string $contentId		コンテンツID
	 * @return string				画像URL
	 */
	function getDefaultThumbUrl($contentType, $contentId)
	{
		global $gEnvManager;
		
		return $gEnvManager->getResourceUrl() . self::CONTENT_DIR . $contentType . self::THUMBNAIL_DIR . $contentId . '_' . self::DEFAULT_THUMBNAIL_TYPE . '.' . self::DEFAULT_THUMBNAIL_EXT;
	}
	/**
	 * サムネール画像のファイルパスを取得
	 *
	 * @param string $contentType	コンテンツタイプ
	 * @param string $contentId		コンテンツID
	 * @return string				画像パス
	 */
	function getDefaultThumbPath($contentType, $contentId)
	{
		global $gEnvManager;
		
		return $gEnvManager->getResourcePath() . self::CONTENT_DIR . $contentType . self::THUMBNAIL_DIR . $contentId . '_' . self::DEFAULT_THUMBNAIL_TYPE . '.' . self::DEFAULT_THUMBNAIL_EXT;
	}
	/**
	 * サムネールを作成
	 *
	 * @param string $path		元の画像ファイル
	 * @param string $destPath	サムネールファイルパス
	 * @param int $size			サムネールの縦横サイズ
	 * @param int $type			サムネールの画像タイプ
	 * @param bool $useCrop		画像切り取りありかどうか
	 * @param string $bgColor	背景色。設定しない場合は空文字列
	 * @param int $imageQuality	画像品質
	 * @return bool				true=作成,false=作成失敗
	 */
	function createThumb($path, $destPath, $size, $type, $useCrop = true, $bgColor = '#FFFFFF', $imageQuality = 100)
	{
		$imageSize = @getimagesize($path);
		if ($imageSize){
			$imageType = $imageSize[2];
		} else {
			return false;
		}
		
		// 画像作成
		switch ($imageType){
			case IMAGETYPE_GIF:
				$imageObj = @imagecreatefromgif($path);
				break;
			case IMAGETYPE_JPEG:
				$imageObj = @imagecreatefromjpeg($path);
				break;
			case IMAGETYPE_PNG:
				$imageObj = @imagecreatefrompng($path);
				break;
			default:
				return false;
		}

		// 画像サイズ取得
		$x = 0;
		$y = 0;
		$newX = 0;
		$newY = 0;
		$width = imagesx($imageObj);
		$height = imagesy($imageObj);
		if ($useCrop){		// 画像切り取りありのとき
			if ($width > $height){
				$newHeight	= $size;
				$newWidth	= round($size * $width / $height);

				$x = ceil(($width - $height) / 2);
				$y = 0;
			} else {
				$newWidth	= $size;
				$newHeight	= round($size * $height / $width);

				$x = 0;
				$y = ceil(($height - $width) / 2);
			}
		} else {
			if ($width > $height){
				$newHeight = $height * ($size / $width);
				$newWidth = $size;
			} else {
				$newWidth = $width * ($size / $height);
				$newHeight = $size;
			}
		
			$newX = 0;
			$newY = 0;
			if ($newWidth < $size) $newX = round(($size - $newWidth) / 2);
			if ($newHeight < $size) $newY = round(($size - $newHeight) / 2);
		}

		// TrueColorイメージを作成
		$thumbObj = imagecreatetruecolor($size, $size);
		
		// 背景色の設定		
		if (!$useCrop){			// 切り取りなしのとき
			// サムネールの背景色を取得
			$bgColorR = intval(substr($bgColor, 1, 2), 16);
			$bgColorG = intval(substr($bgColor, 3, 2), 16);
			$bgColorB = intval(substr($bgColor, 5, 2), 16);

			// TrueColorイメージを作成
		//	$thumbObj = imagecreatetruecolor($size, $size);
			$bgcolor = imagecolorallocate($thumbObj, $bgColorR, $bgColorG, $bgColorB);		// 背景色設定
			imagefill($thumbObj, 0, 0, $bgcolor);
		}
		
		// 画像リサイズ
		// imagecopyresampledの方がimagecopyresizedよりも画質が良いのでこちらを使用
		if (function_exists("imagecopyresampled")){
			if (!imagecopyresampled($thumbObj, $imageObj, $newX, $newY, $x, $y, $newWidth, $newHeight, $width, $height)){
				if (!imagecopyresized($thumbObj, $imageObj, $newX, $newY, $x, $y, $newWidth, $newHeight, $width, $height)) return false;
			}
		} else {
			if (!imagecopyresized($thumbObj, $imageObj, $newX, $newY, $x, $y, $newWidth, $newHeight, $width, $height)) return false;
		}
		
		// 画像出力
		switch ($type){
			case IMAGETYPE_GIF:
				$ret = @imagegif($thumbObj, $destPath, $imageQuality);
				break;
			case IMAGETYPE_JPEG:
				$ret = @imagejpeg($thumbObj, $destPath, $imageQuality);
				break;
			case IMAGETYPE_PNG:
				//$ret = @imagepng($thumbObj, $destPath, $imageQuality);		// PNGでは$imageQualityを使用すると画像が0サイズで作成される
				$ret = @imagepng($thumbObj, $destPath);
				break;
		}
		// イメージを破棄
		$ret = imagedestroy($imageObj);
		$ret = imagedestroy($thumbObj);
		return $ret;
	}
	/**
	 * サムネールを作成
	 *
	 * @param string $path		ファイルパス
	 * @param int $type			画像タイプ
	 * @param string $destPath	出力ファイル保存のパス
	 * @param int $destType		出力画像タイプ
	 * @param int $size			サムネールの縦横サイズ
	 * @return bool				true=成功、false=失敗
	 */
/*	function createThumbImage($path, $type, $destPath, $destType, $size)
	{
		// 画像作成
		switch ($type){
			case IMAGETYPE_GIF:
				$image = @imagecreatefromgif($path);
				break;
			case IMAGETYPE_JPEG:
				$image = @imagecreatefromjpeg($path);
				break;
			case IMAGETYPE_PNG:
				$image = @imagecreatefrompng($path);
				break;
			default:
				return false;
		}

		// 画像サイズ取得
		$width = imagesx($image);
		$height = imagesy($image);

		if ($width > $height){
			$n_height = $height * ($size / $width);
			$n_width = $size;
		} else {
			$n_width = $width * ($size / $height);
			$n_height = $size;
		}
		
		$x = 0;
		$y = 0;
		if ($n_width < $size) $x = round(($size - $n_width) / 2);
		if ($n_height < $size) $y = round(($size - $n_height) / 2);
		
		// サムネールの背景色を取得
		$bgColor = self::$_configArray[photo_mainCommonDef::CF_THUMBNAIL_BG_COLOR];	// サムネール背景色
		$bgColorR = intval(substr($bgColor, 1, 2), 16);
		$bgColorG = intval(substr($bgColor, 3, 2), 16);
		$bgColorB = intval(substr($bgColor, 5, 2), 16);

		// TrueColorイメージを作成
		$thumb = imagecreatetruecolor($size, $size);
		//$bgcolor = imagecolorallocate($thumb, 255, 255, 255);		// 背景色設定
		$bgcolor = imagecolorallocate($thumb, $bgColorR, $bgColorG, $bgColorB);		// 背景色設定
		imagefill($thumb, 0, 0, $bgcolor);
		
		// 画像リサイズ
		// imagecopyresampledの方がimagecopyresizedよりも画質が良いのでこちらを使用
		if (function_exists("imagecopyresampled")){
			if (!imagecopyresampled($thumb, $image, $x, $y, 0, 0, $n_width, $n_height, $width, $height)){
				if (!imagecopyresized($thumb, $image, $x, $y, 0, 0, $n_width, $n_height, $width, $height)) return false;
			}
		} else {
			if (!imagecopyresized($thumb, $image, $x, $y, 0, 0, $n_width, $n_height, $width, $height)) return false;
		}

		// 画像出力
		switch ($destType){
			case IMAGETYPE_GIF:
				$ret = @imagegif($thumb, $destPath, self::IMAGE_QUALITY);
				break;
			case IMAGETYPE_JPEG:
				$ret = @imagejpeg($thumb, $destPath, self::IMAGE_QUALITY);
				break;
			case IMAGETYPE_PNG:
				$ret = @imagepng($thumb, $destPath, self::IMAGE_QUALITY);
				break;
		}
		// イメージを破棄
		$ret = imagedestroy($image);
		$ret = imagedestroy($thumb);
		return $ret;
	}*/
}
?>
