<?php
/**
 * コンテナクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2012 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: admin_default_contentContentWidgetContainer.php 4974 2012-06-16 02:24:17Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getWidgetContainerPath('default_content') . '/admin_default_contentBaseWidgetContainer.php');

class admin_default_contentContentWidgetContainer extends admin_default_contentBaseWidgetContainer
{
	private $serialNo;		// 選択中の項目のシリアル番号
	private $serialArray = array();		// 表示されているコンテンツシリアル番号
	private $selectedItem = array();	// 選択中の項目
	private $completed;			// データ登録完了かどうか
	private $isExistsContent;		// コンテンツ項目が存在するかどうか
	private $isMultiLang;			// 多言語対応画面かどうか
	const ICON_SIZE = 16;		// アイコンのサイズ
	const INC_INDEX = 1;		// メニュー項目表示順の増加分
	const ADMIN_WIDGET_ID = 'admin_main';		// 管理ウィジェットのウィジェットID
	const CALENDAR_ICON_FILE = '/images/system/calendar.png';		// カレンダーアイコン
	const ACTIVE_ICON_FILE = '/images/system/active.png';			// 公開中アイコン
	const INACTIVE_ICON_FILE = '/images/system/inactive.png';		// 非公開アイコン
	const ADD_TO_MENU_ICON_FILE = '/images/system/addtomenu.png';		// メニューに追加用アイコン
	const PREVIEW_ICON_FILE = '/images/system/preview.png';		// プレビュー用アイコン
	const LANG_ICON_PATH = '/images/system/flag/';		// 言語アイコンパス
	const MSG_UPDATE_CONTENT = 'コンテンツを更新しました';			// コンテンツ更新メッセージ
	const DEFAULT_SEARCH_KEY = '1';			// デフォルトの検索キー(更新日時)
	const DEFAULT_SEARCH_ORDER = '1';			// デフォルトの検索ソート順(降順)
	const DEFAULT_LIST_COUNT = 20;			// 最大リスト表示数
	const DEFAULT_PASSWORD = '********';	// 設定済みを示すパスワード
	const LOG_MSG_ADD_CONTENT = '汎用コンテンツ(%s)を追加しました。タイトル: %s';
	const LOG_MSG_UPDATE_CONTENT = '汎用コンテンツ(%s)を更新しました。タイトル: %s';
	const LOG_MSG_DEL_CONTENT = '汎用コンテンツ(%s)を削除しました。タイトル: %s';
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		// 親クラスを呼び出す
		parent::__construct();
		
		$this->isMultiLang = $this->gEnv->isMultiLanguageSite();			// 多言語対応画面かどうか
	}
	/**
	 * テンプレートファイルを設定
	 *
	 * _assign()でデータを埋め込むテンプレートファイルのファイル名を返す。
	 * 読み込むディレクトリは、「自ウィジェットディレクトリ/include/template」に固定。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。そのまま_assign()に渡る
	 * @return string 						テンプレートファイル名。テンプレートライブラリを使用しない場合は空文字列「''」を返す。
	 */
	function _setTemplate($request, &$param)
	{
		$filename = '';
		$task = $request->trimValueOf('task');
		switch ($task){
			case self::TASK_CONTENT:		// コンテンツ管理
			default:
				$filename = 'admin_main.tmpl.html';
				break;
			case self::TASK_CONTENT_DETAIL:		// 詳細画面
				$filename = 'admin_main_detail.tmpl.html';
				break;
			case self::TASK_ADD_TO_MENU:		// コンテンツへのリンクをメニュー項目に追加
				$filename = 'admin_list.tmpl.html';
				break;
		}
		return $filename;
	}
	/**
	 * テンプレートにデータ埋め込む
	 *
	 * _setTemplate()で指定したテンプレートファイルにデータを埋め込む。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。_setTemplate()と共有。
	 * @return								なし
	 */
	function _assign($request, &$param)
	{
		$task = $request->trimValueOf('task');
		switch ($task){
			case self::TASK_CONTENT:		// コンテンツ管理
			default:
				$this->createList($request);
				break;
			case self::TASK_CONTENT_DETAIL:		// 詳細画面
				$this->createDetail($request);
				break;
			case self::TASK_ADD_TO_MENU:		// コンテンツへのリンクをメニュー項目に追加
				$this->createAddToMenu($request);
				break;
		}
	}
	/**
	 * コンテンツ一覧画面作成
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param								なし
	 */
	function createList($request)
	{
		// ユーザ情報、表示言語
		$userId = $this->gEnv->getCurrentUserId();
		$this->langId = $this->gEnv->getDefaultLanguage();
		
		// 一覧表示数
		$maxListCount = self::DEFAULT_LIST_COUNT;
		
		// ##### 検索条件 #####
		$pageNo = $request->trimValueOf('page');				// ページ番号
		if (empty($pageNo)) $pageNo = 1;
		$searchKeyword = $request->trimValueOf('search_keyword');			// 検索キーワード
		$searchKey = $request->trimValueOf('search_key');			// 検索ソートキー
		if ($searchKey == '') $searchKey = self::DEFAULT_SEARCH_KEY;
		$searchOrder = $request->trimValueOf('search_order');			// 検索ソート順
		if ($searchOrder == '') $searchOrder = self::DEFAULT_SEARCH_ORDER;

		$act = $request->trimValueOf('act');
		if ($act == 'delete'){		// 項目削除の場合
			$listedItem = explode(',', $request->trimValueOf('seriallist'));
			$delItems = array();
			for ($i = 0; $i < count($listedItem); $i++){
				// 項目がチェックされているかを取得
				$itemName = 'item' . $i . '_selected';
				$itemValue = ($request->trimValueOf($itemName) == 'on') ? 1 : 0;
				
				if ($itemValue){		// チェック項目
					$delItems[] = $listedItem[$i];
				}
			}
			if (count($delItems) > 0){
				// 削除するコンテンツの情報を取得
				$delContentInfo = array();
				for ($i = 0; $i < count($delItems); $i++){
					$ret = self::$_mainDb->getContentBySerial($delItems[$i], $row);
					if ($ret){
						$newInfoObj = new stdClass;
						$newInfoObj->contentId = $row['cn_id'];		// コンテンツID
						$newInfoObj->name = $row['cn_name'];		// コンテンツ名前
						$delContentInfo[] = $newInfoObj;
					}
				}
				
				// 多言語対応状態に関わらずコンテンツIDで削除
				//if ($this->isMultiLang){		// 多言語対応のとき
					$ret = self::$_mainDb->delContentItemById($delItems);
				//} else {
				//	$ret = self::$_mainDb->delContentItem($delItems);
				//}
				if ($ret){		// データ削除成功のとき
					$this->setGuidanceMsg('データを削除しました');
					
					// キャッシュデータのクリア
					for ($i = 0; $i < count($delItems); $i++){
						$this->clearCacheBySerial($delItems[$i]);
					}
					
					// 運用ログを残す
					for ($i = 0; $i < count($delContentInfo); $i++){
						$infoObj = $delContentInfo[$i];
						$contentAttr = default_contentCommonDef::$_deviceTypeName;
						$this->gOpeLog->writeUserInfo(__METHOD__, sprintf(self::LOG_MSG_DEL_CONTENT, $contentAttr, $infoObj->name), 2100, 'ID=' . $infoObj->contentId);
					}
				} else {
					$this->setAppErrorMsg('データ削除に失敗しました');
				}
			}
		} else if ($act == 'search'){		// 検索のとき
			$pageNo = 1;		// ページ番号初期化
		} else if ($act == 'selpage'){			// ページ選択
		}
		// コンテンツ総数を取得
		$totalCount = self::$_mainDb->getContentCount(default_contentCommonDef::$_contentType, $this->langId, $searchKeyword, $searchKey, $searchOrder);

		// 表示するページ番号の修正
		$pageCount = (int)(($totalCount -1) / $maxListCount) + 1;		// 総ページ数
		if ($pageNo < 1) $pageNo = 1;
		if ($pageNo > $pageCount) $pageNo = $pageCount;
		$this->firstNo = ($pageNo -1) * $maxListCount + 1;		// 先頭番号
		
		// ページング用リンク作成
		$pageLink = '';
		if ($pageCount > 1){	// ページが2ページ以上のときリンクを作成
			for ($i = 1; $i <= $pageCount; $i++){
				if ($i == $pageNo){
					$link = '&nbsp;' . $i;
				} else {
					$link = '&nbsp;<a href="#" onclick="selpage(\'' . $i . '\');return false;">' . $i . '</a>';
				}
				$pageLink .= $link;
			}
		}
		
		// 一覧の表示タイプを設定
		if ($this->isMultiLang){		// 多言語対応の場合
			$this->tmpl->setAttribute('show_multilang', 'visibility', 'visible');
		} else {
			$this->tmpl->setAttribute('show_singlelang', 'visibility', 'visible');
		}
		
		// コンテンツリストを取得
		self::$_mainDb->searchContent(default_contentCommonDef::$_contentType, $this->langId, $maxListCount, $pageNo, $searchKeyword, $searchKey, $searchOrder, array($this, 'itemListLoop'));
		if (!$this->isExistsContent) $this->tmpl->setAttribute('itemlist', 'visibility', 'hidden');// コンテンツ項目がないときは、一覧を表示しない
		
		// 画面にデータを埋め込む
		// 検索条件
		if (empty($searchKey)){			// コンテンツIDをキーにする場合
			$this->tmpl->addVar('_widget', 'search_content_id_checked', 'checked');
		} else {
			$this->tmpl->addVar('_widget', 'search_update_dt_checked', 'checked');
		}
		if (empty($searchOrder)){			// 昇順にソートするとき
			$this->tmpl->addVar('_widget', 'search_asc_checked', 'checked');
		} else {
			$this->tmpl->addVar('_widget', 'search_desc_checked', 'checked');
		}
		
		// 検索結果
		$this->tmpl->addVar("_widget", "page_link", $pageLink);
		$this->tmpl->addVar("_widget", "total_count", $totalCount);
		$this->tmpl->addVar("_widget", "search_keyword", $searchKeyword);	// 検索キーワード
		
		// その他
		$this->tmpl->addVar("_widget", "serial_list", implode($this->serialArray, ','));// 表示項目のシリアル番号を設定
		$this->tmpl->addVar("_widget", "target_widget", $this->gEnv->getCurrentWidgetId());// メニュー選択ウィンドウ表示用
		$this->tmpl->addVar("_widget", "device_type", default_contentCommonDef::$_deviceType);		// デバイスタイプ
	}
	/**
	 * コンテンツ詳細画面作成
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param								なし
	 */
	function createDetail($request)
	{
		// ユーザ情報、表示言語
		$this->langId = $this->gEnv->getDefaultLanguage();
		
		// ウィンドウ表示状態
		$openby = $request->trimValueOf(M3_REQUEST_PARAM_OPEN_BY);
		
		// 言語を取得
		if ($this->isMultiLang){		// 多言語対応の場合
			$langId = $request->trimValueOf('item_lang');				// 現在メニューで選択中の言語
			if (!empty($langId)) $this->langId = $langId;
		}
		$act = $request->trimValueOf('act');
		$this->serialNo = $request->trimValueOf('serial');		// 選択項目のシリアル番号
		$contentId = $request->trimValueOf('contentid');		// コンテンツID
		$name = $request->trimValueOf('item_name');
		$html = $request->valueOf('item_html');		// HTMLタグを可能とする
		$key = $request->trimValueOf('item_key');		// 外部参照用キー
		$visible = ($request->trimValueOf('item_visible') == 'on') ? 1 : 0;		// チェックボックス
		$limited = ($request->trimValueOf('item_limited') == 'on') ? 1 : 0;		// チェックボックス
		$default = ($request->trimValueOf('item_default') == 'on') ? 1 : 0;		// チェックボックス
		$metaTitle = $request->trimValueOf('item_meta_title');		// ページタイトル名
		$metaDesc = $request->trimValueOf('item_meta_desc');			// ページ要約
		$metaKeyword = $request->trimValueOf('item_meta_keyword');	// ページキーワード
		$password = $request->trimValueOf('password');
		
		$start_date = $request->trimValueOf('item_start_date');		// 公開期間開始日付
		if (!empty($start_date)) $start_date = $this->convertToProperDate($start_date);
		$start_time = $request->trimValueOf('item_start_time');		// 公開期間開始時間
		if (empty($start_date)){
			$start_time = '';					// 日付が空のときは時刻も空に設定する
		} else {
			if (empty($start_time)) $start_time = '00:00';		// 日付が入っているときは時間にデフォルト値を設定
		}
		if (!empty($start_time)) $start_time = $this->convertToProperTime($start_time, 1/*時分フォーマット*/);
		
		$end_date = $request->trimValueOf('item_end_date');		// 公開期間終了日付
		if (!empty($end_date)) $end_date = $this->convertToProperDate($end_date);
		$end_time = $request->trimValueOf('item_end_time');		// 公開期間終了時間
		if (empty($end_date)){
			$end_time = '';					// 日付が空のときは時刻も空に設定する
		} else {
			if (empty($end_time)) $end_time = '00:00';		// 日付が入っているときは時間にデフォルト値を設定
		}
		if (!empty($end_time)) $end_time = $this->convertToProperTime($end_time, 1/*時分フォーマット*/);
		
		$reloadData = false;		// データの再読み込み
		$hasPassword = false;		// パスワードが設定されているかどうか
		$histryIndex = -1;	// 履歴番号
		if ($act == 'add'){		// 項目追加の場合
			// 入力チェック
			$this->checkInput($name, '名前');

			// 期間範囲のチェック
			if (!empty($start_date) && !empty($end_date)){
				if (strtotime($start_date . ' ' . $start_time) >= strtotime($end_date . ' ' . $end_time)) $this->setUserErrorMsg('公開期間が不正です');
			}
			
			// エラーなしの場合は、データを登録
			if ($this->getMsgCount() == 0){
				// 保存データ作成
				if (empty($start_date)){
					$startDt = $this->gEnv->getInitValueOfTimestamp();
				} else {
					$startDt = $start_date . ' ' . $start_time;
				}
				if (empty($end_date)){
					$endDt = $this->gEnv->getInitValueOfTimestamp();
				} else {
					$endDt = $end_date . ' ' . $end_time;
				}
				
				if ($this->isMultiLang){		// 多言語対応の場合
					//$contentId = $request->trimValueOf('contentid');		// コンテンツID
					$ret = self::$_mainDb->addContentItem(default_contentCommonDef::$_contentType, $contentId, $this->langId, $name, ''/*説明*/, $html, $visible, $default, $limited, $key, $password, 
														$metaTitle, $metaDesc, $metaKeyword, $startDt, $endDt, $newSerial);
				} else {
					$ret = self::$_mainDb->addContentItem(default_contentCommonDef::$_contentType, 0, $this->langId, $name, ''/*説明*/, $html, $visible, $default, $limited, $key, $password, 
														$metaTitle, $metaDesc, $metaKeyword, $startDt, $endDt, $newSerial);
				}
				if ($ret){
					$this->setGuidanceMsg('データを追加しました');
					// シリアル番号更新
					$this->serialNo = $newSerial;
					$reloadData = true;		// データの再読み込み
					
					// 親ウィンドウを更新
					$this->gPage->updateParentWindow($this->serialNo);
					
					// コンテンツ更新情報をサーバへ登録
					$ret = $this->registContentInfoBySerial($this->serialNo, true/*新規*/);
					if ($ret) $this->setGuidanceMsg('更新情報をサーバへアップしました');
					
					// 運用ログを残す
					$ret = self::$_mainDb->getContentBySerial($this->serialNo, $row);
					if ($ret){
						$contentId = $row['cn_id'];		// コンテンツID
						$name = $row['cn_name'];		// コンテンツ名前
					}
					$contentAttr = default_contentCommonDef::$_deviceTypeName;
					if ($this->isMultiLang) $contentAttr .= $this->getLangName($this->langId);		// 多言語対応の場合
					$this->gOpeLog->writeUserInfo(__METHOD__, sprintf(self::LOG_MSG_ADD_CONTENT, $contentAttr, $name), 2100, 'ID=' . $contentId);
				} else {
					$this->setAppErrorMsg('データ追加に失敗しました');
				}
			}
		} else if ($act == 'update'){		// 項目更新の場合
			// 入力チェック
			$this->checkInput($name, '名前');
			
			// 期間範囲のチェック
			if (!empty($start_date) && !empty($end_date)){
				if (strtotime($start_date . ' ' . $start_time) >= strtotime($end_date . ' ' . $end_time)) $this->setUserErrorMsg('公開期間が不正です');
			}
			
			// エラーなしの場合は、データを更新
			if ($this->getMsgCount() == 0){
				// 保存データ作成
				if (empty($start_date)){
					$startDt = $this->gEnv->getInitValueOfTimestamp();
				} else {
					$startDt = $start_date . ' ' . $start_time;
				}
				if (empty($end_date)){
					$endDt = $this->gEnv->getInitValueOfTimestamp();
				} else {
					$endDt = $end_date . ' ' . $end_time;
				}
				
				$ret = self::$_mainDb->updateContentItem($this->serialNo, $name, $html, $visible, $default, $limited, $key, $password, $metaTitle, $metaDesc, $metaKeyword, $startDt, $endDt, $newSerial);
				if ($ret){
					$this->setGuidanceMsg('データを更新しました');
					// シリアル番号更新
					$this->serialNo = $newSerial;
					$reloadData = true;		// データの再読み込み
					
					// キャッシュデータのクリア
					$this->clearCacheBySerial($this->serialNo);
					
					// 親ウィンドウを更新
					$this->gPage->updateParentWindow($this->serialNo);
					
					// コンテンツ更新情報をサーバへ登録
					$ret = $this->registContentInfoBySerial($this->serialNo);
					if ($ret) $this->setGuidanceMsg('更新情報をサーバへアップしました');
					
					// 運用ログを残す
					$ret = self::$_mainDb->getContentBySerial($this->serialNo, $row);
					if ($ret){
						$contentId = $row['cn_id'];		// コンテンツID
						$name = $row['cn_name'];		// コンテンツ名前
					}
					$contentAttr = default_contentCommonDef::$_deviceTypeName;
					if ($this->isMultiLang) $contentAttr .= $this->getLangName($this->langId);		// 多言語対応の場合
					$this->gOpeLog->writeUserInfo(__METHOD__, sprintf(self::LOG_MSG_UPDATE_CONTENT, $contentAttr, $name), 2100, 'ID=' . $contentId);
				} else {
					$this->setAppErrorMsg('データ更新に失敗しました');
				}
			}				
		} else if ($act == 'delete'){		// 項目削除の場合
			if (empty($this->serialNo)){
				$this->setUserErrorMsg('削除項目が選択されていません');
			}
			// エラーなしの場合は、データを削除
			if ($this->getMsgCount() == 0){
				// 削除するコンテンツの情報を取得
				$ret = self::$_mainDb->getContentBySerial($this->serialNo, $row);
				if ($ret){
					$contentId = $row['cn_id'];		// コンテンツID
					$name = $row['cn_name'];		// コンテンツ名前
				}

				if ($this->isMultiLang){		// 多言語対応のとき
					if ($this->langId == $this->gEnv->getDefaultLanguage()){		// デフォルト言語のときは全削除
						$ret = self::$_mainDb->delContentItemById(array($this->serialNo));
					} else {
						$ret = self::$_mainDb->delContentItem(array($this->serialNo));
					}
				} else {
					// 多言語対応状態に関わらずコンテンツIDで削除
					$ret = self::$_mainDb->delContentItemById(array($this->serialNo));
					//$ret = self::$_mainDb->delContentItem(array($this->serialNo));
				}
				if ($ret){		// データ削除成功のとき
					$this->setGuidanceMsg('データを削除しました');
					$reloadData = true;		// データの再読み込み
					
					// キャッシュデータのクリア
					$this->clearCacheBySerial($this->serialNo);
					
					// 親ウィンドウを更新
					$this->gPage->updateParentWindow();
					
					// 運用ログを残す
					$contentAttr = default_contentCommonDef::$_deviceTypeName;
					if ($this->isMultiLang && $this->langId != $this->gEnv->getDefaultLanguage()) $contentAttr .= $this->getLangName($this->langId);		// 多言語対応の場合
					$this->gOpeLog->writeUserInfo(__METHOD__, sprintf(self::LOG_MSG_DEL_CONTENT, $contentAttr, $name), 2100, 'ID=' . $contentId);
				} else {
					$this->setAppErrorMsg('データ削除に失敗しました');
				}
			}
		} else if ($act == 'get_history'){		// 履歴データの取得のとき
			$reloadData = true;		// データの再読み込み
		} else if ($act == 'selectlang'){		// 言語選択
			// 多言語対応の場合の処理
			//$contentId = $request->trimValueOf('contentid');		// コンテンツID
			
			// コンテンツを取得
			$ret = self::$_mainDb->getContentByContentId(default_contentCommonDef::$_contentType, $contentId, $this->langId, $row);
			if ($ret){
				$this->serialNo = $row['cn_serial'];		// コンテンツシリアル番号
				$reloadData = true;		// データの再読み込み
			} else {
				$this->serialNo = 0;
			}
		} else {
			// ##### コンテンツIDが設定されているとき(他ウィジェットからの表示)は、データを取得 #####
			//$contentId = $request->trimValueOf('contentid');		// コンテンツID
			if (empty($contentId)){
				if (empty($this->serialNo)){		// 新規項目追加のとき
					$visible = 1;		// 非公開で登録されてしまうので、初期状態は表示に設定
				} else {
					$reloadData = true;		// データの再読み込み
				}
			} else {
				// 多言語対応の場合は、言語を取得
				if ($this->isMultiLang){		// 多言語対応の場合
					$langId = $request->trimValueOf(M3_REQUEST_PARAM_OPERATION_LANG);		// lang値を取得
					if (!empty($langId)) $this->langId = $langId;
				}
		
				// コンテンツを取得
				$ret = self::$_mainDb->getContentByContentId(default_contentCommonDef::$_contentType, $contentId, $this->langId, $row);
				if ($ret){
					$this->serialNo = $row['cn_serial'];		// コンテンツシリアル番号
					$reloadData = true;		// データの再読み込み
				} else {
					$visible = 1;		// 非公開で登録されてしまうので、初期状態は表示に設定
					$this->serialNo = 0;
				}
			}
		}
		if ($reloadData){		// データの再読み込み
			$ret = self::$_mainDb->getContentBySerial($this->serialNo, $row);
			if ($ret){
				$contentId = $row['cn_id'];		// コンテンツID
				$name = $row['cn_name'];		// コンテンツ名前
				$html = str_replace(M3_TAG_START . M3_TAG_MACRO_ROOT_URL . M3_TAG_END, $this->getUrl($this->gEnv->getRootUrl()), $row['cn_html']);				// HTML
				$key = $row['cn_key'];					// 外部参照用キー
				$update_user = $this->convertToDispString($row['lu_name']);// 更新者
				$update_dt = $this->convertToDispDateTime($row['cn_create_dt']);
			
				// 項目表示、デフォルト値チェックボックス
				$visible = $row['cn_visible'];
				$default = $row['cn_default'];
				$limited = $row['cn_user_limited'];		// ユーザ制限
				$metaTitle = $row['cn_meta_title'];		// ページタイトル名(METAタグ)
				$metaDesc = $row['cn_meta_description'];		// ページ要約(METAタグ)
				$metaKeyword = $row['cn_meta_keywords'];		// ページキーワード(METAタグ)
				$start_date = $this->convertToDispDate($row['cn_active_start_dt']);	// 公開期間開始日
				$start_time = $this->convertToDispTime($row['cn_active_start_dt'], 1/*時分*/);	// 公開期間開始時間
				$end_date = $this->convertToDispDate($row['cn_active_end_dt']);	// 公開期間終了日
				$end_time = $this->convertToDispTime($row['cn_active_end_dt'], 1/*時分*/);	// 公開期間終了時間
				
				// パスワード
				if (!empty($row['cn_password'])) $hasPassword = true;		// パスワードが設定されている
				
				// 履歴番号
				if ($row['cn_deleted']) $histryIndex = $row['cn_history_index'];
			} else {
				$this->serialNo = 0;
			}
		}
		// 一覧の表示タイプを設定
		if ($this->isMultiLang){		// 多言語対応の場合
			$this->tmpl->setAttribute('show_multilang', 'visibility', 'visible');
			$this->tmpl->addVar("show_multilang", "sel_item_name", $name);		// 名前
			
			if (empty($contentId)){		// 新規追加の場合
				$defaultLangName = $this->gEnv->getDefaultLanguageNameByCurrentLanguage();		// デフォルト言語の現在の表示名を取得
				$this->tmpl->addVar("default_lang", "default_lang", $defaultLangName);
				$this->tmpl->setAttribute('default_lang', 'visibility', 'visible');
			} else {		// コンテンツが選択されているとき
				// 言語選択メニュー作成
				self::$_mainDb->getAvailableLang(array($this, 'langLoop'));
				$this->tmpl->setAttribute('select_lang', 'visibility', 'visible');
				
				// パスワード入力エリア
				if ($this->langId != $this->gEnv->getDefaultLanguage()) $this->tmpl->addVar("_widget", "password_disabled", "disabled");
			}
			
			// 言語イメージ
			$langImage = $this->createLangImage($contentId);
			$this->tmpl->addVar("show_multilang", "lang", $langImage);
		} else {
			$this->tmpl->setAttribute('show_singlelang', 'visibility', 'visible');
			$this->tmpl->addVar("show_singlelang", "sel_item_name", $name);		// 名前
		}
		
		// ### 入力値を再設定 ###
		$this->tmpl->addVar("_widget", "sel_item_html", $html);		// HTML
		$this->tmpl->addVar("_widget", "sel_item_key", $key);		// 外部参照用キー
		$this->tmpl->addVar("_widget", "meta_title", $metaTitle);		// ページタイトル名(METAタグ)
		$this->tmpl->addVar("_widget", "meta_desc", $metaDesc);		// ページ要約(METAタグ)
		$this->tmpl->addVar("_widget", "meta_keyword", $metaKeyword);		// ページキーワード(METAタグ)
		$this->tmpl->addVar("_widget", "update_user", $update_user);	// 更新者
		$this->tmpl->addVar("_widget", "update_dt", $update_dt);	// 更新日時
		$this->tmpl->addVar("_widget", "start_date", $start_date);	// 公開期間開始日
		$this->tmpl->addVar("_widget", "start_time", $start_time);	// 公開期間開始時間
		$this->tmpl->addVar("_widget", "end_date", $end_date);	// 公開期間終了日
		$this->tmpl->addVar("_widget", "end_time", $end_time);	// 公開期間終了時間
		if ($hasPassword) $this->tmpl->addVar("_widget", "password", self::DEFAULT_PASSWORD);// 入力済みを示すパスワードの設定
		
		// 項目表示、項目利用可否チェックボックス
		$visibleStr = '';
		if ($visible) $visibleStr = 'checked';
		$this->tmpl->addVar("_widget", "sel_item_visible", $visibleStr);
		$defaultStr = '';
		if ($default) $defaultStr = 'checked';
		$this->tmpl->addVar("_widget", "sel_item_default", $defaultStr);
		$limitedStr = '';
		if ($limited) $limitedStr = 'checked';
		$this->tmpl->addVar("_widget", "sel_item_limited", $limitedStr);
	
		$this->tmpl->addVar("_widget", "serial", $this->serialNo);		// 選択中のシリアル番号
		$this->tmpl->addVar("_widget", "target_widget", $this->gEnv->getCurrentWidgetId());// メニュー選択ウィンドウ表示用
		$this->tmpl->addVar("_widget", "device_type", default_contentCommonDef::$_deviceType);		// デバイスタイプ
			
		// パスの設定
		$this->tmpl->addVar('_widget', 'admin_url', $this->getUrl($this->gEnv->getDefaultAdminUrl()));// 管理者URL

		// プレビュー用URL
		switch (default_contentCommonDef::$_deviceType){		// デバイスごとの処理
			case 0:		// PC
			default:
				$previewUrl = $this->gEnv->getDefaultUrl() . '?' . M3_REQUEST_PARAM_CONTENT_ID . '=' . $contentId;
				break;
			case 1:		// 携帯
				$previewUrl = $this->gEnv->getDefaultMobileUrl() . '?' . M3_REQUEST_PARAM_CONTENT_ID . '=' . $contentId;
				break;
			case 2:		// スマートフォン
				$previewUrl = $this->gEnv->getDefaultSmartphoneUrl() . '?' . M3_REQUEST_PARAM_CONTENT_ID . '=' . $contentId;
				break;
		}
		if ($histryIndex >= 0) $previewUrl .= '&' . M3_REQUEST_PARAM_HISTORY . '=' . $histryIndex;
		$previewUrl .= '&' . M3_REQUEST_PARAM_OPERATION_COMMAND . '=' . M3_REQUEST_CMD_PREVIEW;
		if ($this->isMultiLang) $previewUrl .= '&' . M3_REQUEST_PARAM_OPERATION_LANG . '=' . $this->langId;		// 多言語対応の場合は言語IDを付加
		$this->tmpl->addVar('_widget', 'preview_url', $previewUrl);// プレビュー用URL(一般画面)
		
		$this->tmpl->addVar('_widget', 'custom_value_task', 'usercustom');		// ユーザ定義値参照用
		$this->tmpl->addVar('_widget', 'admin_widget_id', self::ADMIN_WIDGET_ID);// ユーザ定義値参照用(管理ウィジェットのウィジェットID)
		$this->tmpl->addVar('_widget', 'calendar_img', $this->getUrl($this->gEnv->getRootUrl() . self::CALENDAR_ICON_FILE));	// カレンダーアイコン
		
		// ボタンの表示制御
		if (empty($this->serialNo)){		// 新規追加項目を選択しているとき
			if ($this->isMultiLang){		// 多言語対応の場合
				if (empty($contentId)){
					$this->tmpl->addVar("_widget", "sel_item_id", '');			// コンテンツID
					$this->tmpl->addVar("_widget", "item_id", '新規');			// コンテンツID
				} else {
					$this->tmpl->addVar("_widget", "sel_item_id", $contentId);			// コンテンツID
					$this->tmpl->addVar("_widget", "item_id", $contentId);			// コンテンツID
				}
			} else {
				$this->tmpl->addVar("_widget", "sel_item_id", '');			// コンテンツID
				$this->tmpl->addVar("_widget", "item_id", '新規');			// コンテンツID
			}
			$this->tmpl->setAttribute('add_button', 'visibility', 'visible');// 「新規追加」ボタン
			$this->tmpl->addVar('_widget', 'preview_btn_disabled', 'disabled');// プレビューボタン使用不可
			$this->tmpl->addVar('_widget', 'history_btn_disabled', 'disabled');// 履歴ボタン使用不可
		} else {
			// 履歴番号
			$itemId = $contentId;			// コンテンツID
			if ($histryIndex >= 0) $itemId .= '(' . ($histryIndex +1) . ')';

			$this->tmpl->addVar("_widget", "sel_item_id", $contentId);			// コンテンツID
			$this->tmpl->addVar("_widget", "item_id", $itemId);			// コンテンツID
			$this->tmpl->setAttribute('del_button', 'visibility', 'visible');// 「削除」ボタン
		}
		// 「戻る」ボタンの表示
		if ($openby == 'simple' || $openby == 'tabs') $this->tmpl->setAttribute('cancel_button', 'visibility', 'hidden');		// 詳細画面のみの表示またはタブ表示のときは戻るボタンを隠す
	}
	/**
	 * コンテンツへのリンクをメニューに追加するための画面作成
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param								なし
	 */
	function createAddToMenu($request)
	{
		// ユーザ情報、表示言語
		$userId = $this->gEnv->getCurrentUserId();
		$this->langId = $this->gEnv->getDefaultLanguage();

		$act = $request->trimValueOf('act');
		$contentId = $request->trimValueOf('contentid');		// コンテンツID
		$serialList = $request->trimValueOf('seriallist');
		if (!empty($serialList)){
			$listedItem = explode(',', $serialList);
			for ($i = 0; $i < count($listedItem); $i++){
				// 項目がチェックされているかを取得
				$itemName = 'item' . $i . '_selected';
				$itemValue = ($request->trimValueOf($itemName) == 'on') ? 1 : 0;
				if ($itemValue) $this->selectedItem[] = $listedItem[$i];// チェック項目
			}
		}
		if ($act == 'addtomenu'){			// メニューに項目を追加
			// URLの作成
			switch (default_contentCommonDef::$_deviceType){		// デバイスごとの処理
				case 0:		// PC
				default:
					$url = M3_TAG_START . M3_TAG_MACRO_ROOT_URL . M3_TAG_END . '/' . $this->gEnv->getDefaultPageId() . '.php?contentid=' . $contentId;
					break;
				case 1:		// 携帯
					$url = M3_TAG_START . M3_TAG_MACRO_ROOT_URL . M3_TAG_END . '/' . M3_DIR_NAME_MOBILE . '/' . $this->gEnv->getDefaultPageId() . '.php?contentid=' . $contentId;
					break;
				case 2:		// スマートフォン
					$url = M3_TAG_START . M3_TAG_MACRO_ROOT_URL . M3_TAG_END . '/' . M3_DIR_NAME_SMARTPHONE . '/' . $this->gEnv->getDefaultPageId() . '.php?contentid=' . $contentId;
					break;
			}

			// コンテンツ名を取得
			$menutItemName = '';
			$ret = self::$_mainDb->getContentByContentId(default_contentCommonDef::$_contentType, $contentId, $this->langId, $row);
			if ($ret) $menutItemName = $row['cn_name'];		// 名前は取得値を設定

			// メニュー項目追加
			for ($i = 0; $i < count($this->selectedItem); $i++){
				$ret = self::$_mainDb->addMenuItem($this->selectedItem[$i], $menutItemName, $url);
				if (!$ret) break;
			}
			if ($ret){
				// 親ウィンドウを更新
				//$this->gPage->updateParentWindow();		// 設定画面からの小ウィンドウ表示なので親ウィンドウ(設定画面)は更新しない
					
				$this->setGuidanceMsg('メニューにリンクを追加しました');
				$this->completed = true;			// データ登録完了かどうか
			} else {
				$this->setAppErrorMsg('メニューのリンク追加に失敗しました');
			}
		}

		// メニューID選択メニュー作成
		self::$_mainDb->getMenuIdList(default_contentCommonDef::$_deviceType, array($this, 'menuIdListLoop'));
		
		$this->tmpl->addVar("_widget", "content_id", $contentId);// コンテンツID
		$this->tmpl->addVar("_widget", "serial_list", implode($this->serialArray, ','));// 表示項目のシリアル番号を設定
		
		// 一覧項目がないときは、一覧を表示しない
		if (!$this->isExistsContent) $this->tmpl->setAttribute('itemlist', 'visibility', 'hidden');
		
		if ($this->completed){// 	データ追加完了のとき
			$this->tmpl->addVar('_widget', 'add_disabled', 'disabled');// 「追加」ボタン
		}
	}
	/**
	 * 取得したデータをテンプレートに設定する
	 *
	 * @param int $index			行番号(0～)
	 * @param array $fetchedRow		フェッチ取得した行
	 * @param object $param			未使用
	 * @return bool					true=処理続行の場合、false=処理終了の場合
	 */
	function itemListLoop($index, $fetchedRow, $param)
	{
		$serial = $this->convertToDispString($fetchedRow['cn_serial']);
		$contentId = $fetchedRow['cn_id'];		// コンテンツID
		
		// 表示状態
		$visible = '';
		if ($fetchedRow['cn_visible']) $visible = 'checked';

		// ユーザ制限
		$limited = '';
		if ($fetchedRow['cn_user_limited']) $limited = 'checked';

		// デフォルト時の項目かどうか
		$default = '';
		if ($fetchedRow['cn_default']) $default = 'checked';
		
		// 対応言語を取得
		$lang = '';
		if ($this->isMultiLang){		// 多言語対応の場合
			$lang = $this->createLangImage($contentId);
		}
		// 総参照数
		$totalViewCount = $this->gInstance->getAnalyzeManager()->getTotalContentViewCount(default_contentCommonDef::$_viewContentType, $serial);
		
		// 公開状況の設定
		$now = date("Y/m/d H:i:s");	// 現在日時
		$startDt = $fetchedRow['cn_active_start_dt'];
		$endDt = $fetchedRow['cn_active_end_dt'];
		
		$isActive = false;		// 公開状態
		if ($fetchedRow['cn_visible']) $isActive = $this->isActive($startDt, $endDt, $now);// 表示可能
		
		if ($isActive){		// コンテンツが公開状態のとき
			$iconUrl = $this->gEnv->getRootUrl() . self::ACTIVE_ICON_FILE;			// 公開中アイコン
			$iconTitle = '公開中';
		} else {
			$iconUrl = $this->gEnv->getRootUrl() . self::INACTIVE_ICON_FILE;		// 非公開アイコン
			$iconTitle = '非公開';
		}
		$statusImg = '<img src="' . $this->getUrl($iconUrl) . '" width="' . self::ICON_SIZE . '" height="' . self::ICON_SIZE . '" border="0" alt="' . $iconTitle . '" title="' . $iconTitle . '" />';
		
		// 操作用ボタン
		$addToMenuImg = $this->getUrl($this->gEnv->getRootUrl() . self::ADD_TO_MENU_ICON_FILE);		// メニューに追加用アイコン
		$addToMenuStr = 'メニューに追加';
		$statusUrl = $this->gEnv->getDefaultUrl() . '?' . M3_REQUEST_PARAM_CONTENT_ID . '=' . $contentId;// 現在の表示画面用URL
		
		// プレビュー用URL
		switch (default_contentCommonDef::$_deviceType){		// デバイスごとの処理
			case 0:		// PC
			default:
				$previewUrl = $this->gEnv->getDefaultUrl() . '?' . M3_REQUEST_PARAM_CONTENT_ID . '=' . $contentId;
				break;
			case 1:		// 携帯
				$previewUrl = $this->gEnv->getDefaultMobileUrl() . '?' . M3_REQUEST_PARAM_CONTENT_ID . '=' . $contentId;
				break;
			case 2:		// スマートフォン
				$previewUrl = $this->gEnv->getDefaultSmartphoneUrl() . '?' . M3_REQUEST_PARAM_CONTENT_ID . '=' . $contentId;
				break;
		}
		$previewUrl .= '&' . M3_REQUEST_PARAM_OPERATION_COMMAND . '=' . M3_REQUEST_CMD_PREVIEW;// プレビュー用URL
//		if ($this->isMultiLang) $previewUrl .= '&' . M3_REQUEST_PARAM_OPERATION_LANG . '=' . $this->langId;		// 多言語対応の場合は言語IDを付加
		$previewImg = $this->getUrl($this->gEnv->getRootUrl() . self::PREVIEW_ICON_FILE);
		$previewStr = 'プレビュー';
		
		$row = array(
			'index' => $index,													// 項目番号
			'serial' => $serial,			// シリアル番号
			'id' => $this->convertToDispString($contentId),			// ID
			'name' => $this->convertToDispString($fetchedRow['cn_name']),		// 名前
			'lang' => $lang,													// 対応言語
			'view_count' => $totalViewCount,									// 総参照数
			'status' => $statusImg,												// 公開状況
			'update_user' => $this->convertToDispString($fetchedRow['lu_name']),	// 更新者
			'update_dt' => $this->convertToDispDateTime($fetchedRow['cn_create_dt']),	// 更新日時
			'visible' => $visible,											// メニュー項目表示制御
			'limited' => $limited,											// ユーザ制限
			'default' => $default,											// デフォルト項目
			'add_to_menu_img' => $addToMenuImg,											// メニューに追加用の画像
			'add_to_menu_str' => $addToMenuStr,											// メニューに追加用の文字列
			'status_url' => $statusUrl,											// 現在の表示画面用URL
			'preview_url' => $previewUrl,											// プレビュー用のURL
			'preview_img' => $previewImg,											// プレビュー用の画像
			'preview_str' => $previewStr									// プレビュー文字列
		);
		if ($this->isMultiLang){		// 多言語対応のとき
			$this->tmpl->addVars('itemlist2', $row);
			$this->tmpl->parseTemplate('itemlist2', 'a');
		} else {
			$this->tmpl->addVars('itemlist', $row);
			$this->tmpl->parseTemplate('itemlist', 'a');
		}
		
		// 表示中のコンテンツIDを保存
		$this->serialArray[] = $fetchedRow['cn_serial'];
		
		$this->isExistsContent = true;		// コンテンツ項目が存在するかどうか
		return true;
	}
	/**
	 * 取得した言語をテンプレートに設定する
	 *
	 * @param int $index			行番号(0～)
	 * @param array $fetchedRow		フェッチ取得した行
	 * @param object $param			未使用
	 * @return bool					true=処理続行の場合、false=処理終了の場合
	 */
	function langLoop($index, $fetchedRow, $param)
	{
		$selected = '';
		if ($fetchedRow['ln_id'] == $this->langId){
			$selected = 'selected';
		}
		if ($this->gEnv->getCurrentLanguage() == 'ja'){		// 日本語表示の場合
			$name = $this->convertToDispString($fetchedRow['ln_name']);
		} else {
			$name = $this->convertToDispString($fetchedRow['ln_name_en']);
		}

		$row = array(
			'value'    => $this->convertToDispString($fetchedRow['ln_id']),			// 言語ID
			'name'     => $name,			// 言語名
			'selected' => $selected														// 選択中かどうか
		);
		$this->tmpl->addVars('lang_list', $row);
		$this->tmpl->parseTemplate('lang_list', 'a');
		return true;
	}
	/**
	 * 取得したデータをテンプレートに設定する
	 *
	 * @param int $index			行番号(0～)
	 * @param array $fetchedRow		フェッチ取得した行
	 * @param object $param			未使用
	 * @return bool					true=処理続行の場合、false=処理終了の場合
	 */
	function menuIdListLoop($index, $fetchedRow, $param)
	{
		$id = $fetchedRow['mn_id'];
		
		$checkStr = '';
		if (in_array($id, $this->selectedItem)) $checkStr = 'checked ';
		if ($this->completed) $checkStr .= 'disabled ';			// 追加完了のとき
		
		$row = array(
			'index' => $index,													// 項目番号
			'serial' => $this->convertToDispString($id),			// シリアル番号
			'id' => $this->convertToDispString($id),			// ID
			'name' => $this->convertToDispString($fetchedRow['mn_name']),		// 名前
			'check' => $checkStr		// チェック状態
		);
		$this->tmpl->addVars('itemlist', $row);
		$this->tmpl->parseTemplate('itemlist', 'a');
		
		// 表示中項目IDを保存
		$this->serialArray[] = $id;
		$this->isExistsContent = true;		// コンテンツ項目が存在するかどうか
		return true;
	}
	/**
	 * キャッシュデータをクリア
	 *
	 * @param int $serial		削除対象のコンテンツシリアル番号
	 * @return					なし
	 */
	function clearCacheBySerial($serial)
	{
		$ret = self::$_mainDb->getContentBySerial($serial, $row);		// コンテンツID取得
		if ($ret){
			$contentId = $row['cn_id'];		// コンテンツID
			$urlParam = array();
			$urlParam[] = M3_REQUEST_PARAM_CONTENT_ID . '=' . $contentId;		// コンテンツID
			$urlParam[] = M3_REQUEST_PARAM_CONTENT_ID_SHORT . '=' . $contentId;		// コンテンツID略式
			$this->clearCache($urlParam);
		}
	}
	/**
	 * 期間から公開可能かチェック
	 *
	 * @param timestamp	$startDt		公開開始日時
	 * @param timestamp	$endDt			公開終了日時
	 * @param timestamp	$now			基準日時
	 * @return bool						true=公開可能、false=公開不可
	 */
	function isActive($startDt, $endDt, $now)
	{
		$isActive = false;		// 公開状態

		if ($startDt == $this->gEnv->getInitValueOfTimestamp() && $endDt == $this->gEnv->getInitValueOfTimestamp()){
			$isActive = true;		// 公開状態
		} else if ($startDt == $this->gEnv->getInitValueOfTimestamp()){
			if (strtotime($now) < strtotime($endDt)) $isActive = true;		// 公開状態
		} else if ($endDt == $this->gEnv->getInitValueOfTimestamp()){
			if (strtotime($now) >= strtotime($startDt)) $isActive = true;		// 公開状態
		} else {
			if (strtotime($startDt) <= strtotime($now) && strtotime($now) < strtotime($endDt)) $isActive = true;		// 公開状態
		}
		return $isActive;
	}
	/**
	 * コンテンツ更新情報をサーバへ登録
	 *
	 * @param int $serial		コンテンツシリアル番号
	 * @param bool $isNew		新規または更新
	 * @return					なし
	 */
	function registContentInfoBySerial($serial, $isNew=false)
	{
		$ret = self::$_mainDb->getContentBySerial($serial, $row);		// コンテンツID取得
		if ($ret){
			$contentId = $row['cn_id'];		// コンテンツID
			$contentTitle = $row['cn_name'];		// コンテンツタイトル
			$contentLink = $url = $this->gEnv->getDefaultUrl() . '?contentid=' . $contentId;		// コンテンツリンク先
			$contentDt = $row['cn_create_dt'];		// コンテンツ更新日時
			
			$ret = $this->gInstance->getConnectManager()->registUpdateInfo(''/*汎用コンテンツ*/, ''/*送信先はデフォルトURL*/, ''/*デフォルトウィジェットに送信*/, 
								$this->gEnv->getCurrentWidgetId(), self::MSG_UPDATE_CONTENT, $contentTitle, $contentLink, $contentDt);
		}
		return $ret;
	}
	/**
	 * 言語アイコンを作成
	 *
	 * @param int $contentId	コンテンツID
	 * @return atring			言語アイコンタグ
	 */
	function createLangImage($contentId)
	{
		$imageTag = '';
		$ret = self::$_mainDb->getLangByContentId(default_contentCommonDef::$_contentType, $contentId, $rows);
		if ($ret){
			$count = count($rows);
			for ($i = 0; $i < $count; $i++){
				if ($this->gEnv->getCurrentLanguage() == 'ja'){	// 日本語の場合
					$langName = $rows[$i]['ln_name'];
				} else {
					$langName = $rows[$i]['ln_name_en'];
				}
				// 言語アイコン
				$iconTitle = $langName;
				$iconUrl = $this->gEnv->getRootUrl() . self::LANG_ICON_PATH . $rows[$i]['ln_image_filename'];		// 画像ファイル
				$iconTag = '<img src="' . $this->getUrl($iconUrl) . '" border="0" alt="' . $iconTitle . '" title="' . $iconTitle . '" />';
				$imageTag .= $iconTag;
			}
		}
		return $imageTag;
	}
	/**
	 * 言語表示名を取得
	 *
	 * @param string $langId	言語ID
	 * @return atring			表示名
	 */
	function getLangName($langId)
	{
		$ret = self::$_mainDb->getLang($langId, $row);
		if ($this->gEnv->getCurrentLanguage() == 'ja'){		// 日本語表示の場合
			$name = $this->convertToDispString($row['ln_name']);
		} else {
			$name = $this->convertToDispString($row['ln_name_en']);
		}
		return $name;
	}
}
?>
