<?php
/**
 * 画像処理マネージャー
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2012 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: imageManager.php 5117 2012-08-18 09:00:10Z fishbone $
 * @link       http://www.magic3.org
 */
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/core.php');

class ImageManager extends Core
{
	private $db;					// DBオブジェクト
	const CONTENT_DIR = '/etc/';
	const THUMBNAIL_DIR = '/thumb';
	const SITE_LOGO_DIR = '/etc/site/thumb/';		// サイトロゴ格納ディレクトリ
	const DEFAULT_THUMBNAIL_EXT = 'png';
	const DEFAULT_THUMBNAIL_SIZE = 72;
	const DEFAULT_THUMBNAIL_TYPE = '72c';
	const CF_SITE_LOGO_FILENAME = 'site_logo_filename';		// サイトロゴファイル
	const CF_THUMB_FORMAT = 'thumb_format';		// サムネールフォーマット
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		global $gInstanceManager;
		
		// 親クラスを呼び出す
		parent::__construct();
		
		// システムDBオブジェクト取得
	//	$this->db = $gInstanceManager->getSytemDbObject();
	}
	/**
	 * システム共通のデフォルトのサムネールを作成
	 *
	 * @param string $contentType	コンテンツタイプ
	 * @param int    $deviceType	デバイスタイプ(0=PC、1=携帯、2=スマートフォン)
	 * @param string $contentId		コンテンツID
	 * @param string $path			元の画像ファイル
	 * @param array  $destFilename	作成した画像のファイル名
	 * @param string $destDir		作成した画像のディレクトリ
	 * @param string $destDirUrl	作成した画像のディレクトリURL
	 * @return bool					true=作成,false=作成失敗
	 */
	function createSystemDefaultThumb($contentType, $deviceType, $contentId, $path, &$destFilename, &$destDir = NULL, &$destDirUrl = NULL)
	{
		global $gSystemManager;

		// パラメータエラーチェック
		if (strlen($contentId) == 0) return false;
		
		$destFilename = array();		// 画像ファイル名
		
		// 画像格納用のディレクトリ作成
		$thumbDir = $this->getSystemThumbPath($contentType, $deviceType);
		if (!file_exists($thumbDir)) mkdir($thumbDir, M3_SYSTEM_DIR_PERMISSION, true/*再帰的*/);
		if (!is_null($destDir)) $destDir = $thumbDir;
		
		// 画像ディレクトリのURL
		if (!is_null($destDirUrl)) $destDirUrl = $this->getSystemThumbUrl($contentType, $deviceType);

		// 画像のフォーマットを取得
		$formatArray = explode(';', $gSystemManager->getSystemConfig(self::CF_THUMB_FORMAT));
		
		for ($i = 0; $i < count($formatArray); $i++){
			$format = trim($formatArray[$i]);
			$ret = preg_match('/(\d+)(.*)\.(gif|png|jpg|jpeg|bmp)$/i', $format, $matches);
			if ($ret){
				$thumbSize = $matches[1];
				$thumbAttr = strtolower($matches[2]);
				$ext = strtolower($matches[3]);

				$imageType = IMAGETYPE_JPEG;
				switch ($ext){
					case 'jpg':
					case 'jpeg':
						$imageType = IMAGETYPE_JPEG;
						break;
					case 'gif':
						$imageType = IMAGETYPE_GIF;
						break;
					case 'png':
						$imageType = IMAGETYPE_PNG;
						break;
					case 'bmp':
						$imageType = IMAGETYPE_BMP;
						break;
				}
				$thumbFilename = $contentId . '_' . $format;
				$thumbPath = $thumbDir . '/' . $thumbFilename;

				// サムネールの作成
				if ($thumbAttr == 'c'){		// 切り取りサムネールの場合
					$ret = $this->gInstance->getImageManager()->createThumb($path, $thumbPath, $thumbSize, $imageType, true);
				} else {
					$ret = $this->gInstance->getImageManager()->createThumb($path, $thumbPath, $thumbSize, $imageType, false);
				}
				if ($ret){
					$destFilename[] = $thumbFilename;
				} else {
					return false;
				}
			}
		}
		return true;
	}
	/**
	 * システム共通のデフォルトのサムネールを削除
	 *
	 * @param string $contentType	コンテンツタイプ
	 * @param int    $deviceType	デバイスタイプ(0=PC、1=携帯、2=スマートフォン)
	 * @param array  $filename		ファイル名
	 * @return bool					true=成功,false=失敗
	 */
	function delSystemDefaultThumb($contentType, $deviceType, $filename)
	{
		global $gSystemManager;
		
		// 画像格納用のディレクトリ取得
		$thumbDir = $this->getSystemThumbPath($contentType, $deviceType);
		
		$ret = true;
		for ($i = 0; $i < count($filename); $i++){
			$thumbnailPath = $thumbDir . '/' . $filename[$i];
			if (!@unlink($thumbnailPath)) $ret = false;
		}
		return $ret;
	}
	/**
	 * システム共通のサムネール画像のURLを取得
	 *
	 * @param string     $contentType	コンテンツタイプ
	 * @param int        $deviceType	デバイスタイプ(0=PC、1=携帯、2=スマートフォン)
	 * @param string     $filename		ファイル名(空のときは格納ディレクトリを返す)
	 * @return string					画像URL
	 */
	function getSystemThumbUrl($contentType, $deviceType, $filename = '')
	{
		global $gEnvManager;
				
		switch ($deviceType){
			case 0:		// PC
			default:
				$deviceDir = '';
				break;
			case 1:		// 携帯
				$deviceDir = '/m';
				break;
			case 2:		// スマートフォン
				$deviceDir = '/s';
				break;
		}
		
		$destUrl = $gEnvManager->getResourceUrl() . self::CONTENT_DIR . $contentType . $deviceDir . self::THUMBNAIL_DIR;		// 画像格納用のディレクトリ
		if (empty($filename)){
			return $destUrl;
		} else {
			return $destUrl . '/' . $filename;
		}
	}
	/**
	 * システム共通のサムネール画像のパスを取得
	 *
	 * @param string     $contentType	コンテンツタイプ
	 * @param int        $deviceType	デバイスタイプ(0=PC、1=携帯、2=スマートフォン)
	 * @param string     $filename		ファイル名(空のときは格納ディレクトリを返す)
	 * @return string					画像パス
	 */
	function getSystemThumbPath($contentType, $deviceType, $filename = '')
	{
		global $gEnvManager;
				
		switch ($deviceType){
			case 0:		// PC
			default:
				$deviceDir = '';
				break;
			case 1:		// 携帯
				$deviceDir = '/m';
				break;
			case 2:		// スマートフォン
				$deviceDir = '/s';
				break;
		}
		
		$destDir = $gEnvManager->getResourcePath() . self::CONTENT_DIR . $contentType . $deviceDir . self::THUMBNAIL_DIR;		// 画像格納用のディレクトリ
		if (empty($filename)){
			return $destDir;
		} else {
			return $destDir . '/' . $filename;
		}
	}
	/**
	 * デフォルトのサムネールを作成
	 *
	 * @param string $contentType	コンテンツタイプ
	 * @param string $contentId		コンテンツID
	 * @param string $path			元の画像ファイル
	 * @return bool				true=作成,false=作成失敗
	 */
	function createDefaultThumb($contentType, $contentId, $path)
	{
		global $gEnvManager;
		
		// サムネール画像のパス
		$destPath = $gEnvManager->getResourcePath() . self::CONTENT_DIR . $contentType . self::THUMBNAIL_DIR . '/' . $contentId . '_' . self::DEFAULT_THUMBNAIL_TYPE . '.' . self::DEFAULT_THUMBNAIL_EXT;

		// 画像格納用のディレクトリ作成
		$destDir = dirname($destPath);
		if (!file_exists($destDir)) mkdir($destDir, M3_SYSTEM_DIR_PERMISSION, true/*再帰的*/);
					
		$ret = $this->createThumb($path, $destPath, self::DEFAULT_THUMBNAIL_SIZE, IMAGETYPE_PNG);
		return $ret;
	}
	/**
	 * サムネール画像のURLを取得
	 *
	 * @param string $contentType	コンテンツタイプ
	 * @param string $contentId		コンテンツID
	 * @return string				画像URL
	 */
	function getDefaultThumbUrl($contentType, $contentId)
	{
		global $gEnvManager;
		
		return $gEnvManager->getResourceUrl() . self::CONTENT_DIR . $contentType . self::THUMBNAIL_DIR . '/' . $contentId . '_' . self::DEFAULT_THUMBNAIL_TYPE . '.' . self::DEFAULT_THUMBNAIL_EXT;
	}
	/**
	 * サムネール画像のファイルパスを取得
	 *
	 * @param string $contentType	コンテンツタイプ
	 * @param string $contentId		コンテンツID
	 * @return string				画像パス
	 */
	function getDefaultThumbPath($contentType, $contentId)
	{
		global $gEnvManager;
		
		return $gEnvManager->getResourcePath() . self::CONTENT_DIR . $contentType . self::THUMBNAIL_DIR . '/' . $contentId . '_' . self::DEFAULT_THUMBNAIL_TYPE . '.' . self::DEFAULT_THUMBNAIL_EXT;
	}
	/**
	 * サムネールを作成
	 *
	 * @param string $path		元の画像ファイル
	 * @param string $destPath	サムネールファイルパス
	 * @param int $size			サムネールの縦横サイズ
	 * @param int $type			サムネールの画像タイプ
	 * @param bool $useCrop		画像切り取りありかどうか
	 * @param string $bgColor	背景色。設定しない場合は空文字列
	 * @param int $imageQuality	画像品質
	 * @return bool				true=作成,false=作成失敗
	 */
	function createThumb($path, $destPath, $size, $type, $useCrop = true, $bgColor = '#FFFFFF', $imageQuality = 100)
	{
		$imageSize = @getimagesize($path);
		if ($imageSize){
			$imageType = $imageSize[2];
		} else {
			return false;
		}
		
		// 画像作成
		switch ($imageType){
			case IMAGETYPE_GIF:
				$imageObj = @imagecreatefromgif($path);
				break;
			case IMAGETYPE_JPEG:
				$imageObj = @imagecreatefromjpeg($path);
				break;
			case IMAGETYPE_PNG:
				$imageObj = @imagecreatefrompng($path);
				break;
			default:
				return false;
		}

		// 画像サイズ取得
		$x = 0;
		$y = 0;
		$newX = 0;
		$newY = 0;
		$width = imagesx($imageObj);
		$height = imagesy($imageObj);
		if ($useCrop){		// 画像切り取りありのとき
			if ($width > $height){
				$newHeight	= $size;
				$newWidth	= round($size * $width / $height);

				$x = ceil(($width - $height) / 2);
				$y = 0;
			} else {
				$newWidth	= $size;
				$newHeight	= round($size * $height / $width);

				$x = 0;
				$y = ceil(($height - $width) / 2);
			}
		} else {
			if ($width > $height){
				$newHeight = $height * ($size / $width);
				$newWidth = $size;
			} else {
				$newWidth = $width * ($size / $height);
				$newHeight = $size;
			}
		
			$newX = 0;
			$newY = 0;
			if ($newWidth < $size) $newX = round(($size - $newWidth) / 2);
			if ($newHeight < $size) $newY = round(($size - $newHeight) / 2);
		}

		// TrueColorイメージを作成
		$thumbObj = imagecreatetruecolor($size, $size);
		
		// 背景色の設定		
		if (!$useCrop){			// 切り取りなしのとき
			// サムネールの背景色を取得
			$bgColorR = intval(substr($bgColor, 1, 2), 16);
			$bgColorG = intval(substr($bgColor, 3, 2), 16);
			$bgColorB = intval(substr($bgColor, 5, 2), 16);

			// TrueColorイメージを作成
		//	$thumbObj = imagecreatetruecolor($size, $size);
			$bgcolor = imagecolorallocate($thumbObj, $bgColorR, $bgColorG, $bgColorB);		// 背景色設定
			imagefill($thumbObj, 0, 0, $bgcolor);
		}
		
		// 画像リサイズ
		// imagecopyresampledの方がimagecopyresizedよりも画質が良いのでこちらを使用
		if (function_exists("imagecopyresampled")){
			if (!imagecopyresampled($thumbObj, $imageObj, $newX, $newY, $x, $y, $newWidth, $newHeight, $width, $height)){
				if (!imagecopyresized($thumbObj, $imageObj, $newX, $newY, $x, $y, $newWidth, $newHeight, $width, $height)) return false;
			}
		} else {
			if (!imagecopyresized($thumbObj, $imageObj, $newX, $newY, $x, $y, $newWidth, $newHeight, $width, $height)) return false;
		}
		
		// 画像出力
		switch ($type){
			case IMAGETYPE_GIF:
				$ret = @imagegif($thumbObj, $destPath, $imageQuality);
				break;
			case IMAGETYPE_JPEG:
				$ret = @imagejpeg($thumbObj, $destPath, $imageQuality);
				break;
			case IMAGETYPE_PNG:
				//$ret = @imagepng($thumbObj, $destPath, $imageQuality);		// PNGでは$imageQualityを使用すると画像が0サイズで作成される
				$ret = @imagepng($thumbObj, $destPath);
				break;
		}
		// イメージを破棄
		$ret = imagedestroy($imageObj);
		$ret = imagedestroy($thumbObj);
		return $ret;
	}
	/**
	 * サイトロゴ画像のURLを取得
	 *
	 * @return string				画像URL
	 */
	function getSiteLogoUrl()
	{
		global $gEnvManager;
		global $gSystemManager;
		
		$siteLogoFiles = explode(';', $gSystemManager->getSystemConfig(self::CF_SITE_LOGO_FILENAME));		// サイトロゴファイル名
		$path = $gEnvManager->getResourcePath() . self::SITE_LOGO_DIR . $siteLogoFiles[count($siteLogoFiles) -1];
		$url = '';
		if (is_readable($path)) $url = $gEnvManager->getResourceUrl() . self::SITE_LOGO_DIR . $siteLogoFiles[count($siteLogoFiles) -1];
		return $url;
	}
	/**
	 * コンテンツから先頭の画像のパスを取得
	 *
	 * @param string $html		検索コンテンツ
	 * @return string			画像パス(取得できない場合は空文字列)
	 */
	function getFirstImagePath($html)
	{
		global $gEnvManager;
		
		$exp = '/<img[^<]*?src\s*=\s*[\'"]+(.+?)[\'"]+[^>]*?>/si';
		$ret = preg_match($exp, $html, $matches);
		if ($ret){
			$path = $gEnvManager->getMacroPath($matches[1]);		// 画像URL
			$path = str_replace(M3_TAG_START . M3_TAG_MACRO_ROOT_URL . M3_TAG_END, $gEnvManager->getSystemRootPath(), $path);// マクロ変換
			if (is_readable($path)) return $path;
		}
		return '';
	}
}
?>
