package mandelbrotExplorer;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class MandelbrotImage {
	private static final long serialVersionUID = 1L;
	private MandelbrotSet mandel;
	private Rectangle viewRect;
	protected BufferedImage image;
	private Component imageObserver;
	private ExecutorService drawingExe;
	private ConcurrentLinkedQueue<MandelbrotDrawing> mds;
	
	public MandelbrotImage(Component imageObserver) {
		this.imageObserver = imageObserver;

		this.drawingExe = Executors.newCachedThreadPool();
		this.mds = new ConcurrentLinkedQueue<MandelbrotDrawing>();
	}

	public int getWidth(){
		return image.getWidth();
	}

	public int getHeight(){
		return image.getHeight();
	}

	public boolean imageIsNull(){
		if (image == null) { 
			return true;
		} else {
			return false;
		}
	}
	
	public void createNewImage(int w,int h,double rLeft,double rRight,double iTop,double iBottom){
		image = new BufferedImage(w,h, BufferedImage.TYPE_INT_BGR);
		mandel = new MandelbrotSet( w , h , rLeft, rRight, iTop, iBottom);
		viewRect = new Rectangle(0,0,w,h);
		
	}
	
	public void drawAll(){
		if( this.imageIsNull() || mandel==null ){ return; }
		for(MandelbrotDrawing md:mds){
			md.stopDrawing();
		}
		this.mds = new ConcurrentLinkedQueue<MandelbrotDrawing>();
		this.addDrawingThread(mandel , viewRect );
	}
	
	public String toString(){
		double rLeft = mandel.getLeft()+viewRect.getX()*mandel.getScaleR();
		double rRight= mandel.getLeft()+(viewRect.getX()+viewRect.getWidth())*mandel.getScaleR();
		double iTop  = mandel.getTop()+viewRect.getY()*mandel.getScaleI();
		double iBottom=mandel.getTop()+(viewRect.getY()+viewRect.getHeight())*mandel.getScaleI();

		return String.format("}fu[W ͈́i%3.10g`%3.10g) ",rLeft,rRight)
		+String.format("͈́i%3.10g`%3.10g)",iBottom,iTop);
	}

	///////////////// gAk
	public void changeMagnification(int z){
		
		double zoomRate = 1.5;
		Point2D.Double center = new Point2D.Double(
					(mandel.getLeft()+viewRect.getX()*mandel.getScaleR())+(double)getWidth()/2.0*mandel.getScaleR()
					,(mandel.getTop()+viewRect.getY()*mandel.getScaleI())+(double)getHeight()/2.0*mandel.getScaleI());
		Point2D.Double leftTop;
		if( z > 0){	// g
			leftTop = new Point2D.Double(
					center.getX()-((double)getWidth()*mandel.getScaleR()/zoomRate)/2.0
					,center.getY()-((double)getHeight()*mandel.getScaleI()/zoomRate)/2.0  );
			mandel = new MandelbrotSet(leftTop
					,mandel.getScaleR()/zoomRate,mandel.getScaleI()/zoomRate);
			viewRect.setBounds(0, 0, image.getWidth() ,image.getHeight() );
			//  gimage\
			synchronized (image) {
				BufferedImage bi = new BufferedImage( (int)((double)getWidth()*zoomRate),(int)((double)getHeight()*zoomRate), BufferedImage.TYPE_INT_BGR);
				AffineTransformOp ato = new AffineTransformOp(
						AffineTransform.getScaleInstance(zoomRate,zoomRate)
						,null);
				ato.filter(image,bi);
				Graphics2D g = image.createGraphics();
				g.setColor(new Color(0,0,0));
				g.fillRect(0, 0, image.getWidth(), image.getHeight());
				g.drawImage(bi, null, -(bi.getWidth()/2-image.getWidth()/2), -(bi.getHeight()/2-image.getHeight()/2));
			}
			this.imageObserver.repaint();
		}else{		// k
			leftTop = new Point2D.Double(
					center.getX()-((double)getWidth()*mandel.getScaleR()*zoomRate)/2.0
					,center.getY()-((double)getHeight()*mandel.getScaleI()*zoomRate)/2.0  );
			mandel = new MandelbrotSet(leftTop
					,mandel.getScaleR()*zoomRate,mandel.getScaleI()*zoomRate);
			viewRect.setBounds(0, 0, image.getWidth() ,image.getHeight() );
			//	kimage\
			BufferedImage bi = new BufferedImage((int)(getWidth()/zoomRate),(int)(getHeight()/zoomRate), BufferedImage.TYPE_INT_BGR);
			AffineTransformOp ato = new AffineTransformOp(
					AffineTransform.getScaleInstance(1.0/zoomRate,1.0/zoomRate)
					,null);
			synchronized (image) {
				ato.filter(image,bi);
				Graphics2D g = image.createGraphics();
				g.setColor(new Color(0,0,0));
				g.fillRect(0, 0, image.getWidth(), image.getHeight());
				g.drawImage(bi, null, image.getWidth()/2-bi.getWidth()/2, image.getHeight()/2-bi.getHeight()/2);
			}
			this.imageObserver.repaint();
		}
		this.drawAll();
	}
	
	////////////////////////// ړ
	public void translate(Point distance){
		this.translate( distance.x , distance.y );
	}

	public void translate(int dx,int dy){

		dx = -dx;
		dy = -dy;
		Rectangle oRect = new Rectangle(viewRect);
		viewRect.translate(dx, dy);

		synchronized (image) {
			// imagẻ摜炷AhbOł炵炷A
			// hbOɂČ̂́A݂AŎۂɂ炷B
			BufferedImage bc = new BufferedImage(image.getWidth(), image
					.getHeight(), BufferedImage.TYPE_INT_BGR);
			bc.createGraphics().drawImage(image, null, -dx, -dy);
			image.createGraphics().drawImage(bc, null, 0, 0);
		}		
		imageObserver.repaint();

		// ړʐVɌ̈ɑ΂addDrawingThread()Ă`threadJn
		Rectangle nr= new Rectangle();
		if (dx >= 0 && dy >= 0 ) {
			nr.setBounds(viewRect.x, viewRect.y+viewRect.height-dy, viewRect.width, dy);
			this.addDrawingThread(mandel , nr);
			
			nr.setBounds(oRect.x+oRect.width, viewRect.y, dx, viewRect.height-dy);
			this.addDrawingThread(mandel , nr);
		}
		if (dx >= 0 && dy < 0 ) {
			nr.setBounds(viewRect.x, viewRect.y, viewRect.width, -dy);
			this.addDrawingThread(mandel , nr);
			
			nr.setBounds(viewRect.x+viewRect.width-dx, oRect.y, dx, viewRect.height+dy);
			this.addDrawingThread(mandel , nr);
		}
		if (dx < 0 && dy >= 0 ) {
			nr.setBounds(viewRect.x, viewRect.y, -dx, viewRect.height);
			this.addDrawingThread(mandel , nr);
			
			nr.setBounds(oRect.x, viewRect.y+viewRect.height-dy, viewRect.width+dx, dy);
			this.addDrawingThread(mandel , nr);
		}
		if (dx < 0 && dy < 0 ) {
			nr.setBounds(viewRect.x, viewRect.y, -dx, viewRect.height);
			this.addDrawingThread(mandel , nr);
			
			nr.setBounds(oRect.x, viewRect.y, viewRect.width+dx, -dy);
			this.addDrawingThread(mandel , nr);
		}
	}

	//////////////////////////resize
	public void setSize(int w,int h){
		
		Rectangle oRect = new Rectangle(viewRect);
		viewRect.setSize(w, h);
		
		//***************ÂimagêĂĐVTCYimage
		synchronized (image) {
			BufferedImage bi = new BufferedImage(image.getWidth(),image.getHeight(), BufferedImage.TYPE_INT_BGR); 
			bi.createGraphics().drawImage(image, null, 0, 0);
			image = new BufferedImage(w, h, BufferedImage.TYPE_INT_BGR);
			image.createGraphics().drawImage(bi,null, 0, 0 );
		}
		imageObserver.repaint();

		// resizeʐVɌ̈ɑ΂addDrawingThread()Ă`threadJn
		Rectangle nr= new Rectangle();
		if( oRect.width < viewRect.width ){
			nr.setBounds(viewRect.x+oRect.width,viewRect.y , viewRect.width-oRect.width, viewRect.height);
			this.addDrawingThread(mandel , nr);
		}
		if( oRect.height < viewRect.height ){
			if( oRect.width < viewRect.width ){
				nr.setBounds(viewRect.x, viewRect.y+oRect.height, oRect.width, viewRect.height-oRect.height);
			}else{
				nr.setBounds(viewRect.x, viewRect.y+oRect.height, viewRect.width, viewRect.height-oRect.height);
			}
			this.addDrawingThread(mandel , nr);
		}
	}

	////////////////////////// `Thread֌W
	private void addDrawingThread(final MandelbrotSet ms ,final Rectangle rect){
		MandelbrotDrawing mandelbrotDrawing = new MandelbrotDrawing(ms, rect, this);
		this.drawingExe.submit( mandelbrotDrawing );
		mds.add(mandelbrotDrawing);
	}
	
	protected void paintImagePiece(ImagePiece ip){ // MandelbrotDrawingThreadĂяo
		System.out.println("paintImagePiece "+ip);
		if( ip!=null ){
			synchronized (image) {
				image.createGraphics().drawImage(ip.getImage(), null,
						ip.getRect().x-viewRect.x, ip.getRect().y-viewRect.y);
			}
			imageObserver.repaint();
		}
		
		for(MandelbrotDrawing md:mds){
			if( md.isDone() ){ mds.remove(md); }
		}
	}

	@Override
	protected void finalize() throws Throwable {
		this.drawingExe.shutdown();
		super.finalize();
	}
}
