package mandelbrotExplorer;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.util.ArrayList;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.Future;

public class MandelbrotImage {
	private MandelbrotSet mandel;
	private Rectangle viewRect;
	private BufferedImage image;
	private BufferedImage smallImage;
	private Component imageObserver;
	private MandelbrotThreadExecutor drawingExe;
	private CompletionService<ImagePiece> cs;
	private ArrayList<DrawingTask> dts;

	public MandelbrotImage(Component imageObserver) {
		this.imageObserver = imageObserver;

		this.drawingExe = new MandelbrotThreadExecutor(this);
		this.cs = new ExecutorCompletionService<ImagePiece>(drawingExe);
		this.dts =new ArrayList<DrawingTask>();
	}

	public int getWidth(){
		return image.getWidth();
	}

	public int getHeight(){
		return image.getHeight();
	}

	public void createNewImage(int w,int h,MandelbrotSet mandelbrotSet){
		image = new BufferedImage(w,h, BufferedImage.TYPE_INT_BGR);
		mandel = mandelbrotSet;
		viewRect = new Rectangle(0,0,w,h);
	}

	public void createNewImage(int w,int h,Rectangle2D.Double region){
		image = new BufferedImage(w,h, BufferedImage.TYPE_INT_BGR);
		mandel = new MandelbrotSet( w , h , region);
		viewRect = new Rectangle(0,0,w,h);

	}

	public void createNewImage(int w,int h,Rectangle2D.Double region,double aspectRatio){
		image = new BufferedImage(w,h, BufferedImage.TYPE_INT_BGR);
		mandel = new MandelbrotSet( w , h , region , aspectRatio);
		viewRect = new Rectangle(0,0,w,h);

	}

	public void drawAll(){
		if( this.imageIsNull() || mandel==null ){ return; }

		drawingExe.addTaskCount(1);
		
		synchronized (dts) {
			// ݎsDrawingTaskAׂĒ~
			for (int i = 0; i < dts.size(); ++i) {
				if( dts.get(i).getState()!=DrawingTask.TERMINATED ){
					dts.get(i).stop();
				}
			}
		}		
		this.addDrawingTask(mandel , viewRect );
	}

	public MandelbrotSet getMandelbrotSet(){
		MandelbrotSet m = new MandelbrotSet( 
				new Point2D.Double(this.getRegion().x,this.getRegion().y) ,mandel.getScaleR(), mandel.getScaleI()
		);
		if( smallImage!=null ){
			m.setSmallImage(smallImage);
		}
		return m;
	}

	public Rectangle2D.Double getRegion(){
		if( mandel==null ){return null; }
		double rLeft = mandel.getLeft()+viewRect.getX()*mandel.getScaleR();
		double iTop  = mandel.getTop()+viewRect.getY()*mandel.getScaleI();
		double rWidth= viewRect.getWidth()*mandel.getScaleR();
		double iHeight=viewRect.getHeight()*mandel.getScaleI();
		return new Rectangle2D.Double(rLeft,iTop,rWidth,iHeight);
	}

	public void drawGraphics(Graphics g,Point p,Component imageObserver){
		g.drawImage(image , p.x, p.y, imageObserver);
	}

	public boolean imageIsNull(){
		if (image == null) { 
			return true;
		} else {
			return false;
		}
	}

	public String toString(){
		double rLeft = mandel.getLeft()+viewRect.getX()*mandel.getScaleR();
		double rRight= mandel.getLeft()+(viewRect.getX()+viewRect.getWidth())*mandel.getScaleR();
		double iTop  = mandel.getTop()+viewRect.getY()*mandel.getScaleI();
		double iBottom=mandel.getTop()+(viewRect.getY()+viewRect.getHeight())*mandel.getScaleI();

		return String.format("}fu[W ͈́i%3.10g`%3.10g) ",rLeft,rRight)
		+String.format("͈́i%3.10g`%3.10g)",iBottom,iTop);
	}

	///////////////// gAk
	public void changeMagnification(int z){

		double zoomRate = 1.5;
		Point2D.Double center = new Point2D.Double(
				(mandel.getLeft()+viewRect.getX()*mandel.getScaleR())+(double)getWidth()/2.0*mandel.getScaleR()
				,(mandel.getTop()+viewRect.getY()*mandel.getScaleI())+(double)getHeight()/2.0*mandel.getScaleI());
		Point2D.Double leftTop;
		if( z > 0){	// g
			leftTop = new Point2D.Double(
					center.getX()-((double)getWidth()*mandel.getScaleR()/zoomRate)/2.0
					,center.getY()-((double)getHeight()*mandel.getScaleI()/zoomRate)/2.0  );
			mandel = new MandelbrotSet(leftTop
					,mandel.getScaleR()/zoomRate,mandel.getScaleI()/zoomRate);
			synchronized (viewRect) {
				viewRect.setBounds(0, 0, image.getWidth(), image.getHeight());
				//  gimage\
				synchronized (image) {
					BufferedImage bi = new BufferedImage(
							(int) ((double) getWidth() * zoomRate),
							(int) ((double) getHeight() * zoomRate),
							BufferedImage.TYPE_INT_BGR);
					AffineTransformOp ato = new AffineTransformOp(
							AffineTransform
									.getScaleInstance(zoomRate, zoomRate), null);
					ato.filter(image, bi);
					Graphics2D g = image.createGraphics();
					g.setColor(new Color(0, 0, 0));
					g.fillRect(0, 0, image.getWidth(), image.getHeight());
					g.drawImage(bi, null, -(bi.getWidth() / 2 - image
							.getWidth() / 2), -(bi.getHeight() / 2 - image
							.getHeight() / 2));
				}
			}			
			this.imageObserver.repaint();
		}else{		// k
			leftTop = new Point2D.Double(
					center.getX()-((double)getWidth()*mandel.getScaleR()*zoomRate)/2.0
					,center.getY()-((double)getHeight()*mandel.getScaleI()*zoomRate)/2.0  );
			mandel = new MandelbrotSet(leftTop
					,mandel.getScaleR()*zoomRate,mandel.getScaleI()*zoomRate);
			synchronized (viewRect) {
				viewRect.setBounds(0, 0, image.getWidth(), image.getHeight());
				//	kimage\
				BufferedImage bi = new BufferedImage(
						(int) (getWidth() / zoomRate),
						(int) (getHeight() / zoomRate),
						BufferedImage.TYPE_INT_BGR);
				AffineTransformOp ato = new AffineTransformOp(AffineTransform
						.getScaleInstance(1.0 / zoomRate, 1.0 / zoomRate), null);
				synchronized (image) {
					ato.filter(image, bi);
					Graphics2D g = image.createGraphics();
					g.setColor(new Color(0, 0, 0));
					g.fillRect(0, 0, image.getWidth(), image.getHeight());
					g.drawImage(bi, null, image.getWidth() / 2 - bi.getWidth()
							/ 2, image.getHeight() / 2 - bi.getHeight() / 2);
				}
			}			
			this.imageObserver.repaint();
		}
		this.drawAll();
	}

	//////////////////////////resize
	public void setSize(int w,int h){

		Rectangle oRect;
		synchronized (viewRect) {
			oRect = new Rectangle(viewRect);
			viewRect.setSize(w, h);
			//***************ÂimagêĂĐVTCYimage
			synchronized (image) {
				BufferedImage bi = new BufferedImage(image.getWidth(), image
						.getHeight(), BufferedImage.TYPE_INT_BGR);
				bi.getRaster().setRect(image.getRaster());
				image = new BufferedImage(w, h, BufferedImage.TYPE_INT_BGR);
				image.getRaster().setRect(0, 0, bi.getRaster());
			}
		}		
		imageObserver.repaint();

		// resizeʐVɌ̈ɑ΂addDrawingTask()Ă`threadJn
		Rectangle nr= new Rectangle();
		if( oRect.width < viewRect.width ){
			nr.setBounds(viewRect.x+oRect.width,viewRect.y , viewRect.width-oRect.width, viewRect.height);
			drawingExe.addTaskCount(1);
			this.addDrawingTask(mandel , nr);
		}
		if( oRect.height < viewRect.height ){
			if( oRect.width < viewRect.width ){
				nr.setBounds(viewRect.x, viewRect.y+oRect.height, oRect.width, viewRect.height-oRect.height);
			}else{
				nr.setBounds(viewRect.x, viewRect.y+oRect.height, viewRect.width, viewRect.height-oRect.height);
			}
			drawingExe.addTaskCount(1);
			this.addDrawingTask(mandel , nr);
		}
	}

	////////////////////////// ړ
	public void translate(Point distance){
		this.translate( distance.x , distance.y );
	}

	public void translate(int dx,int dy){
		dx = -dx;
		dy = -dy;
		Rectangle oRect = new Rectangle(viewRect);
		//DebugThread.printf("ړCxgsynchronized\n");
		synchronized (viewRect) {
			viewRect.translate(dx, dy);
			//DebugThread.printf("viewRectړ:[%d,%d]->[%d,%d]\n", oRect.x,oRect.y, viewRect.x, viewRect.y);
			drawingExe.addTaskCount(2);
			// imagẻ摜炷AhbOł炵炷A
			// hbOɂČ̂́A݂AŎۂɂ炷B
			//DebugThread.printf("imageړJn\n");
			BufferedImage bc = new BufferedImage(image.getWidth(), image.getHeight(), BufferedImage.TYPE_INT_BGR);
			WritableRaster raster = bc.getRaster();
			raster.setRect(image.getRaster());
			image.createGraphics().clearRect(0, 0, image.getWidth(), image.getHeight());
			image.getRaster().setRect(-dx, -dy, raster);
			//DebugThread.printf("imageړI\n");
		}		

		// ړʐVɌ̈ɑ΂addDrawingTask()ĕ`threadJn
		Rectangle nr= new Rectangle();
		if (dx >= 0 && dy >= 0 ) {
			nr.setBounds(viewRect.x, viewRect.y+viewRect.height-dy, viewRect.width, dy);
			this.addDrawingTask(mandel , nr);

			nr.setBounds(oRect.x+oRect.width, viewRect.y, dx, viewRect.height-dy);
			this.addDrawingTask(mandel , nr);
		}
		if (dx >= 0 && dy < 0 ) {
			nr.setBounds(viewRect.x, viewRect.y, viewRect.width, -dy);
			this.addDrawingTask(mandel , nr);

			nr.setBounds(viewRect.x+viewRect.width-dx, oRect.y, dx, viewRect.height+dy);
			this.addDrawingTask(mandel , nr);
		}
		if (dx < 0 && dy >= 0 ) {
			nr.setBounds(viewRect.x, viewRect.y, -dx, viewRect.height);
			this.addDrawingTask(mandel , nr);

			nr.setBounds(oRect.x, viewRect.y+viewRect.height-dy, viewRect.width+dx, dy);
			this.addDrawingTask(mandel , nr);
		}
		if (dx < 0 && dy < 0 ) {
			nr.setBounds(viewRect.x, viewRect.y, -dx, viewRect.height);
			this.addDrawingTask(mandel , nr);

			nr.setBounds(oRect.x, viewRect.y, viewRect.width+dx, -dy);
			this.addDrawingTask(mandel , nr);
		}

	}

	////////////////////////// `Thread֌W
	private void addDrawingTask(final MandelbrotSet ms ,final Rectangle rect){
		DrawingTask mandelbrotTask = new DrawingTask(ms, rect);
		//DebugThread.printf("DrawingTask submit:%s\n",rect);
		this.cs.submit(mandelbrotTask);
		synchronized (dts) {
			this.dts.add(mandelbrotTask);
		}		
	}
	
	protected void paintImagePiece(){ // MandelbrotThreadExecutordrawingExeafterExecute()Ăяo
		ImagePiece ip;
		try {
			//DebugThread.printf("paintImagePiece synchronized\n");
			synchronized (viewRect) {
				//DebugThread.printf("paintImagePieceJn\n");
				Future<ImagePiece> f;
				while ((f = cs.poll()) != null) {
					ip = f.get();
					if( ip.getImage()!= null) {
						image.getRaster().setRect(ip.getRect().x - viewRect.x,
								ip.getRect().y - viewRect.y, ip.getImage());
						//DebugThread.printf("paintImage %s [%d,%d]\n", ip , ip.getRect().x-viewRect.x, ip.getRect().y-viewRect.y);
						imageObserver.repaint();
					}
				}
				if( drawingExe.isEmptyTask() ){
					this.makeSmallImage();
					//DebugThread.printf("smallImage쐬\n");
				}
				ArrayList<DrawingTask> removeList = new ArrayList<DrawingTask>();
				synchronized (dts) {
					for (int i = 0; i < dts.size(); ++i) {
						if (dts.get(i).getState() == DrawingTask.TERMINATED) {
							removeList.add(dts.get(i));
						}
					}
					dts.removeAll(removeList);
				}
			}
			imageObserver.repaint();
			((Mandelbrot) imageObserver).sendRegionUpdate();
		} catch (InterruptedException e) {
			e.printStackTrace();
		} catch (ExecutionException e) {
			e.printStackTrace();
		}
	}

	public BufferedImage makeSmallImage(){
		// ICONp̏smallImageĂB
		smallImage = new BufferedImage(64, 64,
				BufferedImage.TYPE_INT_BGR);
		synchronized (image) {
			AffineTransformOp ato = new AffineTransformOp(
					AffineTransform.getScaleInstance(
							64.0 / (double) image.getWidth(),
							64.0 / (double) image.getHeight()), null);
			ato.filter(image, smallImage);
		}
		return smallImage;
	}

	@Override
	protected void finalize() throws Throwable {
		this.drawingExe.shutdown();
		super.finalize();
	}
}
