
/* Drop Tables */

DROP TABLE IF EXISTS MNJ_CTXT;
DROP TABLE IF EXISTS MNJ_CTXT_ITEM;
DROP TABLE IF EXISTS MNJ_KWD;
DROP TABLE IF EXISTS MNJ_MURMUR;
DROP TABLE IF EXISTS MNJ_REL_MURMUR_CTXT;
DROP TABLE IF EXISTS MNJ_USER;




/* Create Tables */

-- Ctxt of keyword. Ctxt is nested keyword.
CREATE TABLE MNJ_CTXT
(
	-- ID of CTXT
	CTXT_ID INT NOT NULL,
	-- true: entry is valid.
	-- false: entry is invalid.
	VALID_FLG BOOLEAN DEFAULT 'true' NOT NULL,
	-- true: alias of parent keyword. Sometimes translation of parent keyword.
	-- false: Non alias.
	ALIAS_FLG BOOLEAN DEFAULT 'false' NOT NULL,
	-- 0: For all locales.
	-- 47: en_US
	-- 97: ja_JP
	LOCALE_ID INT DEFAULT 47 NOT NULL,
	-- 1:target kwd
	-- 0:non keyword
	KWD_ID INT NOT NULL,
	CTXT_FULL_PATH VARCHAR(4000) NOT NULL,
	-- Level of last Keyword's deapth.
	CTXT_ENDPOINT_LEVEL INT NOT NULL,
	-- Sort priority. Small value means higher priority.
	SORT_PRIORITY INT DEFAULT 10000 NOT NULL,
	-- Owner of this row.
	OWNER_USER_ID INT NOT NULL,
	-- last update time (timestamp) of row
	UPD_DT TIMESTAMP NOT NULL,
	-- last update user id of row
	UPD_USER_ID INT NOT NULL,
	PRIMARY KEY (CTXT_ID)
) WITHOUT OIDS;


-- Each item of CTXT. Each item bound to Keyword.
CREATE TABLE MNJ_CTXT_ITEM
(
	-- ID of CTXT
	CTXT_ID INT NOT NULL,
	-- Level of Keyword's deapth.
	CTXT_LEVEL INT NOT NULL,
	-- 1:target kwd
	-- 0:non keyword
	KWD_ID INT NOT NULL,
	-- last update time (timestamp) of row
	UPD_DT TIMESTAMP NOT NULL,
	-- last update user id of row
	UPD_USER_ID INT NOT NULL,
	PRIMARY KEY (CTXT_ID, CTXT_LEVEL)
) WITHOUT OIDS;


-- Keyword item of manjyu.
CREATE TABLE MNJ_KWD
(
	-- 1:target kwd
	-- 0:non keyword
	KWD_ID INT NOT NULL UNIQUE,
	-- true: entry is valid.
	-- false: entry is invalid.
	VALID_FLG BOOLEAN DEFAULT 'true' NOT NULL,
	-- 0: For all locales.
	-- 47: en_US
	-- 97: ja_JP
	LOCALE_ID INT DEFAULT 0 NOT NULL,
	-- Name of keyword.
	KWD_NAME VARCHAR(254) NOT NULL UNIQUE,
	-- Normalized name of Keyword.
	KWD_NAME_NORMAL VARCHAR(254) NOT NULL,
	-- Owner of this row.
	OWNER_USER_ID INT NOT NULL,
	-- last update time (timestamp) of row
	UPD_DT TIMESTAMP NOT NULL,
	-- last update user id of row
	UPD_USER_ID INT NOT NULL,
	PRIMARY KEY (KWD_ID)
) WITHOUT OIDS;


-- Murmur of Manjyu.
-- Most text will contain into murmur.
CREATE TABLE MNJ_MURMUR
(
	-- ID of murmur.
	MURMUR_ID INT NOT NULL,
	-- true: entry is valid.
	-- false: entry is invalid.
	VALID_FLG BOOLEAN DEFAULT 'true' NOT NULL,
	-- 0: For all locales.
	-- 47: en_US
	-- 97: ja_JP
	LOCALE_ID INT DEFAULT 47 NOT NULL,
	-- 0: Symbolic link (reserved)
	-- 1: Translatied text
	-- 11: Plain text. (Sometimes Title text...)
	-- 12: Description text.
	-- 13: Code text  (reserved). Preformatted text.
	-- 21: Plain URL
	-- 22: Image URL (reserved)
	-- 23: Wiki URL
	-- 91: GUID (reserved)
	-- 92: Creator (reserved)
	MURMUR_TYPE INT DEFAULT 11 NOT NULL,
	-- Visibility of murmur. Negative value means imvisible.
	MURMUR_VISIBILITY INT DEFAULT 1 NOT NULL,
	-- Murmured date. Like pubDate in RSS.
	MURMUR_DATE TIMESTAMP NOT NULL,
	-- Murmur text. Text, URL, URI, ... etc.
	MURMUR_TEXT VARCHAR(4000) NOT NULL,
	-- Normalized Murmur text.
	MURMUR_TEXT_NORMAL VARCHAR(4000) NOT NULL,
	-- 0:none
	-- 
	REFER_MURMUR_ID INT DEFAULT 0 NOT NULL,
	-- Owner of this row.
	OWNER_USER_ID INT NOT NULL,
	-- last update time (timestamp) of row
	UPD_DT TIMESTAMP NOT NULL,
	-- last update user id of row
	UPD_USER_ID INT NOT NULL,
	PRIMARY KEY (MURMUR_ID)
) WITHOUT OIDS;


-- Relation table from Murmur to CTXT.
CREATE TABLE MNJ_REL_MURMUR_CTXT
(
	-- ID of murmur.
	MURMUR_ID INT NOT NULL,
	-- ID of CTXT
	CTXT_ID INT NOT NULL,
	-- Owner of this row.
	OWNER_USER_ID INT NOT NULL,
	-- Sort priority. Small value means higher priority.
	SORT_PRIORITY INT DEFAULT 10000 NOT NULL,
	-- true: entry is valid.
	-- false: entry is invalid.
	VALID_FLG BOOLEAN DEFAULT 'true' NOT NULL,
	-- last update time (timestamp) of row
	UPD_DT TIMESTAMP NOT NULL,
	-- last update user id of row
	UPD_USER_ID INT NOT NULL,
	PRIMARY KEY (MURMUR_ID, CTXT_ID, OWNER_USER_ID)
) WITHOUT OIDS;


-- User entry for Manjyu
CREATE TABLE MNJ_USER
(
	-- 0: Anonymous
	-- 1: Admin
	-- 2: ToshikiIga
	-- 3-10000: reserved.
	-- 10001...: User defined user.
	USER_ID INT NOT NULL,
	-- true: entry is valid.
	-- false: entry is invalid.
	VALID_FLG BOOLEAN DEFAULT 'true' NOT NULL,
	-- User name in Manjyu. Used in logon user.
	USER_CD VARCHAR(254) NOT NULL UNIQUE,
	-- Default locale of user.
	DEFAULT_LOCALE_ID INT DEFAULT 47 NOT NULL,
	-- Password value. Will be SHA256ed value... soon... someday
	USER_PASSWD VARCHAR(254) NOT NULL,
	-- true: administrator
	-- false: general user
	USER_FLG_ADMIN BOOLEAN DEFAULT 'false' NOT NULL,
	-- true: Editor for Keyword
	-- false: No edit rights.
	USER_FLG_EDITOR_KWD BOOLEAN DEFAULT 'false' NOT NULL,
	-- true: Editor for Murmur.
	-- false: No edit rights.
	USER_FLG_EDITOR_MURMUR BOOLEAN DEFAULT 'false' NOT NULL,
	-- last update time (timestamp) of row
	UPD_DT TIMESTAMP NOT NULL,
	-- last update user id of row
	UPD_USER_ID INT NOT NULL,
	PRIMARY KEY (USER_ID)
) WITHOUT OIDS;



/* Comments */

COMMENT ON TABLE MNJ_CTXT IS 'Ctxt of keyword. Ctxt is nested keyword.';
COMMENT ON COLUMN MNJ_CTXT.CTXT_ID IS 'ID of CTXT';
COMMENT ON COLUMN MNJ_CTXT.VALID_FLG IS 'true: entry is valid.
false: entry is invalid.';
COMMENT ON COLUMN MNJ_CTXT.ALIAS_FLG IS 'true: alias of parent keyword. Sometimes translation of parent keyword.
false: Non alias.';
COMMENT ON COLUMN MNJ_CTXT.LOCALE_ID IS '0: For all locales.
47: en_US
97: ja_JP';
COMMENT ON COLUMN MNJ_CTXT.KWD_ID IS '1:target kwd
0:non keyword';
COMMENT ON COLUMN MNJ_CTXT.CTXT_ENDPOINT_LEVEL IS 'Level of last Keyword''s deapth.';
COMMENT ON COLUMN MNJ_CTXT.SORT_PRIORITY IS 'Sort priority. Small value means higher priority.';
COMMENT ON COLUMN MNJ_CTXT.OWNER_USER_ID IS 'Owner of this row.';
COMMENT ON COLUMN MNJ_CTXT.UPD_DT IS 'last update time (timestamp) of row';
COMMENT ON COLUMN MNJ_CTXT.UPD_USER_ID IS 'last update user id of row';
COMMENT ON TABLE MNJ_CTXT_ITEM IS 'Each item of CTXT. Each item bound to Keyword.';
COMMENT ON COLUMN MNJ_CTXT_ITEM.CTXT_ID IS 'ID of CTXT';
COMMENT ON COLUMN MNJ_CTXT_ITEM.CTXT_LEVEL IS 'Level of Keyword''s deapth.';
COMMENT ON COLUMN MNJ_CTXT_ITEM.KWD_ID IS '1:target kwd
0:non keyword';
COMMENT ON COLUMN MNJ_CTXT_ITEM.UPD_DT IS 'last update time (timestamp) of row';
COMMENT ON COLUMN MNJ_CTXT_ITEM.UPD_USER_ID IS 'last update user id of row';
COMMENT ON TABLE MNJ_KWD IS 'Keyword item of manjyu.';
COMMENT ON COLUMN MNJ_KWD.KWD_ID IS '1:target kwd
0:non keyword';
COMMENT ON COLUMN MNJ_KWD.VALID_FLG IS 'true: entry is valid.
false: entry is invalid.';
COMMENT ON COLUMN MNJ_KWD.LOCALE_ID IS '0: For all locales.
47: en_US
97: ja_JP';
COMMENT ON COLUMN MNJ_KWD.KWD_NAME IS 'Name of keyword.';
COMMENT ON COLUMN MNJ_KWD.KWD_NAME_NORMAL IS 'Normalized name of Keyword.';
COMMENT ON COLUMN MNJ_KWD.OWNER_USER_ID IS 'Owner of this row.';
COMMENT ON COLUMN MNJ_KWD.UPD_DT IS 'last update time (timestamp) of row';
COMMENT ON COLUMN MNJ_KWD.UPD_USER_ID IS 'last update user id of row';
COMMENT ON TABLE MNJ_MURMUR IS 'Murmur of Manjyu.
Most text will contain into murmur.';
COMMENT ON COLUMN MNJ_MURMUR.MURMUR_ID IS 'ID of murmur.';
COMMENT ON COLUMN MNJ_MURMUR.VALID_FLG IS 'true: entry is valid.
false: entry is invalid.';
COMMENT ON COLUMN MNJ_MURMUR.LOCALE_ID IS '0: For all locales.
47: en_US
97: ja_JP';
COMMENT ON COLUMN MNJ_MURMUR.MURMUR_TYPE IS '0: Symbolic link (reserved)
1: Translatied text
11: Plain text. (Sometimes Title text...)
12: Description text.
13: Code text  (reserved). Preformatted text.
21: Plain URL
22: Image URL (reserved)
23: Wiki URL
91: GUID (reserved)
92: Creator (reserved)';
COMMENT ON COLUMN MNJ_MURMUR.MURMUR_VISIBILITY IS 'Visibility of murmur. Negative value means imvisible.';
COMMENT ON COLUMN MNJ_MURMUR.MURMUR_DATE IS 'Murmured date. Like pubDate in RSS.';
COMMENT ON COLUMN MNJ_MURMUR.MURMUR_TEXT IS 'Murmur text. Text, URL, URI, ... etc.';
COMMENT ON COLUMN MNJ_MURMUR.MURMUR_TEXT_NORMAL IS 'Normalized Murmur text.';
COMMENT ON COLUMN MNJ_MURMUR.REFER_MURMUR_ID IS '0:none
';
COMMENT ON COLUMN MNJ_MURMUR.OWNER_USER_ID IS 'Owner of this row.';
COMMENT ON COLUMN MNJ_MURMUR.UPD_DT IS 'last update time (timestamp) of row';
COMMENT ON COLUMN MNJ_MURMUR.UPD_USER_ID IS 'last update user id of row';
COMMENT ON TABLE MNJ_REL_MURMUR_CTXT IS 'Relation table from Murmur to CTXT.';
COMMENT ON COLUMN MNJ_REL_MURMUR_CTXT.MURMUR_ID IS 'ID of murmur.';
COMMENT ON COLUMN MNJ_REL_MURMUR_CTXT.CTXT_ID IS 'ID of CTXT';
COMMENT ON COLUMN MNJ_REL_MURMUR_CTXT.OWNER_USER_ID IS 'Owner of this row.';
COMMENT ON COLUMN MNJ_REL_MURMUR_CTXT.SORT_PRIORITY IS 'Sort priority. Small value means higher priority.';
COMMENT ON COLUMN MNJ_REL_MURMUR_CTXT.VALID_FLG IS 'true: entry is valid.
false: entry is invalid.';
COMMENT ON COLUMN MNJ_REL_MURMUR_CTXT.UPD_DT IS 'last update time (timestamp) of row';
COMMENT ON COLUMN MNJ_REL_MURMUR_CTXT.UPD_USER_ID IS 'last update user id of row';
COMMENT ON TABLE MNJ_USER IS 'User entry for Manjyu';
COMMENT ON COLUMN MNJ_USER.USER_ID IS '0: Anonymous
1: Admin
2: ToshikiIga
3-10000: reserved.
10001...: User defined user.';
COMMENT ON COLUMN MNJ_USER.VALID_FLG IS 'true: entry is valid.
false: entry is invalid.';
COMMENT ON COLUMN MNJ_USER.USER_CD IS 'User name in Manjyu. Used in logon user.';
COMMENT ON COLUMN MNJ_USER.DEFAULT_LOCALE_ID IS 'Default locale of user.';
COMMENT ON COLUMN MNJ_USER.USER_PASSWD IS 'Password value. Will be SHA256ed value... soon... someday';
COMMENT ON COLUMN MNJ_USER.USER_FLG_ADMIN IS 'true: administrator
false: general user';
COMMENT ON COLUMN MNJ_USER.USER_FLG_EDITOR_KWD IS 'true: Editor for Keyword
false: No edit rights.';
COMMENT ON COLUMN MNJ_USER.USER_FLG_EDITOR_MURMUR IS 'true: Editor for Murmur.
false: No edit rights.';
COMMENT ON COLUMN MNJ_USER.UPD_DT IS 'last update time (timestamp) of row';
COMMENT ON COLUMN MNJ_USER.UPD_USER_ID IS 'last update user id of row';



