/*******************************************************************************
 * ManjyuBase64
 * Copyright (C) 2012 Toshiki IGA
 * 
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2012 Toshiki IGA and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      Toshiki IGA - initial API and implementation
 *******************************************************************************/
/*******************************************************************************
 * Copyright 2012 Toshiki IGA and others.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/
package org.manjyu.base64;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;

/**
 * ManjyuBase64 is Free MIME Base64 Library for Java. ManjyuBase64Util is main class of ManjyuBase64. 
 * 
 * <ul>
 * <li>ManjyuBase64 require Java 1.5 or later.</li>
 * <li>ManjyuBase64 depends Java SE API only.</li>
 * </ul>
 * 
 * @author Toshiki Iga
 */
public class ManjyuBase64Util {
	/**
	 * Hide constructor.
	 */
	private ManjyuBase64Util() {
	}

	/////////////////////////////////////////////////////////////////
	// RFC 2045

	/**
	 * Encode stream as RFC 2045 base64.
	 * 
	 * @param inStream Input stream for source.
	 * @param base64Writer Base64 encoded writer.
	 * @throws IOException 
	 */
	public static void encodeRfc2045(final InputStream inStream, final Writer base64Writer) throws IOException {
		new Base64Rfc2045().encode(inStream, base64Writer);
	}

	/**
	 * Encode byte array as RFC 2045 base64.
	 * 
	 * @param input Byte array.
	 * @return Base64 encoded string.
	 * @throws IOException
	 */
	public static String encodeRfc2045(final byte[] input) throws IOException {
		final StringWriter writer = new StringWriter();
		new Base64Rfc2045().encode(new ByteArrayInputStream(input), writer);
		writer.flush();
		return writer.toString();
	}

	/**
	 * Encode stream as RFC 2045 base64 with newline. 
	 * 
	 * max 76 characters in one line.
	 * 
	 * @param inStream Input stream for source.
	 * @param base64Writer Base64 encoded writer.
	 * @throws IOException
	 */
	public static void encodeRfc2045WithNewLine(final InputStream inStream, final Writer base64Writer)
			throws IOException {
		// 57 means 76 characters in one line.
		final byte[] buf = new byte[57];
		for (;;) {
			final int result = inStream.read(buf);
			if (result < 0) {
				break;
			}

			final InputStream inStreamBuf = new ByteArrayInputStream(buf, 0, result);
			new Base64Rfc2045().encode(inStreamBuf, base64Writer);
			if (result == buf.length) {
				base64Writer.write(0x0d);
				base64Writer.write(0x0a);
			}
		}
	}

	/**
	 * Decode input character reader as RFC 2045 base64.
	 * 
	 * @param base64Reader Base64 encoded reader.
	 * @param outStream Original byte stream.
	 * @throws IOException
	 */
	public static void decodeRfc2045(final Reader base64Reader, final OutputStream outStream) throws IOException {
		new Base64Rfc2045().decode(base64Reader, outStream);
	}

	/**
	 * Decode string as RFC 2045 base64.
	 * 
	 * @param base64Input Base64 encoded string.
	 * @return Original byte array.
	 * @throws IOException
	 */
	public static byte[] decodeRfc2045(final String base64Input) throws IOException {
		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		new Base64Rfc2045().decode(new StringReader(base64Input), outStream);
		outStream.flush();
		return outStream.toByteArray();
	}

	/////////////////////////////////////////////////////////////////
	// URL: base64url

	/**
	 * Encode stream as URL base 64.
	 * 
	 * <p>Difference between RFC 2045 and URL based.</p>
	 * <ul>
	 * <li>RFC 2045: +, URL: -</li>
	 * <li>RFC 2045: /, URL: _</li>
	 * </ul>
	 * 
	 * @param inStream
	 * @param base64Writer
	 * @throws IOException
	 */
	public static void encodeUrlAsBase64url(final InputStream inStream, final Writer base64Writer) throws IOException {
		new Base64ForUrl().encode(inStream, base64Writer);
	}

	/**
	 * Encode stream as URL base64 with newline. 
	 * 
	 * max 76 characters in one line.
	 * 
	 * <p>Difference between RFC 2045 and URL based.</p>
	 * <ul>
	 * <li>RFC 2045: +, URL: -</li>
	 * <li>RFC 2045: /, URL: _</li>
	 * </ul>
	 * 
	 * @param inStream byte input stream.
	 * @param base64Writer encoded charater writer.
	 * @throws IOException
	 */
	public static void encodeUrlAsBase64urlWithNewLine(final InputStream inStream, final Writer base64Writer)
			throws IOException {
		// 57 means 76 characters in one line.
		final byte[] buf = new byte[57];
		for (;;) {
			final int result = inStream.read(buf);
			if (result < 0) {
				break;
			}

			final InputStream inStreamBuf = new ByteArrayInputStream(buf, 0, result);
			new Base64ForUrl().encode(inStreamBuf, base64Writer);
			if (result == buf.length) {
				base64Writer.write(0x0d);
				base64Writer.write(0x0a);
			}
		}
	}

	/**
	 * Decode input character reader as URL base64.
	 * 
	 * <p>Difference between RFC 2045 and URL based.</p>
	 * <ul>
	 * <li>RFC 2045: +, URL: -</li>
	 * <li>RFC 2045: /, URL: _</li>
	 * </ul>
	 * 
	 * @param base64Reader
	 * @param outStream decoded bytes.
	 * @throws IOException
	 */
	public static void decodeUrlAsBase64url(final Reader base64Reader, final OutputStream outStream) throws IOException {
		new Base64ForUrl().decode(base64Reader, outStream);
	}
}
