/*
 * Copyright 2000-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.marevol.utils.util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Calendar;
import java.util.Date;

public class DateUtil
{
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(DateUtil.class);

    /**
     * Allocates a Date object and initializes it so that it represents 
     * midnight, local time, at the beginning of the day specified by the 
     * year, month, and date arguments.
     * 
     * @param year
     * @param month
     * @param date
     * @return
     */
    public static Date get(int year, int month, int date)
    {
        if (log.isDebugEnabled())
        {
            log.debug("get(int, int, int) -  : year=" + year + ", month=" + month + ", date=" + date);
        }

        Calendar cal = Calendar.getInstance();
        cal.set(year + 1900, month, date);
        return cal.getTime();
    }

    /**
     * Allocates a Date object and initializes it so that it represents the 
     * instant at the start of the minute specified by the year, month, date, 
     * hrs, and min arguments, in the local time zone.
     * 
     * @param year
     * @param month
     * @param date
     * @param hrs
     * @param min
     * @return
     */
    public static Date get(int year, int month, int date, int hrs, int min)
    {
        if (log.isDebugEnabled())
        {
            log.debug("get(int, int, int, int, int) -  : year=" + year + ", month=" + month + ", date=" + date
                    + ", hrs=" + hrs + ", min=" + min);
        }

        Calendar cal = Calendar.getInstance();
        cal.set(year + 1900, month, date, hrs, min);
        return cal.getTime();
    }

    /**
     * Allocates a Date object and initializes it so that it represents the 
     * instant at the start of the second specified by the year, month, date, 
     * hrs, min, and sec arguments, in the local time zone.
     * 
     * @param year
     * @param month
     * @param date
     * @param hrs
     * @param min
     * @param sec
     * @return
     */
    public static Date get(int year, int month, int date, int hrs, int min, int sec)
    {
        if (log.isDebugEnabled())
        {
            log.debug("get(int, int, int, int, int, int) -  : year=" + year + ", month=" + month + ", date=" + date
                    + ", hrs=" + hrs + ", min=" + min + ", sec=" + sec);
        }

        Calendar cal = Calendar.getInstance();
        cal.set(year + 1900, month, date, hrs, min, sec);
        return cal.getTime();

    }

    /**
     * Allocates a Date object and initializes it so that it represents the 
     * instant at the start of the second specified by the year, month, date, 
     * hrs, min, sec and millisec arguments, in the local time zone.
     * 
     * @param year
     * @param month
     * @param date
     * @param hrs
     * @param min
     * @param sec
     * @param millisec
     * @return
     */
    public static Date get(int year, int month, int date, int hrs, int min, int sec, int millisec)
    {
        if (log.isDebugEnabled())
        {
            log.debug("get(int, int, int, int, int, int) -  : year=" + year + ", month=" + month + ", date=" + date
                    + ", hrs=" + hrs + ", min=" + min + ", sec=" + sec + ", millisec=" + millisec);
        }

        Calendar cal = Calendar.getInstance();
        cal.set(year + 1900, month, date, hrs, min, sec);
        cal.set(Calendar.MILLISECOND, millisec);
        return cal.getTime();

    }

    /**
     * Returns a value that is the result of subtracting 1900 from the year 
     * that contains or begins with the instant in time represented by this 
     * Date object, as interpreted in the local time zone.
     * 
     * @param d
     * @return
     */
    public static int getYear(Date d)
    {
        Calendar cal = Calendar.getInstance();
        cal.setTime(d);
        return cal.get(Calendar.YEAR) - 1900;
    }

    /**
     * Returns a number representing the month that contains or begins with the
     * instant in time represented by this Date object. The value returned is 
     * between 0 and 11, with the value 0 representing January.
     * 
     * @param d
     * @return
     */
    public static int getMonth(Date d)
    {
        Calendar cal = Calendar.getInstance();
        cal.setTime(d);
        return cal.get(Calendar.MONTH);
    }

    /**
     * Returns the day of the month represented by this Date object. The value 
     * returned is between 1 and 31 representing the day of the month that 
     * contains or begins with the instant in time represented by this Date 
     * object, as interpreted in the local time zone.
     * 
     * @param d
     * @return
     */
    public static int getDate(Date d)
    {
        Calendar cal = Calendar.getInstance();
        cal.setTime(d);
        return cal.get(Calendar.DAY_OF_MONTH);
    }

    /**
     * Returns the day of the week represented by this date. The returned value 
     * (0 = Sunday, 1 = Monday, 2 = Tuesday, 3 = Wednesday, 4 = Thursday, 
     * 5 = Friday, 6 = Saturday) represents the day of the week that contains or 
     * begins with the instant in time represented by this Date object, as 
     * interpreted in the local time zone.
     * 
     * @param d
     * @return
     */
    public static int getDay(Date d)
    {
        Calendar cal = Calendar.getInstance();
        cal.setTime(d);
        return cal.get(Calendar.DAY_OF_WEEK) - 1;
    }

    /**
     * Returns the hour represented by this Date object. The returned value is 
     * a number (0 through 23) representing the hour within the day that contains 
     * or begins with the instant in time represented by this Date object, as 
     * interpreted in the local time zone.
     * 
     * @param d
     * @return
     */
    public static int getHours(Date d)
    {
        Calendar cal = Calendar.getInstance();
        cal.setTime(d);
        return cal.get(Calendar.HOUR_OF_DAY);
    }

    /**
     * Returns the number of minutes past the hour represented by this date, 
     * as interpreted in the local time zone. The value returned is between 0 
     * and 59.
     * 
     * @param d
     * @return
     */
    public static int getMinutes(Date d)
    {
        Calendar cal = Calendar.getInstance();
        cal.setTime(d);
        return cal.get(Calendar.MINUTE);
    }

    /**
     * Returns the number of seconds past the minute represented by this date. 
     * The value returned is between 0 and 61. The values 60 and 61 can only 
     * occur on those Java Virtual Machines that take leap seconds into account.
     * 
     * @param d
     * @return
     */
    public static int getSeconds(Date d)
    {
        Calendar cal = Calendar.getInstance();
        cal.setTime(d);
        return cal.get(Calendar.SECOND);
    }

}
