/*
 * Copyright (c) 2006-2009 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.wizards.layout;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;

import jp.sf.maskat.ui.ISharedImages;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.Messages;
import jp.sf.maskat.ui.editors.layout.LayoutGraphicalEditor;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWizard;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;


/**
 * This is a sample new wizard. Its role is to create a new file resource in the
 * provided container. If the container resource (a folder or a project) is
 * selected in the workspace when the wizard is opened, it will accept it as the
 * target container. The wizard creates one
 */
public class NewLayoutFileWizard extends Wizard implements INewWizard {

	/** The wizard ID */
	public static final String WIZARD_ID = MaskatUIPlugin.PLUGIN_ID
			+ ".newLayoutFileWizard"; //$NON-NLS-1$
	
	private NewLayoutFileWizardPage page;

	private ISelection selection;

	/**
	 * Constructor for NewLayoutFileWizard.
	 */
	public NewLayoutFileWizard() {
		super();
		this.setWindowTitle(
				Messages.getString("wizard.layout.windowtitle")); //$NON-NLS-1$
		setNeedsProgressMonitor(true);
	}

	/**
	 * Adding the page to the wizard.
	 */
	public void addPages() {
		page = new NewLayoutFileWizardPage(selection);
		ImageDescriptor image = MaskatUIPlugin.getImageDescriptor(
				ISharedImages.IMG_WIZBAN_NEWFILE);
		
		page.setImageDescriptor(image);
		
		addPage(page);
	}

	/**
	 * This method is called when 'Finish' button is pressed in the wizard. We
	 * will create an operation and run it using wizard as execution context.
	 */
	public boolean performFinish() {
		final String containerName = page.getContainerName();
		final String fileName = page.getFileName();
		final String layoutName = page.getLayoutName();
		IRunnableWithProgress op = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException {
				try {
					doFinish(containerName, fileName, layoutName, monitor);
				} catch (CoreException e) {
					throw new InvocationTargetException(e);
				} finally {
					monitor.done();
				}
			}
		};
		try {
			getContainer().run(true, false, op);
		} catch (InterruptedException e) {
			return false;
		} catch (InvocationTargetException e) {
			Throwable realException = e.getTargetException();
			MessageDialog.openError(getShell(), "Error", realException.getMessage());
			return false;
		}
		return true;
	}

	/**
	 * The worker method. It will find the container, create the file if missing
	 * or just replace its contents, and open the editor on the newly created
	 * file.
	 */
	private void doFinish(String containerName, String fileName, String layoutName,
			IProgressMonitor monitor) throws CoreException {
		
		// create a sample file
		monitor.beginTask("Creating " + fileName, 2);
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IResource resource = root.findMember(new Path(containerName));
		if (!resource.exists() || !(resource instanceof IContainer)) {
			throwCoreException(containerName +
				Messages.getString("wizard.layout.msg.error.notcontainer")); //$NON-NLS-1$
		}
		
		IContainer container = (IContainer) resource;
		final IFile file = container.getFile(new Path(fileName));
		try {
			InputStream stream = openContentStream(layoutName);
			if (file.exists()) {
				String[] msgArgs = {fileName};
				throwCoreException(Messages.format(
					"wizard.layout.msg.error.existfile", msgArgs)); 
				
//				throwCoreException(
//					Messages.getString("wizard.layout.msg.error.existfile")); //$NON-NLS-1$
			} else {
				file.create(stream, true, monitor);
			}
			stream.close();
		} catch (IOException e) {
		}
		
		monitor.worked(1);
		monitor.setTaskName(
				Messages.getString("wizard.layout.taskname")); //$NON-NLS-1$		

		getShell().getDisplay().asyncExec(new Runnable() {
			public void run() {
				IWorkbenchPage page = PlatformUI.getWorkbench()
						.getActiveWorkbenchWindow().getActivePage();
				try {
					IDE.openEditor(page, file, LayoutGraphicalEditor.EDITOR_ID, true);
				} catch (PartInitException e) {
				}
			}
		});
		monitor.worked(1);
	}

	private String getTemplate(String templateName) throws IOException {
		InputStream in = null;
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		try {
			in = this.getClass().getResourceAsStream(templateName);
			BufferedReader reader = new BufferedReader(new InputStreamReader(in));
			PrintWriter pw = new PrintWriter(baos);
			
			String line = null;
			while ((line = reader.readLine()) != null) {
				pw.println(line);
			}
			pw.close();
			
		} finally {
			if (in != null) {
				in.close();
			}
		}
		return baos.toString();		
	}
	
	/**
	 * We will initialize file contents with a sample text.
	 * 
	 * @throws UnsupportedEncodingException
	 */

	private InputStream openContentStream(String layoutName)
			throws UnsupportedEncodingException, IOException {
		String contents = getTemplate("layout.template"); //$NON-NLS-1$
		contents = contents.replaceAll("\\Q${layoutName}\\E", layoutName); //$NON-NLS-1$
		return new ByteArrayInputStream(contents.getBytes("UTF-8")); //$NON-NLS-1$
	}

	private void throwCoreException(String message) throws CoreException {
		IStatus status = new Status(IStatus.ERROR, MaskatUIPlugin.PLUGIN_ID, IStatus.OK,
				message, null);
		throw new CoreException(status);
	}

	/**
	 * We will accept the selection in the workbench to see if we can initialize
	 * from it.
	 * 
	 * @see IWorkbenchWizard#init(IWorkbench, IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		this.selection = selection;
	}
}
