/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.betwixt;

import java.util.Map;

import jp.sf.maskat.core.MaskatElement;
import jp.sf.maskat.core.event.Component;
import jp.sf.maskat.core.layout.Desc;
import jp.sf.maskat.core.layout.JavaScript;


import org.apache.commons.betwixt.AttributeDescriptor;
import org.apache.commons.betwixt.ElementDescriptor;
import org.apache.commons.betwixt.strategy.ValueSuppressionStrategy;

/**
 * レイアウト定義XMLの要素に定義されている属性の出力制御を行うクラスです。
 * <p>
 * 属性に定義されている値がデフォルト値である場合はその属性を出力しません。
 */
public class LayoutValueSuppressionStrategy extends ValueSuppressionStrategy {

    private MaskatElement currentElement = null;

	/**
     * {@inheritDoc}
     */
    public boolean suppressAttribute(AttributeDescriptor attributeDescriptor,
            String value) {
        if (attributeDescriptor.getPropertyName() != null && value != null) {
        	if (value.length() <= 0 && attributeDescriptor.getPropertyType() == Integer.class) {
        		return true;
        	}
            return isDefaultValue(attributeDescriptor.getPropertyName(), value);
        }
        return false;
    }

	/**
	 * 指定された値がデフォルト値であるか判定します。
	 * 
	 * @param attribute 属性名
	 * @param value 値
	 * @return true デフォルト値
	 */
    private boolean isDefaultValue(String attribute, String value) {
        Map defaultMap = currentElement.getDefaultValues();
        if (defaultMap.containsKey(attribute)) {
            Object v = defaultMap.get(attribute);
            if (!(v instanceof String)) {
            	v = (new StringBuffer()).append(v).toString();
            }
            return value.equals((String) v);
        }
        return false;
    }

	/**
     * {@inheritDoc}
     */
    public boolean suppressElement(ElementDescriptor element,
            String namespaceUri, String localName, String qualifiedName,
            Object value) {
        if (value instanceof MaskatElement) {
            currentElement = (MaskatElement) value;
            if (value instanceof Component) {
                return !((Component) value).hasEnabledEvents();
            }
            if (value instanceof Desc) {
                return ((Desc) value).getComment() == null;
            }
        } else if (value instanceof JavaScript) {
            return ((JavaScript) value).getContent() == null;
        }
        return ValueSuppressionStrategy.DEFAULT.suppressElement(element,
                namespaceUri, localName, qualifiedName, value);
    }
}
