/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.betwixt;

import java.beans.IntrospectionException;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringWriter;
import java.util.Iterator;
import java.util.Map;

import jp.sf.maskat.core.event.EventDef;
import jp.sf.maskat.core.layout.LayoutDef;

import org.apache.commons.betwixt.XMLIntrospector;
import org.apache.commons.betwixt.io.BeanReader;
import org.apache.commons.betwixt.io.BeanWriter;
import org.apache.commons.io.IOUtils;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * イベント定義XML、レイアウト定義XMLの入出力を行うクラスです。
 * <p>
 * このクラスはマスカットエディタからの呼び出しによりイベント定義XMLと
 * レイアウト定義XMLの入出力を行います。
 * </p>
 */
public class MaskatBeanIO {

	/**
	 * イベント定義XML用 betwixt設定ファイル名
	 */
	public static final String EVENT_DEF_CONFIG = "eventConfig.betwixt";

	/**
	 * レイアウト定義XML用 betwixt設定ファイル名
	 */
	public static final String LAYOUT_DEF_CONFIG = "layoutConfig.betwixt";

	/**
	 * イベント定義XMLを読み込み、EventDefオブジェクトとして取得します。
	 * 
	 * @param input イベント定義XMLのInputSource
	 * @return EventDefオブジェクト
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 */
	public static EventDef readEventDef(InputSource input)
		throws SAXException, IOException, IntrospectionException {

		// Now convert this to a bean using betwixt
		// Create BeanReader
		BeanReader beanReader = new BeanReader();

		// Configure the reader
		beanReader.getXMLIntrospector().getConfiguration()
				.setAttributesForPrimitives(false);
		beanReader.getBindingConfiguration().setMapIDs(false);

		XMLIntrospector intro = beanReader.getXMLIntrospector();
		InputStream in = null;
		try {
			in = MaskatBeanIO.class.getResourceAsStream(EVENT_DEF_CONFIG);
			intro.register(new InputSource(in));
		} finally {
			IOUtils.closeQuietly(in);
		}
		// Register beans so that betwixt knows what the xml is to be converted
		// to
		// Since the element mapped to a Bean isn't called the same
		// as Betwixt would have guessed, need to register the path as well
		beanReader.registerBeanClass("eventDef", EventDef.class);

		// Now we parse the xml
		EventDef eventDef = (EventDef) beanReader.parse(input);

		return eventDef;
	}

	/**
	 * イベント定義XMLを読み込み、EventDefオブジェクトとして取得します。
	 * 
	 * @param reader イベント定義XMLのReader
	 * @return EventDefオブジェクト
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 */
	public static EventDef readEventDef(Reader reader)
			throws SAXException, IOException, IntrospectionException {
		return readEventDef(new InputSource(reader));
	}

	/**
	 * イベント定義XMLを読み込みEventDefに格納します。
	 * 
	 * @param uri イベント定義XMLのuri文字列
	 * @return EventDefオブジェクト
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 */
	public static EventDef readEventDef(String uri)
			throws SAXException, IOException, IntrospectionException {
		return readEventDef(new InputSource(uri));
	}

	/**
	 * イベント定義XMLを読み込み、EventDefオブジェクトとして取得します。
	 * 
	 * @param stream イベント定義XMLのInputStream
	 * @return EventDefオブジェクト
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 */
	public static EventDef readEventDef(InputStream stream)
			throws SAXException, IOException, IntrospectionException {
		return readEventDef(new InputSource(stream));
	}

	/**
	 * EventDefオブジェクトからイベント定義XML表現の文字列を取得します。
	 *  
	 * @param eventDef EventDefオブジェクト
	 * @return イベント定義XML表現の文字列
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル操作でエラーが発生した場合
	 * @throws IntrospectionException JavaBeanからの変換でエラーが発生した場合
	 */
	public static String writeEventDef(EventDef eventDef)
			throws IntrospectionException, IOException, SAXException {
		// Start by preparing the writer
		// We'll write to a string
		StringWriter outputWriter = new StringWriter();

		// Betwixt just writes out the bean as a fragment
		// So if we want well-formed xml, we need to add the prolog
		outputWriter.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
		String lineSeparator = System.getProperty("line.separator");
		outputWriter.write(lineSeparator);

		// Create a BeanWriter which writes to our prepared stream
		BeanWriter beanWriter = new BeanWriter(outputWriter);

		// Configure betwixt
		// For more details see java docs or later in the main documentation
		beanWriter.getXMLIntrospector().getConfiguration()
				.setAttributesForPrimitives(false);
		beanWriter.getBindingConfiguration().setMapIDs(false);

		beanWriter.getBindingConfiguration().setValueSuppressionStrategy(
				new EventValueSuppressionStrategy());

		beanWriter.enablePrettyPrint();
		beanWriter.setInitialIndentLevel(0);
		beanWriter.setWriteEmptyElements(false);

		XMLIntrospector intro = beanWriter.getXMLIntrospector();
		InputStream in = null;
		try {
			in = MaskatBeanIO.class.getResourceAsStream(EVENT_DEF_CONFIG);
			intro.register(new InputSource(in));
		} finally {
			IOUtils.closeQuietly(in);
		}
		// If the base element is not passed in, Betwixt will guess
		// But let's write bean as base element
		beanWriter.write(eventDef);

		// Write to System.out
		// (We could have used the empty constructor for BeanWriter
		// but this way is more instructive)
		String out = outputWriter.toString();

		// Betwixt writes fragments not documents so does not automatically
		// close
		// writers or streams.
		// We will do no more writing so close the writer now.
		outputWriter.close();

		return out;
	}

	/**
	 * レイアウト定義XMLを読み込み、LayoutDefオブジェクトとして取得します。
	 * 
	 * @param input レイアウト定義XMLのInputSource
	 * @return LayoutDefオブジェクト
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 */
	public static LayoutDef readLayoutDef(InputSource input)
			throws IntrospectionException, IOException, SAXException {
		// Now convert this to a bean using betwixt
		// Create BeanReader
		BeanReader beanReader = new MaskatBeanReader();
		beanReader.setNamespaceAware(true);
        MaskatNamespacePrefixMapper namespaceMapper = new MaskatNamespacePrefixMapper();
        beanReader.getXMLIntrospector().getConfiguration().setPrefixMapper(namespaceMapper); 
		
        // Configure the reader
        beanReader.getXMLIntrospector().getConfiguration().setAttributesForPrimitives(true);
        beanReader.getBindingConfiguration().setMapIDs(false);
        beanReader.getBindingConfiguration().setClassNameAttribute("maskatClassName");
        beanReader.getReadConfiguration().setBeanCreationChain(new DynaBeanCreationChain());
        
        XMLIntrospector intro = beanReader.getXMLIntrospector();
		InputStream in = MaskatBeanIO.class.getResourceAsStream(LAYOUT_DEF_CONFIG);
		intro.register(new InputSource(in));
		
        // Register beans so that betwixt knows what the xml is to be converted to
        // Since the element mapped to a Bean isn't called the same 
        // as Betwixt would have guessed, need to register the path as well
        beanReader.registerBeanClass("layoutDef", LayoutDef.class);
        beanReader.setEntityResolver(new MaskatEntityResolver());
        
        // Now we parse the xml
        LayoutDef layoutDef = (LayoutDef) beanReader.parse(input);
        if (layoutDef != null) {
        	layoutDef.setNamespaces(namespaceMapper.getPrefixes());
        }
        return layoutDef;
	}

	/**
	 * レイアウト定義XMLを読み込み、LayoutDefオブジェクトとして取得します。
	 * 
	 * @param reader レイアウト定義XMLのReader
	 * @return LayoutDefオブジェクト
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 */
	public static LayoutDef readLayoutDef(Reader reader)
			throws IntrospectionException, IOException, SAXException {
		return readLayoutDef(new InputSource(reader));
	}

	/**
	 * レイアウト定義XMLを読み込み、LayoutDefオブジェクトとして取得します。
	 * 
	 * @param uri レイアウト定義XMLのパス
	 * @return LayoutDefオブジェクト
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 */
	public static LayoutDef readLayoutDef(String uri)
			throws IntrospectionException, IOException, SAXException {
		return readLayoutDef(new InputSource(uri));
	}

	/**
	 * レイアウト定義XMLを読み込み、LayoutDefオブジェクトとして取得します。
	 * 
	 * @param stream レイアウト定義XMLのInputStream
	 * @return LayoutDefオブジェクト
	 * @throws IntrospectionException JavaBeanへの変換でエラーが発生した場合
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル読み込みでエラーが発生した場合
	 */
	public static LayoutDef readLayoutDef(InputStream stream)
		throws IntrospectionException, IOException, SAXException {
		return readLayoutDef(new InputSource(stream));
	}

	/**
	 * LayoutDefオブジェクトからレイアウト定義XML表現の文字列を取得します。
	 *  
	 * @param layoutDef LayoutDefオブジェクト
	 * @return レイアウト定義XML表現の文字列
	 * @throws SAXException XML パーサでエラーが発生した場合
	 * @throws IOException ファイル操作でエラーが発生した場合
	 * @throws IntrospectionException JavaBeanからの変換でエラーが発生した場合
	 */
	public static String writeLayoutDef(LayoutDef layoutDef)
		throws IntrospectionException, IOException, SAXException {
		// Start by preparing the writer
		// We'll write to a string
		StringWriter outputWriter = new StringWriter();

		// Betwixt just writes out the bean as a fragment
		// So if we want well-formed xml, we need to add the prolog
		outputWriter.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");

		// Create a BeanWriter which writes to our prepared stream
		BeanWriter beanWriter = new MaskatBeanWriter(outputWriter);

		// Configure betwixt
		// For more details see java docs or later in the main documentation
		beanWriter.getXMLIntrospector().getConfiguration()
				.setAttributesForPrimitives(true);
		// set this to avoid the conflict with className of
		// Label/DivHtml/Checkbox/Radio
		beanWriter.getBindingConfiguration().setClassNameAttribute(
				"maskatClassName");
		beanWriter.getBindingConfiguration().setMapIDs(false);
		beanWriter.getBindingConfiguration().setValueSuppressionStrategy(
				new LayoutValueSuppressionStrategy());
		beanWriter.enablePrettyPrint();
		beanWriter.setInitialIndentLevel(0);
		beanWriter.setWriteEmptyElements(false);

		MaskatNamespacePrefixMapper namespaceMapper =
				new MaskatNamespacePrefixMapper();
		Map namespaces = layoutDef.getNamespaces();
		for (Iterator ite = namespaces.keySet().iterator(); ite.hasNext();) {
			String key = (String) ite.next();
			namespaceMapper.setPrefix((String) namespaces.get(key), key);
		}
		beanWriter.getXMLIntrospector().getConfiguration().setPrefixMapper(
				namespaceMapper);

		XMLIntrospector intro = beanWriter.getXMLIntrospector();

		InputStream in = null;
		try {
			in = MaskatBeanIO.class.getResourceAsStream(LAYOUT_DEF_CONFIG);
			intro.register(new InputSource(in));
		} finally {
			IOUtils.closeQuietly(in);
		}
		// If the base element is not passed in, Betwixt will guess
		// But let's write bean as base element
		beanWriter.write(layoutDef);

		// Write to System.out
		// (We could have used the empty constructor for BeanWriter
		// but this way is more instructive)
		String out = outputWriter.toString();

		// Betwixt writes fragments not documents so does not automatically
		// close
		// writers or streams.
		// We will do no more writing so close the writer now.
		outputWriter.close();

		return out;
	}
}
