/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.layout;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.List;

import jp.sf.maskat.core.event.Event;
import jp.sf.maskat.core.event.EventDef;
import jp.sf.maskat.core.event.Source;
import jp.sf.maskat.core.event.Target;

/**
 * コンポーネント要素の抽象クラスです。
 * 
 * 動的に生成するDynaComponentを利用しないで静的にコンポーネントを
 * 拡張する場合、このクラスを継承する必要があります。
 */
public abstract class AbstractComponent extends AbstractLayoutElement
		implements Component {

	/**
	 * コンポーネント名
	 */
	private String name;

	/**
	 * タブインデックス
	 */
	private int tabIndex;

	/**
	 * left（座標）
	 */
	private int left;

	/**
	 * top（座標）
	 */
	private int top;

	/**
	 * width（サイズ）
	 */
	private int width;

	/**
	 * height（サイズ）
	 */
	private int height;

	/**
	 * モデルの状態変化を通知するリスナ
	 */
	private PropertyChangeSupport listeners = new PropertyChangeSupport(this);

	/**
	 * デフォルトコンストラクタです
	 */
	public AbstractComponent() {
		super();
	}

	/**
     * {@inheritDoc}
     */
	public String getName() {
		return name;
	}

	/**
     * {@inheritDoc}
     */
	public void setName(String name) {
		String oldName = this.name;
		this.name = name;
		updateComponentIdInEvent(oldName, name);
		firePropertyChange("name", oldName, name);
	}

	/**
     * {@inheritDoc}
     */
	public int getTabIndex() {
		return tabIndex;
	}

	/**
     * {@inheritDoc}
     */
	public void setTabIndex(int tabIndex) {
		this.tabIndex = tabIndex;
	}

	/**
     * {@inheritDoc}
     */
	public int getHeight() {
		return height;
	}

	/**
     * {@inheritDoc}
     */
	public void setHeight(int height) {
		int oldHeight = this.height;
		this.height = height;
		if (oldHeight != height) {
			firePropertyChange("constraint", "", null);
		}
	}

	/**
     * {@inheritDoc}
     */
	public int getLeft() {
		return left;
	}

	/**
     * {@inheritDoc}
     */
	public void setLeft(int left) {
		int oldLeft = this.left;
		this.left = left;
		if (oldLeft != left) {
			firePropertyChange("constraint", "", null);
		}
	}

	/**
     * {@inheritDoc}
     */
	public int getTop() {
		return top;
	}

	/**
     * {@inheritDoc}
     */
	public void setTop(int top) {
		int oldTop = this.top;
		this.top = top;
		if (oldTop != top) {
			firePropertyChange("constraint", "", null);
		}
	}

	/**
     * {@inheritDoc}
     */
	public int getWidth() {
		return width;
	}

	/**
     * {@inheritDoc}
     */
	public void setWidth(int width) {
		int oldWidth = this.width;
		this.width = width;
		if (oldWidth != width) {
			firePropertyChange("constraint", "", null);
		}
	}

	/**
     * {@inheritDoc}
     */
	public void setConstraint(int left, int top, int width, int height) {
		this.left = left;
		this.top = top;
		this.width = width;
		this.height = height;
		firePropertyChange("constraint", "", null);
	}

//	private List javaScripts = new ArrayList();
//
//	public void addJavaScript(String content) {
//		JavaScript javascriptDef = new JavaScript();
//		javascriptDef.setContent(content);
//		javaScripts.add(javascriptDef);
//	}

	/**
     * {@inheritDoc}
     */
	public Component getContainer() {
		if (getParent() instanceof Component) {
			return (Component) getParent();
		}
		return null;
	}

	/**
     * {@inheritDoc}
     */
	public String[] getEventTypes() {
		return new String[0];
	}

	/**
     * {@inheritDoc}
     */
	public boolean hasEvent(String eventType) {
		String[] eventTypes = getEventTypes();
		if (eventTypes != null && eventTypes.length > 0) {
			for (int i = 0; i < eventTypes.length; i++) {
				if (eventTypes[i].equals(eventType)) {
					return true;
				}
			}
		}
		return false;
	}

	/**
     * {@inheritDoc}
     */
	public Event getEvent(String eventType) {
		if (hasEvent(eventType)) {
			EventDef eventDef = getLayout().getLayoutDef().getEventDef();

			jp.sf.maskat.core.event.Component component = eventDef
					.findComponent(getName());
			if (component != null) {
				return component.findEvent(eventType);
			}
		}
		return null;
	}

	/**
     * {@inheritDoc}
     */
	public void setEvent(Event event) {
		if (hasEvent(event.getId())) {
			EventDef eventDef = getLayout().getLayoutDef().getEventDef();
			jp.sf.maskat.core.event.Component component = eventDef
					.findComponent(getName());
			if (component == null) {
				component = new jp.sf.maskat.core.event.Component();
				component.setId(getName());
				eventDef.addChild(component);
			}
			component.setEvent(event);
			firePropertyChange("#event", null, event);
		}
	}

	/**
     * {@inheritDoc}
     */
	public void addPropertyChangeListener(PropertyChangeListener l) {
		listeners.addPropertyChangeListener(l);
	}

	/**
	 * オブジェクトプロパティに対するバウンドプロパティの変更の報告をサポートします
	 * このメソッドは、プロパティが変更された時に呼び出されPropertyChangeSupport#
	 * firePropertyChangeメソッドに委託されます。
	 * 
	 * {@link PropertyChangeSupport#firePropertyChange(String, Object, Object)}
	 *  
	 * @param propName 値が変更されたプロパティ
	 * @param oldValue プロパティの変更前の値
	 * @param newValue プロパティの変更後の値
	 */
	public void firePropertyChange(String propName, Object oldValue,
			Object newValue) {
		listeners.firePropertyChange(propName, oldValue, newValue);
	}

	/**
     * {@inheritDoc}
     */
	public void removePropertyChangeListener(PropertyChangeListener l) {
		listeners.removePropertyChangeListener(l);
	}

	/**
     * {@inheritDoc}
     */
	public Object clone() throws CloneNotSupportedException {
		AbstractComponent def = (AbstractComponent) super.clone();
		def.listeners = new PropertyChangeSupport(def);
		return def;
	}

	/**
	 * コンポーネント名の変更時にイベント定義にあるコンポーネント名も
	 * 合わせて新しいコンポーネント名に修正します
	 * 
	 * @param oldId 修正前のコンポーネント名
	 * @param newId 修正を行うコンポーネント名
	 */
	private void updateComponentIdInEvent(String oldId, String newId) {
		if (oldId == null || oldId.equals(newId)) {
			return;
		}
		if (getLayout() == null || getLayout().getLayoutDef() == null
				|| getLayout().getLayoutDef().getEventDef() == null) {
			return;
		}

		EventDef eventDef = getLayout().getLayoutDef().getEventDef();
		jp.sf.maskat.core.event.Component component = eventDef
				.findComponent(oldId);
		if (component != null) {
			component.setId(newId);
		}

		// TODO should update all Source.obj , Source.idxRef, Target.in,
		// Target.out who equals to oldId ?
		List sources = getAllDescendants(eventDef, Source.class);
		for (int i = 0; i < sources.size(); i++) {
			Source source = (Source) sources.get(i);
			if (oldId.equals(source.getObj())) {
				source.setObj(newId);
			}
			if (oldId.equals(source.getIdxRef())) {
				source.setIdxRef(newId);
			}
		}
		List targets = getAllDescendants(eventDef, Target.class);
		for (int i = 0; i < targets.size(); i++) {
			Target target = (Target) targets.get(i);
			if (oldId.equals(target.getIn())) {
				target.setIn(newId);
			}
			if (oldId.equals(target.getOut())) {
				target.setOut(newId);
			}
		}
	}

}
