/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.layout;

import java.beans.PropertyChangeListener;

import jp.sf.maskat.core.event.Event;

/**
 * コンポーネントのインターフェースです
 * 
 * レイアウト上に配置するコンポーネント部品のモデルはこのインターフェースを
 * インプリメントする必要があります。
 */
public interface Component extends LayoutElement {

	/**
	 * このコンポーネントが属しているコンテナを取得します
	 * 
	 * @return 属しているコンテナ
	 *          親がコンテナでない場合、nullを返します
	 */
	public Component getContainer();

	/**
	 * コンポーネント名を取得します
	 * 
	 * @return コンポーネント名
	 */
	public String getName();

	/**
	 * コンポーネント名を設定します
	 * 
	 * @param name コンポーネント名
	 */
	public void setName(String name);

	/**
	 * タブインデックスを取得します
	 * 
	 * @return タブインデックス
	 */
	public int getTabIndex();

	/**
	 * タブインデックスを設定します
	 * 
	 * @param tabIndex タブインデックス
	 */
	public void setTabIndex(int tabIndex);

	/**
	 * コンポーネントの高さを取得します
	 * 
	 * @return コンポーネントの高さ
	 */
	public int getHeight();

	/**
	 * コンポーネントの高さを設定します
	 * 
	 * @param height コンポーネントの高さ
	 */
	public void setHeight(int height);

	/**
	 * コンポーネントのleft位置を取得します
	 * 
	 * @return left位置
	 */
	public int getLeft();

	/**
	 * コンポーネントのleft位置を設定します
	 * 
	 * @param left left位置
	 */
	public void setLeft(int left);

	/**
	 * コンポーネントのtop位置を取得する
	 * 
	 * @return コンポーネントのtop位置
	 */
	public int getTop();

	/**
	 * コンポーネントのtop位置を設定する
	 * 
	 * @param top top位置
	 */
	public void setTop(int top);

	/**
	 * コンポーネントの幅を取得する
	 * 
	 * @return コンポーネントの幅
	 */
	public int getWidth();

	/**
	 * コンポーネントの幅を設定する
	 * 
	 * @param width コンポーネントの幅
	 */
	public void setWidth(int width);

	/**
	 * コンポーネントの表示条件を設定します
	 * 
	 * @param left コンポーネントのleft
	 * @param top コンポーネントのtop
	 * @param width コンポーネントのwidth
	 * @param height コンポーネントのheight
	 */
	public void setConstraint(int left, int top, int width, int height);

	/**
	 * コンポーネントに定義可能であるイベントタイプを配列で取得します
	 * 
	 * @return イベントタイプの配列
	 */
	public String[] getEventTypes();

	/**
	 * 指定されたイベントタイプがコンポーネントに定義可能であるか
	 * 判定します
	 * 
	 * @param eventType イベントタイプ
	 * @return 存在する場合、trueを返します
	 */
	public boolean hasEvent(String eventType);

	/**
	 * 指定したイベントタイプで登録されたイベントを取得します
	 * 
	 * @param eventType イベントタイプ
	 * @return イベント
	 */
	public Event getEvent(String eventType);

	/**
	 * 指定したイベントを設定します
	 * 
	 * @param event イベント 
	 */
	public void setEvent(Event event);

	/**
	 * PropertyChangeListenerを追加します
	 * 
	 * @param l 追加するリスナー
	 */
	public void addPropertyChangeListener(PropertyChangeListener l);

	/**
	 * PropertyChangeListenerを削除します
	 * 
	 * @param l 削除するリスナー
	 */
	public void removePropertyChangeListener(PropertyChangeListener l);
}
