/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.layout;

import java.util.HashMap;
import java.util.Map;

/**
 * コンポーネントライブラリークラスです
 * 
 * このクラスは各部品ライブラリごとに生成され以下の情報を管理します 
 * ・プリフィックス
 * ・名前空間
 * ・デフォルトの有無
 *    - デフォルトとはレイアウト定義XMLに出力した際にプリフィックス
 *      が付かない部品ライブラリです
 * ・コンポーネント情報（名前とBean）
 */
public class ComponentLibrary {

	/**
	 * 部品ライブラリのプリフィックス
	 */
	private String prefix;

	/**
	 * 部品ライブラリの名前空間
	 */
	private String namespaceURI;

	/**
	 * デフォルトライブラリフラグ
	 */
	private boolean isDefault;

	/**
	 * このライブラリに属するコンポーネントを登録するマップ
	 * key：ノード名、value: DynaComponentClass
	 */
	private Map types = new HashMap();

	/**
	 * コンストラクタです
	 * 
	 * @param prefix 部品ライブラリのプリフィックス
	 * @param namespaceURI 部品ライブラリの名前空間
	 * @param isDefault デフォルトライブラリフラグ
	 */
	public ComponentLibrary(String prefix, String namespaceURI,
			boolean isDefault) {
		super();
		this.prefix = prefix;
		this.namespaceURI = namespaceURI;
		this.isDefault = isDefault;
	}

	/**
	 * 部品ライブラリのプリフィックスを取得します
	 * 
	 * @return 部品ライブラリのプリフィックス
	 */
	public String getPrefix() {
		return prefix;
	}

	/**
	 * 部品ライブラリの名前空間を取得します
	 * 
	 * @return 部品ライブラリの名前空間
	 */
	public String getNamespaceURI() {
		return namespaceURI;
	}

	/**
	 * デフォルトライブラリフラグを取得します
	 * 
	 * @return デフォルトライブラリフラグ
	 */
	public boolean isDefault() {
		return isDefault;
	}

	/**
	 * デフォルトライブラリフラグを設定します
	 * 
	 * @param isDefault デフォルトライブラリフラグ
	 */
	public void setDefault(boolean isDefault) {
		this.isDefault = isDefault;
	}

	/**
	 * コンポーネントタイプを取得します
	 * 
	 * @param name タグ名
	 * @return コンポーネントタイプ
	 */
	public Object getComponentType(String name) {
		if (types.isEmpty() || !types.containsKey(name)) {
			return null;
		}
		return types.get(name);
	}

	/**
	 * 指定したタグ名のコンポーネントが存在するか判定します
	 * 
	 * @param name タグ名
	 * @return 存在する場合にはtrue
	 */
	public boolean containsComponent(String name) {
		return !(types.isEmpty() || !types.containsKey(name));
	}

	/**
	 * このライブラリに登録されている全てのコンポーネントタイプを
	 * 配列で取得する
	 * 
	 * @return コンポーネントタイプの配列
	 */
	public Object[] getComponentTypes() {
		return types.values().toArray();
	}

	/**
	 * コンポーネントタイプを登録します
	 * 
	 * @param name タグ名
	 * @param type コンポーネントタイプ
	 */
	public void registerComponentType(String name, Object type) {
		types.put(name, type);
	}

}
