/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.layout;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.maskat.core.MaskatCorePlugin;
import jp.sf.maskat.core.layout.ComponentLibrary;
import jp.sf.maskat.core.layout.DynaProperty;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;

/**
 * 部品ライブラリプラグインの拡張ポイントからコンポーネントの
 * 情報を取得しComponetLibraryクラスに登録します
 * {@link ComponentLibrary}
 */
public class ComponentRegistry {

	/**
	 * 部品ライブラリマップ
	 */
	private static Map registry = new HashMap();

	/**
	 * プロパティ型（定義名とJavaクラスとのマッピング）
	 */
	private static Map propertyTypes;

	/**
	 * デフォルトライブラリリスト
	 */
	private static ArrayList defaultLibrarys = new ArrayList();

	static {
		propertyTypes = new HashMap();
		propertyTypes.put("boolean", java.lang.Boolean.class);
		propertyTypes.put("int", java.lang.Integer.class);
		propertyTypes.put("float", java.lang.Float.class);
		propertyTypes.put("double", java.lang.Double.class);
		propertyTypes.put("string", java.lang.String.class);

		/*
		 * 拡張ポイント jp.sf.maskat.core.widgets として定義されている
		 * 部品ライブラリを登録する
		 */
		IExtensionPoint point = Platform.getExtensionRegistry()
				.getExtensionPoint(MaskatCorePlugin.PLUGIN_ID, "widgets");
		IExtension[] extensions = point.getExtensions();
		for (int i = 0; i < extensions.length; i++) {
			registerLibraries(extensions[i].getConfigurationElements());
		}
	}

	/**
	 * コンフィグレーションから部品情報をComponentLibraryクラスへ
	 * 登録します。全ての部品ライブラリに対してこの処理を行います
	 * 
	 * @param elements 部品ライブラリのIConfigurationElement
	 */
	private static void registerLibraries(IConfigurationElement[] elements) {
		ComponentLibrary library = null;
		for (int i = 0; i < elements.length; i++) {
			IConfigurationElement element = elements[i];
			if (!"library".equals(element.getName())) {
				continue;
			}
			String prefix = element.getAttribute("prefix");
			String namespaceURI = element.getAttribute("namespaceURI");
			String defaultlib = element.getAttribute("default");
			boolean isDefault = new Boolean(defaultlib).booleanValue();

			try {
				if (element.getAttribute("class") != null) {
					library = (ComponentLibrary) element
							.createExecutableExtension("class");
				} else {
					library = new ComponentLibrary(prefix, namespaceURI,
							isDefault);
					registerComponentTypes(library, element
							.getChildren("widget"));
				}
				if (isDefault) {
					defaultLibrarys.add(library);
				}
				registry.put(namespaceURI, library);
			} catch (CoreException e) {
				MaskatCorePlugin.log(e.getStatus());
			}
		}
	}

	/**
	 * コンフィグレーションから登録先のComponentLibraryに全ての
	 * 部品を登録します
	 * 
	 * @param library 登録先のComponentLibrary
	 * @param elements　部品のIConfigurationElement
	 */
	private static void registerComponentTypes(ComponentLibrary library,
			IConfigurationElement[] elements) {
		DynaComponentClass type = null;
		for (int i = 0; i < elements.length; i++) {
			IConfigurationElement element = elements[i];
			type = new DynaComponentClass(library,
					element.getAttribute("name"), collectDynaProperties(element
							.getChildren("property")),
					collectEventTypes(element.getChildren("event")),
					collectDefaultValues(element.getChildren("property")),
					collectChildrenProperties(element.getChildren("child")),
					collectFlags(element));
			library.registerComponentType(type.getName(), type);
		}
	}

	/**
	 * 指定されたコンフィグレーションからプロパティを取得し
	 * マップとして返却します
	 * 
	 * @param elements プロパティのIConfigurationElement
	 * @return プロパティマップ
	 */
	private static Map collectChildrenProperties(
			IConfigurationElement[] elements) {
		HashMap result = new HashMap();
		for (int i = 0; i < elements.length; i++) {
			IConfigurationElement element = elements[i];
			if (!"child".equals(element.getName())) {
				continue;
			}
			HashMap map = new HashMap();
			map.put("repeat", element.getAttribute("repeat"));
			String nameSpace = element.getAttribute("namespaceURI");
			Map ns = (Map) result.get(nameSpace);
			if (ns == null) {
				ns = new HashMap();
				result.put(nameSpace, ns);
			}
			ns.put(element.getAttribute("name"), map);
		}
		return result;
	}

	/**
	 * 指定されたコンフィグレーションからタイプを取得しフラグとして
	 * 返却します
	 * 
	 * @param element 部品ライブラリのIConfigurationElement
	 * 
	 * @return フラグ （以下のフラグが立ちます）
	 *    type == "container"： DynaComponentClass.CONTAINER
	 *    type == "component"： DynaComponentClass.COMPONENT
	 *    type == "item"：      DynaComponentClass.ITEM;
	 *    nameが有効：          DynaComponentClass.WIDGET_NAME
	 *    tabFocusが有効：      DynaComponentClass.NO_FOCUS
	 *    widthが有効:          DynaComponentClass.RESIZABLE_WIDTH
	 *    heightが有効：        DynaComponentClass.RESIZABLE_HEIGHT
	 *    topが有効：           DynaComponentClass.MOVABLE_TOP
	 *    leftが有効：          DynaComponentClass.MOVABLE_LEFT
	 */
	private static int collectFlags(IConfigurationElement element) {
		int flag = DynaComponentClass.NONE;

		if ("container".equals(element.getAttribute("type"))) {
			flag |= DynaComponentClass.CONTAINER;
		} else if ("component".equals(element.getAttribute("type"))) {
			flag |= DynaComponentClass.COMPONENT;
		} else if ("item".equals(element.getAttribute("type"))) {
			flag |= DynaComponentClass.ITEM;
		}
		if (Boolean.valueOf(element.getAttribute("widgetName")).
				booleanValue()) {
			flag |= DynaComponentClass.WIDGET_NAME;
		}
		if (!Boolean.valueOf(element.getAttribute("tabFocus")).booleanValue()) {
			flag |= DynaComponentClass.NO_FOCUS;
		}
		if ("width".equals(element.getAttribute("size"))) {
			flag |= DynaComponentClass.RESIZABLE_WIDTH;
		} else if ("height".equals(element.getAttribute("size"))) {
			flag |= DynaComponentClass.RESIZABLE_HEIGHT;
		} else if ("both".equals(element.getAttribute("size"))) {
			flag |= DynaComponentClass.RESIZABLE_BOTH;
		}
		if ("top".equals(element.getAttribute("position"))) {
			flag |= DynaComponentClass.MOVABLE_TOP;
		} else if ("left".equals(element.getAttribute("position"))) {
			flag |= DynaComponentClass.MOVABLE_LEFT;
		} else if ("both".equals(element.getAttribute("position"))) {
			flag |= DynaComponentClass.MOVABLE_BOTH;
		}
		return flag;
	}

	/**
	 * 指定されたコンフィグレーションからDynaPropertyを生成
	 * し配列として取得します
	 * 
	 * @param elements プロパティのIConfigurationElement
	 * @return DynaPropertyクラスの配列
	 */
	private static DynaProperty[] collectDynaProperties(
			IConfigurationElement[] elements) {
		List result = new ArrayList();

		for (int i = 0; i < elements.length; i++) {
			IConfigurationElement element = elements[i];
			if (!"property".equals(element.getName())) {
				continue;
			}
			boolean isCdataSection = false;
			String name = element.getAttribute("name");
			if ("#cdata-section".equalsIgnoreCase(name)) {
				name = "context";
				isCdataSection = true;
			} else if ("#text".equalsIgnoreCase(name)) {
				name = "context";
			}
			Class type = (Class) propertyTypes
					.get(element.getAttribute("type"));
			DynaProperty property = new DynaProperty(name, type);
			property.setCdataSection(isCdataSection);

			String required = element.getAttribute("required");
			property.setRequired(Boolean.valueOf(required).booleanValue());

			IConfigurationElement[] items = element.getChildren("enum");
			String[] selectItems = new String[items.length];
			for (int j = 0; j < items.length; j++) {
				selectItems[j] = items[j].getAttribute("value");
			}
			property.setItems(selectItems);

			String editable = element.getAttribute("editable");
			if (editable == null || !"false".equalsIgnoreCase(editable)) {
				property.setEditable(true);
			} else {
				property.setEditable(false);
			}
			String deprecated = element.getAttribute("deprecated");
			boolean isDeprecated = (deprecated != null &&
					"true".equalsIgnoreCase(deprecated));
			property.setDeprecated(isDeprecated);
			
			result.add(property);
		}
		return (DynaProperty[]) result.toArray(new DynaProperty[result.size()]);
	}

	/**
	 * 指定されたコンフィグレーションから定義されているイベントタイプ
	 * を配列として取得します
	 * 
	 * @param elements プロパティのIConfigurationElement
	 * @return イベントタイプの配列
	 */
	private static String[] collectEventTypes(IConfigurationElement[] elements) {
		List result = new ArrayList();

		for (int i = 0; i < elements.length; i++) {
			IConfigurationElement element = elements[i];
			if (!"event".equals(element.getName())) {
				continue;
			}
			result.add(element.getAttribute("type"));
		}
		return (String[]) result.toArray(new String[result.size()]);
	}

	/**
	 * 指定されたコンフィグレーションからデフォルト値をマップとして
	 * 取得します
	 * 
	 * @param elements プロパティのIConfigurationElement
	 * @return デフォルト値のマップ
	 *          key: プロパティ名, value: デフォルト値
	 */
	private static Map collectDefaultValues(IConfigurationElement[] elements) {
		Map result = new HashMap();

		for (int i = 0; i < elements.length; i++) {
			IConfigurationElement element = elements[i];
			if (!"property".equals(element.getName())) {
				continue;
			}
			String name = element.getAttribute("name");
			if ("#cdata-section".equalsIgnoreCase(name)
					|| "#text".equalsIgnoreCase(name)) {
				name = "context";
			}
			String value = element.getAttribute("default");
			Class type = (Class) propertyTypes
					.get(element.getAttribute("type"));
			if (value == null) {
				if (Number.class.isAssignableFrom(type)) {
					continue;
				} else if (Boolean.class.isAssignableFrom(type)) {
					continue;
				} else {
					value = "";
				}
			}
			Object initValue = null;

			try {
				Constructor constructor = type
						.getConstructor(new Class[] { String.class });
				initValue = constructor.newInstance(new Object[] { value });

			} catch (NoSuchMethodException e) {
				initValue = value;
			} catch (IllegalArgumentException e) {
				initValue = value;
			} catch (InstantiationException e) {
				initValue = value;
			} catch (IllegalAccessException e) {
				initValue = value;
			} catch (InvocationTargetException e) {
				initValue = value;
			}
			result.put(name, initValue);
		}
		return result;
	}

	/**
	 * 登録済みの部品ライブラリの中から指定された名前空間とタグ名に該当する
	 * コンポーネントタイプを取得します
	 * 
	 * @param namespaceURI 名前空間
	 * @param name タグ名
	 * @return 該当するコンポーネントタイプ。存在しない場合にはnull
	 */
	public static Object getComponentType(String namespaceURI, String name) {
		if (namespaceURI == null || !registry.containsKey(namespaceURI)) {
			ComponentLibrary[] libraries = getLibraries();
			for (int i = 0; i < libraries.length; i++) {
				ComponentLibrary library = libraries[i];
				if (library.containsComponent(name) && library.isDefault()) {
					return library.getComponentType(name);
				}
			}
			return null;
		}
		return getLibrary(namespaceURI).getComponentType(name);
	}

	/**
	 * 登録済みの部品ライブラリの中から指定された名前空間に該当する
	 * ComponentLibraryを取得します
	 * 
	 * @param namespaceURI 名前空間
	 * @return 該当するComponentLibrary。存在しない場合にはnull
	 */
	public static ComponentLibrary getLibrary(String namespaceURI) {
		if (!registry.containsKey(namespaceURI)) {
			return null;
		}
		return (ComponentLibrary) registry.get(namespaceURI);
	}

	/**
	 * 登録されている全てのComponentLibraryを配列で取得します
	 * 
	 * @return ComponentLibraryの配列
	 */
	public static ComponentLibrary[] getLibraries() {
		return (ComponentLibrary[]) registry.values().toArray(
				new ComponentLibrary[registry.size()]);
	}

	/**
	 * デフォルトコンストラクタです
	 */
	private ComponentRegistry() {
		super();
	}

	/**
	 * デフォルトライブラリを配列で取得します
	 * 
	 * @return デフォルトライブラリの配列
	 */
	public static ComponentLibrary[] getDefaultLibrarys() {
		return (ComponentLibrary[]) defaultLibrarys
				.toArray(new ComponentLibrary[0]);
	}
}
