/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.core.layout;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import jp.sf.maskat.core.event.EventDef;

/**
 * LayoutDefのモデルクラスです
 * 
 * レイアウト定義XMLのLayoutDefタグに該当するモデルクラスです
 */
public class LayoutDef extends AbstractLayoutElement {

	/**
	 * このLayoutDefに定義されている部品ライブラリの名前空間
	 * namespaceがキーで prefixが格納されている
	 */
	private Map namespaces;

	/**
	 * このLayoutDefに定義されているjavaScript
	 * JavaScriptGlobalオブジェクトが格納されている
	 */
	private List javaScripts = new ArrayList();

	/**LayoutDef
	 * このLayoutDefに定義されているEventDefオブジェクト
	 */
	private EventDef eventDef;

	/**
	 * このLayoutDefに定義されている部品ライブラリの名前空間を取得します
	 * 
	 * @return 部品ライブラリ名前空間マップ
	 *          namespaceがキーで prefixが格納されている
	 */
	public Map getNamespaces() {
		return namespaces;
	}

	/**
	 * このLayoutDefに定義されている部品ライブラリの名前空間を設定します
	 * 
	 * @param namespaces 部品ライブラリ名前空間マップ
	 *                    namespaceがキーで prefixが格納されている
	 */
	public void setNamespaces(Map namespaces) {
		this.namespaces = namespaces;
	}

	/**
	 * 指定した部品ライブラリの名前空間からプリフィックスを取得します
	 * 
	 * @param namespaceURI 部品ライブラリの名前空間
	 * @return プリフィックス
	 */
	public String getPrefix(String namespaceURI) {
		return (String) namespaces.get(namespaceURI);
	}

	/**
	 * Layoutを配列で取得します
	 * @deprecated マスカットIDE v2.0からLayoutDefには１つのLayoutしか定義
	 *               できない仕様となりました。次バージョンからこのメソッドは
	 *               削除される可能性があります。
	 * @return このLayoutDefが持っているLayoutの配列
	 */
	public Layout[] getLayouts() {
		return (Layout[]) getChildren().toArray(new Layout[] {});
	}

	/**
	 * EventDefを取得します
	 * 
	 * @return このLayoutDefが持っているEventDef
	 */
	public EventDef getEventDef() {
		return eventDef;
	}

	/**
	 * EventDefを設定します
	 * 
	 * @param eventDef EventDefオブジェクト
	 */
	public void setEventDef(EventDef eventDef) {
		this.eventDef = eventDef;
	}

	/**
	 * 子要素としてjavaScriptを追加します
	 * 
	 * @deprecated javaScriptGlobalクラスは今後削除される可能性があります
	 * @param content 追加するjavaScriptコード
	 */
	public void addJavaScriptGlobal(String content) {
		JavaScriptGlobal javascriptDef = new JavaScriptGlobal();
		javascriptDef.setContent(content);
		javaScripts.add(javascriptDef);
	}

	/**
     * {@inheritDoc}
     */
	public void addChild(Object child) {
		if (child instanceof Layout) {
			((Layout) child).setLayoutDef(this);
		}
		super.addChild(child);
	}

	/**
     * {@inheritDoc}
     */
	public void addChildToIdx(Object child, int idx) {
		if (child instanceof Layout) {
			((Layout) child).setLayoutDef(this);
		}
		super.addChildToIdx(child, idx);
	}
}
