/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors;

import java.util.List;

import jp.sf.maskat.ui.Messages;
import jp.sf.maskat.ui.editors.layout.LayoutGraphicalEditor;
import jp.sf.maskat.ui.editors.layout.actions.CreateComponentAction;
import jp.sf.maskat.ui.editors.layout.editpolicies.PopupBarDescriptorRegistory;
import jp.sf.maskat.ui.editors.layout.editpolicies.PopupBarEditPolicy;

import org.eclipse.gef.EditPart;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.action.SubMenuManager;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.ui.actions.ActionFactory;

/**
 * レイアウトエディタのコンテキストメニューマネージャクラスです
 * 
 * レイアウトエディタ上で表示するコンテキストメニューに「新規追加」メニューを
 * 追加する処理を行います。新規追加メニューには現在選択されている部品の構成要素
 * を作成するメニューを表示します。
 * 
 */
public class LayoutEditorMenuManager extends MenuManager {

	/*
	 * 新規追加メニュー
	 */
    private SubMenuManager menuWrapper;
    
    /*
     * セレクションプロバイダ
     */
    private ISelectionProvider provider;
    
    /*
     * メニュータイトル
     */
    private static final String MENUTITLE = Messages.getString(
    		"layout.contextmenu.newItem"); //$NON-NLS-1$
    
    /**
     * コンストラクタです
     * 
     * @param editor　エディタ
     */
	public LayoutEditorMenuManager(LayoutGraphicalEditor editor) {
		provider = editor.getSite().getSelectionProvider();
		menuWrapper = new SubMenuManager(this);
		menuWrapper.setVisible(true);
		
		addMenuListener(new IMenuListener() {
	        public void menuAboutToShow(IMenuManager manager) {
	        	menuWrapper.removeAll();
	        	EditPart editPart = getAdditionalEditPart(getSelectedEditPart());
	        	if (editPart != null) {
	        		//CreateComponentAction[] commands =
	        		List commands = PopupBarDescriptorRegistory.getMenuActions(editPart);
	        		if (commands.size() > 0) { 
		        		MenuManager menu = new MenuManager(MENUTITLE);
	        			for (int i = 0; i < commands.size(); i++) {
	        				Object command = commands.get(i);
	        				if (command instanceof IAction) {
	        					menu.add((IAction) command);
	        				} else if (command instanceof IContributionItem) {
	        					menu.add((IContributionItem) command);
	        				}
	        			}
	        			menuWrapper.insertBefore(ActionFactory.UNDO.getId(), menu);
	        			menuWrapper.insertBefore(ActionFactory.UNDO.getId(), new Separator());
	        		}
	        	}
	        }
		});
	}
	
	/**
	 * 現在選択されているエディトパートを取得します
	 * 
	 * @return 選択されているエディトパート
	 */
	private EditPart getSelectedEditPart() {
		Object obj = provider.getSelection();
		if (obj instanceof IStructuredSelection) {
			Object editPart = ((IStructuredSelection) obj).getFirstElement();
			if (editPart instanceof EditPart) {
				return (EditPart) editPart;
			}
		}
		return null;
	}
	
	/**
	 * 追加対象であるエディトパートを取得します。
	 * 
	 * 階層構造の上位へ向けて PopupBarEditPolicy.ROLE がインストールされているエディト
	 * パートを検索します。これによりPopupBarで追加可能である項目とコンテキストメニュー
	 * による項目が同一になります。
	 * 
	 * @param select 選択されているエディトパート
	 * @return 追加対象であるエディトパート
	 */
	private EditPart getAdditionalEditPart(EditPart select) {
		while (select != null && select.getEditPolicy(PopupBarEditPolicy.ROLE) == null) {
			select = select.getParent();
		}
		return select;
	}
}
