/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors.layout;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.sf.maskat.core.layout.ComponentRegistry;
import jp.sf.maskat.core.layout.UnknownComponentClass;
import jp.sf.maskat.ui.ISharedImages;
import jp.sf.maskat.ui.MaskatNature;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.editors.layout.requests.BeanCreationFactory;
import jp.sf.maskat.ui.editors.layout.tools.AdvancedCreationEntry;
import jp.sf.maskat.ui.editors.layout.tools.AdvancedSelectionToolEntry;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.gef.palette.CreationToolEntry;
import org.eclipse.gef.palette.MarqueeToolEntry;
import org.eclipse.gef.palette.PaletteContainer;
import org.eclipse.gef.palette.PaletteDrawer;
import org.eclipse.gef.palette.PaletteEntry;
import org.eclipse.gef.palette.PaletteGroup;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.palette.ToolEntry;
import org.eclipse.gef.requests.CreationFactory;
import org.eclipse.gef.ui.palette.PaletteCustomizer;
import org.eclipse.gef.ui.palette.customize.DefaultEntryPage;
import org.eclipse.gef.ui.palette.customize.DrawerEntryPage;
import org.eclipse.gef.ui.palette.customize.EntryPage;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;

/**
 * レイアウトエディタのパレットをカスタマイズするクラスです
 * 
 * 以下の情報を編集することが可能です。設定情報はプレファレンスストアに
 * 保存されワークスペースごとに共有されます。
 * 
 * ・Drawerの表示、非表示、初期時のオープン、クローズ指定、PIN設定
 * ・CreationToolの表示、非表示
 * ・Drawer, CreationToolの表示順番
 */
public class PluggablePaletteCustomizer extends PaletteCustomizer {

	/**
	 * パレットに表示するデフォルトコンポーネントイメージ
	 */
	private static ImageDescriptor DEFAULT_IMAGE_DESCRIPTOR = MaskatUIPlugin
			.getImageDescriptor(ISharedImages.IMG_COMPONENTICON);

	/**
	 * パレットルート
	 */
	private PaletteRoot paletteRoot;

	/**
	 * Drawer, CreationToolのマップ
	 */
	private Map entries;
	
	/**
	 * Drawerとnamespaeceの対応マップ
	 */
	private Map drawerMap;
	
	/**
	 * プロジェクト
	 */
	private IProject project;
	
	/**
	 * デフォルトコンストラクタです
	 */
	public PluggablePaletteCustomizer() {
		super();
		paletteRoot = new PaletteRoot();
		entries = new HashMap();
		drawerMap = new HashMap();
		initializePalette();
		revertToSaved();
	}

	/**
	 * パレットルートを取得します
	 * 
	 * @return パレットルート
	 */
	public PaletteRoot getPaletteRoot() {
		return paletteRoot;
	}

	/**
	 * パレットの初期化を行います
	 * 
	 * 拡張ポイント「paletteEntries」に定義されているDrawer, CreationToolを
	 * パレットに追加します。またDrawer, CreationToolの他にSelectionToolを追加します
	 */
	private void initializePalette() {
		PaletteGroup group = new PaletteGroup("tools"); //$NON-NLS-1$
		//ToolEntry entry = new SelectionToolEntry();
		ToolEntry entry = new AdvancedSelectionToolEntry();
		paletteRoot.setDefaultEntry(entry);
		group.add(entry);
		group.add(new MarqueeToolEntry());
		paletteRoot.add(group);

		IExtensionPoint point = Platform.getExtensionRegistry()
				.getExtensionPoint(MaskatUIPlugin.PLUGIN_ID, "paletteEntries"); //$NON-NLS-1$
		IExtension[] extensions = point.getExtensions();

		/*
		 * Drawerを最初にパレットへ追加します
		 */
		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] elements = extensions[i]
					.getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				if ("drawer".equals(elements[j].getName())) {
					addPaletteDrawer(elements[j],
							extensions[i].getNamespaceIdentifier());
				}
			}
		}
		/*
		 * CreationToolをパレットに追加します
		 */
		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] elements = extensions[i]
					.getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				if ("creationTool".equals(elements[j].getName())) {
					addCreationToolEntry(elements[j]);
				}
			}
		}
	}
	
	/**
	 * カスタマイズされた情報が定義されている場合にはそれに従い
	 * 並び替えおよび初期設定を反映させます。
	 * 
	 * @param entry カスタマイズされたエントリ
	 * @param palent entryの親エントリ
	 */
	private void createPaletteEntry(StoreEntry entry, StoreContainer palent) {
		if (!(entry instanceof RootEntry)) {
			PaletteEntry p = (PaletteEntry) entries.get(palent.getId());
			PaletteEntry c = (PaletteEntry) entries.get(entry.getId());
			if (c != null) {
				if (p instanceof PaletteDrawer) {
					((PaletteDrawer) p).remove(c);
					((PaletteDrawer) p).add(c);

				} else if (p == null) {
					paletteRoot.add(c);
				}
				if (!(c instanceof PaletteDrawer)) {
					entries.remove(entry.getId());
				} else {
					if (c instanceof PaletteDrawer
							&& entry instanceof StoreDrawer) {
						StoreDrawer dr = (StoreDrawer) entry;
						int state = dr.isInitiallyOpen() ? PaletteDrawer.INITIAL_STATE_OPEN
								: PaletteDrawer.INITIAL_STATE_CLOSED;

						if (dr.isInitiallyOpen() && dr.isInitiallyPinned()) {
							state = PaletteDrawer.INITIAL_STATE_PINNED_OPEN;
						}
						((PaletteDrawer) c).setInitialState(state);
					}
				}
				c.setVisible(entry.isVisible());
			}
		}
		if (entry instanceof StoreContainer) {
			StoreContainer container = (StoreContainer) entry;
			List children = container.getChildren();
			for (int i = 0; i < children.size(); i++) {
				createPaletteEntry((StoreEntry) children.get(i), container);
			}
		}
	}
	
    /**
     * {@inheritDoc}
     */
	public boolean canDelete(PaletteEntry entry) {
		return false;
	}

    /**
     * {@inheritDoc}
     */
	public List getNewEntryFactories() {
		return new ArrayList();
	}

    /**
     * {@inheritDoc}
     */
	public EntryPage getPropertiesPage(PaletteEntry entry) {
		if (entry instanceof PaletteDrawer) {
			return new WidgetDrawerEntryPage();
		}
		return new WidgetEntryPage();
	}

	/**
	 * Drawer編集ページクラスです
	 * 説明文を編集できないように DrawerEntryPage を拡張しています。
	 */
	private class WidgetDrawerEntryPage extends DrawerEntryPage {
	    /**
	     * {@inheritDoc}
	     */		
		protected Text createDescText(Composite panel) {
			Text text = super.createDescText(panel);
			text.setEditable(false);
			return text;
		}
	    /**
	     * {@inheritDoc}
	     */
		protected Text createNameText(Composite panel) {
			Text text = super.createNameText(panel);
			text.setEditable(false);
			return text;
		}
	}

	/**
	 * CreationTool編集ページクラスです
	 * 説明文を編集できないように DefaultEntryPage を拡張しています。
	 */
	private class WidgetEntryPage extends DefaultEntryPage {
	    /**
	     * {@inheritDoc}
	     */
		protected Text createDescText(Composite panel) {
			Text text = super.createDescText(panel);
			text.setEditable(false);
			return text;
		}
	    /**
	     * {@inheritDoc}
	     */
		protected Text createNameText(Composite panel) {
			Text text = super.createNameText(panel);
			text.setEditable(false);
			return text;
		}
	}

	/**
	 * パレットに Drawer を追加します。
	 * 
	 * @param element drawerのIConfigurationElementクラス
	 * @param pluginId このDrawerが属しているプラグインのID
	 */
	private void addPaletteDrawer(IConfigurationElement element, String pluginId) {
		PaletteDrawer drawer = new PaletteDrawer(element.getAttribute("label"),
				MaskatUIPlugin.getImageDescriptor(element, "icon"));
		String id = element.getAttribute("id");
		drawer.setId(id);
		entries.put(id, drawer);
		drawerMap.put(drawer, pluginId);

		/*
		 * 親パスがある場合には、親Drapwerの子要素として登録
		 */
		String path = element.getAttribute("path");
		if (path != null && entries.containsKey(path)) {
			PaletteEntry entryForPath = (PaletteEntry) entries.get(path);
			if (entryForPath instanceof PaletteContainer) {
				((PaletteContainer) entryForPath).add(drawer);
			} else {
				entryForPath.getParent().add(drawer);
			}
		}
	}

	/**
	 * パレットに CreationTool を追加します
	 * 
	 * @param element creationToolのIConfigurationElementクラス
	 */
	private void addCreationToolEntry(IConfigurationElement element) {
		String namespaceURI = element.getAttribute("namespaceURI");
		String name = element.getAttribute("name");
		CreationFactory factory = null;

		try {
			if (element.getAttribute("factory") != null) {
				factory = (CreationFactory) element
						.createExecutableExtension("factory");
			} else {
				IConfigurationElement[] children = element
						.getChildren("property");
				Map properties = null;
				if (children != null) {
					properties = new HashMap();
					for (int i = 0; i < children.length; i++) {
						String key = children[i].getAttribute("name");
						String value = children[i].getAttribute("value");
						properties.put(key, value);
					}
				}
				Object type = ComponentRegistry.getComponentType(namespaceURI,
						name);
				type = type == null ? new UnknownComponentClass() : type;
				factory = new BeanCreationFactory(type, properties);
			}
		} catch (CoreException e) {
			MaskatUIPlugin.log(e.getStatus());
		}

		ImageDescriptor iconSmall = MaskatUIPlugin.getImageDescriptor(element,
				"iconSmall");
		ImageDescriptor iconLarge = MaskatUIPlugin.getImageDescriptor(element,
				"iconLarge");
		CreationToolEntry entry = new AdvancedCreationEntry(element
				.getAttribute("label"), element.getAttribute("description"),
				factory, (iconSmall != null) ? iconSmall
						: DEFAULT_IMAGE_DESCRIPTOR,
				(iconLarge != null) ? iconLarge : DEFAULT_IMAGE_DESCRIPTOR);

		String id = element.getAttribute("id");
		entry.setId(id);
		entries.put(id, entry);

		/*
		 * 親パスがある場合には、親Drapwerの子要素として登録
		 */
		String path = element.getAttribute("path");
		if (path != null && entries.containsKey(path)) {
			PaletteEntry entryForPath = (PaletteEntry) entries.get(path);
			if (entryForPath instanceof PaletteContainer) {
				((PaletteContainer) entryForPath).add(entry);
			} else {
				entryForPath.getParent().add(entry);
			}
		}
	}

    /**
     * {@inheritDoc}
     */
	public void revertToSaved() {
	}

    /**
     * {@inheritDoc}
     */
	public void save() {
		//IPreferenceStore store = MaskatUIPlugin.getDefault()
		//		.getPreferenceStore();
		StringBuffer sb = new StringBuffer();
		List children = paletteRoot.getChildren();
		for (int i = 0; i < children.size(); i++) {
			PaletteEntry entry = (PaletteEntry) children.get(i);
			if (!(entry instanceof PaletteGroup)) {
				sb.append(getStoreString(entry));
			}
		}
		//store.setValue(PreferenceConstants.PALETTE_ENTRIES, sb.toString());
		try {
			if (project != null) {
				MaskatNature nature = MaskatNature.getNature(project);
				nature.setPaletteEntries(sb.toString());
				nature.getPreferenceStore().save();
			}
			
		} catch (IOException e) {
			MaskatUIPlugin.log(new Status(IStatus.ERROR,
					MaskatUIPlugin.PLUGIN_ID, 0, e.getMessage(), e)); 
		}
	}

	/**
	 * パレットカスタマイズ情報をプレファレンスストアに格納する文字列に
	 * 変換します。
	 * 
	 * @param entry パレットエントリ
	 * @return プレファレンスストアに格納する文字列
	 */
	private String getStoreString(PaletteEntry entry) {
		StringBuffer sb = new StringBuffer();
		sb.append(entry.getId());
		sb.append(":");
		sb.append(entry.getParent() != null ? entry.getParent().getId() : "");
		sb.append(":");
		sb.append(entry.isVisible());

		if (entry instanceof PaletteDrawer) {
			PaletteDrawer drawer = (PaletteDrawer) entry;
			sb.append(":");
			sb.append(drawer.isInitiallyOpen());
			sb.append(":");
			sb.append(drawer.isInitiallyPinned());
			sb.append(";");

			List children = ((PaletteContainer) entry).getChildren();
			for (int i = 0; i < children.size(); i++) {
				sb.append(getStoreString((PaletteEntry) children.get(i)));
			}

		} else {
			sb.append(";");
		}
		return sb.toString();
	}

	/**
	 * プレファレンスストアに格納されている設定値（文字列）をパースして
	 * StoreEntryオブジェクトへ変換します。
	 * 
	 * 設定値フォーマット
	 * [entry];[entry];[entry] ; によってエントリが区切られています
	 * 
	 * エントリはDrawerとCreationToolの２つがあり各エントリは : によって
	 * 値が区切られています
	 * 
	 * ・Drawer entry = [id]:[parentId]:[hide]:[open]:[pin]
	 * ・CreationTool entry = [id]:[parentId]:[hide]
	 *  
	 * @param value 設定値（文字列）
	 * @return StoreEntryオブジェクト
	 */
	private StoreEntry parseStoreEntry(String value) {
		RootEntry root = new RootEntry();
		if (value == null) {
			return root;
		}
		String[] entries = value.split(";");
		for (int i = 0; i < entries.length; i++) {
			if ("".equals(entries[i].trim())) {
				continue;
			}
			String[] values = entries[i].split(":");
			if (values.length < 3) {
				continue;
			}
			String id = values[0];
			String parentId = values[1];
			boolean visible = new Boolean(values[2]).booleanValue();
			StoreEntry entry;

			if (values.length == 5) {
				entry = new StoreDrawer(id, visible, new Boolean(values[3])
						.booleanValue(), new Boolean(values[4]).booleanValue());
			} else {
				entry = new StoreEntry(id, visible);
			}
			root.addChild(entry, parentId);
		}
		return root;
	}

	/**
	 * プレファレンスストアに定義されたパレットカスタマイズ情報の 
	 * 格納用クラスです。
	 */
	private class StoreEntry {
		private String id = "";

		private boolean visible = false;

		public StoreEntry() {
		}

		public StoreEntry(String id, boolean visible) {
			this.id = id;
			this.visible = visible;
		}

		public String getId() {
			return id;
		}

		public void setId(String id) {
			this.id = id;
		}

		public boolean isVisible() {
			return visible;
		}

		public void setVisible(boolean visible) {
			this.visible = visible;
		}
	}
	
	/**
	 * プレファレンスストアに定義されたパレットカスタマイズ情報の 
	 * RootEntry 格納用クラスです。
	 *
	 */
	private class RootEntry extends StoreContainer {
	}

	/**
	 * プレファレンスストアに定義されたパレットカスタマイズ情報の 
	 * Drawer 格納用クラスです。
	 *
	 */
	private class StoreDrawer extends StoreContainer {
		private boolean initiallyOpen = false;

		private boolean initiallyPinned = false;

		public StoreDrawer(String id, boolean visible, boolean open, boolean pin) {
			super(id, visible);
			this.initiallyOpen = open;
			this.initiallyPinned = pin;
		}

		public boolean isInitiallyOpen() {
			return initiallyOpen;
		}

		public void setInitiallyOpen(boolean initiallyOpen) {
			this.initiallyOpen = initiallyOpen;
		}

		public boolean isInitiallyPinned() {
			return initiallyPinned;
		}

		public void setInitiallyPinned(boolean initiallyPinned) {
			this.initiallyPinned = initiallyPinned;
		}

	}

	/**
	 * プレファレンスストアに定義されたパレットカスタマイズ情報の 
	 * Container 格納用クラスです。
	 *
	 */
	private class StoreContainer extends StoreEntry {
		private List children = new ArrayList();

		public StoreContainer() {
			super("", false);
		}

		public StoreContainer(String id, boolean visible) {
			super(id, visible);
		}

		public List getChildren() {
			return children;
		}

		public void addChild(StoreEntry child) {
			children.add(child);
		}

		public boolean addChild(StoreEntry child, String parentId) {
			if ("".equals(parentId) || getId().equals(parentId)) {
				addChild(child);
				return true;
			} else {
				for (int i = 0; i < children.size(); i++) {
					StoreEntry entry = (StoreEntry) children.get(i);
					if (entry instanceof StoreContainer) {
						boolean result = ((StoreContainer) entry).addChild(
								child, parentId);
						if (result) {
							return true;
						}
					}
				}
			}
			return false;
		}

		public void setChildren(List children) {
			this.children = children;
		}
	}
	
	public IProject getProject() {
		return project;
	}

	public void setProject(IProject project) {
		this.project = project;
	}

	public void updatePaletteEntry(List plugins, String paletteEntries) {
		/*
		 * インストールされていない部品ライブラリは非表示にする
		 */
		for (Iterator ite = drawerMap.keySet().iterator(); ite.hasNext();) {
			PaletteDrawer drawer = (PaletteDrawer) ite.next();
			String id = (String) drawerMap.get(drawer);
			drawer.setVisible(plugins.contains(id));
		}
		/*
		 * プリファレンスストアに登録されているパレット変更情報を反映させる
		 */
		createPaletteEntry(parseStoreEntry(paletteEntries), null);

		/*
		 * プレファレンスストアに登録されているパレット情報以外のエントリが
		 * 新たに追加された場合、パレットに追加する（新たな部品ライブラリの追加等）
		 */
		for (Iterator ite = entries.keySet().iterator(); ite.hasNext();) {
			PaletteEntry p = (PaletteEntry) entries.get(ite.next());
			PaletteEntry px = p;
			PaletteEntry root = null;
			while (px.getParent() != paletteRoot) {
				px = px.getParent();
				if (px == null) {
					if (!(p instanceof PaletteDrawer)) {
						if (root != null) {
							paletteRoot.add(root);
						} else {
							p.setVisible(false);
						}
					}
					break;
				}
				root = px;
			}
		}
	}
}
