/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors.layout.actions;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import jp.sf.maskat.core.MaskatElement;
import jp.sf.maskat.core.layout.Component;
import jp.sf.maskat.core.layout.DynaComponent;
import jp.sf.maskat.core.layout.DynaComponentClass;
import jp.sf.maskat.core.layout.LayoutElement;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.Messages;
import jp.sf.maskat.ui.editors.layout.LayoutGraphicalEditor;
import jp.sf.maskat.ui.editors.layout.commands.AddBasicDefCommand;
import jp.sf.maskat.ui.editors.layout.editparts.LayoutElementEditPart;
import jp.sf.maskat.ui.editors.layout.tools.AdvancedSelectionTool;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.draw2d.FigureCanvas;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PrecisionRectangle;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.RequestConstants;
import org.eclipse.gef.Tool;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.internal.GEFMessages;
import org.eclipse.gef.requests.ChangeBoundsRequest;
import org.eclipse.gef.ui.actions.Clipboard;
import org.eclipse.gef.ui.actions.SelectionAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.menus.CommandContributionItem;

/**
 * ペーストアクションクラスです
 * 
 * レイアウトエディタのメニュー、コンテキストメニューから呼び出され
 * クリップボートに入っているLayoutElementクラスの貼り付けを行います。
 *
 */
public class PasteComponentAction extends SelectionAction {

    private LayoutElement targetDef = null;

    private LayoutGraphicalEditor editor = null;
    
    private boolean menuPaste = false;

	/**
	 * コンストラクタです
	 * 
	 * @param part アクションを行うWorkbenchPart
	 */
    public PasteComponentAction(LayoutGraphicalEditor editor) {
        super(editor);
        this.editor = editor;

        setId(ActionFactory.PASTE.getId());
        setText(GEFMessages.PasteAction_Label);
        setToolTipText(GEFMessages.PasteAction_Tooltip);
        ISharedImages sharedImages = PlatformUI.getWorkbench()
                .getSharedImages();
        setImageDescriptor(sharedImages
                .getImageDescriptor(ISharedImages.IMG_TOOL_PASTE));
        setDisabledImageDescriptor(sharedImages
                .getImageDescriptor(ISharedImages.IMG_TOOL_PASTE_DISABLED));

    }

	/**
     * ペーストが行えるか判定します。
     * 
     * @return ペーストが行える場合、trueを返却します
	 */
    protected boolean calculateEnabled() {
        if (!checkClipboard()) {
            return false;
        }
        List objects = this.getSelectedObjects();
        if (objects.isEmpty()) {
            return false;
        }
        if (objects.size() > 1) {
            return false;
        }
        Object selectedObj = objects.get(0);
        if (!(selectedObj instanceof EditPart)) {
            return false;
        }
        Object model = ((EditPart) selectedObj).getModel();
        if (!(model instanceof LayoutElement)) {
            return false;
        }
        targetDef = (LayoutElement) model;

        List basicDefList = (List) Clipboard.getDefault().getContents();
        return canBePasted(basicDefList);
    }

    /**
     * クリップボートにマスカット要素が入っているか判定します
     * 
     * @return マスカット要素が List で格納されている場合、trueを返却します
     */
    private boolean checkClipboard() {
        Object clipboardContent = Clipboard.getDefault().getContents();
        if (!(clipboardContent instanceof List)) {
            return false;
        }
        for (int i = 0; i < ((List) clipboardContent).size(); i++) {
            if (!(((List) clipboardContent).get(i) instanceof MaskatElement)) {
                return false;
            }
        }
        return true;
    }

    public void runWithEvent(Event event) {
    	menuPaste = event.widget instanceof MenuItem;
    	if (menuPaste && event.widget.getData() instanceof CommandContributionItem) {
    		menuPaste = false;
    	}
    	super.runWithEvent(event);
	}

	/**
     * ペーストコマンドを実行しクリップボードに入っているマスカット要素を
     * 貼り付けます。
     */
    public void run() {
        if (!checkClipboard()) {
            return;
        }
        GraphicalViewer viewer = (GraphicalViewer) editor.getAdapter(GraphicalViewer.class);
        Tool tool = viewer.getEditDomain().getActiveTool();
        if (!(tool instanceof AdvancedSelectionTool)) {
        	return;
        }
        AdvancedSelectionTool advancedTool = (AdvancedSelectionTool) tool;
        Point point = advancedTool.getMouseLocation();
        if (point.x < 0 && this.menuPaste) {
        	this.menuPaste = false;
        }
        Command command = null;
        List basicDefList = (List) Clipboard.getDefault().getContents();
        List source = new ArrayList();
        for (Iterator it = basicDefList.iterator(); it.hasNext();) {
            LayoutElement def = (LayoutElement) it.next();
            try {
                LayoutElement clone = (LayoutElement) def.clone();
                source.add(clone);
            } catch (CloneNotSupportedException e) {
                MessageDialog.openError(
                		this.getWorkbenchPart().getSite().getShell(),
                		Messages.getString("layout.cmd.msg.error.title"), //$NON-NLS-1$
                        Messages.getString("layout.cmd.paste.msg.error")); //$NON-NLS-1$
                MaskatUIPlugin.log(new Status(IStatus.ERROR,
                        MaskatUIPlugin.PLUGIN_ID, IStatus.ERROR,
                        e.getMessage(), e));
                return;
            }
        }
        /*
         * 現在のマウス座標を貼り付ける部品の座標とします。
         * ・コンテキストメニューからの配置
         * ・キー (Ctl + V)による配置
         * 
         * 貼り付け対象が複数存在する場合は、全部品の一番小さな座標が基準座標
         * となります。
         */
        GraphicalEditPart editPart = (GraphicalEditPart) getSelectedObjects().get(0);
        PrecisionRectangle rect = new PrecisionRectangle(editPart.getFigure().getBounds().getCopy());
        editPart.getFigure().translateToAbsolute(rect);
        point.x = point.x < rect.x ? 0 : point.x - rect.x;
        point.y = point.y < rect.y ? 0 : point.y - rect.y;

		ArrayList modifiedTop = new ArrayList();
		ArrayList modifiedLeft = new ArrayList();
		Point offset = new Point(Integer.MAX_VALUE, Integer.MAX_VALUE);
		
		for (Iterator it = source.iterator(); it.hasNext();) {
			LayoutElement element = (LayoutElement) it.next();
			if (element instanceof Component && targetDef instanceof Component) {
				Component compDef = (Component) element;
				if (compDef instanceof DynaComponent) {
					DynaComponentClass dynaClass = (DynaComponentClass)
						((DynaComponent) compDef).getDynaClass();
					if (dynaClass.isTopMovable()) {
						modifiedTop.add(compDef);
						offset.y = Math.min(compDef.getTop(), offset.y);
					}
					if (dynaClass.isLeftMovable()) {
						modifiedLeft.add(compDef);
						offset.x = Math.min(compDef.getLeft(), offset.x);
					}
				} else {
					modifiedTop.add(compDef);
					modifiedLeft.add(compDef);
					offset.y = Math.min(compDef.getTop(), offset.y);
					offset.x = Math.min(compDef.getLeft(), offset.x);
				}
			}
			if (command == null) {
				command = genCommand(targetDef, element);
			} else {
				command = command.chain(genCommand(targetDef, element));
			}
		}
		offset.x = (offset.x == Integer.MAX_VALUE) ? 0 : offset.x;
		offset.y = (offset.y == Integer.MAX_VALUE) ? 0 : offset.y;
		if (!this.menuPaste) {
			point.x = 10;
			point.y = 10;
			offset.x = 0;
			offset.y = 0;
		}
		for (Iterator it = modifiedTop.iterator(); it.hasNext();) {
			Component component = (Component) it.next();
			component.setTop(point.y + component.getTop() - offset.y);
		}
		for (Iterator it = modifiedLeft.iterator(); it.hasNext();) {
			Component component = (Component) it.next();
			component.setLeft(point.x + component.getLeft() - offset.x);
		}
		execute(command);

        List toCopy = new ArrayList();
        for (Iterator it = source.iterator(); it.hasNext();) {
            LayoutElement element = (LayoutElement) it.next();
            try {
                toCopy.add(element.clone());
            } catch (CloneNotSupportedException e) {
                /* suppress */
            }
        }
        if (canBePasted(toCopy)) {
            Clipboard.getDefault().setContents(toCopy);
        } else {
            this.refresh();
        }
    }
    
    /**
     * ペーストコマンドを生成します
     * 
     * @param parentDef 貼り付け元要素
     * @param childDef 貼り付ける要素
     * @return 生成されたペーストコマンド
     */
    private Command genCommand(LayoutElement parentDef, LayoutElement childDef) {
        return new AddBasicDefCommand(targetDef, childDef);
    }

    /**
     * 指定されたマスカット要素が、貼り付け元の子要素として格納できるか
     * 判定します。
     * @param basicDefList 貼り付けるマスカット要素の配列
     * @return 貼り付けることが可能な場合、trueを返却します
     */
    private boolean canBePasted(List basicDefList) {
        GraphicalViewer viewer = (GraphicalViewer) this.editor
                .getAdapter(GraphicalViewer.class);

        List sourceEPList = new ArrayList();
        for (Iterator it = basicDefList.iterator(); it.hasNext();) {
            LayoutElement def = (LayoutElement) it.next();
            GraphicalEditPart ep = (GraphicalEditPart) viewer
                    .getEditPartFactory().createEditPart(null, def);

            ep.setModel(def);
            FigureCanvas canvas = (FigureCanvas) viewer.getControl();
            Rectangle rec = canvas.getViewport().getBounds();
            ep.getFigure()
                    .setBounds(new Rectangle(rec.width, rec.height, 0, 0));
            sourceEPList.add(ep);
        }
        List objects = this.getSelectedObjects();
        EditPart targetEP = (EditPart) objects.get(0);
        if (targetEP instanceof LayoutElementEditPart) {
            for (int i = 0; i < sourceEPList.size(); i++) {
                if (!((LayoutElementEditPart) targetEP)
                        .canAddChild(((EditPart) sourceEPList.get(i))
                                .getModel())) {
                    return false;
                }
                EditPart sourceEditPart = (EditPart) sourceEPList.get(i);
                if (sourceEditPart instanceof LayoutElementEditPart) {
                    if (!((LayoutElementEditPart) sourceEditPart)
                            .canAddParent(targetEP.getModel())) {
                        return false;
                    }
                }
            }
        }

        ChangeBoundsRequest request = new ChangeBoundsRequest(
                RequestConstants.REQ_ADD);
        request.setEditParts(sourceEPList);
        Command command = targetEP.getCommand(request);
        return command == null ? false : command.canExecute();
    }
}
