/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors.layout.editpolicies;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.maskat.core.MaskatCorePlugin;
import jp.sf.maskat.core.layout.ComponentRegistry;
import jp.sf.maskat.core.layout.UnknownComponentClass;
import jp.sf.maskat.ui.ISharedImages;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.editors.layout.actions.CreateComponentAction;
import jp.sf.maskat.ui.editors.layout.requests.BeanCreationFactory;
import jp.sf.maskat.ui.editors.layout.tools.PopupBarTool;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.requests.CreationFactory;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;

public class PopupBarDescriptorRegistory {
	
	/**
	 * EditPartに定義されているPopupBar定義を保持するマップ
	 * EditPartのClassオブジェクトをキーにDescriptorのリストが格納されています
	 */
	private static Map entries = new HashMap();
	
	/**
	 * 初期化時に拡張ポイントからPopupBar定義を取得します
	 */
	static {
		CreationFactoryRegistory registory = new CreationFactoryRegistory();
		
		/*
		 * 拡張ポイント jp.sf.maskat.ui.popupEntriesとして定義されている
		 * PoupuBarエントリを取得します
		 */
		IExtensionPoint point = Platform.getExtensionRegistry()
				.getExtensionPoint(MaskatUIPlugin.PLUGIN_ID, "popupEntries");
		IExtension[] extensions = point.getExtensions();
		for (int i = 0; i < extensions.length; i++) {
			registorDescriptor(extensions[i].getConfigurationElements(), registory);
		}
	}

	private static class CreationFactoryRegistory {
		private static Map entries = new HashMap();
		
		public CreationFactoryRegistory() {
			/*
			 * 拡張ポイント jp.sf.maskat.ui.paletteEntriesとして定義されている
			 * 各部品のCreationFactoryを取得します
			 */
			IExtensionPoint point = Platform.getExtensionRegistry()
				.getExtensionPoint(MaskatUIPlugin.PLUGIN_ID, "paletteEntries");
			IExtension[] extensions = point.getExtensions();
			
			for (int i = 0; i < extensions.length; i++) {
				IConfigurationElement[] elements = extensions[i]
						.getConfigurationElements();
				for (int j = 0; j < elements.length; j++) {
					if ("creationTool".equals(elements[j].getName())) {
						addCreationToolEntry(elements[j]);
					}
				}
			}			
		}
		
		private static void addCreationToolEntry(IConfigurationElement element) {
			String namespaceURI = element.getAttribute("namespaceURI");
			String name = element.getAttribute("name");
			CreationFactory factory = null;

			try {
				if (element.getAttribute("factory") != null) {
					factory = (CreationFactory) element
							.createExecutableExtension("factory");
				} else {
					IConfigurationElement[] children = element
							.getChildren("property");
					Map properties = null;
					if (children != null) {
						properties = new HashMap();
						for (int i = 0; i < children.length; i++) {
							String key = children[i].getAttribute("name");
							String value = children[i].getAttribute("value");
							properties.put(key, value);
						}
					}
					Object type = ComponentRegistry.getComponentType(namespaceURI,
							name);
					type = type == null ? new UnknownComponentClass() : type;
					factory = new BeanCreationFactory(type, properties);
				}
				Map map = (Map) entries.get(namespaceURI);
				if (map == null) {
					map = new HashMap();
					entries.put(namespaceURI, map);
				}
				map.put(name, factory);
				
			} catch (CoreException e) {
				MaskatUIPlugin.log(e.getStatus());
			}
		}
		
		public static CreationFactory getCreationFactory(
				String namespaceURI, String name) {
			
			Map map = (Map) entries.get(namespaceURI);
			return map != null ? (CreationFactory) map.get(name) : null;
		}
	}

	/**
	 * PopupBarのeditPartごとの情報を取得します
	 * 
	 * @param elements プラグイン単位のpopupEntriesエレメント
	 */
	private static void registorDescriptor(
			IConfigurationElement[] elements, CreationFactoryRegistory registory) {
		for (int i = 0; i < elements.length; i++) {
			IConfigurationElement element = elements[i];
			if (!"editPart".equals(element.getName())) {
				continue;
			}
			Class editPartClass = null;
			try {
				editPartClass = element.createExecutableExtension("class").getClass();
			} catch (CoreException e) {
				MaskatCorePlugin.log(e.getStatus());
				continue;
			}
			List list = (List) entries.get(editPartClass);
			if (list == null) {
				list = new ArrayList();
				entries.put(editPartClass, list);
			}
			IConfigurationElement[] children = element.getChildren();
			for (int j = 0; j < children.length; j++) {
				IConfigurationElement child = children[j];
				if ("create".equals(child.getName())) {
					String namespaceURI = child.getAttribute("namespaceURI");
					String name = child.getAttribute("name");
					String message = child.getAttribute("message");
					ImageDescriptor img = MaskatUIPlugin.getImageDescriptor(child, "icon");
					Image icon = img != null ? img.createImage() : null;
					CreationFactory factory = registory.getCreationFactory(namespaceURI, name);
					list.add(new CreateDescriptor(message, icon, factory));
				} else if ("separator".equals(child.getName())) {
					Image img = MaskatUIPlugin.getImage(ISharedImages.IMAGE_SEPARATOR);
					list.add(new SeparatorDescriptor(img));
				}
			}
		}
	}

	/**
	 * PopupBarに表示されるボタンの幅
	 */
	private static final int POPUP_BUTTON_WIDTH = 30;
	
	/**
	 * PopupBarに表示されるセパレータの幅
	 */
	private static final int POPUP_SEPARATOR_WIDHT = 16;
	
	/**
	 * PopupBarに表示されるボタンの高さ
	 */
	private static final int POPUP_BUTTON_HEIGHT = 20;
	
	/**
	 * PopupBarに表示するボタンの情報を持つクラスです 
	 */
	private static class Descriptor {
		private String message;
		private Image icon;
		protected Dimension size;
		public Descriptor(String message, Image icon) {
			this.message = message;
			this.icon = icon;
		}
		public String getMessage() {
			return this.message;
		}
		public Image getIcon() {
			return this.icon;
		}
		public Dimension getDimension() {
			return this.size;
		}
	}
	
	/**
	 * PopupBarに表示されるセパレータの情報を持つクラスです
	 */
	private static class SeparatorDescriptor extends Descriptor {
		public SeparatorDescriptor(Image icon) {
			super(null, icon);
			this.size = new Dimension(POPUP_SEPARATOR_WIDHT, POPUP_BUTTON_HEIGHT);
		}
	}
	
	/**
	 * PopupBarに表示する部品追加ボタンの情報を持つクラスです
	 */
	private static class CreateDescriptor extends Descriptor {
		private CreationFactory factory;
		public CreateDescriptor(String message, Image icon,
				CreationFactory factory) {
			super(message, icon);
			this.factory = factory;
			this.size = new Dimension(POPUP_BUTTON_WIDTH, POPUP_BUTTON_HEIGHT);
		}
		public CreationFactory getFacotry() {
			return this.factory;
		}
	}
	
	/**
	 * 指定されたEditPartに定義されたPopupBarのディスクリプタを取得します
	 * 
	 * @param targetEditPart EditPart
	 * @return ディスクリプタの配列
	 */
	public static PopupBarDescriptor[] getDescriptors(EditPart targetEditPart) {
		List list = (List) entries.get(targetEditPart.getClass());
		if (list == null) {
			return new PopupBarDescriptor[0];
		}
		List descriptors = new ArrayList();
		for (int i = 0; i < list.size(); i++) {			
			Descriptor desciptor = (Descriptor) list.get(i);
			PopupBarTool tracker = null;
			if (desciptor instanceof CreateDescriptor) {
				CreationFactory factory =
						((CreateDescriptor) desciptor).getFacotry();
				tracker = new PopupBarTool(targetEditPart, factory);
			}
			descriptors.add(new PopupBarDescriptor(desciptor.getMessage(),
					desciptor.getIcon(), desciptor.getDimension(), tracker));
		}
		return (PopupBarDescriptor[]) descriptors.toArray(
				new PopupBarDescriptor[descriptors.size()]);
	}

	/**
	 * 指定されたEditPartに定義された「新規追加」コマンドリストを取得します
	 *  
	 * @param targetEditPart 対象EditPart
	 * @return 「新規追加」コマンドのリスト
	 */
	public static List getMenuActions(EditPart targetEditPart) {
		List list = (List) entries.get(targetEditPart.getClass());
		if (list == null) {
			return new ArrayList();
		}
		List descriptors = new ArrayList();
		for (int i = 0; i < list.size(); i++) {			
			Descriptor desciptor = (Descriptor) list.get(i);
			if (desciptor instanceof CreateDescriptor) {
				descriptors.add(new CreateComponentAction(desciptor.getMessage(),
									desciptor.getIcon(),
									((CreateDescriptor) desciptor).getFacotry(),
									targetEditPart));
			} else if (desciptor instanceof SeparatorDescriptor) {
				descriptors.add(new Separator());
			}
		}
		return descriptors;
	}

	
	/**
	 * 指定されたEditPartにPopupBarのディスクリプタが定義されているか判定します
	 * 
	 * @param targetEditPart
	 * @return 定義されている場合には true を返します。
	 */
	public static boolean containsDescriptor(EditPart targetEditPart) {
		List list = (List) entries.get(targetEditPart.getClass());
		return list != null && list.size() > 0;
	}
}
