/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors.layout.tools;

import java.util.ArrayList;
import java.util.List;

import jp.sf.maskat.core.layout.Component;
import jp.sf.maskat.ui.editors.layout.editparts.LayoutEditPart;
import jp.sf.maskat.ui.editors.layout.outline.OutLineTreeEditPart;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gef.requests.ChangeBoundsRequest;
import org.eclipse.gef.tools.SelectionTool;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;

/**
 * グラフィカルエディタ上でのデフォルトセレクションツールです。
 * 以下の機能を拡張したセレクションツールです。
 * 
 * ・ツール利用時のマウス座標を取得する
 * ・キーによる部品の移動処理
 */
public class AdvancedSelectionTool extends SelectionTool {
	
	/**
	 * キーによる部品移動サイズ
	 */
	private int MOVE_SIZE = 10;
	
	/**
	 * 現在のマウス座標を返します
	 * 
	 * @return マウスの絶対座標
	 */
	public Point getMouseLocation() {
		return getCurrentInput().getMouseLocation();
	}

	/**
	 * キーを押されたときに部品の移動コマンドを実行します
	 * 矢印キーによる移動を行います。１回の移動で10px移動します。
	 * またコントロールキーを押下しながら移動された場合は1px単位で移動します。
	 */
	protected boolean handleKeyDown(KeyEvent e) {
		Point point = null;
		int moveSize = (e.stateMask & SWT.CTRL) != 0 ? 1 : MOVE_SIZE;
		switch (e.keyCode) {
		case SWT.ARROW_UP:
			point = new Point(0, -moveSize);
			break;
		case SWT.ARROW_DOWN:
			point = new Point(0, moveSize);
			break;
		case SWT.ARROW_LEFT:
			point = new Point(-moveSize, 0);
			break;
		case SWT.ARROW_RIGHT:
			point = new Point(moveSize, 0);
			break;
		}
		Command command = createCommand(point);
		if (command != null && command.canExecute()) {
			executeCommand(command);
		}
		return super.handleKeyDown(e);
	}

	/**
	 * 部品移動コマンドを生成します。
	 * 
	 * @param point 移動座標
	 * @return 移動コマンド
	 */
	private Command createCommand(Point point) {
		CompoundCommand commands = null;
		if (point != null) {
			//List editParts = getCurrentViewer().getSelectedEditParts();
			List editParts = getMoveEditParts();
			int size = editParts.size();
			if (size > 0) {
				commands = new CompoundCommand();
				for (int i = 0; i < size; i++) {
					EditPart editPart = (EditPart) editParts.get(i);
					if (editPart.getModel() instanceof Component) {
						ChangeBoundsRequest request = new ChangeBoundsRequest(REQ_MOVE);
						request.setMoveDelta(point);
						request.setEditParts(editPart);
						Command command = editPart.getCommand(request);
						if (command != null && command.canExecute()) {
							commands.add(command);
						}
					}
				}
			}
		}
		return commands;
	}

	/*
	 * 親子関係で重複しているコンポーネントを省いたリストを作成します
	 * 
	 * @return 重複しているコンポーネントを省いたリスト 
	 */
	private List getMoveEditParts() {
		List list = new ArrayList();
		List editParts = getCurrentViewer().getSelectedEditParts();
		
		for (int i = 0; i < editParts.size(); i++) {
			EditPart editPart = (EditPart) editParts.get(i);
			EditPart e = editPart.getParent();
			
			if (e instanceof OutLineTreeEditPart) {
				list.add(editPart);
			} else {
				while (!(e instanceof LayoutEditPart)) {
					if (editParts.contains(e)) {
						break;
					}
					e = e.getParent();
				}
				if (e instanceof LayoutEditPart) {
					list.add(editPart);
				}
			}
		}
		return list;
	}
}
