/*
 * Copyright (c) 2006-2010 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package jp.sf.maskat.ui.editors.preview;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import jp.sf.maskat.core.MaskatElementVisitor;
import jp.sf.maskat.core.betwixt.MaskatBeanIO;
import jp.sf.maskat.core.layout.Component;
import jp.sf.maskat.core.layout.LayoutDef;
import jp.sf.maskat.runtime.MaskatRuntimePlugin;
import jp.sf.maskat.ui.MaskatNature;
import jp.sf.maskat.ui.MaskatUIPlugin;
import jp.sf.maskat.ui.Messages;

import org.apache.commons.io.IOUtils;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

public class LayoutPreviewBrowser extends EditorPart {

	/** The editor ID */
	public static final String EDITOR_ID = MaskatUIPlugin.PLUGIN_ID
			+ ".layoutPreviewBrowser"; //$NON-NLS-1$

	/** The swt browser control used to render layout */
	private Browser browser;

	public LayoutPreviewBrowser() {
		super();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createPartControl(Composite parent) {
		ScrolledComposite composite = new ScrolledComposite(
				parent, SWT.H_SCROLL | SWT.V_SCROLL);
		composite.setLayout(new FillLayout());
		composite.setExpandHorizontal(true);
		composite.setExpandVertical(true);
		browser = new Browser(composite, SWT.NONE);
		composite.setContent(browser);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.EditorPart#init(org.eclipse.ui.IEditorSite,
	 *      org.eclipse.ui.IEditorInput)
	 */
	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		setSite(site);
		setInput(input);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.EditorPart#isDirty()
	 */
	public boolean isDirty() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
	 */
	public boolean isSaveAsAllowed() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.EditorPart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor) {
		throw new UnsupportedOperationException();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.EditorPart#doSaveAs()
	 */
	public void doSaveAs() {
		throw new UnsupportedOperationException();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	public void setFocus() {
		try {
			//browser.setText(generateContents());
			browser.setUrl(generateContents().toString());
			
		} catch (Exception e) {
			browser.setText(Messages.getString("preview.create.msg.error")); //$NON-NLS-1$
		}
		browser.setFocus();
	}

	private URL generateContents() throws IOException, CoreException {
		IFile file = (IFile) getEditorInput().getAdapter(IFile.class);
		String frameworkBase = MaskatRuntimePlugin.getFrameworkBaseURL().toString();
		try {
			MaskatNature nature = MaskatNature.getNature(file.getProject());
			if (nature != null) {
				URI uri = nature.getProject().getLocationURI();
				String fwPath = nature.getMaskatFrameworkPath().toString();
				if (fwPath != null && !fwPath.startsWith("/")) {
					fwPath = "/" + fwPath;
				}
				uri = new URI(uri.toString() + fwPath + "/"); //$NON-NLS-1$
				frameworkBase = FileLocator.toFileURL(uri.toURL()).toString();
			}
		} catch (URISyntaxException e) {}
		
		InputStream in = null;
		final Rectangle rec = browser.getBounds();
		try {
			in = file.getContents(true);
			LayoutDef layoutDef = MaskatBeanIO.readLayoutDef(in);
			layoutDef.getLayouts()[0].accept(new MaskatElementVisitor() {
				public void visit(Object def) {
					if (def instanceof Component) {
						Component comp = (Component) def;
						rec.width = Math.max(rec.width, comp.getLeft() + comp.getWidth() + 50);
						rec.height = Math.max(rec.height, comp.getTop() + comp.getHeight() + 50);
					}
				}
			});				
		} catch (Exception e) {
			throw new IOException(e.getMessage());
		} finally {
			IOUtils.closeQuietly(in);
		}
		((ScrolledComposite) browser.getParent()).setMinSize(rec.width, rec.height);
		
		/* create tempFolder */
		File idxFile = File.createTempFile("_maskat", "idx"); //$NON-NLS-1$
		File tmpFolder = new File(idxFile.getParentFile().getAbsoluteFile(), "_maskat"); //$NON-NLS-1$
		tmpFolder.mkdir();
		tmpFolder.deleteOnExit();
		idxFile.delete();
		
		/* container html */		
		InputStream containerIn = null;
		OutputStream containerOut = null;
		File containerFile = null;
		try {
			containerFile = new File(tmpFolder, "preview.html"); //$NON-NLS-1$
			containerFile.createNewFile();
			containerFile.deleteOnExit();
			containerOut = new BufferedOutputStream(new FileOutputStream(containerFile));
			containerIn = getClass().getResourceAsStream("preview.html"); //$NON-NLS-1$
			String html = IOUtils.toString(containerIn, "UTF-8"); //$NON-NLS-1$
			containerOut.write(html.replaceAll(
					"\\Q${maskatFW}\\E", frameworkBase).getBytes("UTF-8")); //$NON-NLS-1$
		} finally {
			IOUtils.closeQuietly(containerIn);
			IOUtils.closeQuietly(containerOut);
		}
		/* layout XML */
		copyFile(file, new File(tmpFolder, "layout.xml")); //$NON-NLS-1$

		/* event XML */
		copyFile("layout_e.xml",
				new File(tmpFolder, "layout_e.xml"));  //$NON-NLS-1$
		
		/* transition XML */
		copyFile("transition.xml",
				new File(tmpFolder, "transition.xml")); //$NON-NLS-1$

		/* properties.json */
		String path = file.getParent().getName();
		try {
			IFile jsonResouce = file.getProject().getFile(
					path + "/properties.json");
			copyFile(jsonResouce,
				new File(tmpFolder, "properties.json")); //$NON-NLS-1$
		} catch (CoreException e) {}
		
		return containerFile != null ? containerFile.toURL() : null;
	}
	
	private void copyFile(String resourceName, File createFile)
		throws IOException {
		copyFile(getClass().getResourceAsStream(resourceName), createFile);
	}
	
	private void copyFile(IFile resource, File createFile)
		throws IOException, CoreException {
		copyFile(resource.getContents(true), createFile);
	}
	
	private void copyFile(InputStream in, File createFile)
		throws IOException {
		OutputStream out = null;
		try {
			if (in != null) {
				createFile.createNewFile();
				createFile.deleteOnExit();
				out = new BufferedOutputStream(
						new FileOutputStream(createFile));
				IOUtils.copy(in, out);
			}
		} finally {
			IOUtils.closeQuietly(in);
			IOUtils.closeQuietly(out);			
		}			
	}
}
