/*
 * Copyright (C) 2000-2002 ASANO Masahiro
 */

#include <string.h>
#include "crash.h"
#include "asm.h"

PRIVATE addr_t dis();
struct commandtable command_dis =
	{"dis", dis, "address [nbytes]", "disassemble memory"};

PRIVATE addr_t opad;
PRIVATE int opsize;

const struct mnemonic code1[256] = {
/*00*/	{ MOD0_1,	2,	"add %s,%s"	},
/*01*/	{ MOD0_4,	2,	"add %s,%s"	},
/*02*/	{ MOD1_1,	2,	"add %s,%s"	},
/*03*/	{ MOD1_4,	2,	"add %s,%s"	},
/*04*/	{ BYTE1,	2,	"add al,%x"	},
/*05*/	{ BYTE4,	5,	"add eax,%x"	},
/*06*/	{ 0,		1,	"push es"	},
/*07*/	{ 0,		1,	"pop es"	},
/*08*/	{ MOD0_1,	2,	"or %s,%s"	},
/*09*/	{ MOD0_4,	2,	"or %s,%s"	},
/*0a*/	{ MOD1_1,	2,	"or %s,%s"	},
/*0b*/	{ MOD1_4,	2,	"or %s,%s"	},
/*0c*/	{ BYTE1,	2,	"or al,%x"	},
/*0d*/	{ BYTE4,	5,	"or eax,%x"	},
/*0e*/	{ 0,		1,	"push cs"	},
/*0f*/	{ OP2B,		2,	""	},
/*10*/	{ MOD0_1,	2,	"adc %s,%s"	},
/*11*/	{ MOD0_4,	2,	"adc %s,%s"	},
/*12*/	{ MOD1_1,	2,	"adc %s,%s"	},
/*13*/	{ MOD1_4,	2,	"adc %s,%s"	},
/*14*/	{ BYTE1,	2,	"adc al,%x"	},
/*15*/	{ BYTE4,	5,	"adc eax,%x"	},
/*16*/	{ 0,		1,	"push ss"	},
/*17*/	{ 0,		1,	"pop ss"	},
/*18*/	{ MOD0_1,	2,	"sbb %s,%s"	},
/*19*/	{ MOD0_4,	2,	"sbb %s,%s"	},
/*1a*/	{ MOD1_1,	2,	"sbb %s,%s"	},
/*1b*/	{ MOD1_4,	2,	"sbb %s,%s"	},
/*1c*/	{ BYTE1,	2,	"sbb al,%x"	},
/*1d*/	{ BYTE4,	5,	"sbb eax,%x"	},
/*1e*/	{ 0,		1,	"push ds"	},
/*1f*/	{ 0,		1,	"pop ds"	},
/*20*/	{ MOD0_1,	2,	"and %s,%s"	},
/*21*/	{ MOD0_4,	2,	"and %s,%s"	},
/*22*/	{ MOD1_1,	2,	"and %s,%s"	},
/*23*/	{ MOD1_4,	2,	"and %s,%s"	},
/*24*/	{ BYTE1,	2,	"and al,%x"	},
/*25*/	{ BYTE4,	5,	"and eax,%x"	},
/*26*/	{ 0,		1,	"es:"	},
/*27*/	{ 0,		1,	"daa"	},
/*28*/	{ MOD0_1,	2,	"sub %s,%s"	},
/*29*/	{ MOD0_4,	2,	"sub %s,%s"	},
/*2a*/	{ MOD1_1,	2,	"sub %s,%s"	},
/*2b*/	{ MOD1_4,	2,	"sub %s,%s"	},
/*2c*/	{ BYTE1,	2,	"sub al,%x"	},
/*2d*/	{ BYTE4,	5,	"sub eax,%x"	},
/*2e*/	{ 0,		1,	"cs:"	},
/*2f*/	{ 0,		1,	"das"	},
/*30*/	{ MOD0_1,	2,	"xor %s,%s"	},
/*31*/	{ MOD0_4,	2,	"xor %s,%s"	},
/*32*/	{ MOD1_1,	2,	"xor %s,%s"	},
/*33*/	{ MOD1_4,	2,	"xor %s,%s"	},
/*34*/	{ BYTE1,	2,	"xor al,%x"	},
/*35*/	{ BYTE4,	5,	"xor eax,%x"	},
/*36*/	{ 0,		1,	"ss:"	},
/*37*/	{ 0,		1,	"aaa"	},
/*38*/	{ MOD0_1,	2,	"cmp %s,%s"	},
/*39*/	{ MOD0_4,	2,	"cmp %s,%s"	},
/*3a*/	{ MOD1_1,	2,	"cmp %s,%s"	},
/*3b*/	{ MOD1_4,	2,	"cmp %s,%s"	},
/*3c*/	{ BYTE1,	2,	"cmp al,%x"	},
/*3d*/	{ BYTE4,	5,	"cmp eax,%x"	},
/*3e*/	{ 0,		1,	"ds:"	},
/*3f*/	{ 0,		1,	"aas"	},
/*40*/	{ 0,		1,	"inc eax"	},
/*41*/	{ 0,		1,	"inc ecx"	},
/*42*/	{ 0,		1,	"inc edx"	},
/*43*/	{ 0,		1,	"inc ebx"	},
/*44*/	{ 0,		1,	"inc esp"	},
/*45*/	{ 0,		1,	"inc ebp"	},
/*46*/	{ 0,		1,	"inc esi"	},
/*47*/	{ 0,		1,	"inc edi"	},
/*48*/	{ 0,		1,	"dec eax"	},
/*49*/	{ 0,		1,	"dec ecx"	},
/*4a*/	{ 0,		1,	"dec edx"	},
/*4b*/	{ 0,		1,	"dec ebx"	},
/*4c*/	{ 0,		1,	"dec esp"	},
/*4d*/	{ 0,		1,	"dec ebp"	},
/*4e*/	{ 0,		1,	"dec esi"	},
/*4f*/	{ 0,		1,	"dec edi"	},
/*50*/	{ 0,		1,	"push eax"	},
/*51*/	{ 0,		1,	"push ecx"	},
/*52*/	{ 0,		1,	"push edx"	},
/*53*/	{ 0,		1,	"push ebx"	},
/*54*/	{ 0,		1,	"push esp"	},
/*55*/	{ 0,		1,	"push ebp"	},
/*56*/	{ 0,		1,	"push esi"	},
/*57*/	{ 0,		1,	"push edi"	},
/*58*/	{ 0,		1,	"pop eax"	},
/*59*/	{ 0,		1,	"pop ecx"	},
/*5a*/	{ 0,		1,	"pop edx"	},
/*5b*/	{ 0,		1,	"pop ebx"	},
/*5c*/	{ 0,		1,	"pop esp"	},
/*5d*/	{ 0,		1,	"pop ebp"	},
/*5e*/	{ 0,		1,	"pop esi"	},
/*5f*/	{ 0,		1,	"pop edi"	},
/*60*/	{ 0,		1,	"pusha"	},
/*61*/	{ 0,		1,	"popa"	},
/*62*/	{ MOD1_4,	2,	"bound %s,%s"	},
/*63*/	{ MOD1_4,	2,	"arpl %s,%s"	},
/*64*/	{ 0,		1,	"fs:"	},
/*65*/	{ 0,		1,	"gs:"	},
/*66*/	{ OPSIZE,	1,	"Operand Size"	},
/*67*/	{ 1,		1,	"Address Size"	},
/*68*/	{ BYTE4,	5,	"push %x"	},
/*69*/	{ IMUL,		2,	"imul %s,%s,%x"	},
/*6a*/	{ BYTE1,	2,	"push %x"	},
/*6b*/	{ IMUL,		2,	"imul %s,%s,%x"	},
/*6c*/	{ 0,		1,	"insb"	},
/*6d*/	{ 0,		1,	"insd"	},
/*6e*/	{ 0,		1,	"outsb"	},
/*6f*/	{ 0,		1,	"outsd"	},
/*70*/	{ RELAD1,	2,	"jo %x"	},
/*71*/	{ RELAD1,	2,	"jno %x"	},
/*72*/	{ RELAD1,	2,	"jc %x"	},
/*73*/	{ RELAD1,	2,	"jnc %x"	},
/*74*/	{ RELAD1,	2,	"jz %x"	},
/*75*/	{ RELAD1,	2,	"jnz %x"	},
/*76*/	{ RELAD1,	2,	"jbe %x"	},
/*77*/	{ RELAD1,	2,	"jnbe %x"	},
/*78*/	{ RELAD1,	2,	"js %x"	},
/*79*/	{ RELAD1,	2,	"jns %x"	},
/*7a*/	{ RELAD1,	2,	"jp %x"	},
/*7b*/	{ RELAD1,	2,	"jnp %x"	},
/*7c*/	{ RELAD1,	2,	"jl %x"	},
/*7d*/	{ RELAD1,	2,	"jnl %x"	},
/*7e*/	{ RELAD1,	2,	"jle %x"	},
/*7f*/	{ RELAD1,	2,	"jnle %x"	},
/*80*/	{ GROUPA,	2,	"Group A"	},
/*81*/	{ GROUPA,	2,	"Group A"	},
/*82*/	{ GROUPA,	2,	"Group A"	},
/*83*/	{ GROUPA,	2,	"Group A"	},
/*84*/	{ MOD1_1,	2,	"test %s,%s"	},
/*85*/	{ MOD1_4,	2,	"test %s,%s"	},
/*86*/	{ MOD0_1,	2,	"xchg %s,%s"	},
/*87*/	{ MOD0_4,	2,	"xchg %s,%s"	},
/*88*/	{ MOD0_1,	2,	"mov %s,%s"	},
/*89*/	{ MOD0_4,	2,	"mov %s,%s"	},
/*8a*/	{ MOD1_1,	2,	"mov %s,%s"	},
/*8b*/	{ MOD1_4,	2,	"mov %s,%s"	},
/*8c*/	{ SREG0,	2,	"mov %s,%s"	},
/*8d*/	{ MOD1_4,	2,	"lea %s,%s"	},
/*8e*/	{ SREG1,	2,	"mov %s,%s"	},
/*8f*/	{ POP,		2,	"pop %s"	},
/*90*/	{ 0,		1,	"nop"	},
/*91*/	{ 0,		1,	"xchg eax,ecx"	},
/*92*/	{ 0,		1,	"xchg eax,edx"	},
/*93*/	{ 0,		1,	"xchg eax,ebx"	},
/*94*/	{ 0,		1,	"xchg eax,esp"	},
/*95*/	{ 0,		1,	"xchg eax,ebp"	},
/*96*/	{ 0,		1,	"xchg eax,esi"	},
/*97*/	{ 0,		1,	"xchg eax,edi"	},
/*98*/	{ 0,		1,	"cwde"	},
/*99*/	{ 0,		1,	"cdq"	},
/*9a*/	{ BYTE6,	7,	"call %x:%x"	},
/*9b*/	{ 0,		1,	"wait"	},
/*9c*/	{ 0,		1,	"pushf"	},
/*9d*/	{ 0,		1,	"popf"	},
/*9e*/	{ 0,		1,	"sahf"	},
/*9f*/	{ 0,		1,	"lahf"	},
/*a0*/	{ ADDR4,	5,	"mov al,[%x]"	},
/*a1*/	{ ADDR4,	5,	"mov eax,[%x]"	},
/*a2*/	{ ADDR4,	5,	"mov [%x],al"	},
/*a3*/	{ ADDR4,	5,	"mov [%x],eax"	},
/*a4*/	{ 0,		1,	"movsb"	},
/*a5*/	{ 0,		1,	"movsd"	},
/*a6*/	{ 0,		1,	"cmpsb"	},
/*a7*/	{ 0,		1,	"cmpsd"	},
/*a8*/	{ BYTE1,	2,	"test al,%x"	},
/*a9*/	{ BYTE4,	5,	"test eax,%x"	},
/*aa*/	{ 0,		1,	"stosb"	},
/*ab*/	{ 0,		1,	"stosd"	},
/*ac*/	{ 0,		1,	"lodsb"	},
/*ad*/	{ 0,		1,	"lodsd"	},
/*ae*/	{ 0,		1,	"scasb"	},
/*af*/	{ 0,		1,	"scasd"	},
/*b0*/	{ BYTE1,	2,	"mov al,%x"	},
/*b1*/	{ BYTE1,	2,	"mov cl,%x"	},
/*b2*/	{ BYTE1,	2,	"mov dl,%x"	},
/*b3*/	{ BYTE1,	2,	"mov bl,%x"	},
/*b4*/	{ BYTE1,	2,	"mov ah,%x"	},
/*b5*/	{ BYTE1,	2,	"mov ch,%x"	},
/*b6*/	{ BYTE1,	2,	"mov dh,%x"	},
/*b7*/	{ BYTE1,	2,	"mov bh,%x"	},
/*b8*/	{ BYTE4,	5,	"mov eax,%x"	},
/*b9*/	{ BYTE4,	5,	"mov ecx,%x"	},
/*ba*/	{ BYTE4,	5,	"mov edx,%x"	},
/*bb*/	{ BYTE4,	5,	"mov ebx,%x"	},
/*bc*/	{ BYTE4,	5,	"mov esp,%x"	},
/*bd*/	{ BYTE4,	5,	"mov ebp,%x"	},
/*be*/	{ BYTE4,	5,	"mov esi,%x"	},
/*bf*/	{ BYTE4,	5,	"mov edi,%x"	},
/*c0*/	{ GROUPB,	2,	"Shift Group B"	},
/*c1*/	{ GROUPB,	2,	"Shift Group B"	},
/*c2*/	{ BYTE2,	3,	"ret %x"	},
/*c3*/	{ 0,		1,	"ret"	},
/*c4*/	{ MOD1_4,	2,	"les"	},
/*c5*/	{ MOD1_4,	2,	"lds"	},
/*c6*/	{ IMM,		2,	"mov.b %s,%x"	},
/*c7*/	{ IMM,		2,	"mov.d %s,%x"	},
/*c8*/	{ ENTER,	4,	"enter %x,%x"	},
/*c9*/	{ 0,		1,	"leave"	},
/*ca*/	{ BYTE2,	3,	"retf %x"	},
/*cb*/	{ 0,		1,	"retf"	},
/*cc*/	{ 0,		1,	"int 3"	},
/*cd*/	{ BYTE1,	2,	"int %x"	},
/*ce*/	{ 0,		1,	"into"	},
/*cf*/	{ 0,		1,	"iret"	},
/*d0*/	{ GROUPB,	2,	"Shift Group B"	},
/*d1*/	{ GROUPB,	2,	"Shift Group B"	},
/*d2*/	{ GROUPB,	2,	"Shift Group B"	},
/*d3*/	{ GROUPB,	2,	"Shift Group B"	},
/*d4*/	{ 0,		2,	"aam"	},
/*d5*/	{ 0,		2,	"aad"	},
/*d6*/	{ 0,		1,	"setalc"	},
/*d7*/	{ 0,		1,	"xalt"	},
/*d8*/	{ NPX,		2,	"esc 0"	},
/*d9*/	{ NPX,		2,	"esc 1"	},
/*da*/	{ NPX,		2,	"esc 2"	},
/*db*/	{ NPX,		2,	"esc 3"	},
/*dc*/	{ NPX,		2,	"esc 4"	},
/*dd*/	{ NPX,		2,	"esc 5"	},
/*de*/	{ NPX,		2,	"esc 6"	},
/*df*/	{ NPX,		2,	"esc 7"	},
/*e0*/	{ RELAD1,	2,	"loopnz %x"	},
/*e1*/	{ RELAD1,	2,	"loopz %x"	},
/*e2*/	{ RELAD1,	2,	"loop %x"	},
/*e3*/	{ RELAD1,	2, 	"jecxz %x"	},
/*e4*/	{ BYTE1,	2,	"in al,%x"	},
/*e5*/	{ BYTE1,	2,	"in eax,%x"	},
/*e6*/	{ BYTE1,	2,	"out %xh,al"	},
/*e7*/	{ BYTE1,	2,	"out %xh,eax"	},
/*e8*/	{ RELAD4,	5,	"call %x"	},
/*e9*/	{ RELAD4,	5,	"jmp %x"	},
/*ea*/	{ BYTE6,	7,	"jmp %x:%x"	},
/*eb*/	{ RELAD1,	2,	"jmp %x"	},
/*ec*/	{ 0,		1,	"in al,dx"	},
/*ed*/	{ 0,		1,	"in eax,dx"	},
/*ee*/	{ 0,		1,	"out dx,al"	},
/*ef*/	{ 0,		1,	"out dx,eax"	},
/*f0*/	{ 0,		1,	"lock:"	},
/*f1*/	{ 0,		1,	"lock:"	},
/*f2*/	{ 0,		1,	"repne:"	},
/*f3*/	{ 0,		1,	"rep:"	},
/*f4*/	{ 0,		1,	"halt"	},
/*f5*/	{ 0,		1,	"cmc"	},
/*f6*/	{ GROUPC,	2,	"Unary Group C"	},
/*f7*/	{ GROUPC,	2,	"Unary Group C"	},
/*f8*/	{ 0,		1,	"clc"	},
/*f9*/	{ 0,		1,	"stc"	},
/*fa*/	{ 0,		1,	"cli"	},
/*fb*/	{ 0,		1,	"sti"	},
/*fc*/	{ 0,		1,	"cld"	},
/*fd*/	{ 0,		1,	"std"	},
/*fe*/	{ GROUPD,	2,	"Group D"	},
/*ff*/	{ GROUPE,	2,	"Group E"	},
};

const struct mnemonic code2[256] = {
/*00*/	{ GROUPF,	3,	"Group F"	},
/*01*/	{ GROUPG,	3,	"Group G"	},
/*02*/	{ MOD1_4,	3,	"lar"	},
/*03*/	{ MOD1_4,	3,	"lsl"	},
/*04*/	{ 1,		2,	""	},
/*05*/	{ 0,		2,	"loadall"	},
/*06*/	{ 0,		2,	"clts"	},
/*07*/	{ 0,		2,	"loadall"	},
/*08*/	{ 0,		2,	"invd"	},
/*09*/	{ 0,		2,	"wbinvd"	},
/*0a*/	{ 1,		2,	""	},
/*0b*/	{ 0,		2,	"ud2"	},
/*0c*/	{ 1,		2,	""	},
/*0d*/	{ 1,		2,	""	},
/*0e*/	{ 1,		2,	""	},
/*0f*/	{ 1,		2,	""	},
/*10*/	{ 1,		3,	"movups"	},
/*11*/	{ 1,		3,	"movups"	},
/*12*/	{ 1,		3,	"movlps"	},
/*13*/	{ 1,		3,	"movlps"	},
/*14*/	{ 1,		3,	"unpcklps"	},
/*15*/	{ 1,		3,	"unpckhps"	},
/*16*/	{ 1,		3,	"movhps"	},
/*17*/	{ 1,		3,	"movhps"	},
/*18*/	{ PREF,		3,	"prefetch %s"	},
/*19*/	{ 1,		2,	""	},
/*1a*/	{ 1,		2,	""	},
/*1b*/	{ 1,		2,	""	},
/*1c*/	{ 1,		2,	""	},
/*1d*/	{ 1,		2,	""	},
/*1e*/	{ 1,		2,	""	},
/*1f*/	{ 1,		2,	""	},
/*20*/	{ CREG0,	3,	"mov %s,cr%x"	},
/*21*/	{ CREG0,	3,	"mov %s,dr%x"	},
/*22*/	{ CREG1,	3,	"mov cr%x,%s"	},
/*23*/	{ CREG1,	3,	"mov dr%x,%s"	},
/*24*/	{ CREG0,	3,	"mov %s,tr%x"	},
/*25*/	{ 1,		2,	""	},
/*26*/	{ CREG1,	3,	"mov tr%x,%s"	},
/*27*/	{ 1,		2,	""	},
/*28*/	{ 1,		3,	"movaps"	},
/*29*/	{ 1,		3,	"movaps"	},
/*2a*/	{ 1,		3,	"cvtpi2ps"	},
/*2b*/	{ 1,		3,	"movntps"	},
/*2c*/	{ 1,		3,	"cvtps2pi"	},
/*2d*/	{ 1,		3,	"cvtps2pi"	},
/*2e*/	{ 1,		3,	"ucomiss"	},
/*2f*/	{ 1,		3,	"comiss"	},
/*30*/	{ 0,		2,	"wrmsr"	},
/*31*/	{ 0,		2,	"rdtsc"	},
/*32*/	{ 0,		2,	"rdmsr"	},
/*33*/	{ 0,		2,	"rdpmc"	},
/*34*/	{ 0,		2,	"sysenter"	},
/*35*/	{ 0,		2,	"sysexit"	},
/*36*/	{ 1,		2,	""	},
/*37*/	{ 1,		2,	""	},
/*38*/	{ 1,		2,	""	},
/*39*/	{ 1,		2,	""	},
/*3a*/	{ 1,		2,	""	},
/*3b*/	{ 1,		2,	""	},
/*3c*/	{ 1,		2,	""	},
/*3d*/	{ 1,		2,	""	},
/*3e*/	{ 1,		2,	""	},
/*3f*/	{ 1,		2,	""	},
/*40*/	{ MOD1_4,	3,	"cmovo %s,%s"	},
/*41*/	{ MOD1_4,	3,	"cmovno %s,%s"	},
/*42*/	{ MOD1_4,	3,	"cmovc %s,%s"	},
/*43*/	{ MOD1_4,	3,	"cmovnc %s,%s"	},
/*44*/	{ MOD1_4,	3,	"cmovz %s,%s"	},
/*45*/	{ MOD1_4,	3,	"cmovnz %s,%s"	},
/*46*/	{ MOD1_4,	3,	"cmovbe %s,%s"	},
/*47*/	{ MOD1_4,	3,	"cmovnbe %s,%s"	},
/*48*/	{ MOD1_4,	3,	"cmovs %s,%s"	},
/*49*/	{ MOD1_4,	3,	"cmovns %s,%s"	},
/*4a*/	{ MOD1_4,	3,	"cmovp %s,%s"	},
/*4b*/	{ MOD1_4,	3,	"cmovnp %s,%s"	},
/*4c*/	{ MOD1_4,	3,	"cmovl %s,%s"	},
/*4d*/	{ MOD1_4,	3,	"cmovnl %s,%s"	},
/*4e*/	{ MOD1_4,	3,	"cmovle %s,%s"	},
/*4f*/	{ MOD1_4,	3,	"cmovnle %s,%s"	},
/*50*/	{ 1,		3,	"movmskps"	},
/*51*/	{ 1,		3,	"sqrtps"	},
/*52*/	{ 1,		3,	"rsqrtps"	},
/*53*/	{ 1,		3,	"rcpps"	},
/*54*/	{ 1,		3,	"andps"	},
/*55*/	{ 1,		3,	"andnps"	},
/*56*/	{ 1,		3,	"prps"	},
/*57*/	{ 1,		3,	"xorps"	},
/*58*/	{ 1,		3,	"addps"	},
/*59*/	{ 1,		3,	"mulps"	},
/*5a*/	{ 1,		2,	""	},
/*5b*/	{ 1,		2,	""	},
/*5c*/	{ 1,		3,	"subps"	},
/*5d*/	{ 1,		3,	"minps"	},
/*5e*/	{ 1,		3,	"divps"	},
/*5f*/	{ 1,		3,	"maxps"	},
/*60*/	{ MMX,		3,	"punpcklbw"	},
/*61*/	{ MMX,		3,	"punpcklwd"	},
/*62*/	{ MMX,		3,	"punpckldq"	},
/*63*/	{ MMX,		3,	"packsswb"	},
/*64*/	{ MMX,		3,	"pcmpgtb"	},
/*65*/	{ MMX,		3,	"pcmpgtw"	},
/*66*/	{ MMX,		3,	"pcmpgtd"	},
/*67*/	{ MMX,		3,	"packuswb"	},
/*68*/	{ MMX,		3,	"punpckhbw"	},
/*69*/	{ MMX,		3,	"punpckhwd"	},
/*6a*/	{ MMX,		3,	"punpckhdq"	},
/*6b*/	{ MMX,		3,	"packssdw"	},
/*6c*/	{ 1,		2,	""	},
/*6d*/	{ 1,		2,	""	},
/*6e*/	{ MMX,		3,	"movd"	},
/*6f*/	{ MMX,		3,	"movq"	},
/*70*/	{ 1,		2,	""	},
/*71*/	{ MMX,		3,	"pshimw"	},
/*72*/	{ MMX,		3,	"pshimd"	},
/*73*/	{ MMX,		3,	"pshimq"	},
/*74*/	{ MMX,		3,	"pcmpeqb"	},
/*75*/	{ MMX,		3,	"pcmpeqw"	},
/*76*/	{ MMX,		3,	"pcmpeqd"	},
/*77*/	{ 0,		2,	"emms"	},
/*78*/	{ 1,		2,	"MMX UD"	},
/*79*/	{ 1,		2,	"MMX UD"	},
/*7a*/	{ 1,		2,	"MMX UD"	},
/*7b*/	{ 1,		2,	"MMX UD"	},
/*7c*/	{ 1,		2,	"MMX UD"	},
/*7d*/	{ 1,		2,	"MMX UD"	},
/*7e*/	{ MMX,		3,	"movd"	},
/*7f*/	{ MMX,		3,	"movq"	},
/*80*/	{ RELAD4,	6,	"jo %x"	},
/*81*/	{ RELAD4,	6,	"jno %x"	},
/*82*/	{ RELAD4,	6,	"jc %x"	},
/*83*/	{ RELAD4,	6,	"jnc %x"	},
/*84*/	{ RELAD4,	6,	"jz %x"	},
/*85*/	{ RELAD4,	6,	"jnz %x"	},
/*86*/	{ RELAD4,	6,	"jbe %x"	},
/*87*/	{ RELAD4,	6,	"jnbe %x"	},
/*88*/	{ RELAD4,	6,	"js %x"	},
/*89*/	{ RELAD4,	6,	"jns %x"	},
/*8a*/	{ RELAD4,	6,	"jp %x"	},
/*8b*/	{ RELAD4,	6,	"jnp %x"	},
/*8c*/	{ RELAD4,	6,	"jl %x"	},
/*8d*/	{ RELAD4,	6,	"jnl %x"	},
/*8e*/	{ RELAD4,	6,	"jle %x"	},
/*8f*/	{ RELAD4,	6,	"jnle %x"	},
/*90*/	{ SET,		3,	"seto %s"	},
/*91*/	{ SET,		3,	"setno %s"	},
/*92*/	{ SET,		3,	"setc %s"	},
/*93*/	{ SET,		3,	"setnc %s"	},
/*94*/	{ SET,		3,	"setz %s"	},
/*95*/	{ SET,		3,	"setnz %s"	},
/*96*/	{ SET,		3,	"setbe %s"	},
/*97*/	{ SET,		3,	"setnbe %s"	},
/*98*/	{ SET,		3,	"sets %s"	},
/*99*/	{ SET,		3,	"setns %s"	},
/*9a*/	{ SET,		3,	"setp %s"	},
/*9b*/	{ SET,		3,	"setnp %s"	},
/*9c*/	{ SET,		3,	"setl %s"	},
/*9d*/	{ SET,		3,	"setnl %s"	},
/*9e*/	{ SET,		3,	"setle %s"	},
/*9f*/	{ SET,		3,	"setnle %s"	},
/*a0*/	{ 0,		2,	"push fs"	},
/*a1*/	{ 0,		2,	"pop fs"	},
/*a2*/	{ 0,		2,	"cpuid"	},
/*a3*/	{ MOD1_4,	3,	"bt %s,%s"	},
/*a4*/	{ ARG3,		3,	"shld %s,%s,%x"	},
/*a5*/	{ MOD0_4,	3,	"shld %s,%s,cl"	},
/*a6*/	{ 1,		3,	"xbts"	},
/*a7*/	{ 1,		3,	"ibts"	},
/*a8*/	{ 0,		2,	"push gs"	},
/*a9*/	{ 0,		2,	"pop gs"	},
/*aa*/	{ 1,		2,	"rsm"	},
/*ab*/	{ MOD1_4,	3,	"bts %s,%s"	},
/*ac*/	{ ARG3,		3,	"shrd %s,%s,%x"	},
/*ad*/	{ MOD0_4,	3,	"shrd %s,%s,cl"	},
/*ae*/	{ GROUP16,	3,	"group16"	},
/*af*/	{ MOD1_4,	3,	"imul %s,%s"	},
/*b0*/	{ MOD0_1,	3,	"cmpxchg %s,%s"	},
/*b1*/	{ MOD0_4,	3,	"cmpxchg %s,%s"	},
/*b2*/	{ MOD1_4,	3,	"lss %s,%s"	},
/*b3*/	{ MOD1_4,	3,	"btr %s,%s"	},
/*b4*/	{ MOD1_4,	3,	"lfs %s,%s"	},
/*b5*/	{ MOD1_4,	3,	"lgs %s,%s"	},
/*b6*/	{ MOD0_4,	3,	"movzx %s,%s"	},
/*b7*/	{ MOD1_4,	3,	"movzx %s,%s"	},
/*b8*/	{ 1,		2,	""	},
/*b9*/	{ 1,		2,	""	},
/*ba*/	{ GROUPH,	4,	"Group H"	},
/*bb*/	{ MOD1_4,	3,	"btc %s,%s"	},
/*bc*/	{ MOD1_4,	3,	"bsf %s,%s"	},
/*bd*/	{ MOD1_4,	3,	"bsr %s,%s"	},
/*be*/	{ MOD0_4,	3,	"movsx.b %s,%s"	},
/*bf*/	{ MOD1_4,	3,	"movsx.w %s,%s"	},
/*c0*/	{ MOD0_1,	3,	"xadd %s,%s"	},
/*c1*/	{ MOD0_4,	3,	"xadd %s,%s"	},
/*c2*/	{ 1,		3,	"cmpss"	},
/*c3*/	{ 1,		2,	""	},
/*c4*/	{ 1,		3,	"pinsnw"	},
/*c5*/	{ 1,		3,	"pextrw"	},
/*c6*/	{ 1,		3,	"shufps"	},
/*c7*/	{ 1,		3,	""	},
/*c8*/	{ 0,		2,	"bswap eax"	},
/*c9*/	{ 0,		2,	"bswap ecx"	},
/*ca*/	{ 0,		2,	"bswap edx"	},
/*cb*/	{ 0,		2,	"bswap ebx"	},
/*cc*/	{ 0,		2,	"bswap esp"	},
/*cd*/	{ 0,		2,	"bswap ebp"	},
/*ce*/	{ 0,		2,	"bswap esi"	},
/*cf*/	{ 0,		2,	"bswap edi"	},
/*d0*/	{ 1,		2,	""	},
/*d1*/	{ MMX,		3,	"psrlw"	},
/*d2*/	{ MMX,		3,	"psrld"	},
/*d3*/	{ MMX,		3,	"psrlq"	},
/*d4*/	{ 1,		2,	""	},
/*d5*/	{ MMX,		3,	"pmulw"	},
/*d6*/	{ 1,		2,	""	},
/*d7*/	{ MMX,		3,	"pmovmskb"	},
/*d8*/	{ MMX,		3,	"psubusb"	},
/*d9*/	{ MMX,		3,	"psubusw"	},
/*da*/	{ MMX,		3,	"pminub"	},
/*db*/	{ MMX,		3,	"pand"	},
/*dc*/	{ MMX,		3,	"paddusb"	},
/*dd*/	{ MMX,		3,	"paddusw"	},
/*de*/	{ MMX,		3,	"pmaxub"	},
/*df*/	{ MMX,		3,	"pandn"	},
/*e0*/	{ MMX,		3,	"pavgb"	},
/*e1*/	{ MMX,		3,	"psraw"	},
/*e2*/	{ MMX,		3,	"psrad"	},
/*e3*/	{ MMX,		3,	"pavgw"	},
/*e4*/	{ MMX,		3,	"pmulhuw"	},
/*e5*/	{ MMX,		3,	"pmulhw"	},
/*e6*/	{ 1,		2,	""	},
/*e7*/	{ MMX,		3,	"movntq"	},
/*e8*/	{ MMX,		3,	"psubsb"	},
/*e9*/	{ MMX,		3,	"psubsw"	},
/*ea*/	{ MMX,		3,	"pminsw"	},
/*eb*/	{ MMX,		3,	"por"	},
/*ec*/	{ MMX,		3,	"paddsb"	},
/*ed*/	{ MMX,		3,	"paddsw"	},
/*ee*/	{ MMX,		3,	"pmaxsw"	},
/*ef*/	{ MMX,		3,	"pxor"	},
/*f0*/	{ 1,		2,	""	},
/*f1*/	{ MMX,		3,	"psllw"	},
/*f2*/	{ MMX,		3,	"pslld"	},
/*f3*/	{ MMX,		3,	"psllq"	},
/*f4*/	{ 1,		2,	""	},
/*f5*/	{ MMX,		3,	"pmaddwd"	},
/*f6*/	{ MMX,		3,	"psadbw"	},
/*f7*/	{ 1,		2,	""	},
/*f8*/	{ MMX,		3,	"psubb"	},
/*f9*/	{ MMX,		3,	"psubw"	},
/*fa*/	{ MMX,		3,	"psubd"	},
/*fb*/	{ 1,		2,	""	},
/*fc*/	{ MMX,		3,	"paddb"	},
/*fd*/	{ MMX,		3,	"paddw"	},
/*fe*/	{ MMX,		3,	"paddd"	},
/*ff*/	{ 1,		2,	""	},
};

PRIVATE const char *npx_d9[] = {
/*e0*/	"fchs",		"fabs",		NULL,		NULL,
/*e4*/	"ftst",		"fxam",		"ftsp",		NULL,
/*e8*/	"fld1",		"fldl2t",	"fldl2e",	"fldpi",
/*ec*/	"fldlg2",	"fldln2",	"fldz",		NULL,
/*f0*/	"f2xm1",	"fyl2x",	"fptan",	"fpatan",
/*f4*/	"fxtract",	"fprem1",	"fdecstp",	"fincstp",
/*f8*/	"fprem",	"fyl2xp1",	"fsqrt",	"fsincos",
/*fc*/	"frndint",	"fscale",	"fsin",		"fcos",
};

PRIVATE const char *npx_db[] = {
/*e0*/	"feni",		"fdisi",	"fclex",	"finit",
/*e4*/	"fsetpm",	"frstpm",	NULL,		NULL,
/*e8*/	"fsbp0",	NULL,		"fsbp2",	"fsbp1",
/*ec*/	"frint2",	NULL,		NULL,		NULL,
/*f0*/	NULL,		"f4x4",		NULL,		NULL,
/*f4*/	NULL,		NULL,		NULL,		NULL,
/*f8*/	NULL,		NULL,		NULL,		NULL,
/*fc*/	NULL,		NULL,		NULL,		NULL,
};

PRIVATE const char *npx_2b[8][8] = {
{/*d8*/	"add st,st(%d)",
	"fmul st,st(%d)",
	"fcom st(%d)",
	"fcomp st(%d)",
	"fsub st,st(%d)",
	"fsubr st,st(%d)",
	"fdiv st(%d)",
	"fdivr st,st(%d)" },
{/*d9*/	"fld st(%d)",
	"fxch st(%d)",
	"fnop",
	"fstp st(%d)",
	NULL,
	NULL,
	NULL,
	NULL },
{/*da*/	"fcmovb st,st(%d)",
	"fcmove st,st(%d)",
	"fcmovbe st,st(%d)",
	"fcmovu st,st(%d)",
	NULL,
	"ficompp",
	NULL,
	NULL },
{/*db*/	"fcmovnb st,st(%d)",
	"fcmovne st,st(%d)",
	"fcmovnbe st,st(%d)",
	"fcmovne st,st(%d)",
	NULL,
	"fucomi st,st(%d)",
	"fcomi st,st(%d)",
	NULL },
{/*dc*/	"add st(%d),st",
	"fmul st(%d),st",
	"fcom st(%d),st",
	"fcomp st(%d),st",
	"fsubr st(%d),st",
	"fsub st(%d),st",
	"fdivr st(%d),st",
	"fdiv st(%d),st" },
{/*dd*/	"ffree st(%d)",
	"fxch st(%d)",
	"fst st(%d)",
	"fstp st(%d)",
	"fucom st(%d)",
	"fucomp st(%d)",
	NULL,
	"frichop" },
{/*de*/	"addp st(%d),st",
	"fmulp st(%d),st",
	"fcomp st(%d),st",
	"fcompp st(%d)",
	"fsubrp st(%d),st",
	"fsubp st(%d),st",
	"fdivrp st(%d),st",
	"fdivp st(%d),st" },
{/*df*/	NULL,
	NULL,
	NULL,
	NULL,
	"fstsw ax",
	"fucomp st,st(%d)",
	"fcomp st,st(%d)",
	"frinear" }
};

PRIVATE const char *npx_opt[8][8] = {
{/*d8*/	"fadd",  "fmul",  "fcom",  "fcomp",
	"fsub",  "fsubr", "fdiv"   "fdivr" },
{/*d9*/	"fld",	 NULL,    "fst",   "fstp",
	"fldenv", "fldcw","fstenv", "fstcw"},
{/*da*/	"fiadd", "fimul", "ficom", "ficomp",
	"fisub", "fisubr","fidiv", "fidivr"},
{/*db*/	"fild",  NULL,    "fist",  "fistp",
	NULL,    "fld",   NULL,    "fstp"  },
{/*dc*/	"fadd",  "fmul",  "fcom",  "fcomp",
	"fsub",  "fsubr", "fdiv"   "fdivr" },
{/*dd*/	"fld", 	 NULL,    "fst",   "fstp",
	"frstor",NULL,    "fsave", "fstsw" },
{/*de*/	"fiadd", "fimul", "ficom", "ficomp",
	"fisub", "fisubr","fidiv", "fidivr"},
{/*df*/	"fild",  NULL,    "fist",  "fistp",
	"fbld",  "fild",  "fbstp", "fst"   },
};

const char *
getreg8(r)
int r;
{
	switch (r) {
	case 0:	return "al";
	case 1:	return "cl";
	case 2:	return "dl";
	case 3:	return "bl";
	case 4:	return "ah";
	case 5:	return "ch";
	case 6:	return "dh";
	case 7:	return "bh";
	default:return "??";
	}
}

const char *
getreg16(r)
int r;
{
	switch (r) {
	case 0:	return "ax";
	case 1:	return "cx";
	case 2:	return "dx";
	case 3:	return "bx";
	case 4:	return "sp";
	case 5:	return "bp";
	case 6:	return "si";
	case 7:	return "di";
	default:return "??";
	}
}

const char *
getreg32(r)
int r;
{
	switch (r) {
	case 0:	return "eax";
	case 1:	return "ecx";
	case 2:	return "edx";
	case 3:	return "ebx";
	case 4:	return "esp";
	case 5:	return "ebp";
	case 6:	return "esi";
	case 7:	return "edi";
	default:return "??";
	}
}

const char *
getsreg(n)
int n;
{
	switch (n) {
	case 0:	return "es";
	case 1:	return "cs";
	case 2:	return "ss";
	case 3:	return "ds";
	case 4:	return "fs";
	case 5:	return "gs";
	default:return "??";
	}
}

const char *
getsib(mod, cur, nb)
	int mod;
	const unsigned char *cur;
	int *nb;
{
	int sib;
	static char buf[16];
	int scale;

	sib = *cur;
	if (mod == 0 && (sib & 7) == 5) {
		*nb += 4;
		opad = *(int *)(cur + 1);
		sprintf(buf, "%lx", opad);
	} else {
		strcpy(buf, getreg32(sib & 7));
	}
	if (((sib >> 3) & 7) == 4) {
		return buf;
	}
	strcat(buf, "+");
	scale = 1 << (sib >> 6);
	if (scale > 1) {
		sprintf(buf + strlen(buf), "%d*", scale);
	}
	sprintf(buf + strlen(buf), getreg32((sib >> 3) & 7));
	return buf;
}

const char *
getaddrmod(w, cur, nb)
int w;
unsigned const char *cur;
int *nb;
{
	static char buf[32];
	const char *p = NULL;
	int amd = *cur;

	switch (amd & 0xc0) {
	case 0:
		switch (amd & 7) {
		case 0:	return "[eax]";
		case 1:	return "[ecx]";
		case 2:	return "[edx]";
		case 3:	return "[ebx]";
		case 4:	*nb += 1;
			p = getsib(0, cur + 1, nb);
			sprintf(buf, "[%s]", p);
			return buf;
		case 5:	*nb += 4;
			opad = *(addr_t *)(cur + 1);
			sprintf(buf, "[%lx]", opad);
			return buf;
		case 6:	return "[esi]";
		case 7:	return "[edi]";
		}
		break;

	case 0x40:
		switch (amd & 7) {
		case 0:	p = "[eax%c%x]";	break;
		case 1:	p = "[ecx%c%x]";	break;
		case 2:	p = "[edx%c%x]";	break;
		case 3: p = "[ebx%c%x]";	break;
		case 4:	*nb += 2;
			p = getsib(1, cur + 1, nb);
			if ((signed char)*(cur + 2) >= 0) {
				sprintf(buf, "[%s+%x]", p, (signed char)*(cur + 2));
			} else {
				sprintf(buf, "[%s-%x]", p, -(signed char)*(cur + 2));
			}
			return buf;
		case 5:	p = "[ebp%c%x]";	break;
		case 6:	p = "[esi%c%x]";	break;
		case 7:	p = "[edi%c%x]";	break;
		}
		*nb += 1;
		if ((signed char)*(cur + 1) >= 0) {
			sprintf(buf, p, '+', (signed char)*(cur + 1));
		} else {
			sprintf(buf, p, '-', -(signed char)*(cur + 1));
		}
		return buf;

	case 0x80:
		switch (amd & 7) {
		case 0:	p = "[eax%c%x]";	break;
		case 1:	p = "[ecx%c%x]";	break;
		case 2:	p = "[edx%c%x]";	break;
		case 3: p = "[ebx%c%x]";	break;
		case 4:	*nb += 5;
			p = getsib(2, cur + 1, nb);
			opad = *(addr_t *)(cur + 2);
			if (opad < 0xffff0000) {
				sprintf(buf, "[%s+%lx]", p, opad);
			} else {
				sprintf(buf, "[%s-%lx]", p, -opad);
			}
			return buf;
		case 5:	p = "[ebp%c%x]";	break;
		case 6:	p = "[esi%c%x]";	break;
		case 7:	p = "[edi%c%x]";	break;
		}
		*nb += 4;
		if (opad == 0) {
			opad = *(unsigned int *)(cur + 1);
		}
		if (*(unsigned int *)(cur + 1) < 0xffff0000) {
			sprintf(buf, p, '+', *(unsigned int *)(cur + 1));
		} else {
			sprintf(buf, p, '-', -*(unsigned int *)(cur + 1));
		}
		return buf;

	case 0xc0:
		switch (w) {
		case 0:
			return getreg8(amd & 7);
		case 1:
			if (opsize)
				return getreg16(amd & 7);
			return getreg32(amd & 7);
		}
	}
	return NULL;
}

const char *
get_npx(c, cur ,nb)
int c;
unsigned const char *cur;
int *nb;
{
	const char *s, *op;
	static char buf[32];

	if (c == 0xd9 && *cur >= 0xe0)
		return npx_d9[*cur - 0xe0];
	if (c == 0xdb && *cur >= 0xe0)
		return npx_db[*cur - 0xe0];
	if (*cur >= 0xc0) {
		s = npx_2b[c & 7][(*cur - 0xc0) / 8];
		if (s == NULL)
			return NULL;
		sprintf(buf, s, *cur & 7);
		return buf;
	}
	s = getaddrmod(1, cur, nb);
	if ((op = npx_opt[c & 7][(*cur / 8) & 7]) == 0)
		return NULL;
	sprintf(buf, "%s %s", op, s);
	return buf;
}

const char *
get_groupa_op(n)
	int n;
{
	switch (n) {
	case 0:	return "add %s,%x";
	case 1:	return "or %s,%x";
	case 2:	return "adc %s,%x";
	case 3:	return "sbb %s,%x";
	case 4:	return "and %s,%x";
	case 5:	return "sub %s,%x";
	case 6:	return "xor %s,%x";
	case 7:	return "cmp %s,%x";
	default:return "?";
	}
}

const char *
get_groupb_op(n)
	int n;
{
	switch (n) {
	case 0:	return "rol %s,";
	case 1:	return "ror %s,";
	case 2:	return "rcl %s,";
	case 3:	return "rcr %s,";
	case 4:	return "shl %s,";
	case 5:	return "shr %s,";
	case 6:	return "sal %s,";
	case 7:	return "sar %s,";
	default:return "?";
	}
}

const char *
get_groupc_op(n)
	int n;
{
	switch (n) {
	case 0:	return "test %s,%x";
	case 1:	return "test %s,%x";
	case 2:	return "not %s";
	case 3:	return "neg %s";
	case 4:	return "mul %s";
	case 5:	return "imul %s";
	case 6:	return "div %s";
	case 7:	return "idiv %s";
	default:return "?";
	}
}

const char *
get_groupd_op(n)
	int n;
{
	switch (n) {
	case 0:	return "inc %s";
	case 1:	return "dec %s";
	default:return "?";
	}
}

const char *
get_groupe_op(n)
	int n;
{
	switch (n) {
	case 0:	return "inc %s";
	case 1:	return "dec %s";
	case 2:	return "call %s";	/* near */
	case 3:	return "call %s";	/* far */
	case 4:	return "jmp near %s";
	case 5:	return "jmp far %s";
	case 6:	return "push %s";
	default:return "?";
	}
}

const char *
get_groupf_op(n)
	int n;
{
	switch (n) {
	case 0:	return "sldt %s";
	case 1:	return "str %s";
	case 2:	return "lldt %s";
	case 3:	return "ltr %s";
	case 4:	return "verr %s";
	case 5:	return "verw %s";
	default:return "?";
	}
}

const char *
get_groupg_op(n)
	int n;
{
	switch (n) {
	case 0:	return "sgdt %s";
	case 1:	return "sidt %s";
	case 2:	return "lgdt %s";
	case 3:	return "lidt %s";
	case 4:	return "smsw %s";
	case 6:	return "lmsw %s";
	case 7:	return "invlpg %s";
	default:return "?";
	}
}

const char *
get_grouph_op(n)
	int n;
{
	switch (n) {
	case 4:	return "bt %s,%x";
	case 5:	return "bts %s,%x";
	case 6:	return "btr %s,%x";
	case 7:	return "btc %s,%x";
	default:return "?";
	}
}

const char *
get_group16_op(n)
	int n;
{
	switch (n) {
	case 0:	return "fxsave %s";
	case 1:	return "fxrstor %s";
	case 2:	return "ldmxcsr %s";
	case 3:	return "stmxcsr %s";
	case 7:	return "sfence %s";
	default:return "?";
	}
}

const char *
getop2_0f(addr, cur, nb)
	addr_t addr;
	unsigned char *cur;
	int *nb;
{
	int c, i;
	static char buf[32];
	const struct mnemonic *p;
	const char *s;

	c = *cur;
	p = &code2[c];
	*nb = p->nbyte;
	switch (p->type) {
	case DIRECT:
		return p->op;
	case RELAD4:
		opad = addr + *nb + *(int *)(cur + 1);
		sprintf(buf, p->op, opad);
		return buf;
	case SET:
		sprintf(buf, p->op, getaddrmod(1, cur + 1, nb));
		return buf;
	case MOD0_4:
		sprintf(buf, p->op,
			getaddrmod(1, cur + 1, nb),
			opsize? getreg16((cur[1] >> 3) & 7):
				getreg32((cur[1] >> 3) & 7));
		return buf;
	case MOD1_4:
		sprintf(buf, p->op,
			opsize? getreg16((cur[1] >> 3) & 7):
				getreg32((cur[1] >> 3) & 7),
			getaddrmod(1, cur + 1, nb));
		return buf;
	case CREG0:
		sprintf(buf, p->op,
			getreg32(cur[1] & 7),
			(cur[1] >> 3) & 7);
		return buf;
	case CREG1:
		sprintf(buf, p->op,
			(cur[1] >> 3) & 7,
			getreg32(cur[1] & 7));
		return buf;
	case GROUPF:
		s = getaddrmod(1, cur + 1, nb);
		sprintf(buf, get_groupf_op((cur[1] >> 3) & 7), s);
		return buf;
	case GROUPG:
		s = getaddrmod(1, cur + 1, nb);
		sprintf(buf, get_groupg_op((cur[1] >> 3) & 7), s);
		return buf;
	case GROUPH:
		s = getaddrmod(1, cur + 1, nb);
		sprintf(buf, get_grouph_op((cur[1] >> 3) & 7),
			s, cur[*nb - 2]);
		return buf;
	case GROUP16:
		s = getaddrmod(1, cur + 1, nb);
		sprintf(buf, get_group16_op((cur[1] >> 3) & 7), s);
		return buf;
	case ARG3:
		s = getaddrmod(1, cur + 1, nb);
		if  (c & 1)
			i = 0;
		else {
			i = *(cur + *nb - 1);
			*nb += 1;
		}
		sprintf(buf, p->op,
			opsize? getreg16((cur[1] >> 3) & 7):
				getreg32((cur[1] >> 3) & 7),
			s, i);
		return buf;
	case MMX:
		getaddrmod(1, cur + 1, nb);
		sprintf(buf, "? (%s)", p->op);
		return buf;
	case PREF:
		sprintf(buf, p->op, getreg32(cur[1] & 7));
		return buf;
	}
	sprintf(buf, "? (%s)", p->op);
	return buf;
}

PRIVATE int
print_dis(addr, first)
	addr_t addr;
	int first;
{
	unsigned char buf[16];
	char prt[80];
	const struct symtable *sym;
	int nb, i = 0;
	const struct mnemonic *p;
	const char *s;

	memread(addr, sizeof(buf), buf, "disassemble memory");

	opad = 0;
	sym = searchsym_byaddr(addr);
	if (sym == NULL) {
		mprintf("  -       ");
	} else if (addr == sym->addr) {
		mprintf("%-9s ", sym->name);
		if (strlen(sym->name) > 9) {
			mprintf("\n+0        ");
		}
	} else if (first) {
		sprintf(prt, "%s+%lx", sym->name, addr - sym->addr);
		if (strlen(prt) > 9) {
			mprintf("%s\n+%-8lx ", sym->name, addr - sym->addr);
		} else {
			mprintf("%-9s ", prt);
		}
	} else {
		mprintf("+%-8lx ", addr - sym->addr);
	}
	mprintf("%08lx:", addr);

	p = &code1[buf[0]];
	nb = p->nbyte;
	switch (p->type) {
	case DIRECT:
		sprintf(prt, p->op);
		break;
	case BYTE1:
		sprintf(prt, p->op, buf[1]);
		break;
	case BYTE2:
		sprintf(prt, p->op, *(unsigned short *)&buf[1]);
		break;
	case BYTE4:
		if (opsize) {
			sprintf(prt, p->op, (int)*(unsigned short *)&buf[1]);
			nb -= 2;
		} else {
			unsigned u = *(unsigned int *)&buf[1];
			if (u < 0xffffff00) {
				opad = u;
				sprintf(prt, p->op, u);
			} else {
				char buf[16], *q = buf;
				for (i = 0; i < sizeof(buf) - 1; i++) {
					if (p->op[i] == 0)
						break;
					else if (p->op[i] == '%')
						*q++ = '-';
					*q++ = p->op[i];
				}
				*q++ = 0;
				sprintf(prt, buf, -u);
			}
		}
		break;
	case BYTE6:
		opad = *(unsigned int *)&buf[1];
		sprintf(prt, p->op, *(unsigned short *)&buf[5], opad);
		break;
	case ADDR4:
		opad = *(unsigned int *)&buf[1];
		sprintf(prt, p->op, opad);
		break;
	case RELAD1:
		opad = addr + nb + *(signed char *)&buf[1];
		sprintf(prt, p->op, opad);
		break;
	case RELAD2:
		opad = addr + nb + *(signed short *)&buf[1];
		sprintf(prt, p->op, opad);
		break;
	case RELAD4:
		opad = addr + nb + *(signed int *)&buf[1];
		sprintf(prt, p->op, opad);
		break;
	case MOD0_1:
		sprintf(prt, p->op,
			getaddrmod(0, buf + 1, &nb),
			getreg8((buf[1] >> 3) & 7));
		break;
	case MOD0_4:
		sprintf(prt, p->op,
			getaddrmod(1, buf + 1, &nb),
			opsize? getreg16((buf[1] >> 3) & 7):
				getreg32((buf[1] >> 3) & 7));
		break;
	case MOD1_1:
		sprintf(prt, p->op, getreg8((buf[1] >> 3) & 7),
			getaddrmod(0, buf + 1, &nb));
		break;
	case MOD1_4:
		sprintf(prt, p->op,
			opsize? getreg16((buf[1] >> 3) & 7):
				getreg32((buf[1] >> 3) & 7),
			getaddrmod(1, buf + 1, &nb));
		break;
	case OP2B:
		strcpy(prt, getop2_0f(addr, buf + 1, &nb));
		break;
	case GROUPA:
		s = getaddrmod(buf[0] & 1, buf + 1, &nb);
		switch (buf[0] & 3) {
		case 0:	i = buf[nb];		nb += 1;	break;
		case 1:	if (opsize) {
				i = *(unsigned short *)&buf[nb];
				nb += 2;
			} else {
				i = opad = *(int *)&buf[nb];
				nb += 4;
			}
			break;
		case 2:	i = buf[nb];		nb += 1;	break;
		case 3:	i = *(signed char *)&buf[nb];	nb += 1;break;
		}
		sprintf(prt, get_groupa_op((buf[1] >> 3) & 7), s, i);
		break;
	case GROUPB:
		s = getaddrmod(buf[0] & 1, buf + 1, &nb);
		sprintf(prt, get_groupb_op((buf[1] >> 3) & 7), s);
		switch (buf[0]) {
		case 0xc0: case 0xc1:
			sprintf(prt + strlen(prt), "%x", buf[nb++]);	break;
		case 0xd0: case 0xd1:
			strcat(prt, "1");	break;
		case 0xd2: case 0xd3:
			strcat(prt, "cl");	break;
		}
		break;
	case GROUPC:	/* f6 or f7 */
		s = getaddrmod(buf[0] & 1, buf + 1, &nb);
		switch ((buf[1] >> 3) & 7) {
		case 0: case 1:
			switch (buf[0] & 1) {
			case 0:	i = buf[nb];		nb += 1;	break;
			case 1:	if (opsize) {
					i = *(unsigned short *)&buf[nb];
					nb += 2;
				} else {
					i = *(int *)&buf[nb];
					if (opad == 0)
						opad = i;
					nb += 4;
				}
				break;
			}
		}
		sprintf(prt, get_groupc_op((buf[1] >> 3) & 7), s, i);
		break;
	case GROUPD:
		s = getaddrmod(0, buf + 1, &nb);
		sprintf(prt, get_groupd_op((buf[1] >> 3) & 7), s);
		break;
	case GROUPE:
		s = getaddrmod(1, buf + 1, &nb);
		sprintf(prt, get_groupe_op((buf[1] >> 3) & 7), s);
		break;
	case IMM:
		s = getaddrmod(buf[0] & 1, buf + 1, &nb);
		switch (buf[0] & 1) {
		case 0:	i = buf[nb];		nb += 1;	break;
		case 1:	if (opsize) {
				i = *(unsigned short *)&buf[nb];
				nb += 2;
			} else {
				i = *(int *)&buf[nb];
				if (opad == 0)
					opad = i;
				nb += 4;
			}
			break;
		}
		sprintf(prt, p->op, s, i);
		break;
	case POP:
		s = getaddrmod(1, buf + 1, &nb);
		sprintf(prt, p->op, s);
		break;
	case IMUL:
		s = getaddrmod(buf[0] & 1, buf + 1, &nb);
		switch (buf[0] & 2) {
		case 2:	i = buf[nb];	nb += 1;	break;
		case 0:	if (opsize) {
				i = *(unsigned short *)&buf[nb];
				nb += 2;
			} else {
				i = opad = *(int *)&buf[nb];
				nb += 4;
			}
			break;
		}
		sprintf(prt, p->op,
			opsize? getreg16((buf[1] >> 3) & 7):
				getreg32((buf[1] >> 3) & 7),
			s, i);
		break;
	case SREG0:
		sprintf(prt, p->op,
			getaddrmod(1, buf + 1, &nb),
			getsreg((buf[1] >> 3) & 7));
		break;
	case SREG1:
		sprintf(prt, p->op,
			getsreg((buf[1] >> 3) & 7),
			getaddrmod(1, buf + 1, &nb));
		break;
	case OPSIZE:
		sprintf(prt, "(%s)", p->op);
		break;
	case ENTER:
		sprintf(prt, p->op, *(unsigned short *)&buf[1], buf[3]);
		break;
	case NPX:
		if ((s = get_npx(buf[0], buf + 1, &nb)) == NULL) {
			s = "? (NPX)";
		}
		strcpy(prt, s);
		break;
	default:
		sprintf(prt, "? (%s)", p->op);
		break;
	}
	opsize = (p->type == OPSIZE);

	for (i = 0; i < nb; i++) {
		mprintf(" %02x", buf[i]);
	}
	for (; i < 7; i++) {
		mprintf("   ");
	}

	if (i == 7) {
		mprintf(" ");
	} else {
		mprintf("\n                                         ");
	}
	mprintf("%s", prt);
	if (opad && (s = getsymstr(opad)) != NULL) {
		mprintf("  ;%s", s);
	}
	mprintf("\n");
	return nb;
}

PRIVATE addr_t
dis()
{
	addr_t addr, saddr, size = 16;

	if (argcnt < 2 || argcnt > 3) {
		THROW(usage);
	}

	saddr = getaddr(args[1]);
	if (argcnt == 3) {
		size = getaddr(args[2]);
	}

	opsize = 0;
	addr = print_dis(saddr, 1) + saddr;
	while (addr < saddr + size) {
		addr += print_dis(addr, 0);
	}
	return addr;
}
