#ifndef __MERCURY_REGEX_TRAITS__
#define __MERCURY_REGEX_TRAITS__

#include <cstddef>

namespace mercury
{
	namespace _regex
	{
		typedef enum tagCHARCLASS
		{
			CHARCLASS_NONE   = 0x0000,
			CHARCLASS_UPPER  = 0x0001,
			CHARCLASS_LOWER  = 0x0002,
			CHARCLASS_DIGIT  = 0x0004,
			CHARCLASS_XDIGIT = 0x0008,
			CHARCLASS_PUNCT  = 0x0010,
			CHARCLASS_BLANK  = 0x0020,
			CHARCLASS_SPACE  = 0x0040,
			CHARCLASS_CNTRL  = 0x0080,
			CHARCLASS_GRAPH  = 0x0100,
			CHARCLASS_PRINT  = 0x0200,
			CHARCLASS_UNDER  = 0x0400,
			CHARCLASS_ALPHA  = CHARCLASS_UPPER | CHARCLASS_LOWER,
			CHARCLASS_ALNUM  = CHARCLASS_ALPHA | CHARCLASS_DIGIT,
			CHARCLASS_CSYM   = CHARCLASS_ALNUM | CHARCLASS_UNDER,
		} CHARCLASS;
		template<typename _Char>
		class regex_traits
		{
		public:
			typedef _Char char_type;

		public:
			static _Char meta_escape          (void) { return _Char('\\'); }
			static _Char meta_head            (void) { return _Char('^'); }
			static _Char meta_tail            (void) { return _Char('$'); }
			static _Char meta_disjunction     (void) { return _Char('|'); }
			static _Char meta_repeat0         (void) { return _Char('*'); }
			static _Char meta_repeat1         (void) { return _Char('+'); }
			static _Char meta_repeat01        (void) { return _Char('?'); }
			static _Char meta_repeat_begin    (void) { return _Char('{'); }
			static _Char meta_repeat_separator(void) { return _Char(','); }
			static _Char meta_repeat_end      (void) { return _Char('}'); }
			static _Char meta_repeat_lazy     (void) { return _Char('?'); }
			static _Char meta_any             (void) { return _Char('.'); }
			static _Char meta_set_begin       (void) { return _Char('['); }
			static _Char meta_set_not         (void) { return _Char('^'); }
			static _Char meta_set_range       (void) { return _Char('-'); }
			static _Char meta_set_end         (void) { return _Char(']'); }
			static _Char meta_group_begin     (void) { return _Char('('); }
			static _Char meta_group_end       (void) { return _Char(')'); }
			static _Char escape_bel(void) { return _Char('a'); }
			static _Char escape_ff (void) { return _Char('f'); }
			static _Char escape_lf (void) { return _Char('n'); }
			static _Char escape_cr (void) { return _Char('r'); }
			static _Char escape_ht (void) { return _Char('t'); }
			static _Char escape_vt (void) { return _Char('v'); }
			static _Char escape_esc(void) { return _Char('e'); }

			static _Char escape_upper       (void) { return _Char('u'); }
			static _Char escape_upper_invert(void) { return _Char('U'); }
			static _Char escape_lower       (void) { return _Char('l'); }
			static _Char escape_lower_invert(void) { return _Char('L'); }
			static _Char escape_digit       (void) { return _Char('d'); }
			static _Char escape_digit_invert(void) { return _Char('D'); }
			static _Char escape_space       (void) { return _Char('s'); }
			static _Char escape_space_invert(void) { return _Char('S'); }
			static _Char escape_csym        (void) { return _Char('w'); }
			static _Char escape_csym_invert (void) { return _Char('W'); }

			static _Char escape_any(void) { return _Char('C'); }

		public:
			static _Char cntrl_nul(void) { return m_cntrl_nul; }
			static _Char cntrl_soh(void) { return m_cntrl_soh; }
			static _Char cntrl_stx(void) { return m_cntrl_stx; }
			static _Char cntrl_etx(void) { return m_cntrl_etx; }
			static _Char cntrl_eot(void) { return m_cntrl_eot; }
			static _Char cntrl_enq(void) { return m_cntrl_enq; }
			static _Char cntrl_ack(void) { return m_cntrl_ack; }
			static _Char cntrl_bel(void) { return m_cntrl_bel; }
			static _Char cntrl_bs (void) { return m_cntrl_bs ; }
			static _Char cntrl_ht (void) { return m_cntrl_ht ; }
			static _Char cntrl_lf (void) { return m_cntrl_lf ; }
			static _Char cntrl_vt (void) { return m_cntrl_vt ; }
			static _Char cntrl_ff (void) { return m_cntrl_ff ; }
			static _Char cntrl_cr (void) { return m_cntrl_cr ; }
			static _Char cntrl_so (void) { return m_cntrl_so ; }
			static _Char cntrl_si (void) { return m_cntrl_si ; }
			static _Char cntrl_dle(void) { return m_cntrl_dle; }
			static _Char cntrl_dc1(void) { return m_cntrl_dc1; }
			static _Char cntrl_dc2(void) { return m_cntrl_dc2; }
			static _Char cntrl_dc3(void) { return m_cntrl_dc3; }
			static _Char cntrl_dc4(void) { return m_cntrl_dc4; }
			static _Char cntrl_nak(void) { return m_cntrl_nak; }
			static _Char cntrl_syn(void) { return m_cntrl_syn; }
			static _Char cntrl_etb(void) { return m_cntrl_etb; }
			static _Char cntrl_can(void) { return m_cntrl_can; }
			static _Char cntrl_em (void) { return m_cntrl_em ; }
			static _Char cntrl_sub(void) { return m_cntrl_sub; }
			static _Char cntrl_esc(void) { return m_cntrl_esc; }
			static _Char cntrl_fs (void) { return m_cntrl_fs ; }
			static _Char cntrl_gs (void) { return m_cntrl_gs ; }
			static _Char cntrl_rs (void) { return m_cntrl_rs ; }
			static _Char cntrl_us (void) { return m_cntrl_us ; }
			static _Char cntrl_del(void) { return m_cntrl_del; }
			static _Char char_underscore(void) { return m_char_underscore; }

		public:
			static const _Char *class_upper             (void) { return                 m_class_upper  ; }
			static size_t       class_upper_count       (void) { return mercury_countof(m_class_upper ); }
			static const _Char *class_lower             (void) { return                 m_class_lower  ; }
			static size_t       class_lower_count       (void) { return mercury_countof(m_class_lower ); }
			static const _Char *class_digit             (void) { return                 m_class_digit  ; }
			static size_t       class_digit_count       (void) { return mercury_countof(m_class_digit ); }
			static const _Char *class_xdigit            (void) { return                 m_class_xdigit ; }
			static size_t       class_xdigit_count      (void) { return mercury_countof(m_class_xdigit); }
			static const _Char *class_punct             (void) { return                 m_class_punct  ; }
			static size_t       class_punct_count       (void) { return mercury_countof(m_class_punct) ; }
			static const _Char *class_blank             (void) { return                 m_class_blank  ; }
			static size_t       class_blank_count       (void) { return mercury_countof(m_class_blank) ; }
			static const _Char *class_space             (void) { return                 m_class_space  ; }
			static size_t       class_space_count       (void) { return mercury_countof(m_class_space) ; }
			static const _Char *class_cntrl             (void) { return                 m_class_cntrl  ; }
			static size_t       class_cntrl_count       (void) { return mercury_countof(m_class_cntrl) ; }
			static const _Char *class_graph_invert      (void) { return                 m_class_graph_invert ; }
			static size_t       class_graph_invert_count(void) { return mercury_countof(m_class_graph_invert); }
			static const _Char *class_print_invert      (void) { return                 m_class_print_invert ; }
			static size_t       class_print_invert_count(void) { return mercury_countof(m_class_print_invert); }

		public:
			static int number(const _Char &ch)
			{
				const _Char number_array[] =
				{
					_Char('0'), _Char('1'), _Char('2'), _Char('3'), _Char('4'),
					_Char('5'), _Char('6'), _Char('7'), _Char('8'), _Char('9'),
				};
				for(size_t i = 0; i < mercury_countof(number_array); i++)
				{
					if(ch == number_array[i])
					{
						return static_cast<int>(i);
					}
				}
				return -1;
			}
			template<typename _InputStreamIterator, typename _OutputStreamIterator>
			static int number(const _InputStreamIterator &in, _OutputStreamIterator &out)
			{
				bool done  = false;
				int  value = 0;

				_InputStreamIterator p = in;
				for(; !p.eos(); p++)
				{
					const int now = number(*p);
					if(now == -1) { break; }
					value = value * 10 + now;
					done = true;
				}
				if(!done) { return -1; }

				out = p;
				return value;
			}

		public:
			static _Char next(const _Char &ch) { return ch + 1; }
			template<typename _InputStreamIterator, typename _OutputStreamIterator>
			static CHARCLASS get_charclass(const _InputStreamIterator &in, _OutputStreamIterator &out)
			{
				typedef struct tagCLASSINFO
				{
					CHARCLASS type;
					_Char     name[16];
				} CLASSINFO;
				const CLASSINFO info[] =
				{
					{CHARCLASS_UPPER , {_Char('['), _Char(':'), _Char('u'), _Char('p'), _Char('p'), _Char('e'), _Char('r'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_LOWER , {_Char('['), _Char(':'), _Char('l'), _Char('o'), _Char('w'), _Char('e'), _Char('r'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_ALPHA , {_Char('['), _Char(':'), _Char('a'), _Char('l'), _Char('p'), _Char('h'), _Char('a'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_DIGIT , {_Char('['), _Char(':'), _Char('d'), _Char('i'), _Char('g'), _Char('i'), _Char('t'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_ALNUM , {_Char('['), _Char(':'), _Char('a'), _Char('l'), _Char('n'), _Char('u'), _Char('m'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_XDIGIT, {_Char('['), _Char(':'), _Char('x'), _Char('d'), _Char('i'), _Char('g'), _Char('i'), _Char('t'), _Char(':'), _Char(']'), cntrl_nul()}},
					{CHARCLASS_PUNCT , {_Char('['), _Char(':'), _Char('p'), _Char('u'), _Char('n'), _Char('c'), _Char('t'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_BLANK , {_Char('['), _Char(':'), _Char('b'), _Char('l'), _Char('a'), _Char('n'), _Char('k'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_SPACE , {_Char('['), _Char(':'), _Char('s'), _Char('p'), _Char('a'), _Char('c'), _Char('e'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_CNTRL , {_Char('['), _Char(':'), _Char('c'), _Char('n'), _Char('t'), _Char('r'), _Char('l'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_GRAPH , {_Char('['), _Char(':'), _Char('g'), _Char('r'), _Char('a'), _Char('p'), _Char('h'), _Char(':'), _Char(']'),             cntrl_nul()}},
					{CHARCLASS_PRINT , {_Char('['), _Char(':'), _Char('p'), _Char('r'), _Char('i'), _Char('n'), _Char('t'), _Char(':'), _Char(']'),             cntrl_nul()}},
				};

				for(size_t i = 0; i < mercury_countof(info); i++)
				{
					_InputStreamIterator p = in;

					const _Char *classname = info[i].name;
					while(*classname != cntrl_nul())
					{
						if(p.eos()) { goto next; }
						if(*p++ != *classname++) { goto next; }
					}
					out = p;
					return info[i].type;
next:;
				}
				return CHARCLASS_NONE;
			}

		private:
			static const _Char m_cntrl_nul;
			static const _Char m_cntrl_soh;
			static const _Char m_cntrl_stx;
			static const _Char m_cntrl_etx;
			static const _Char m_cntrl_eot;
			static const _Char m_cntrl_enq;
			static const _Char m_cntrl_ack;
			static const _Char m_cntrl_bel;
			static const _Char m_cntrl_bs ;
			static const _Char m_cntrl_ht ;
			static const _Char m_cntrl_lf ;
			static const _Char m_cntrl_vt ;
			static const _Char m_cntrl_ff ;
			static const _Char m_cntrl_cr ;
			static const _Char m_cntrl_so ;
			static const _Char m_cntrl_si ;
			static const _Char m_cntrl_dle;
			static const _Char m_cntrl_dc1;
			static const _Char m_cntrl_dc2;
			static const _Char m_cntrl_dc3;
			static const _Char m_cntrl_dc4;
			static const _Char m_cntrl_nak;
			static const _Char m_cntrl_syn;
			static const _Char m_cntrl_etb;
			static const _Char m_cntrl_can;
			static const _Char m_cntrl_em ;
			static const _Char m_cntrl_sub;
			static const _Char m_cntrl_esc;
			static const _Char m_cntrl_fs ;
			static const _Char m_cntrl_gs ;
			static const _Char m_cntrl_rs ;
			static const _Char m_cntrl_us ;
			static const _Char m_cntrl_del;

			static const _Char m_char_underscore;

		private:
			static const _Char m_class_upper       [26];
			static const _Char m_class_lower       [26];
			static const _Char m_class_digit       [10];
			static const _Char m_class_xdigit      [22];
			static const _Char m_class_punct       [ 6];
			static const _Char m_class_blank       [ 2];
			static const _Char m_class_space       [ 6];
			static const _Char m_class_cntrl       [33];
			static const _Char m_class_graph_invert[ 6];
			static const _Char m_class_print_invert[ 5];
		};
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_nul = _Char(0x00);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_soh = _Char(0x01);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_stx = _Char(0x02);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_etx = _Char(0x03);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_eot = _Char(0x04);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_enq = _Char(0x05);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_ack = _Char(0x06);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_bel = _Char(0x07);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_bs  = _Char(0x08);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_ht  = _Char(0x09);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_lf  = _Char(0x0a);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_vt  = _Char(0x0b);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_ff  = _Char(0x0c);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_cr  = _Char(0x0d);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_so  = _Char(0x0e);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_si  = _Char(0x0f);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_dle = _Char(0x10);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_dc1 = _Char(0x11);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_dc2 = _Char(0x12);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_dc3 = _Char(0x13);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_dc4 = _Char(0x14);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_nak = _Char(0x15);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_syn = _Char(0x16);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_etb = _Char(0x17);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_can = _Char(0x18);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_em  = _Char(0x19);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_sub = _Char(0x1a);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_esc = _Char(0x1b);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_fs  = _Char(0x1c);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_gs  = _Char(0x1d);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_rs  = _Char(0x1e);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_us  = _Char(0x1f);
		template<typename _Char> const _Char regex_traits<_Char>::m_cntrl_del = _Char(0x7f);
		template<typename _Char> const _Char regex_traits<_Char>::m_char_underscore = _Char('_');

		template<typename _Char> const _Char regex_traits<_Char>::m_class_upper       [] = {_Char('A'), _Char('B'), _Char('C'), _Char('D'), _Char('E'), _Char('F'), _Char('G'), _Char('H'), _Char('I'), _Char('J'), _Char('K'), _Char('L'), _Char('M'), _Char('N'), _Char('O'), _Char('P'), _Char('Q'), _Char('R'), _Char('S'), _Char('T'), _Char('U'), _Char('V'), _Char('W'), _Char('X'), _Char('Y'), _Char('Z')};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_lower       [] = {_Char('a'), _Char('b'), _Char('c'), _Char('d'), _Char('e'), _Char('f'), _Char('g'), _Char('h'), _Char('i'), _Char('j'), _Char('k'), _Char('l'), _Char('m'), _Char('n'), _Char('o'), _Char('p'), _Char('q'), _Char('r'), _Char('s'), _Char('t'), _Char('u'), _Char('v'), _Char('w'), _Char('x'), _Char('y'), _Char('z')};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_digit       [] = {_Char('0'), _Char('1'), _Char('2'), _Char('3'), _Char('4'), _Char('5'), _Char('6'), _Char('7'), _Char('8'), _Char('9')};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_xdigit      [] = {_Char('0'), _Char('1'), _Char('2'), _Char('3'), _Char('4'), _Char('5'), _Char('6'), _Char('7'), _Char('8'), _Char('9'), _Char('A'), _Char('B'), _Char('C'), _Char('D'), _Char('E'), _Char('F'), _Char('a'), _Char('b'), _Char('c'), _Char('d'), _Char('e'), _Char('f')};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_punct       [] = {_Char('.'), _Char(','), _Char('!'), _Char('?'), _Char(':'), _Char(';')};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_blank       [] = {_Char(' ') , m_cntrl_ht};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_space       [] = {_Char(' ') , m_cntrl_ht, m_cntrl_lf, m_cntrl_vt, m_cntrl_ff, m_cntrl_cr};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_cntrl       [] = {m_cntrl_nul, m_cntrl_soh, m_cntrl_stx, m_cntrl_etx, m_cntrl_eot, m_cntrl_enq, m_cntrl_ack, m_cntrl_bel, m_cntrl_bs, m_cntrl_ht, m_cntrl_lf, m_cntrl_vt, m_cntrl_ff, m_cntrl_cr, m_cntrl_so, m_cntrl_si, m_cntrl_dle, m_cntrl_dc1, m_cntrl_dc2, m_cntrl_dc3, m_cntrl_dc4, m_cntrl_nak, m_cntrl_syn, m_cntrl_etb, m_cntrl_can, m_cntrl_em, m_cntrl_sub, m_cntrl_esc, m_cntrl_fs, m_cntrl_gs, m_cntrl_rs, m_cntrl_us, m_cntrl_del};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_graph_invert[] = {_Char(' ') , m_cntrl_ht, m_cntrl_lf, m_cntrl_vt, m_cntrl_ff, m_cntrl_cr};
		template<typename _Char> const _Char regex_traits<_Char>::m_class_print_invert[] = {             m_cntrl_ht, m_cntrl_lf, m_cntrl_vt, m_cntrl_ff, m_cntrl_cr};
	}
}

#endif
