#ifndef __MERCURY_REGEX__
#define __MERCURY_REGEX__

#include <memory>
#include "stream_iterator.h"
#include "_regex/nfa.h"
#include "_regex/parser.h"
#include "_regex/traits.h"


namespace mercury
{
	template<typename _Char, typename _Traits = _regex::regex_traits<_Char>, size_t MAX_NFA_STATES = 32767>
	class basic_regex
	{
	public:
		typedef size_t  size_type;
		typedef _Char   value_type;
		typedef _Traits traits_type;

		typedef       value_type *      pointer;
		typedef const value_type *const_pointer;
		template<typename _InputOutputIterator /* = const_pointer */>
		struct MATCH_INFO
		{
			size_t offset;
			size_t length;
			_InputOutputIterator begin;
			_InputOutputIterator end;
			template<typename _Container>
			_Container get(void) const { return _Container(begin, end); }
		};

	public:
		basic_regex(void)
		{
			_init();
		}
		basic_regex(const_pointer pattern_string)
		{
			_init();
			compile(pattern_string);
		}
		basic_regex(const_pointer pattern_string, const size_t pattern_length)
		{
			_init();
			compile(pattern_string, pattern_length);
		}
		template<typename _Container>
		basic_regex(const _Container &pattern)
		{
			_init();
			compile(pattern);
		}
		template<typename _InputIterator>
		basic_regex(const _InputIterator &pattern_begin, const _InputIterator &pattern_end)
		{
			_init();
			compile(pattern_begin, pattern_end);
		}
		basic_regex &operator=(const_pointer pattern_string)
		{
			compile(pattern_string);
			return *this;
		}
		template<typename _Container>
		basic_regex &operator=(const _Container &pattern)
		{
			compile(pattern);
			return *this;
		}
		void compile(const_pointer pattern_string)
		{
			stream_iterator_nul<_Char, const_pointer> iterator(pattern_string, _Traits::cntrl_nul());
			_compile(iterator);
		}
		void compile(const_pointer pattern_string, const size_t pattern_length)
		{
			stream_iterator_end<_Char, const_pointer> iterator(&pattern_string[0], &pattern_string[pattern_length]);
			_compile(iterator);
		}
		template<typename _Container>
		void compile(const _Container &pattern)
		{
			stream_iterator_end<_Char, typename _Container::const_iterator> iterator(pattern.begin(), pattern.end());
			_compile(iterator);
		}
		template<typename _InputIterator>
		void compile(const _InputIterator &pattern_begin, const _InputIterator &pattern_end)
		{
			stream_iterator_end<_Char, _InputIterator> iterator(pattern_begin, pattern_end);
			_compile(iterator);
		}
		bool is_head(void) const { return m_head; }
		bool is_tail(void) const { return m_tail; }
		bool match(const_pointer text_pointer, MATCH_INFO<const_pointer> &match_info) const
		{
			stream_iterator_nul<_Char, const_pointer> iterator(text_pointer, _Traits::cntrl_nul());
			return _match(iterator, match_info);
		}
		bool match(const_pointer text_pointer, const size_t text_length, MATCH_INFO<const_pointer> &match_info) const
		{
			stream_iterator_end<_Char, const_pointer> iterator(&text_pointer[0], &text_pointer[text_length]);
			return _match(iterator, match_info);
		}
		template<typename _Container, typename _MATCH_INFO>
		bool match(const _Container &text, _MATCH_INFO &match_info) const
		{
			stream_iterator_end<_Char, typename _Container::const_iterator> iterator(text.begin(), text.end());
			return _match(iterator, match_info);
		}
		template<typename _InputIterator>
		bool match(const _InputIterator &text_begin, const _InputIterator &text_end, MATCH_INFO<_InputIterator> &match_info) const
		{
			stream_iterator_end<_Char, _InputIterator> iterator(text_begin, text_end);
			return _match(iterator, match_info);
		}

	private:
		bool m_head;
		bool m_tail;
		_regex::nfa<_Char> m_nfa;
		void _init(void)
		{
			m_head = false;
			m_tail = false;
		}
		template<typename _InputStreamIterator>
		void _compile(const _InputStreamIterator &pattern)
		{
			_regex::parser<_Char, _InputStreamIterator, _Traits> syntax(pattern);
			std::auto_ptr<_regex::parse_tree<_Char> > tree(syntax.parse());
			_regex::state_generator generator(MAX_NFA_STATES);
			m_nfa = tree->build_nfa(generator);

			m_head = syntax.is_head();
			m_tail = syntax.is_tail();
		}
		template<typename _InputStreamIterator, typename _MATCH_INFO>
		bool _match(const _InputStreamIterator &text, _MATCH_INFO &match_info) const
		{
			_MATCH_INFO info;
			info.offset = 0;
			_InputStreamIterator p = text;
			do
			{
				if(m_nfa.match_forward(p, info.end, info.length))
				{
					if(is_tail() && !p.eos(info.end))
					{
						continue;
					}
					info.begin = p.iterator();
					match_info = info;
					return true;
				}
				if(is_head())
				{
					break;
				}
			}
			while(info.offset++, !(p++).eos());
			return false;
		}
	};
	typedef basic_regex< char  >  regex;
	typedef basic_regex<wchar_t> wregex;
}

#endif
