/*
 * Copyright (c) 2003 Shinji Kashihara. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies
 * this distribution, and is available at cpl-v10.html.
 */
package mergedoc.core;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.zip.Deflater;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import javax.swing.event.ChangeListener;
import javax.xml.parsers.SAXParser;

import mergedoc.MergeDocException;
import mergedoc.xml.ConfigManager;
import mergedoc.xml.ReplaceEntry;
import mergedoc.xml.ReplaceHandler;

import org.xml.sax.SAXException;

/**
 * マージマネージャです。
 * @author Shinji Kashihara
 */
public class MergeManager {

    /** マージ設定 */
    private Preference pref;

    /** 処理状態 */
    private WorkingState workingState = new WorkingState();

    /**
     * コンストラクタです。
     */
    public MergeManager() throws MergeDocException {
    }

    /**
     * マージ設定をセットします。
     * @param pref マージ設定
     */
    public void setPreference(Preference pref) {
        this.pref = pref;
        workingState.initialize();
    }

    /**
     * マージ可能な状態か検証します。
     * @throws MergeDocException マージ不可能な状態の場合
     */
    public void validate() throws MergeDocException {

        // API ドキュメントディレクトリのチェック
        File docDir = pref.getDocDirectory();
        if (docDir != null && docDir.getPath().length() > 0) {
            File rootFile = new File(docDir, "allclasses-frame.html");
            if (!rootFile.exists()) {
                throw new MergeDocException(
                "正しい API ドキュメントディレクトリを指定してください。\n" +
                "指定するディレクトリには allclasses-frame.html ファイルが\n" +
                "含まれている必要があります。");
            }
        }

        // 入力ソースアーカイブファイルのチェック
        File inFile = pref.getInputArchive();
        if (inFile == null || inFile.getPath().equals("")) {
            throw new MergeDocException(
            "入力ソースアーカイブファイルを指定してください。");
        }
        if (entrySize() == 0) {
            throw new MergeDocException(
            "正しい入力ソースアーカイブファイルを指定してください。");
        }

        // 出力ソースアーカイブファイルのチェック
        File outFile = pref.getOutputArchive();
        if (outFile == null || outFile.getPath().equals("")) {
            throw new MergeDocException(
            "出力ソースアーカイブファイルを指定してください。");
        }
        if (outFile.equals(inFile)) {
            throw new MergeDocException(
            "入力、出力ソースアーカイブファイルに同じファイルが指定されています。\n" +
            "正しい出力ソースアーカイブファイルを指定してください。");
        }
        if (pref.getOutputArchive().exists()) {
            if (!outFile.canWrite()) {
                throw new MergeDocException(
                "指定された出力ソースアーカイブファイルは書き込み不可です。\n" +
                "正しい出力ソースアーカイブファイルを指定してください。");
            }
        } else {
            try {
                outFile.createNewFile();
                outFile.delete();
            } catch (IOException e) {
                throw new MergeDocException(
                "正しい出力ソースアーカイブファイルを指定してください。", e);
            }
        }
    }

    /**
     * 処理を実行します。<br>
     * @throws MergeDocException コンフィグ情報の取得に失敗した場合
     * @throws SAXException SAX パース例外が発生した場合
     * @throws IOException 入出力例外が発生した場合
     */
    public void execute() throws MergeDocException, SAXException, IOException {
        
        if (workingState.isCanceled()) return;
        ZipInputStream in = null;
        ZipOutputStream out = null;

        try {
            in = new ZipInputStream(new BufferedInputStream(
                new FileInputStream(pref.getInputArchive())));

            out = new ZipOutputStream(new BufferedOutputStream(
                new FileOutputStream(pref.getOutputArchive())));

            out.setLevel(Deflater.BEST_SPEED);

            long start = System.currentTimeMillis();
            merge(in, out);
            long end = System.currentTimeMillis();
            workingState.setWorkTime((end - start) / 1000);

        } finally {

            if (in != null) in.close();
            if (out != null && workingState.getChangedCount() > 0) {
                out.close();
            } 
        }
    }

    /**
     * ZIP 入力ストリーム内の Java ソースファイルを順次読み込み, Javadoc
     * から取得したコメントとマージして出力ソースに書き込みます。
     * @param in ZIP 入力ストリーム 
     * @param out ZIP 出力ストリーム
     * @throws MergeDocException コンフィグ情報の取得に失敗した場合
     * @throws SAXException SAX パース例外が発生した場合
     * @throws IOException 入出力例外が発生した場合
     */
    private void merge(ZipInputStream in, ZipOutputStream out)
        throws MergeDocException, SAXException, IOException
    {
        String target = System.getProperty("target.class", "");
        target = target.replaceAll("\\.", "/");

        for (ZipEntry inEntry = null; (inEntry = in.getNextEntry()) != null;) {

            if (workingState.isCanceled()) return;
            byte[] outBuf = null;

            // 入力 ZipEntry から Java ソースパス取得
            String path = inEntry.getName().replaceFirst("^src/", "");
            if (!path.startsWith(target)) continue;

            if (path.endsWith(Java.FILE_EXT)) {

                // Java インスタンス生成
                String className = path.replaceFirst("\\.java$", "").replaceAll("/", "\\.");
                Java java = new Java(className, in, pref.getInputEncoding());
                in.closeEntry();

                // マージ
                String result = mergeEntry(java);
                result = StringUtils.expand8Tab(result);
                result = doFilter(java.getClassName(), result);
                outBuf = result.getBytes(pref.getOutputEncoding());
                workingState.changeWorkingText(java.getClassName());

            } else {
                
                // マージしないでスルー
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                byte[] buf = new byte[1024];
                for (int size = 0; (size = in.read(buf)) > 0;) {
                    baos.write(buf, 0, size);
                }
                baos.close();
                outBuf = baos.toByteArray();
            }

            // 出力 ZipEntry に書き込み
            ZipEntry outEntry = new ZipEntry(inEntry.getName());
            out.putNextEntry(outEntry);
            out.write(outBuf);
            out.closeEntry();

        }
    }

    /**
     * 指定した Java ソースと Javadoc コメントをマージし, 新しい Java
     * ソースを取得します。Javadoc API HTML が見つからない場合はそのまま
     * Java ソースを返します。
     * @param java Java ソース
     * @return マージ後の Java ソース
     * @throws IOException 入出力例外が発生した場合
     */
    private String mergeEntry(Java java) throws IOException {

        APIDocument apiDoc = new APIDocument(
            pref.getDocDirectory(), java.getClassName(), pref.getDocEncoding());
        if (apiDoc.isEmpty()) {
            return java.getText();
        }

        JavaBuffer javaBuf = new JavaBuffer(java);
        while (javaBuf.nextComment()) {

            Signature sig = javaBuf.getSignature();
            Comment com = apiDoc.getComment(sig);
            javaBuf.setLocalizedComment(com);
        }

        String result = javaBuf.finishToString();
        return result;
    }

    /**
     * XML に定義された置換エントリを元にソース置換処理を行います。
     * @param className クラス名
     * @param source Java ソース文字列
     * @return 処理後のソース文字列
     * @throws MergeDocException コンフィグ情報の取得に失敗した場合
     * @throws SAXException SAX パース例外が発生した場合
     * @throws IOException 入出力例外が発生した場合
     */
    private String doFilter(String className, String source)
        throws MergeDocException, SAXException, IOException
    {
        // クラス別置換定義の処理
        String path = className.replaceAll("\\.", "/") + ".xml";
        ConfigManager config = ConfigManager.getInstance();
        File entryXML = config.getFile(path);
        if (entryXML.exists()) {
            SAXParser saxParser = config.getSAXPerser();
            ReplaceHandler handler = new ReplaceHandler(source);
            saxParser.parse(entryXML, handler);
            source = handler.getResult();
        }

        // グローバル置換定義の処理
        ReplaceEntry[] entries = pref.getGlobalEntries();
        for (int i = 0; i < entries.length; i++) {
            ReplaceEntry entry = entries[i];
            source = entry.replace(source);
        }

        return source;
    }

    /**
     * 処理対象となるエントリ数を取得します。
     * @return 処理対象となるエントリ数
     */
    public int entrySize() {
        int size = 0;
        try {
            ZipFile zip = new ZipFile(pref.getInputArchive());
            for (Enumeration it = zip.entries(); it.hasMoreElements();) {
                ZipEntry entry = (ZipEntry)it.nextElement();
                if (entry.getName().endsWith(Java.FILE_EXT)) {
                    size++;
                }
            }
        } catch (IOException e) {
        }
        return size;
    }

    /**
     * 進捗監視用のリスナをセットします。
     * @param changeListener 進捗監視用のリスナ
     */
    public void setChangeListener(ChangeListener changeListener) {
        workingState.setChangeListener(changeListener);
    }

    /**
     * 処理状態を取得します。
     * @return 処理状態
     */
    public WorkingState getWorkingState() {
        return workingState;
    }
}
